#include "CipeTapIO.h"
#include "hexdump.h"

//========================================================================================
//
//========================================================================================
CipeTapIO::CipeTapIO (CipeAdapter &p_Adapter) : m_Adapter (p_Adapter), m_TaskObject (0)
   {
    unsigned long l_Len;

    m_Handle = CreateFile
       (
        m_Adapter.TapName().c_str(),
        GENERIC_READ | GENERIC_WRITE,
        FILE_SHARE_READ,
        0,
        OPEN_EXISTING,
        FILE_ATTRIBUTE_SYSTEM | FILE_FLAG_OVERLAPPED,
        0
       );

    if (m_Handle == INVALID_HANDLE_VALUE)
       throw CipeTapIOException();
    else if (! DeviceIoControl (m_Handle, CIPE_IOCTL_GET_MAC, m_MAC, sizeof (m_MAC), m_MAC, sizeof (m_MAC), &l_Len, 0))
       throw CipeTapIOException();
    else if (! DeviceIoControl (m_Handle, CIPE_IOCTL_GET_LASTMAC, m_NextMAC, sizeof (m_NextMAC), m_NextMAC, sizeof (m_NextMAC), &l_Len, 0))
       throw CipeTapIOException();
    else
       DbgPrint ("[%s] TAP started on [%s]\n", Name().c_str(), m_Adapter.TapName().c_str());
   }

CipeTapIO::~CipeTapIO()
   {
    CloseHandle (m_Handle);
   }

//========================================================================================
//
//========================================================================================
void CipeTapIO::RequestAsyncReceive()
   {
    if (! CheckForShutdownIndication())
       {
        for (m_Synchronous = 1; m_Synchronous;)
           {
            try
               {
                if (m_Synchronous = ReadFile (m_Handle, m_Buffer.buffer, DescriptorSize(), &m_Buffer.length, &m_Overlapped))
                   {
                    CompleteAsyncReceive();
                   }
               }
            catch (...)
               {
                DbgPrint ("[%s] Exception in TAP CipeTapIO::RequestAsyncReceive() for [%s]\n", Name().c_str(), m_Adapter.TapName().c_str());
               }
           }

        if ((++m_Tx % STATISTICS_UPDATE_FREQUENCY) == 0 && m_Tx > 0)
           {
            UpdateAdapterStatistics();
           }
       }
   }

void CipeTapIO::CompleteAsyncReceive()
   {
    unsigned long l_Status = 0, l_SocketCount = m_SocketList.size();

    if (! m_Synchronous)
       {
        m_Buffer.length = m_Overlapped.InternalHigh;
        l_Status = m_Overlapped.Internal;
       }

    for (unsigned long l_Index = 0, EventReset(); l_Status == 0 && m_Buffer.length && l_Index < l_SocketCount; ++l_Index)
       {
        CipeGenericIO &l_SocketIO = *(m_SocketList [l_Index]);

        if (IsArpRequest (m_Buffer)) // Synthesize an ARP reply on behalf of the peer
           {if (l_SocketIO.Address() == m_Buffer.arp.DestinationIP) ArpReply (l_SocketIO);}
        else if (MatchingMAC (m_Buffer.mac.destination, l_SocketIO.MAC())) // If direct address, send to peer
           l_SocketIO.Send (m_Buffer);
        else if (MatchingMAC (m_Buffer.mac.destination, BroadcastMAC())) // If MAC broadcast, send to peer
           l_SocketIO.Send (m_Buffer);
       }
   }

void CipeTapIO::Send (CipePacketDescriptor &p_Buffer)
   {
    unsigned long l_BytesWritten;

    if (p_Buffer.length >= sizeof (p_Buffer.mac) && p_Buffer.length <= DescriptorSize())
       {
        memcpy (p_Buffer.mac.destination, MAC(), sizeof (MACADDR)); // Ensures that we deliver to the local MAC address

        if (WriteFile (m_Handle, p_Buffer.buffer, p_Buffer.length, &l_BytesWritten, &m_OverlappedEx))
           ;
        else if (WSAGetLastError() == WSA_IO_PENDING)
           GetOverlappedResult (m_Handle, &m_OverlappedEx, &l_BytesWritten, TRUE);
        else
           DbgPrint ("Adapter [%s] failed on write attempt\n", Name().c_str());

        ResetEvent (m_OverlappedEx.hEvent);
       }
   }

//========================================================================================
//
//========================================================================================
void CipeTapIO::Enjoin (CipeGenericIO &p_Object)
   {
    if (p_Object.Name() == "JOB HANDLER")
       m_TaskObject = &p_Object;
    else if (typeid (CipeTapIO) != typeid (p_Object) && Name() == p_Object.AssociateName())
       m_SocketList.push_back (&p_Object);
   }

//===================================================================================================
//                                 Arp Table Manipulation
//===================================================================================================
BOOL CipeTapIO::IsArpRequest (CipePacketDescriptor &p_Buffer)
   {
    return
       (
        p_Buffer.length > 21 &&
        p_Buffer.mac.type == htons (0x0806) &&
        p_Buffer.arp.ProtocolAddressType == htons (0x0800) &&
        p_Buffer.arp.Operation == htons (0x0001)
       );
   }

void CipeTapIO::ArpReply (CipeGenericIO &p_Object)
   {
    CipePacketDescriptor l_ArpReply, &l_ArpRequest = m_Buffer;

    DbgPrint ("[%s] Generating ARP entry for [%s:%s]\n", Name().c_str(), p_Object.Name().c_str(), inet_str (p_Object.Address()).c_str());

    //----------------------------------------------
    // Initialize ARP reply fields
    //----------------------------------------------
    l_ArpReply.mac.type                = htons (0x0806);
    l_ArpReply.arp.MacAddressType      = htons (0x0001);
    l_ArpReply.arp.ProtocolAddressType = htons (0x0800);
    l_ArpReply.arp.MacAddressSize      = 0x06;
    l_ArpReply.arp.ProtocolAddressSize = 0x04;
    l_ArpReply.arp.Operation           = htons (0x02);

    //----------------------------------------------------
    // Set destination MAC address to be local PTP address
    //----------------------------------------------------
    memcpy (l_ArpReply.mac.destination, this->MAC(), sizeof (MACADDR));
    memcpy (l_ArpReply.mac.source, p_Object.MAC(), sizeof (MACADDR));

    //----------------------------------------------
    // Copy Source MAC + IP from ARP request
    //----------------------------------------------
    memcpy (l_ArpReply.arp.SourceMAC, p_Object.MAC(), sizeof (MACADDR));
    l_ArpReply.arp.SourceIP = l_ArpRequest.arp.DestinationIP;

    //----------------------------------------------
    // Copy Destination MAC + IP from ARP request
    //----------------------------------------------
    memcpy (l_ArpReply.arp.DestinationMAC, this->MAC(), sizeof (MACADDR));

    l_ArpReply.arp.DestinationIP = l_ArpRequest.arp.SourceIP;
    l_ArpReply.length = sizeof (CipeMacInfo) + sizeof (CipeArpInfo);
    l_ArpReply.flags = 0;

    Send (l_ArpReply);
   }

//===================================================================================================
//                                       Statistics
//===================================================================================================
unsigned long CipeTapIO::TxErr()
   {
    unsigned long l_TxErr = 0;
    for (unsigned long l_Index = 0; l_Index < m_SocketList.size(); ++l_Index) l_TxErr += m_SocketList [l_Index]->TxErr();
    return l_TxErr;
   }

unsigned long CipeTapIO::RxErr()
   {
    unsigned long l_RxErr = 0;
    for (unsigned long l_Index = 0; l_Index < m_SocketList.size(); ++l_Index) l_RxErr += m_SocketList [l_Index]->RxErr();
    return l_RxErr;
   }

unsigned long CipeTapIO::Tx()
   {
    unsigned long l_Tx = 0;
    for (unsigned long l_Index = 0; l_Index < m_SocketList.size(); ++l_Index) l_Tx += m_SocketList [l_Index]->Tx();
    return l_Tx;
   }

unsigned long CipeTapIO::Rx()
   {
    unsigned long l_Rx = 0;
    for (unsigned long l_Index = 0; l_Index < m_SocketList.size(); ++l_Index) l_Rx += m_SocketList [l_Index]->Rx();
    return l_Rx;
   }

void CipeTapIO::UpdateAdapterStatistics()
   {
    unsigned long l_Length, l_Stats[] = {Tx(), Rx(), TxErr(), RxErr()};
    DeviceIoControl (m_Handle, CIPE_IOCTL_SET_STATISTICS, l_Stats, sizeof (unsigned long) * 4, 0, 0, &l_Length, 0);
   }

//===================================================================================================
//                                   End of Source
//===================================================================================================

