#include "CipeEngine.h"
#include "hexdump.h"

//========================================================================================
//
//========================================================================================
CipeEngine::CipeEngine() throw (CipeEngineException) : m_State (STATE_STOPPED)
   {
    //=================================================
    // Initialize Winsock Library
    //=================================================
    if (WSAStartup ((WORD) 257, &m_WsaData))
       {
        DbgPrint ("Couldn't startup Winsock\n");
        throw CipeEngineException();
       }

    Load();
   }

CipeEngine::~CipeEngine()
   {
    Purge();
    WSACleanup();
   }

//========================================================================================
//
//========================================================================================
void CipeEngine::Load() throw (CipeEngineException)
   {
    unsigned long l_Index, l_Size = 0;

    Purge();

    //=================================================
    // Initialize Task handler
    //=================================================
    push_back (new CipeTaskIO());

    //=================================================
    // Initialize TAP device handles
    //=================================================
    for (l_Index = 0; l_Index < m_PeerList.AdapterList().size(); ++l_Index)
       {
        push_back (new CipeTapIO (m_PeerList.AdapterList() [l_Index]));
       }

    //=================================================
    // Initialize socket handles
    //=================================================
    for (l_Index = 0; l_Index < m_PeerList.size(); ++l_Index) if (m_PeerList [l_Index].Enabled())
       {
        push_back (new CipeSocketIO (m_PeerList [l_Index]));
       }

    l_Size = size();

    //=================================================
    // Initialize Event handles for overlapped I/O
    //=================================================
    if (! (m_EventTable = new HANDLE [l_Size]))
       {
        throw CipeEngineException();
       }

    //=================================================
    // Allow each I/O to have relationships with others
    //=================================================
    for (l_Index = 0; l_Index < l_Size; ++l_Index)
       {
        for (int l_NestedIndex = 0; l_NestedIndex < l_Size; ++l_NestedIndex)
           {
            (*this) [l_Index].Enjoin ((*this) [l_NestedIndex]);
           }
       }
   }

void CipeEngine::Purge()
   {
    for (; size(); pop_back()) delete (back());
   }

//========================================================================================
//
//========================================================================================
void CipeEngine::EventLoop()
   {
    m_State = STATE_RUNNING;

    QueueUpAsyncRequests();

    while (TestForActiveHandles()) switch (int l_Index = Select())
       {
        case WAIT_TIMEOUT:
           {
            switch (m_State)
               {
                case STATE_PENDING:
                   {
                    m_State = STATE_FORCE;
                    break;
                   }

                case STATE_FORCE:
                   {
                    m_State = STATE_STOPPED;
                    Purge();
                    break;
                   }

                case STATE_RUNNING:
                   {
                    for (unsigned long l_Idx = size(); l_Idx; --l_Idx) try
                       {
                        (*this) [l_Idx - 1].TimeoutEvent();
                       }
                    catch (...)
                       {
                       }
                    break;
                   }
               }

            break;
           }

        default:
           {
            for (unsigned long l_Size = size(); l_Index < l_Size; ++l_Index)
               {
                try
                   {
                    HANDLE &l_Handle = (*this) [l_Index].EventHandle();

                    if (l_Handle && WaitForSingleObject (l_Handle, 0) == WAIT_OBJECT_0)
                       {
                        (*this) [l_Index].CompleteAsyncReceive();
                        (*this) [l_Index].RequestAsyncReceive();
                       }
                   }
                catch (...)
                   {
                   }
               }

            break;
           }
       }
   }

BOOL CipeEngine::TestForActiveHandles()
   {
    unsigned long l_Size = size();
    BOOL l_Found = FALSE;

    for (unsigned long l_Index = 0; l_Index < l_Size && ! l_Found; ++l_Index) try
       {
        if ((*this) [l_Index].EventHandle()) l_Found = TRUE;
       }
    catch (...)
       {
       }

    return (l_Found);
   }

void CipeEngine::QueueUpAsyncRequests()
   {
    for (unsigned long l_Index = size(); l_Index; --l_Index) (*this) [l_Index - 1].RequestAsyncReceive();
   }

int CipeEngine::Select()
   {
    int l_ItemCount = 0, l_Result = WAIT_TIMEOUT, l_Size = size();

    for (unsigned long l_Index = 0; l_Index < l_Size; ++l_Index)
       {
        try
           {
            if (m_EventTable [l_ItemCount] = (*this) [l_Index].EventHandle()) ++l_ItemCount;
           }
        catch (...)
           {
           }
       }

    if (l_ItemCount && (l_Result = WaitForMultipleObjects (l_ItemCount, m_EventTable, FALSE, DAEMON_SELECT_TIMEOUT)) != WAIT_TIMEOUT)
       {
        l_Result -= WAIT_OBJECT_0;
       }

    return l_Result;
   }

void CipeEngine::Shutdown()
   {
    if (m_State == STATE_RUNNING)
       {
        unsigned long l_Size = size();
        for (int l_Index = 0; l_Index < l_Size; ++l_Index) if ((*this) [l_Index].EventHandle()) (*this) [l_Index].Shutdown();
        m_State = STATE_PENDING;
       }
   }

CipeGenericIO &CipeEngine::operator[] (unsigned long p_Subscript) throw (CipeSubscriptException)
   {
    if (p_Subscript >= size())
       {
        throw CipeSubscriptException();
       }

    try
       {
        return ((CipeGenericIO &) *((vector <CipeGenericIO *> *) this)->operator[] (p_Subscript));
       }
    catch (...)
       {
       }

    throw CipeSubscriptException();
   }

//========================================================================================
//                                   End of Source
//========================================================================================
