/* armor.c - Armor/Text handling for data  
 *  Copyright (C) 2001, 2002 Timo Schulz
 *  Copyright (C) 1998, 1999, 2000, 2001 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * ChangeLog for basic BASE64 code (base64_encode, base64_decode):
 *
 * Original author: Eric S. Raymond (Fetchmail)
 * Heavily modified by Brendan Cully <brendan@kublai.com> (Mutt)
 * Modify the code for generic use by Timo Schulz <ts@winpt.org>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>

#include "opencdk.h"
#include "main.h"
#include "iobuf.h"

#define LF "\n"

#define CRCINIT 0xB704CE
#define CRC_UPDATE( crc, val, nbytes ) \
do { \
  int j; \
  for ( j=0; j<(nbytes) && (nbytes) > 0; j++ ) { \
      (crc) = ((crc) << 8) ^ crc_table[(((crc) >> 16) & 0xff) ^ (val[j])];  \
      (crc) &= 0x00ffffff; \
    } \
} while (0)

#define BAD -1
#define base64val(c) index64[(unsigned int)(c)]

#define LINE_LENGTH 66
#define LINE_LENGTH_RAW 48
#define MAX_LINELEN 19995

static u32 crc_table[] = {
  0x00000000, 0x00864CFB, 0x018AD50D, 0x010C99F6, 0x0393E6E1, 0x0315AA1A, 
  0x021933EC, 0x029F7F17, 0x07A18139, 0x0727CDC2, 0x062B5434, 0x06AD18CF, 
  0x043267D8, 0x04B42B23, 0x05B8B2D5, 0x053EFE2E, 0x0FC54E89, 0x0F430272, 
  0x0E4F9B84, 0x0EC9D77F, 0x0C56A868, 0x0CD0E493, 0x0DDC7D65, 0x0D5A319E, 
  0x0864CFB0, 0x08E2834B, 0x09EE1ABD, 0x09685646, 0x0BF72951, 0x0B7165AA, 
  0x0A7DFC5C, 0x0AFBB0A7, 0x1F0CD1E9, 0x1F8A9D12, 0x1E8604E4, 0x1E00481F, 
  0x1C9F3708, 0x1C197BF3, 0x1D15E205, 0x1D93AEFE, 0x18AD50D0, 0x182B1C2B, 
  0x192785DD, 0x19A1C926, 0x1B3EB631, 0x1BB8FACA, 0x1AB4633C, 0x1A322FC7, 
  0x10C99F60, 0x104FD39B, 0x11434A6D, 0x11C50696, 0x135A7981, 0x13DC357A, 
  0x12D0AC8C, 0x1256E077, 0x17681E59, 0x17EE52A2, 0x16E2CB54, 0x166487AF, 
  0x14FBF8B8, 0x147DB443, 0x15712DB5, 0x15F7614E, 0x3E19A3D2, 0x3E9FEF29, 
  0x3F9376DF, 0x3F153A24, 0x3D8A4533, 0x3D0C09C8, 0x3C00903E, 0x3C86DCC5, 
  0x39B822EB, 0x393E6E10, 0x3832F7E6, 0x38B4BB1D, 0x3A2BC40A, 0x3AAD88F1, 
  0x3BA11107, 0x3B275DFC, 0x31DCED5B, 0x315AA1A0, 0x30563856, 0x30D074AD, 
  0x324F0BBA, 0x32C94741, 0x33C5DEB7, 0x3343924C, 0x367D6C62, 0x36FB2099, 
  0x37F7B96F, 0x3771F594, 0x35EE8A83, 0x3568C678, 0x34645F8E, 0x34E21375, 
  0x2115723B, 0x21933EC0, 0x209FA736, 0x2019EBCD, 0x228694DA, 0x2200D821, 
  0x230C41D7, 0x238A0D2C, 0x26B4F302, 0x2632BFF9, 0x273E260F, 0x27B86AF4, 
  0x252715E3, 0x25A15918, 0x24ADC0EE, 0x242B8C15, 0x2ED03CB2, 0x2E567049, 
  0x2F5AE9BF, 0x2FDCA544, 0x2D43DA53, 0x2DC596A8, 0x2CC90F5E, 0x2C4F43A5, 
  0x2971BD8B, 0x29F7F170, 0x28FB6886, 0x287D247D, 0x2AE25B6A, 0x2A641791, 
  0x2B688E67, 0x2BEEC29C, 0x7C3347A4, 0x7CB50B5F, 0x7DB992A9, 0x7D3FDE52, 
  0x7FA0A145, 0x7F26EDBE, 0x7E2A7448, 0x7EAC38B3, 0x7B92C69D, 0x7B148A66, 
  0x7A181390, 0x7A9E5F6B, 0x7801207C, 0x78876C87, 0x798BF571, 0x790DB98A, 
  0x73F6092D, 0x737045D6, 0x727CDC20, 0x72FA90DB, 0x7065EFCC, 0x70E3A337, 
  0x71EF3AC1, 0x7169763A, 0x74578814, 0x74D1C4EF, 0x75DD5D19, 0x755B11E2, 
  0x77C46EF5, 0x7742220E, 0x764EBBF8, 0x76C8F703, 0x633F964D, 0x63B9DAB6, 
  0x62B54340, 0x62330FBB, 0x60AC70AC, 0x602A3C57, 0x6126A5A1, 0x61A0E95A, 
  0x649E1774, 0x64185B8F, 0x6514C279, 0x65928E82, 0x670DF195, 0x678BBD6E, 
  0x66872498, 0x66016863, 0x6CFAD8C4, 0x6C7C943F, 0x6D700DC9, 0x6DF64132, 
  0x6F693E25, 0x6FEF72DE, 0x6EE3EB28, 0x6E65A7D3, 0x6B5B59FD, 0x6BDD1506, 
  0x6AD18CF0, 0x6A57C00B, 0x68C8BF1C, 0x684EF3E7, 0x69426A11, 0x69C426EA, 
  0x422AE476, 0x42ACA88D, 0x43A0317B, 0x43267D80, 0x41B90297, 0x413F4E6C, 
  0x4033D79A, 0x40B59B61, 0x458B654F, 0x450D29B4, 0x4401B042, 0x4487FCB9, 
  0x461883AE, 0x469ECF55, 0x479256A3, 0x47141A58, 0x4DEFAAFF, 0x4D69E604, 
  0x4C657FF2, 0x4CE33309, 0x4E7C4C1E, 0x4EFA00E5, 0x4FF69913, 0x4F70D5E8, 
  0x4A4E2BC6, 0x4AC8673D, 0x4BC4FECB, 0x4B42B230, 0x49DDCD27, 0x495B81DC, 
  0x4857182A, 0x48D154D1, 0x5D26359F, 0x5DA07964, 0x5CACE092, 0x5C2AAC69, 
  0x5EB5D37E, 0x5E339F85, 0x5F3F0673, 0x5FB94A88, 0x5A87B4A6, 0x5A01F85D, 
  0x5B0D61AB, 0x5B8B2D50, 0x59145247, 0x59921EBC, 0x589E874A, 0x5818CBB1, 
  0x52E37B16, 0x526537ED, 0x5369AE1B, 0x53EFE2E0, 0x51709DF7, 0x51F6D10C, 
  0x50FA48FA, 0x507C0401, 0x5542FA2F, 0x55C4B6D4, 0x54C82F22, 0x544E63D9, 
  0x56D11CCE, 0x56575035, 0x575BC9C3, 0x57DD8538, 
};

static const char *armor_begin[] = {
    "BEGIN PGP MESSAGE",
    "BEGIN PGP PUBLIC KEY BLOCK",
    "BEGIN PGP PRIVATE KEY BLOCK",
    "BEGIN PGP SIGNATURE",
    NULL
};

static const char *armor_end[] = {
    "END PGP MESSAGE",
    "END PGP PUBLIC KEY BLOCK",
    "END PGP PRIVATE KEY BLOCK",
    "END PGP SIGNATURE",
    NULL
};

static const char *valid_headers[] = {
    "Comment", 
    "Version", 
    "MessageID", 
    "Hash",
    "Charset",
    NULL
};

static char b64chars[64] = {
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
    'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd',
    'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's',
    't', 'u', 'v', 'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7',
    '8', '9', '+', '/'
};

static int index64[128] = {
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,62, -1,-1,-1,63,
    52,53,54,55, 56,57,58,59, 60,61,-1,-1, -1,-1,-1,-1,
    -1, 0, 1, 2,  3, 4, 5, 6,  7, 8, 9,10, 11,12,13,14,
    15,16,17,18, 19,20,21,22, 23,24,25,-1, -1,-1,-1,-1,
    -1,26,27,28, 29,30,31,32, 33,34,35,36, 37,38,39,40,
    41,42,43,44, 45,46,47,48, 49,50,51,-1, -1,-1,-1,-1
};

/* encode a raw binary buffer to a null-terminated base64 strings */
static void
base64_encode( byte *out, const byte *in, size_t len, size_t olen )
{
    if ( !out || !in )
        return;
  
    while ( len >= 3 && olen > 10 ) {
        *out++ = b64chars[in[0] >> 2];
        *out++ = b64chars[((in[0] << 4) & 0x30) | (in[1] >> 4)];
        *out++ = b64chars[((in[1] << 2) & 0x3c) | (in[2] >> 6)];
        *out++ = b64chars[in[2] & 0x3f];
        olen -= 4; len -= 3; in += 3;
    }

    /* clean up remainder */
    if ( len > 0 && olen > 4 ) {
        byte fragment = 0;
        *out++ = b64chars[in[0] >> 2];
        fragment = (in[0] << 4) & 0x30;
        if ( len > 1 )
            fragment |= in[1] >> 4;
        *out++ = b64chars[fragment];
        *out++ = (len < 2) ? '=' : b64chars[(in[1] << 2) & 0x3c];
        *out++ = '=';
    }
    *out = '\0';
} /* base64_encode */

/* Convert '\0'-terminated base64 string to raw byte buffer.
   Returns length of returned buffer, or -1 on error. */
static int 
base64_decode( char *out, const char *in )
{
    int len = 0;
    byte digit1, digit2, digit3, digit4;

    if ( !out || !in )
        return -1;
  
    do {
        digit1 = in[0];
        if ( digit1 > 127 || base64val (digit1) == BAD )
            return -1;
        digit2 = in[1];
        if ( digit2 > 127 || base64val (digit2) == BAD )
            return -1;
        digit3 = in[2];
        if ( digit3 > 127 || ((digit3 != '=')
                              && (base64val (digit3) == BAD) ) )
            return -1;
        digit4 = in[3];
        if ( digit4 > 127 || ((digit4 != '=')
                              && (base64val (digit4) == BAD) ) )
            return -1;
        in += 4;

        /* digits are already sanity-checked */
        *out++ = (base64val(digit1) << 2) | (base64val(digit2) >> 4);
        len++;
        if ( digit3 != '=' ) {
            *out++ = ((base64val(digit2) << 4) & 0xf0) 
                | (base64val(digit3) >> 2);
            len++;
            if ( digit4 != '=' ) {
                *out++ = ((base64val(digit3) << 6) & 0xc0) | base64val(digit4);
                len++;
            }
        }
    } while ( *in && digit4 != '=' );

    return len;
} /* base64_decode */

static void
write_str( CDK_IOBUF buf, const char *str, const char *lf )
{
    if ( str && *str )
        cdk_iobuf_write( buf, str, strlen(str) );
    cdk_iobuf_write( buf, lf? lf : LF, lf? strlen(lf) : strlen(LF) );
} /* write_str */

static int
check_armor( CDK_IOBUF inp )
{
    CDK_BSTRING a;
    int check = 0;
  
    if ( !inp )
        return 0;
    a = cdk_iobuf_read_mem( inp, 0 );
    if ( a && strstr( a->d, "-----BEGIN PGP" )
         && strstr( a->d, "-----END PGP" ) )
        check = 1;
    cdk_free( a );
  
    return check;
} /* check_armor */

static int
is_armored( const byte *buf )
{
    int ctb = 0 , pkttype = 0;

    ctb = *buf;
    if ( !(ctb & 0x80) )
        return 1; /* invalid packet: assume it is armored */
    pkttype =  ctb & 0x40 ? (ctb & 0x3f) : ((ctb>>2) &0xf);

    switch( pkttype ) {
    case PKT_MARKER:
    case PKT_SYMKEY_ENC:
    case PKT_ONEPASS_SIG:
    case PKT_PUBLIC_KEY:
    case PKT_SECRET_KEY:
    case PKT_PUBKEY_ENC:
    case PKT_SIGNATURE:
    case PKT_COMMENT:
    case PKT_OLD_COMMENT:
    case PKT_PLAINTEXT:
    case PKT_COMPRESSED:
    case PKT_ENCRYPTED:
        return 0; /* seems to be a regular packet: not armored */
    }
  
    return 1;
} /* is_armored */

static int
armor_encode( armor_filter_s *afx, CDK_IOBUF buf, size_t size )
{
    char crcbuf[5] = {0}, buffer[128];
    byte crcbuf2[3] = {0}, raw[80];
    size_t nread = 0;
    const char *s, *lf;
    CDK_IOBUF a = NULL;
    int rc = 0;

    if ( afx->idx < 0 || afx->idx > DIM(armor_begin)
         || afx->idx2 < 0 || afx->idx2 > DIM(armor_end) )
        return CDKERR_INV_VALUE;

    rc = cdk_iobuf_create( &a, NULL );
    if ( rc )
        return rc;
    
    lf = afx->le? afx->le : LF;
    s = armor_begin[afx->idx];
    cdk_iobuf_write( a, "-----", 5 );
    cdk_iobuf_write( a, s, strlen(s) );
    write_str( a, "-----", lf );
    write_str( a, "Version: "PACKAGE" "VERSION, lf );
    if ( afx->hdrlines )
        cdk_iobuf_write( a, afx->hdrlines, strlen(afx->hdrlines) );
    write_str( a, NULL, lf );
    while ( size > LINE_LENGTH_RAW ) {
        rc = cdk_iobuf_read( buf, raw, LINE_LENGTH_RAW, &nread );
        if ( rc )
            goto leave;
        size -= nread;
        if ( nread ) {
            base64_encode( buffer, raw, nread, sizeof(buffer)-1 );
            write_str( a, buffer, lf );
            CRC_UPDATE( afx->crc, raw, nread );
        }
    }
    if ( size > 0 ) {
        rc = cdk_iobuf_read( buf, raw, size, &nread );
        if ( !rc && nread ) {
            base64_encode( buffer, raw, size, sizeof(buffer) );
            write_str( a, buffer, lf );
            CRC_UPDATE( afx->crc, raw, size );
        }
    }
    crcbuf2[0] = afx->crc >> 16;
    crcbuf2[1] = afx->crc >>  8;
    crcbuf2[2] = afx->crc      ;
    crcbuf[0] = b64chars[crcbuf2[0] >> 2];
    crcbuf[1] = b64chars[((crcbuf2[0] << 4) & 0x30) | (crcbuf2[1] >> 4)];
    crcbuf[2] = b64chars[((crcbuf2[1] << 2) & 0x3c) | (crcbuf2[2] >> 6)];
    crcbuf[3] = b64chars[crcbuf2[2] & 0x3f];
    cdk_iobuf_put( a, '=' );
    write_str( a, crcbuf, lf );
    s = armor_end[afx->idx2];
    cdk_iobuf_write( a, "-----", 5 );
    cdk_iobuf_write( a, s, strlen(s) );
    write_str( a, "-----", lf );

leave:
    cdk_iobuf_copy( buf, a, 0 );
    cdk_iobuf_close( a );
  
    return 0;
} /* armor_encode */

static unsigned
len_without_trailing_chars( byte *line, unsigned len, const char *trimchars )
{
    byte *p, *mark;
    unsigned n;

    for ( mark=NULL, p=line, n=0; n < len; n++, p++ ) {
        if ( strchr( trimchars, *p ) ) {
            if( !mark )
                mark = p;             
        }
        else
            mark = NULL;     
    }

    return mark? (mark - line) : len;    
} /* len_without_trailing_chars */

static int
_text_filter( text_filter_s *tfx, CDK_IOBUF buf )
{
    int rc = 0, ec = 0;
    size_t len = 0, size = 0;
    unsigned maxlen = 0;
    CDK_IOBUF a;

    if ( !tfx || !buf )
        return CDKERR_INV_VALUE;

    size = cdk_iobuf_get_length( buf );
    if ( !size )
        return CDKERR_INV_VALUE;
    rc = cdk_iobuf_create( &a, NULL );
    if ( rc )
        return rc;
  
    while ( !rc && len < size ) {
        int lf_seen = 0;
        while( len < size && tfx->buffer_pos < tfx->buffer_len && !ec ) { 
            cdk_iobuf_put( a, tfx->buffer[tfx->buffer_pos++] );
            len++;
        }
        if( len >= size )
            continue; /* read the next line */
        maxlen = MAX_LINELEN;
        tfx->buffer_pos = 0;
        tfx->buffer_len = cdk_iobuf_get_line( buf, &tfx->buffer,
                                              &tfx->buffer_size, &maxlen );
        if ( !maxlen )
            tfx->truncated++;
        if ( !tfx->buffer_len ) {
            if ( !len )
                rc = CDKERR_INV_VALUE; /* eof */
            break;
        }
        lf_seen = tfx->buffer[tfx->buffer_len-1] == '\n';
        tfx->buffer_len = trim_trailing_ws( tfx->buffer, tfx->buffer_len );
        if ( lf_seen ) {
            tfx->buffer[tfx->buffer_len++] = '\r';
            tfx->buffer[tfx->buffer_len++] = '\n';
        }  
    }

    cdk_iobuf_copy( buf, a, 0 );
    cdk_iobuf_close( a );
  
    return rc;
} /* _text_filter */

int
cdk_armor_filter_use( CDK_IOBUF inp )
{
    byte buf[1];
    int n = 0;
    int check = 0;

    n = cdk_iobuf_peek( inp, buf, 1 );
    if( n == -1 )
        return 0; /* EOF, doesn't matter whether armored or not */
    if( !n )
        return 1; /* can't check it: try armored */

    check = is_armored( buf );
    if ( check )
        check = check_armor( inp );
  
    return check;
} /* cdk_armor_filter_use */ 

/**
 * cdk_text_filter - Text filter.
 * @ctx: The text filter context.
 * @control: The iobuf control id.
 * @buf: The io buffer.
 *
 * The filter is used to make canonical text: Lines are terminated by
 * CR, LF, trailing white spaces are removed.
 */
int
cdk_text_filter( text_filter_s *tfx, int control, CDK_IOBUF buf )
{
    int rc = 0;

    if ( !tfx || !buf )
        return CDKERR_INV_VALUE;

    if ( control == IOBUF_CTRL_UNDERFLOW ) {
        rc = _text_filter( tfx, buf );
        cdk_free( tfx->buffer );
        tfx->buffer = NULL;
    }
  
    return rc;  
} /* cdk_text_filter */

/**
 * cdk_armor_filter - Decode armored data
 * @ctx: opaque context that is the armor filter context.
 * @control: The control code (FLUSH for writing, UNDERFLOW for reading)
 * @buf: IOBUF that holds the data to en- or decode.
 *
 * Decodes the iobuf src into raw data and store the result into r_dst.
 **/
int
cdk_armor_filter( armor_filter_s *afx, int control, CDK_IOBUF buf )
{
    CDK_IOBUF a = NULL;
    const char *s = NULL;
    char buffer[128];
    byte raw[80];
    u32 crc2 = 0;
    size_t nread = 0, size = 0;
    int idx = 0, idx2 = 0;
    int i, pgp_data = 0;
    int rc = 0;
    
    if ( !afx || !buf )
        return CDKERR_INV_VALUE;

    afx->crc = CRCINIT;
    size = cdk_iobuf_get_length( buf );
    if ( !size )
        return CDKERR_GENERAL;
    cdk_iobuf_rewind( buf );

    if ( control == IOBUF_CTRL_FLUSH )
        return armor_encode( afx, buf, size );

    /* Search the begin of the message */
    while ( !cdk_iobuf_eof( buf ) && !pgp_data ) {
        rc = cdk_iobuf_read_line( buf, buffer, sizeof(buffer)-1, &nread );
        if ( rc )
            goto leave;
        for ( i=0; (s=armor_begin[i]); i++ ) {
            if ( !strncmp( s, buffer+5, strlen(s) ) ) {
                pgp_data = 1;
                idx = i;
                break;
            }
        }
    }
    if ( cdk_iobuf_eof( buf ) || !pgp_data ) {
        rc = CDKERR_ARMOR; /* no data found */
        goto leave;
    }

    /* Parse header until the empty line is reached */
    while ( !cdk_iobuf_eof( buf ) ) {
        rc = cdk_iobuf_read_line( buf, buffer, sizeof(buffer)-1, &nread );
        if ( rc )
            goto leave;
        if ( nread == 1 ) {
            rc = 0;
            break; /* empty line */
        }
        /* From RFC2440: OpenPGP should consider improperly formatted Armor
           Headers to be corruption of the ASCII Armor. A colon and a single
           space separate the key and value. */
        if ( !strstr( buffer, ": " ) ) {
            rc = CDKERR_ARMOR;
            goto leave;
        }
        for ( rc=1, i=0; (s=valid_headers[i]); i++ ) {
            if ( !strncmp( s, buffer, strlen(s) ) )
                rc = 0;
        }
        if ( rc ) {
            /* From RFC2440: Unknown keys should be reported to the user,
               but OpenPGP should continue to process the message. */
            cdk_printf( "Unknown header: `%s'\n", buffer );
            rc = 0;
        }
    }
    
    cdk_iobuf_new( &a, size );
    /* Read the data body */
    while ( !cdk_iobuf_eof(buf) ) {
        rc = cdk_iobuf_read_line( buf, buffer, sizeof(buffer)-1, &nread );
        if ( rc )
            goto leave;
        if ( *buffer == '=' && nread == 6 ) { /* CRC */
            byte crcbuf[3];
            memset( crcbuf, 3, sizeof crcbuf );
            base64_decode( crcbuf, buffer+1 );
            crc2 = (crcbuf[0] << 16) | (crcbuf[1] << 8) | (crcbuf[2]);
            break; /* stop here */
        }
        else {
            int narm = base64_decode( raw, buffer );
            CRC_UPDATE( afx->crc, raw, narm );
            if ( narm > 0 )
                cdk_iobuf_write( a, raw, narm );
        }
    }
    
    /* Search the tail of the message */
    rc = cdk_iobuf_read_line( buf, buffer, sizeof(buffer)-1, &nread );
    if ( !rc ) {
        for ( rc = 1, i = 0; (s=armor_end[i]); i++ ) {
            if ( !strncmp( s, buffer+5, strlen(s) ) ) {
                rc = 0;
                idx2 = i;
                break;
            }
        }
    }
    /* this catches error when no tail was found or the header is
       different then the tail line. */
    if ( rc || idx != idx2 )
        rc = CDKERR_ARMOR;

    afx->crc_okay = (afx->crc == crc2) ? 1 : 0;
    if ( !afx->crc_okay && !rc )
        rc = CDKERR_ARMOR_CRC;
      
leave:
    cdk_iobuf_copy( buf, a, 0 );
    cdk_iobuf_close( a );

    return rc;
} /* cdk_armor_filter */

/**
 * cdk_armor_file - Armor a file.
 * @opts: sexp structure with the following format:
 * (armor-file(
 *   (textmode%d)
 *   (compress%d)
 * ))
 * @file: the plainext file.
 * @output: the output file.
 **/
int
cdk_armor_file( GCRY_SEXP opts, const char *file, const char *output )
{
    CDK_IOBUF inp = NULL, outp = NULL;
    armor_filter_s afx;
    int rc = 0;

    if ( !file || !output )
        return CDKERR_INV_VALUE;
  
    rc = cdk_iobuf_open( &inp, file, IOBUF_MODE_RD );
    if ( rc )
        return rc;

    outp = _cdk_make_plaintext( inp, file, 0 );
    if ( !outp ) {
        rc = CDKERR_INV_VALUE;
        goto leave;
    }

    memset( &afx, 0, sizeof afx );
    afx.idx = afx.idx2 = ARMOR_MESSAGE;
    rc = cdk_armor_filter( &afx, IOBUF_CTRL_FLUSH, outp );
    if ( rc )
        goto leave;

    rc = cdk_iobuf_store( outp, output, 0 );

leave:
    cdk_iobuf_close( outp );
    cdk_iobuf_close( inp );
  
    return rc;
} /* cdk_armor_file */


