/* keydb.c - Key database routines
 *  Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <stdio.h>
#include <string.h>

#include "opencdk.h"
#include "main.h"

#define KEYDB_ENTRIES 4
static struct cdk_keydb_handle_s keydb_table[KEYDB_ENTRIES];

int
cdk_keydb_add_resource(const char *name, int is_secret)
{
    int i;

    if (!name)
        return CDKERR_INV_VALUE;
  
    for (i=0; i<KEYDB_ENTRIES; i++) {
        if (!keydb_table[i].used) {
            keydb_table[i].name = cdk_alloc(strlen(name)+1);
            if ( !keydb_table[i].name )
                return CDKERR_OUT_OF_CORE;
            strcpy(keydb_table[i].name, name);
            keydb_table[i].secret = is_secret;
            keydb_table[i].used = 1;
            keydb_table[i].type = KEYDB_TYPE_KEYRING;
            keydb_table[i].offset = 0;
            keydb_table[i].old_offset = 0;
            break;
        }
    }

    return 0;
} /* cdk_keydb_add_resource */

int
cdk_keydb_remove_resource(int id)
{
    int i;
  
    for (i=0; i<KEYDB_ENTRIES; i++) {
        if ( keydb_table[i].used && id == i ) {
            cdk_free( keydb_table[i].name );
            keydb_table[i].name = NULL;
            keydb_table[i].offset = 0;
            keydb_table[i].old_offset = 0;
            keydb_table[i].secret = 0;
            keydb_table[i].used = 0;
            break;
        }
    }
    return 0;  
} /* cdk_keydb_remove_resource */

const char*
cdk_keydb_get_name( int id )
{
    int i;
  
    for ( i=0; i<KEYDB_ENTRIES; i++ ) {
        if ( keydb_table[i].used && id == i )
            return keydb_table[i].name;
    }
    return NULL;  
} /* cdk_keydb_get_name */

int
cdk_keydb_is_secret( int id )
{
    int i;
  
    for (i=0; i<KEYDB_ENTRIES; i++) {
        if (keydb_table[i].used && id == i)
            return keydb_table[i].secret;
    }
  
    return 0;
} /* cdk_keydb_is_secret */

int
cdk_keydb_find_idx(int is_secret, int *ret_pos)
{
    int i;
  
    if ( !ret_pos )
        return CDKERR_INV_VALUE;
  
    *ret_pos = -1;
    for ( i = 0;  i < KEYDB_ENTRIES; i++ ) {
        if (!keydb_table[i].used)
            continue;
        if ( is_secret == cdk_keydb_is_secret(i) ) {
            *ret_pos = i;
            return 0;
        }
    }
    return CDKERR_GENERAL;
} /* cdk_keydb_find_idx */

CDK_KEYDB_HD
cdk_keydb_get_ctx(int id)
{
    if (id >= 0 && id < KEYDB_ENTRIES)
        return keydb_table[id].used? &keydb_table[id] : NULL;
  
    return NULL;
} /* cdk_keydb_get_ctx */

int
cdk_keydb_open( CDK_KEYDB_HD kdb, CDK_IOBUF *ret_kr )
{
    int rc = 0;
    CDK_IOBUF kr = NULL;
  
    if ( !kdb || !ret_kr )
        return CDKERR_INV_VALUE;

    if ( kdb->type == KEYDB_TYPE_DATA && kdb->buf ) {
        cdk_iobuf_rewind( kdb->buf );
        kr = kdb->buf;        
    }
    else if ( kdb->type == KEYDB_TYPE_KEYRING && kdb->name ) {
        rc = cdk_iobuf_open( &kr, kdb->name, IOBUF_MODE_RD );
        if ( rc )
            goto leave;
    }
    else if ( kdb->type == KEYDB_TYPE_ARMORED && kdb->name ) {
        armor_filter_s afx;
        rc = cdk_iobuf_open( &kr, kdb->name, IOBUF_MODE_RD );
        if ( rc )
            goto leave;
        memset( &afx, 0, sizeof afx );
        rc = cdk_armor_filter( &afx, IOBUF_CTRL_UNDERFLOW, kr );
        if ( rc )
            rc = CDKERR_ARMOR;
    }

    if ( !rc )
        cdk_iobuf_seek( kr, kdb->offset );
leave:
    if ( rc ) {
        cdk_iobuf_close( kr );
        kr = NULL;
    }
    *ret_kr = kr;
  
    return rc;
} /* cdk_keydb_open */

int
cdk_keydb_search( CDK_KEYDB_HD kdb, CDK_KEYDB_SEARCH *ks, CDK_KBNODE *r_key )
{
    CDK_IOBUF kr = NULL;
    CDK_KBNODE p, key = NULL;
    cdkPKT_user_id *uid = NULL;
    cdkPKT_public_key *pk = NULL;
    u32 keyid[2] = {0};
    byte fpr[20] = {0};
    const char *name = NULL;
    int found = 0, eof = 0, i = 0;
    int need_uid = 0, need_fpr = 0, need_keyid = 0;
    int uidlen = 0;
    int rc = 0;
    size_t pos = 0;
  
    if ( !kdb || !r_key || !ks )
        return CDKERR_INV_VALUE;

    rc = cdk_keydb_open( kdb, &kr );
    if ( rc ) {
        *r_key = NULL;
        return rc;
    }
  
    do {
        pos = cdk_iobuf_tell( kr );
        kdb->old_offset = pos;
        rc = cdk_keydb_get_keyblock( kr, &key, &eof );
        if ( !key && eof == 1 ) {
            rc = CDKERR_NOKEY;
            break;
        }
        else if ( rc )
            break;
        pos = cdk_iobuf_tell( kr );
        kdb->offset = pos;
    
        switch ( ks->type ) {
        case KEYDB_SEARCH_EXACT:
        case KEYDB_SEARCH_SUBSTR:
            need_uid = 1;
            break;

        case KEYDB_SEARCH_SHORT_KEYID:
        case KEYDB_SEARCH_KEYID:
            need_keyid = 1;
            break;

        case KEYDB_SEARCH_FPR:
            need_fpr = 1;
            break;
        }
        for ( p = key; p && p->pkt->pkttype; p = p->next ) {
            if ( is_public( p->pkt->pkttype ) ) {
                pk = p->pkt->pkt.public_key;
                if ( need_keyid )
                    cdk_pk_get_keyid( pk, keyid );
                if ( need_fpr )
                    cdk_pk_get_fingerprint( pk, fpr );
            }
            if ( is_secret( p->pkt->pkttype ) ) {
                pk = p->pkt->pkt.secret_key->pk;
                if ( need_keyid )
                    cdk_pk_get_keyid( pk, keyid );
                if ( need_fpr )
                    cdk_pk_get_fingerprint( pk, fpr );
            }
            else if ( p->pkt->pkttype == PKT_USER_ID ) {
                if ( need_uid ) {
                    uid = p->pkt->pkt.user_id;
                    name = uid->name;
                    uidlen = uid->len;
                }
            }

            switch ( ks->type ) {
            case KEYDB_SEARCH_EXACT:
                for ( i=0; name && i<uidlen && ks->u.pattern[i]; i++ ) {
                    if ( name[i] != ks->u.pattern[i] ) {
                        found = 0;
                        break;
                    }
                }
                if ( i == uidlen )
                    found = 1;
                break;
            
            case KEYDB_SEARCH_SUBSTR:
                if ( uidlen > 65536 )
                    return CDKERR_INV_PACKET;
                if ( name && strlen(ks->u.pattern) > uidlen )
                    found = 0;
                else if ( name &&ascii_memistr( name, uidlen, ks->u.pattern ) )
                    found = 1;
                break;

            case KEYDB_SEARCH_SHORT_KEYID:
                if ( keyid[1] == ks->u.keyid[1] ) {
                    found = 1;
                    break;
                }
                break;
        
            case KEYDB_SEARCH_KEYID:
                if ( keyid[1] == ks->u.keyid[1]
                     && keyid[0] == ks->u.keyid[0] ) {
                    found = 1;
                    break;       
                }
                break;

            case KEYDB_SEARCH_FPR:
                if ( !memcmp( ks->u.fpr, fpr, 20 ) ) {
                    found = 1;
                    break;
                }
                break;
            }
            if ( found )
                goto leave;
        }
        if ( !found ) {
            cdk_kbnode_release( key ); key = NULL;
            name = NULL; uidlen = 0;
        }
    } while ( !found && !rc );
  
leave:
    cdk_iobuf_close( kr );
    *r_key = found? key: NULL;
  
    return rc;
} /* cdk_keydb_search */

int
cdk_keydb_get_bykeyid( CDK_KEYDB_HD kdb, u32 *keyid, CDK_KBNODE *ret_pk )
{
    CDK_KEYDB_SEARCH ks;

    if ( !kdb || !keyid || !ret_pk )
        return CDKERR_INV_VALUE;
  
    ks.type = KEYDB_SEARCH_SHORT_KEYID;
    ks.u.keyid[1] = keyid[1];
  
    return cdk_keydb_search( kdb, &ks, ret_pk );
} /* cdk_keydb_get_bykeyid */

int
cdk_keydb_get_byfpr( CDK_KEYDB_HD kdb, const byte *fpr, CDK_KBNODE *ret_pk )
{
    CDK_KEYDB_SEARCH ks;

    if ( !kdb || !fpr || !ret_pk )
        return CDKERR_INV_VALUE;

    ks.type = KEYDB_SEARCH_FPR;
    memcpy(ks.u.fpr, fpr, 20);
  
    return cdk_keydb_search( kdb, &ks, ret_pk );
} /* cdk_keydb_get_byfpr */

int
cdk_keydb_get_bypattern( CDK_KEYDB_HD kdb, const char *patt, CDK_KBNODE *ret_pk )
{
    CDK_KEYDB_SEARCH ks;

    if ( !kdb || !patt || !ret_pk )
        return CDKERR_INV_VALUE;

    ks.type = KEYDB_SEARCH_SUBSTR;
    ks.u.pattern = patt;

    return cdk_keydb_search( kdb, &ks, ret_pk );
} /* cdk_keydb_get_bypattern */

static CDK_KBNODE
keydb_find_byusage( CDK_KBNODE root, int req_usage, int pk )
{
    CDK_KBNODE node;
    int pkttype = 0;

    if ( !req_usage )
        return cdk_kbnode_find( root, PKT_PUBLIC_KEY );

    for ( node = root; node; node = node->next ) {
        pkttype = node->pkt->pkttype;
        if ( pk && is_public( pkttype )
             && (node->pkt->pkt.public_key->pubkey_usage & req_usage) )
            return node;
        else if ( !pk && is_secret( pkttype )
                  && (node->pkt->pkt.secret_key->req_usage & req_usage) )
            return node;
    }

    return NULL;
} /* keydb_find_byusage */

static CDK_KBNODE
keydb_find_bykeyid( CDK_KBNODE root, u32 *keyid, int pk )
{
    CDK_KBNODE node;
    int pkttype = 0;

    for ( node = root; node; node = node->next ) {
        pkttype = node->pkt->pkttype;
        if ( pk && is_public( pkttype )
             && node->pkt->pkt.public_key->keyid[1] == keyid[1] )
            return node;
        else if ( !pk && is_secret( pkttype )
                  && node->pkt->pkt.secret_key->keyid[1] == keyid[1] )
            return node;
    }
  
    return NULL;
} /* keydb_find_bykeyid */      

int
cdk_keydb_get_sk_byname( const char *name, cdkPKT_secret_key **ret_sk,
                         char *pw, int unlock, int alloced )
{
    CDK_KBNODE key, node = NULL;
    cdkPKT_secret_key *sk = NULL;
    CDK_KEYDB_HD khd;
    int req_usage = 0, idx = 0;
    int rc = 0;
  
    if ( !ret_sk )
        return CDKERR_INV_VALUE;

    rc = cdk_keydb_find_idx( 1, &idx );
    if ( rc )
        return CDKERR_NOKEYRING;
    khd = cdk_keydb_get_ctx( idx );
    rc = cdk_keydb_get_bypattern( khd, name, &key );
    if ( rc )
        goto leave;
    if ( alloced ) {
        req_usage = ret_sk[0]->req_usage;
        cdk_free( ret_sk[0] );
        node = keydb_find_byusage( key, req_usage, 0 );
    }
    if ( !node ) {
        node = cdk_kbnode_find( key, PKT_SECRET_KEY );
        if ( !node ) {
            rc = CDKERR_NOKEY;
            goto leave;
        }
    }
    cdk_copy_sk( &sk, node->pkt->pkt.secret_key );
    cdk_kbnode_release( key );

    if ( unlock && pw )
        rc = cdk_seckey_unprotect( sk, pw );

leave:
    *ret_sk = sk;
    return rc;
} /* cdk_keydb_get_sk_byname */

int
cdk_keydb_get_pk_byname( const char *name, cdkPKT_public_key **ret_pk,
                         int alloced )
{
    CDK_KBNODE key, node = NULL;
    cdkPKT_public_key *pk = NULL;
    CDK_KEYDB_HD khd;
    const char *s;
    int req_usage = 0, idx = 0;
    int rc = 0;

    if ( !name || !ret_pk )
        return CDKERR_INV_VALUE;

    rc = cdk_keydb_find_idx( 0, &idx );
    if ( rc ) 
        return CDKERR_NOKEYRING;

    khd = cdk_keydb_get_ctx( idx );
    rc = cdk_keydb_get_bypattern( khd, name, &key );
    if ( rc ) {
        goto leave;
    }
    if ( alloced ) { 
        req_usage = ret_pk[0]->pubkey_usage;
        cdk_free( ret_pk[0] );
        node = keydb_find_byusage( key, req_usage, 1 );
    }
    if ( !node ) {
        node = cdk_kbnode_find( key, PKT_PUBLIC_KEY );
        if ( !node ) {
            rc = CDKERR_NOKEY;
            goto leave;
        }
    }
    cdk_copy_pk( &pk, node->pkt->pkt.public_key );
    for ( node = key; node; node=node->next ) {
        if ( node->pkt->pkttype == PKT_USER_ID ) {
            s = node->pkt->pkt.user_id->name;
            if ( pk && !pk->uid && ascii_memistr( s, strlen(s), name ) ) { 
                cdk_copy_uid( &pk->uid, node->pkt->pkt.user_id );
                break;
            }
        }
    }
    cdk_kbnode_release( key );

leave:
    *ret_pk = pk;
  
    return rc;
} /* cdk_keydb_get_pk_byname */
  
int
cdk_keydb_get_pk( CDK_KEYDB_HD khd, u32 *keyid, cdkPKT_public_key **r_pk )
{
    CDK_KBNODE key = NULL, node = NULL;
    cdkPKT_public_key *pk = NULL;
    CDK_KEYDB_SEARCH ks;
    CDK_KEYDB_HD keydb_hd = NULL;
    int rc = 0, idx = 0;
  
    if ( !keyid || !r_pk )
        return CDKERR_INV_VALUE;

    if ( khd == NULL ) {    
        rc = cdk_keydb_find_idx(0, &idx);
        if ( rc )
            return CDKERR_NOKEYRING;
        keydb_hd = cdk_keydb_get_ctx(idx);
    }
    else
        keydb_hd = khd;

    if ( !keyid[0] ) {
        ks.type = KEYDB_SEARCH_SHORT_KEYID;
        ks.u.keyid[1] = keyid[1];
    }
    else {
        ks.type = KEYDB_SEARCH_KEYID;
        ks.u.keyid[0] = keyid[0];
        ks.u.keyid[1] = keyid[1];
    }
    rc = cdk_keydb_search( keydb_hd, &ks, &key );
    if ( rc )
        goto leave;
    node = keydb_find_bykeyid( key, keyid, 1 );
    if ( !node ) {
        rc = CDKERR_NOKEY;
        goto leave;
    }
    cdk_copy_pk( &pk, node->pkt->pkt.public_key );
    cdk_kbnode_release( key );
  
leave:
    *r_pk = pk;
  
    return rc;
} /* cdk_keydb_get_pk */

int
cdk_keydb_sk_get( CDK_KEYDB_HD khd, u32 *keyid, cdkPKT_secret_key **ret_sk )
{
    CDK_KEYDB_HD skhd;
    CDK_KBNODE key, node;
    cdkPKT_secret_key *sk = NULL;
    int rc = 0;

    if ( !khd ) {
        int pos = 0;
        rc = cdk_keydb_find_idx( 1, &pos );
        if ( rc )
            return CDKERR_NOKEYRING;
        skhd = cdk_keydb_get_ctx( pos );         
    }
    else
        skhd = khd;
    rc = cdk_keydb_get_bykeyid( skhd, keyid, &key );
    if ( rc )
        goto leave;

    node = keydb_find_bykeyid( key, keyid, 0 );
    if ( !node ) {
        rc = CDKERR_NOKEY;
        goto leave;
    }

    cdk_copy_sk( &sk, node->pkt->pkt.secret_key );
    cdk_kbnode_release( key );

leave:
    *ret_sk = sk;
    return rc;
} /* cdk_keydb_sk_get */

static int
keydb_merge_selfsig( CDK_KBNODE key )
{
    CDK_KBNODE k, kb_pk, kb_id;
    struct cdk_subpkt_s *s = NULL;
    cdkPKT_signature *sig = NULL;
    cdkPKT_user_id *uid = NULL;
    const byte *symalg = NULL, *hashalg = NULL, *compalg = NULL;
    size_t nsymalg = 0, nhashalg = 0, ncompalg = 0, n = 0;
  
    if ( !key )
        return CDKERR_INV_VALUE;

    for ( k=key; k && k->pkt->pkttype; k=k->next ) {
        if ( k->pkt->pkttype == PKT_SIGNATURE
             && k->pkt->pkt.signature->sig_class == 0x13 ) {
            kb_id = cdk_kbnode_find_prev( key, k, PKT_USER_ID );
            if ( !kb_id )
                return CDKERR_NOKEY;
            uid = kb_id->pkt->pkt.user_id;
            sig = k->pkt->pkt.signature;
            s = cdk_subpkt_find( sig->hashed, SIGSUBPKT_PRIMARY_UID );
            if ( s )
                uid->is_primary = 1;
            s = cdk_subpkt_find( sig->hashed, SIGSUBPKT_FEATURES );
            if ( s && s->size == 1 && s->data[0] & 0x01 )
                uid->mdc_feature = 1;
            s = cdk_subpkt_find( sig->hashed, SIGSUBPKT_PREFS_SYM );
            if ( s ) {
                symalg = s->data; nsymalg = s->size;
                n += s->size + 1;
            }
            s = cdk_subpkt_find( sig->hashed, SIGSUBPKT_PREFS_HASH );
            if ( s ) {
                hashalg = s->data; nhashalg = s->size;
                n += s->size + 1;
            }
            s = cdk_subpkt_find( sig->hashed, SIGSUBPKT_PREFS_ZIP );
            if ( s ) {       
                compalg = s->data; ncompalg = s->size;
                n += s->size + 1;
            }
            if ( !n || !hashalg || !compalg || !symalg )
                uid->prefs = NULL;
            else {
                uid->prefs = cdk_alloc( sizeof (*uid->prefs) * (n+1) );
                if ( !uid->prefs )
                    return CDKERR_OUT_OF_CORE;
                n = 0;
                for (; nsymalg; nsymalg--, n++) {
                    uid->prefs[n].type = PREFTYPE_SYM;
                    uid->prefs[n].value = *symalg++;
                }
                for (; nhashalg; nhashalg--, n++) {
                    uid->prefs[n].type = PREFTYPE_HASH;
                    uid->prefs[n].value = *hashalg++;
                }
                for (; ncompalg; ncompalg--, n++) {
                    uid->prefs[n].type = PREFTYPE_ZIP;
                    uid->prefs[n].value = *compalg++;
                }
                /* end of list marker */
                uid->prefs[n].type = PREFTYPE_NONE;
                uid->prefs[n].value = 0;
                uid->prefs_size = n;
            
                kb_pk = cdk_kbnode_find_prev( key, k, PKT_PUBLIC_KEY );
                if ( kb_pk && uid->prefs && n ) {
                    cdkPKT_public_key *pk = kb_pk->pkt->pkt.public_key;
                    pk->prefs = _cdk_prefs_copy( uid->prefs );
                    pk->prefs_size = n;
                }
            }
        }
    }

    return 0;
} /* keydb_merge_selfsig */

static void
key_set_expire( cdkPKT_public_key *pk, u32 pk_expired )
{
    u32 key_expire = 0;
    u32 curtime = 0;
  
    if ( pk ) {
        key_expire = 0;
        curtime = make_timestamp( );
        pk->expiredate = pk->timestamp + pk_expired;
        key_expire = pk->expiredate;
        pk->has_expired = (key_expire >= curtime)? 0: 1;
    }
} /* key_set_expire */

static int
keydb_parse_allsigs( CDK_KBNODE key, int check )
{
    CDK_KBNODE k, kb;
    cdkPKT_signature *sig;
    struct cdk_subpkt_s *s = NULL;
    int pk_expired = 0;
    int rc = 0;

    if ( !key )
        return CDKERR_INV_VALUE;

    if ( cdk_kbnode_find( key, PKT_SECRET_KEY ) )
        return 0; /* fixme: we need a special handling then! */
  
    /* reset */	
    for ( k = key; k; k = k->next ) {
        if ( k->pkt->pkttype == PKT_USER_ID )
            k->pkt->pkt.user_id->is_revoked = 0;
        else if ( is_public( k->pkt->pkttype ) )
            k->pkt->pkt.public_key->is_revoked = 0;
    }
    
    for ( k = key; k; k = k->next ) {
        if ( k->pkt->pkttype == PKT_SIGNATURE ) {
            sig = k->pkt->pkt.signature;
            /* Revocation certificates for primary keys */
            if ( sig->sig_class == 0x20 ) {
                kb = cdk_kbnode_find_prev( key, k, PKT_PUBLIC_KEY );
                if ( kb ){
                    kb->pkt->pkt.public_key->is_revoked = 1;
                    if ( check )
                        ;
                }
                else
                    return CDKERR_NOKEY;
            }
            /* Revocation certificates for subkeys */
            else if ( sig->sig_class == 0x28 ) {
                kb = cdk_kbnode_find_prev( key, k, PKT_PUBLIC_SUBKEY );
                if ( kb ) {
                    kb->pkt->pkt.public_key->is_revoked = 1;
                    if ( check )
                        ;
                }
                else
                    return CDKERR_NOKEY;
            }
            /* Revocation certifcates for user ID's */
            else if ( sig->sig_class == 0x30 ) {
                kb = cdk_kbnode_find_prev( key, k, PKT_USER_ID );
                if ( kb ) {
                    kb->pkt->pkt.user_id->is_revoked = 1;
                    if ( check )
                        ;
                }
                else
                    return CDKERR_NOKEY;
            }
            /* Direct certificates for primary keys */
            else if ( sig->sig_class == 0x13 ) {
                kb = cdk_kbnode_find_prev( key, k, PKT_PUBLIC_KEY );
                if ( kb ) {
                    kb->pkt->pkt.public_key->is_invalid = 0;
                    s = cdk_subpkt_find( k->pkt->pkt.signature->hashed,
                                         SIGSUBPKT_KEY_EXPIRE );
                    if ( s ) {
                        pk_expired = buffer_to_u32( s->data );
                        key_set_expire( kb->pkt->pkt.public_key, pk_expired );
                    }
                    if ( check )
                        ;
                }
                else
                    return CDKERR_NOKEY;
            }
            /* Direct certificates for subkeys */
            else if ( sig->sig_class == 0x18 ) {
                kb = cdk_kbnode_find_prev( key, k, PKT_PUBLIC_SUBKEY );
                if ( kb ) {
                    kb->pkt->pkt.public_key->is_invalid = 0;
                    s = cdk_subpkt_find( k->pkt->pkt.signature->hashed,
                                         SIGSUBPKT_KEY_EXPIRE );
                    if ( s ) {
                        pk_expired = buffer_to_u32( s->data );
                        key_set_expire( kb->pkt->pkt.public_key, pk_expired );
                    }
                    if ( check )
                        ;
                }
                else
                    return CDKERR_NOKEY;
            }
        }
	}
	
    return rc;
} /* keydb_parse_allsigs */

int
cdk_keydb_enum_keyblocks( CDK_IOBUF inp, CDK_KBNODE *ret_key )
{
    int eof = 0;
    int rc = 0;

    rc = cdk_keydb_get_keyblock( inp, ret_key, &eof );
    if ( !rc && eof == 1 )
        rc = CDKERR_EOF;
    return rc;
} /* cdk_keydb_enum_keyblocks */

int
cdk_keydb_get_keyblock( CDK_IOBUF inp, CDK_KBNODE *r_key, int *r_eof )
{
    CDK_PACKET *pkt = NULL;
    CDK_KBNODE key = NULL, node = NULL;
    u32 keyid[2] = {0}, main_keyid[2] = {0};
    int rc = 0, rc2 = 0;
    int first_key = 0, got_key = 0;
  
    if ( !inp || !r_key )
        return CDKERR_INV_VALUE;

    while ( 1/*!cdk_iobuf_eof( inp )*/ ) {
        pkt = cdk_alloc_clear( sizeof *pkt );
        if ( !pkt )
            return CDKERR_OUT_OF_CORE;
        rc = cdk_pkt_parse( inp, pkt );
        if ( rc )
            break;
        if ( is_main_key( pkt->pkttype ) || is_sub_key( pkt->pkttype ) ) {
            if ( first_key && is_main_key( pkt->pkttype ) ) {
                cdk_iobuf_goback( inp, pkt->pktsize );
                break;
            }
            if ( is_main_key( pkt->pkttype ) ) {
                if ( is_public( pkt->pkttype) )
                    cdk_pk_get_keyid( pkt->pkt.public_key, main_keyid );
                else
                    cdk_sk_get_keyid( pkt->pkt.secret_key, main_keyid );
                first_key = 1;
            }
            else if ( is_sub_key( pkt->pkttype ) ) {
                if ( is_public( pkt->pkttype ) ) {
                    pkt->pkt.public_key->main_keyid[0] = main_keyid[0];
                    pkt->pkt.public_key->main_keyid[1] = main_keyid[1];
                }
                else {
                    pkt->pkt.secret_key->main_keyid[0] = main_keyid[0];
                    pkt->pkt.secret_key->main_keyid[1] = main_keyid[1];
                }
            }
            /* we save this for the signature */
            if ( is_public( pkt->pkttype ) )
                cdk_pk_get_keyid( pkt->pkt.public_key, keyid );
            else
                cdk_sk_get_keyid( pkt->pkt.secret_key, keyid );
            got_key = 1;
        }
        else if ( pkt->pkttype == PKT_USER_ID )
            ; /* nothing to do for know */
        else if ( pkt->pkttype == PKT_SIGNATURE ) {
            pkt->pkt.signature->key[0] = keyid[0];
            pkt->pkt.signature->key[1] = keyid[1];
        }
        node = cdk_kbnode_new( pkt );
        if ( !key )
            key = node;
        else
            cdk_kbnode_add( key, node );
    }

    if ( got_key ) {
        keydb_merge_selfsig( key );
        rc2 = keydb_parse_allsigs( key, 0 );
        if ( !rc )
            rc = rc2;
	}
  
    *r_key = got_key? key : NULL;
    if ( cdk_iobuf_eof( inp ) ) {
        *r_eof = 1;
        rc = 0;
    }

    return rc;
} /* cdk_keydb_get_keyblock */

int
cdk_keydb_export( CDK_IOBUF out, CDK_STRLIST remusr )
{
    CDK_KEYDB_HD khd = NULL;
    CDK_KBNODE pk, p;
    CDK_STRLIST r;
    int idx = 0;
    int rc = 0;
    
    rc = cdk_keydb_find_idx( 0, &idx );
    if ( rc )
        return rc;
    khd = cdk_keydb_get_ctx( idx );
    if ( !khd )
        return CDKERR_INV_VALUE;

    for ( r = remusr; r; r = r->next ) {
        rc = cdk_keydb_get_bypattern( khd, r->d, &pk );
        if ( rc )
            break;
        for ( p = pk; p; p = p->next ) {
            /* ignore comment packets */
            if ( p->pkt->pkttype == PKT_COMMENT )
                continue;
            /* those packets are not intended for the real wolrd */
            if ( p->pkt->pkttype == PKT_RING_TRUST )
                continue;
            /* we never export local signed signatures */
            if ( p->pkt->pkttype == PKT_SIGNATURE &&
                 !p->pkt->pkt.signature->flags.exportable )
                continue;
            rc = cdk_pkt_build( out, p->pkt );
            printf("pkttype %d rc=%d\n", p->pkt->pkttype, rc);
            if ( rc )
                break;
        }
        cdk_kbnode_release( pk ); pk = NULL;
    }

    return rc;
} /* cdk_keydb_export */    

/* internal functions */
CDK_KBNODE
_cdk_keydb_get_pkblock( u32 *keyid )
{
    CDK_KEYDB_HD khd = NULL;
    CDK_KBNODE kb_pk = NULL;
    int idx = 0;
    int rc = 0;

    rc = cdk_keydb_find_idx( 0, &idx );
    if ( rc )
        return NULL;
    khd = cdk_keydb_get_ctx( idx );
    if ( !khd )
        return NULL;
    rc = cdk_keydb_get_bykeyid( khd, keyid, &kb_pk );
    if ( rc )
        kb_pk = NULL;

    return kb_pk;      
} /* _cdk_keydb_get_pkblock */

int
_cdk_keydb_seckey_available( u32 *keyid )
{
    CDK_KBNODE kb_sk = NULL;
    CDK_KEYDB_HD khd = NULL;
    int idx = 0, avail = 0;
    int rc = 0;
    
    rc = cdk_keydb_find_idx( 1, &idx );
    if ( rc )
        return CDKERR_NOKEYRING;
    khd = cdk_keydb_get_ctx( idx );
    if ( !khd )
        return 0;
    avail = cdk_keydb_get_bykeyid( khd, keyid, &kb_sk );
    cdk_kbnode_release( kb_sk );

    return !avail;
} /* _cdk_keydb_seckey_available */
  
  
