/* keylist.c - Linked key lists
 *  Copyright (C) 2002 Timo Schulz
 *  Copyright (C) 1998, 1999, 2000, 2001, 2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"

static int
pklist_key_is_present( CDK_KEYLIST pkl, cdkPKT_public_key *pk )
{
    for ( ; pkl; pkl = pkl->next) { 
        if ( _cdk_compare_pks( pkl->key.pk, pk ) == 0 )
            return 0;
    }
  
    return -1;
} /* cdk_pklist_key_is_present */

static int
pklist_get_pk( const char *remusr, cdkPKT_public_key **ret_pk )
{
    if ( !remusr || !ret_pk )
        return CDKERR_INV_VALUE;

    return cdk_keydb_get_pk_byname( remusr, ret_pk, 1 );
} /* pklist_get_pk */

static int
pklist_select_mdc( CDK_KEYLIST pkl )
{
    CDK_KEYLIST pkr;
    int mdc = 0;
  
    if( !pkl )
        return CDKERR_INV_VALUE;

    for ( pkr = pkl; pkr; pkr = pkr->next ) {
        if ( pkr->key.pk->uid ) /* selected by user ID */
            mdc = pkr->key.pk->uid->mdc_feature;
        if ( !mdc )
            return 0; /* at least on recipeint does not support it */
    }
  
    return 1; /* can be used */
} /* pklist_select_mdc */

static int
algo_available( int preftype, int algo )
{
    if( preftype == PREFTYPE_SYM )
        return algo && !openpgp_cipher_test_algo( algo );
    else if( preftype == PREFTYPE_HASH )
        return algo && !openpgp_md_test_algo( algo );
    else if( preftype == PREFTYPE_ZIP )
        return !algo || algo == 1 || algo == 2;
    else
        return 0;
} /* algo_available */

static int
pklist_select_algo( CDK_KEYLIST pkl, int preftype )
{
    CDK_KEYLIST pkr;
    u32 bits[8];
    const cdk_prefitem_t *prefs;
    int compr_hack = 0, any = 0;
    int i = 0, j = 0;
  
    if ( !pkl )
        return -1;

    memset( bits, ~0, 8 * sizeof *bits );
    for ( pkr = pkl; pkr; pkr = pkr->next ) {
        u32 mask[8];
        if ( preftype == PREFTYPE_SYM ) { 
            memset(mask, 0, 8 * sizeof *mask);
            mask[0] |= (1<<2); /* 3DES is implicitly there for everyone else */
        }
        if ( pkr->key.pk->uid )
            prefs = pkr->key.pk->uid->prefs;
        else
            prefs = pkr->key.pk->prefs;
        any = 0;
        for ( i=0; prefs && prefs[i].type; i++ ) {
            if ( prefs[i].type == preftype ) {
                mask[prefs[i].value/32] |= 1 << (prefs[i].value%32);
                any = 1; 
            }
        }
        if ( ( !prefs || !any ) && preftype == PREFTYPE_ZIP ) {
            mask[0] |= 3; /* asume no_compression and old pgp */
            compr_hack = 1;
        }
        for ( i=0; i < 8; i++ )
            bits[i] &= mask[i];
        /* Usable algorithms are now in bits:
           We now use the last key from pkl to select the algorithm we want
           to use. There are no preferences for the last key, we select the one
           corresponding to first set bit. */
        i = -1;
        any = 0;
        for ( j = 0; prefs && prefs[j].type; j++ ) {
            if ( prefs[j].type == preftype ) {
                if ( (bits[prefs[j].value/32] & (1<<(prefs[j].value%32))) ) {
                    if ( algo_available( preftype, prefs[j].value ) ) {
                        any = 1;
                        i = prefs[j].value;
                        break;
                    }
                }
            }
        }
        if ( !prefs || !any ) {
            for ( j = 0; j < 256; j++ )
                if ( (bits[j/32] & (1<<(j%32))) ) {
                    if ( algo_available( preftype, j ) ) {
                        i = j;
                        break; 
                    } 
                }
        }
        if ( compr_hack && !i ) {
            /* selected no compression, but we should check whether
               algorithm 1 is also available (the ordering is not relevant
               in this case). */
            if( bits[0] & (1<<1) )
                i = 1; /* yep; we can use compression algo 1 */
        }
    }
  
    return i;
} /* pklist_select_algo */

static int
is_duplicated_entry( CDK_STRLIST list, CDK_STRLIST item )
{
    for (; list && list != item; list = list->next) {
        if ( !strcmp( list->d, item->d ) )
            return 1;
    }
  
    return 0;
} /* is_duplicated_entry */

int
cdk_pklist_select_algo( CDK_KEYLIST pkl )
{
    int algo = 0;
    
    if ( !pkl )
        return OPENPGP_DEF_CIPHER;

    algo = pklist_select_algo( pkl, PREFTYPE_SYM );
    if ( algo == -1 )
        algo = OPENPGP_DEF_CIPHER;

    return algo;
} /* cdk_pklist_select_algo */

int
cdk_pklist_use_mdc( CDK_KEYLIST pkl )
{
    if ( !pkl )
        return 0;
    
    return pklist_select_mdc( pkl );
} /* cdk_pklist_use_mdc */

/**
 * cdk_pklist_release - Release the PK list.
 * @pkl: public key list
 *
 * Release this object.
 **/
void
cdk_pklist_release( CDK_KEYLIST pkl )
{
    CDK_KEYLIST pkr;
  
    for( ; pkl; pkl = pkr ) {
        pkr = pkl->next;
        cdk_pk_release( pkl->key.pk );
        pkl->key.pk = NULL;
        cdk_free( pkl );
    }
} /* cdk_pklist_release */

/**
 * cdk_pklist_build - Build a public key list.
 * @remusr: The recipients in a linked string list.
 * @ret_pkl: The context to save the list.
 * @use: public key usage.
 *
 * Build a public key list which depends on all recipients in
 * the string list.
 **/
int
cdk_pklist_build( CDK_STRLIST remusr, CDK_KEYLIST *ret_pkl, int use )
{
    CDK_KEYLIST pkl = NULL, r = NULL;
    cdkPKT_public_key *pk = NULL;
    int equal = 0;
    int rc = 0;

    for ( ; remusr; remusr = remusr->next ) {
        pk = cdk_alloc_clear( sizeof *pk );
        if ( !pk ) { 
            return CDKERR_OUT_OF_CORE;
            goto leave;
        }
        pk->pubkey_usage = use;
        rc = pklist_get_pk( remusr->d, &pk );
        if ( rc )
            goto leave;
        else {
            equal = pklist_key_is_present( pkl, pk );
            if ( equal == 0 ) {
                cdk_pk_release( pk ); pk = NULL;
                continue; /* key already in list so skip it */
            }
            r = cdk_alloc( sizeof *r );
            if ( !r ) {              
                rc = CDKERR_OUT_OF_CORE;
                goto leave;
            }
            r->type = PKT_PUBLIC_KEY;
            r->key.pk = pk; /*pk = NULL;*/
            r->next = pkl;
            pkl = r;
        }
    }
leave:
    if ( rc ) { 
        cdk_pklist_release( pkl );
        pkl = NULL;
    }
    *ret_pkl = pkl;
  
    return rc;
} /* cdk_pklist_build */

/**
 * cdk_pklist_encrypt - Encrypt the given public key list.
 * @pkl: The public key list.
 * @dek: The session key.
 * @out: The output file.
 *
 * Encrypt the session key with each recipient in the public key
 * list and store the packets in the output file.
 **/
int
cdk_pklist_encrypt( CDK_KEYLIST pkl, CDK_DEK *dek, CDK_IOBUF outp )
{
    cdkPKT_public_key *pk = NULL;
    cdkPKT_pubkey_enc *enc = NULL;
    CDK_PACKET pkt;
    GCRY_MPI frame = NULL;
    int nbits = 0;
    int rc = 0;

    if ( !pkl || !dek || !outp )
        return CDKERR_INV_VALUE;

    if ( pkl->type != PKT_PUBLIC_KEY )
        return CDKERR_INV_MODE;
    
    for ( ; pkl; pkl = pkl->next ) {
        pk = pkl->key.pk;
        cdk_free( enc ); enc = NULL;
        enc = cdk_alloc_clear( sizeof *enc );
        if ( !enc )
            return CDKERR_OUT_OF_CORE;
        enc->pubkey_algo = pk->pubkey_algo;
        cdk_pk_get_keyid( pk, enc->keyid );
        nbits = cdk_pk_get_nbits( pk );
        rc = _cdk_encode_sesskey( &frame, dek, nbits );
        if ( rc ) {
            cdk_free( enc );
            return rc;
        }
        rc = cdk_pk_encrypt( pk, enc, frame );
        gcry_mpi_release( frame );
        if ( rc ) {
            cdk_free( enc );
            break;
        }
        else {
            init_packet( &pkt );
            pkt.pkttype = PKT_PUBKEY_ENC;
            pkt.pkt.pubkey_enc = enc;
            rc = cdk_pkt_build( outp, &pkt );
        }
        if ( rc ) {
            cdk_free( enc );
            break;
        }
    }

    return rc;
} /* cdk_pklist_encrypt */

/**
 * cdk_sklist_release - Release the SK list.
 * @skl: secret key list.
 *
 * Relese the object.
 **/
void
cdk_sklist_release( CDK_KEYLIST skl )
{
    CDK_KEYLIST sk_rover = NULL;

    for ( ; skl; skl = sk_rover ) {
        sk_rover = skl->next;
        cdk_sk_release( skl->key.sk );
        skl->key.sk = NULL;
        cdk_free( skl );
    }
} /* cdk_sklist_release */

/**
 * cdk_sklist_build - Build a secret key list.
 * @locusr: A list of all names which keys shall be used.
 * @ret_skl: The context to save the list.
 * @unlock: Ask for a passphrase to unlock the keys.
 * @use: Public key usage.
 *
 * Build a SK list based on the names in @locusr.
 **/
int
cdk_sklist_build( CDK_STRLIST locusr, CDK_KEYLIST *ret_skl,
                  int unlock, unsigned int use )
{
    CDK_KEYLIST r = NULL, skl = NULL;
    cdkPKT_secret_key *sk = NULL;
    int rc = 0;

    if ( !ret_skl )
        return CDKERR_INV_VALUE;

    if ( !locusr ) { /* use the default one */
        sk = cdk_alloc_clear( sizeof *sk );
        if ( !sk )
            return CDKERR_OUT_OF_CORE;
        sk->req_usage = use;
        rc = cdk_keydb_get_sk_byname( NULL, &sk, NULL, 1, 1 );
        if ( rc ) { 
            cdk_sk_release( sk );
            sk = NULL;
        }
        else {
            r = cdk_alloc( sizeof *r );
            if ( !r )
                return CDKERR_OUT_OF_CORE;
            r->key.sk = sk; /*sk = NULL;*/
            r->next = skl;
            r->mark = 0;
            r->type = PKT_SECRET_KEY;
            skl = r;
        }
    }
    else {
        CDK_STRLIST locusr_orig = locusr;
        for ( ; locusr; locusr = locusr->next ) {
            if ( is_duplicated_entry( locusr_orig, locusr ) )
                continue;          
            sk = cdk_alloc_clear( sizeof *sk );
            if ( !sk )
                return CDKERR_OUT_OF_CORE;
            sk->req_usage = use;
            rc = cdk_keydb_get_sk_byname( locusr->d, &sk, NULL, 0, 1 );
            if ( rc ) {
                cdk_sk_release( sk );
                sk = NULL;
            }
            else {
                r = cdk_alloc_clear( sizeof *r );
                if ( !r )
                    return CDKERR_OUT_OF_CORE;
                r->key.sk = sk; /*sk = NULL;*/
                r->next = skl;
                r->mark = 0;
                skl = r;
            }
        }
    }

    if ( rc ) {    
        cdk_sklist_release( skl );
        skl = NULL;
    }

    *ret_skl = skl;
    return rc;
} /* cdk_sklist_build */

int
cdk_sklist_write_onepass( CDK_KEYLIST skl, CDK_IOBUF outp, int sigclass )
{
    cdkPKT_onepass_sig *ops;
    CDK_KEYLIST r;
    CDK_PACKET pkt;
    int i, skcount = 0;
    int rc = 0;
  
    if ( !skl || !outp )
        return CDKERR_INV_VALUE;

    for ( skcount = 0, r = skl; r; r = r->next )
        skcount++;

    for ( ; skcount; skcount-- ) {
        for ( i = 0, r = skl; r; r = r->next ) {
            if ( ++i == skcount )
                break;           
        }
        ops = cdk_alloc_clear( sizeof *ops );
        if ( !ops )
            return CDKERR_OUT_OF_CORE;

        ops->version = 3;
        cdk_sk_get_keyid( r->key.sk, ops->keyid );
        ops->sig_class = sigclass;
        ops->digest_algo = _cdk_sig_hash_for( r->key.sk->pubkey_algo,
                                              r->key.sk->version );
        ops->pubkey_algo = r->key.sk->pubkey_algo;
        ops->last = (skcount == 1);
  
        init_packet( &pkt );
        pkt.pkttype = PKT_ONEPASS_SIG;
        pkt.pkt.onepass_sig = ops;
        rc = cdk_pkt_build( outp, &pkt );
        cdk_pkt_free( &pkt );
        if ( rc )
            return rc;
    }

    return rc;
} /* cdk_sklist_write_onepass */

/**
 * cdk_sklist_write - Write signature list to file.
 * @skl: secret key list.
 * @outp: the output file.
 * @hash: hash context.
 * @sigclass: signature class.
 *
 * Create a signature for each secret key inside the list and
 * write to the file.
 **/
int
cdk_sklist_write( CDK_KEYLIST skl, CDK_IOBUF outp, GCRY_MD_HD hash,
                  int sigclass )
{
    CDK_KEYLIST r = NULL;
    cdkPKT_signature *sig = NULL;
    GCRY_MD_HD md;
    CDK_PACKET pkt;
    byte digest[24]; /* maximal 192-bit for TIGER-192 */
    int rc = 0;

    if ( !skl || !outp || !hash )
        return CDKERR_INV_VALUE;

    for ( r = skl; r; r = r->next ) {
        sig = cdk_sig_new();
        cdk_sig_create( r->key.sk->pk, sig );
        sig->sig_class = sigclass;
        md = gcry_md_copy( hash );
        cdk_hash_sig_result( sig, md );
        gcry_md_final( md );
      
        memcpy( digest, gcry_md_read( md, sig->digest_algo),
                gcry_md_get_algo_dlen( sig->digest_algo ) );
        rc = cdk_pk_sign( r->key.sk, sig, digest );
        if ( rc )
            return rc;
        init_packet( &pkt );
        pkt.pkttype = PKT_SIGNATURE;
        pkt.pkt.signature = sig;
        rc = cdk_pkt_build( outp, &pkt );
        if ( rc ) {
            cdk_sig_release( sig );
            return rc;
        }
        cdk_sig_release( sig );
        gcry_md_close( md );
    }
  
    return rc;
} /* cdk_sklist_write */

