/* misc.c
 *  Copyright (C) 2002 Timo Schulz
 *  Copyright (C) 1998-2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "opencdk.h"
#include "main.h"

u32
make_timestamp(void)
{
    return (u32)time(NULL);
} /* make_timestamp */

u32
buffer_to_u32( const byte *buffer )
{
    u32 u = 0;
    const byte *p;
  
    if ( !buffer )
        return 0;
  
    p = buffer;
    u = (p[0] << 24) | (p[1] << 16) | (p[2] << 8) | p[3];
  
    return u;    
} /* buffer_to_u32 */

void
u32_to_buffer( u32 u, byte *buffer )
{
    if ( !buffer )
        return;

    buffer[0] = u >> 24;
    buffer[1] = u >> 16;
    buffer[2] = u >>  8;
    buffer[3] = u      ;
} /* u32_to_buffer */

int
openpgp_cipher_test_algo( int algo )
{
    if ( algo < 0 || algo > 110 )
        return CDKERR_INV_ALGO;  
    return gcry_cipher_test_algo(algo);
} /* openpgp_cipher_test_algo */

int
openpgp_pk_test_algo( int algo, unsigned int usage_flags )
{
    size_t n = usage_flags;
  
    if ( algo < 0 || algo > 110 )
        return GCRYERR_INV_PK_ALGO;
    return gcry_pk_algo_info( algo, GCRYCTL_TEST_ALGO, NULL, &n );      
} /* openpgp_pk_test_algo */

int
openpgp_pk_algo_usage( int algo )
{
    int usage = 0;
  
    switch ( algo ) {
    case GCRY_PK_RSA:
        usage = GCRY_PK_USAGE_SIGN | GCRY_PK_USAGE_ENCR;
        break;
    case GCRY_PK_RSA_E:
        usage = GCRY_PK_USAGE_ENCR;
        break;
    case GCRY_PK_RSA_S:
        usage = GCRY_PK_USAGE_SIGN;
        break;
    case GCRY_PK_ELG:
        usage = GCRY_PK_USAGE_SIGN | GCRY_PK_USAGE_ENCR;
        break;
    case GCRY_PK_ELG_E:
        usage = GCRY_PK_USAGE_ENCR;
        break;
    case GCRY_PK_DSA:
        usage = GCRY_PK_USAGE_SIGN;
        break;
    }
  
    return usage;
} /* openpgp_pk_algo_usage */

int
openpgp_md_test_algo( int algo )
{
    if ( algo < 0 || algo > 110 )
        return GCRYERR_INV_MD_ALGO;

    return gcry_md_test_algo(algo);      
} /* openpgp_md_test_algo */

void
cdk_md_close( GCRY_MD_HD *r_hd )
{
    if ( !r_hd || *r_hd == NULL )
        return;
    gcry_md_close( *r_hd );
    *r_hd = NULL;
} /* cdk_md_close */

void
cdk_cipher_close( GCRY_CIPHER_HD *r_hd )
{
    if ( !r_hd || *r_hd == NULL )
        return;
    gcry_cipher_close( *r_hd );
    *r_hd = NULL;
} /* cdk_cipher_close */

int
openpgp_get_opt_val( GCRY_SEXP opt, int id )
{
    GCRY_SEXP s1, s2;
    const char *s, *p;
    size_t n = 0;
    int set = 0;

    if ( !opt )
        return 0;
  
    if ( id == OPENPGP_ARMOR ) s = "armor";
    else if ( id == OPENPGP_TEXTMODE ) s = "textmode";
    else if ( id == OPENPGP_COMPRESS ) s = "compress";
    else if ( id == OPENPGP_DIGEST ) s = "digest";
    else if ( id == OPENPGP_CIPHER ) s = "cipher";
    else return 0;
    s1 = gcry_sexp_find_token( opt, s, strlen(s) );
    if ( !s1 )
        return 0; /* didn't find */
    p = gcry_sexp_nth_data( s1, 0, &n );
    if ( p ) {
        s2 = gcry_sexp_cdr( s1 );
        if ( !s2 )
            return 0;
        p = gcry_sexp_nth_data( s2, 0, &n );
        if ( p && n == 1 && *p == '1' )
            set = 1;                      
    }
  
    return set;   
} /* openpgp_get_opt_val */

static const char*
parse_version_number( const char *s, int *number )
{
    int val = 0;

    if( *s == '0' && isdigit(s[1]) )
        return NULL;
    /* leading zeros are not allowed */
    for ( ; isdigit(*s); s++ ) {
        val *= 10;
        val += *s - '0';     
    }
    *number = val;
    return val < 0? NULL : s;      
}

static const char *
parse_version_string( const char *s, int *major, int *minor, int *micro )
{
    s = parse_version_number( s, major );
    if( !s || *s != '.' )
        return NULL;
    s++;
    s = parse_version_number( s, minor );
    if( !s || *s != '.' )
        return NULL;
    s++;
    s = parse_version_number( s, micro );
    if( !s )
        return NULL;
    return s; /* patchlevel */
}

/**
 * cdk_check_version - Version control handling.
 * @req_version: The requested version
 *
 * Check that the the version of the library is at minimum the requested
 * one and return the version string; return NULL if the condition is
 * not satisfied.  If a NULL is passed to this function, no check is done,
 *but the version string is simply returned.
 **/
const char *
cdk_check_version( const char *req_version )
{
    const char *ver = VERSION;
    int my_major, my_minor, my_micro;
    int rq_major, rq_minor, rq_micro;
    const char *my_plvl, *rq_plvl;
  
    if ( !req_version )
        return ver;
    my_plvl = parse_version_string( ver, &my_major, &my_minor, &my_micro );
    if ( !my_plvl )
        return NULL;
    /* very strange our own version is bogus */
    rq_plvl = parse_version_string( req_version, &rq_major, &rq_minor,
                                    &rq_micro );
    if ( !rq_plvl )
        return NULL;  /* req version string is invalid */
    if ( my_major > rq_major
         || (my_major == rq_major && my_minor > rq_minor)
         || (my_major == rq_major && my_minor == rq_minor
             && my_micro > rq_micro)
         || (my_major == rq_major && my_minor == rq_minor
             && my_micro == rq_micro
             && strcmp( my_plvl, rq_plvl ) >= 0) ) {
        return ver;    
    }
    return NULL;
} /* cdk_check_version */

/**
 * cdk_strlist_free - Free a stringlist object.
 * @sl - the stringlist.
 *
 * Free the object.
 **/
void
cdk_strlist_free( CDK_STRLIST sl )
{
    CDK_STRLIST sl2;

    if (!sl)
        return;
  
    for(; sl; sl = sl2 ) {
        sl2 = sl->next;
        cdk_free(sl);
    }
} /* cdk_strlist_free */

/**
 * cdk_strlist_add - Add an entry to the stringlist.
 * @list: the stringlist.
 * @string: the text to add.
 *
 * Add an entry.
 **/
CDK_STRLIST
cdk_strlist_add( CDK_STRLIST *list, const char *string )
{
    CDK_STRLIST sl;

    if ( !list || !string )
        return NULL;
  
    sl = cdk_alloc( sizeof *sl + strlen(string));
    if ( !sl )
        return NULL;
    sl->flags = 0;
    strcpy(sl->d, string);
    sl->next = *list;
    *list = sl;
  
    return sl;  
} /* cdk_strlist_add */

/**
 * cdk_strlist_append - Append an entry to the stringlist.
 * @list: the stringlist.
 * @string:
 *
 * Append an entry. If @list is not initialized, a new list
 * is allocated.
 **/
CDK_STRLIST
cdk_strlist_append( CDK_STRLIST *list, const char *string )
{
    CDK_STRLIST r, sl;

    if ( !string )
        return NULL;
  
    sl = cdk_alloc( sizeof *sl + strlen(string));
    if ( !sl )
        return NULL;
    sl->flags = 0;
    strcpy(sl->d, string);
    sl->next = NULL;
    if( !*list )
        *list = sl;
    else {
        for ( r = *list; r->next; r = r->next )
            ;
        r->next = sl; 
    }
    return sl;      
} /* cdk_strlist_append */

CDK_STRLIST
cdk_strlist_prev( CDK_STRLIST head, CDK_STRLIST node )
{
    CDK_STRLIST n;
  
    for ( n=NULL; head && head != node; head = head->next )
        n = head;
  
    return n;  
} /* cdk_strlist_prev */

CDK_STRLIST
cdk_strlist_last( CDK_STRLIST node )
{
    if (node) { 
        for ( ; node->next ; node = node->next )
            ;
    }
  
    return node;
} /* cdk_strlist_last */

static int
ascii_toupper(int c)
{
    if (c >= 'a' && c <= 'z')
        c &= ~0x20;
  
    return c;   
} /* ascii_toupper */

const char *
ascii_memistr( const char *buf, size_t buflen, const char *sub )
{
    const byte *t, *s;
    size_t n;

    for ( t=buf, n=buflen, s=sub ; n ; t++, n-- ) {
        if ( ascii_toupper(*t) == ascii_toupper(*s) ) {
            for( buf=t++, buflen = n--, s++;
                 n && ascii_toupper(*t) == ascii_toupper(*s); t++, s++, n-- )
                ;
            if( !*s )
                return buf;
            t = buf;
            n = buflen;
            s = sub ;                        
        }
    }

    return NULL ;
} /* ascii_memistr */

int
ascii_strcasecmp( const char *a, const char *b )
{
    if ( !a || !b )
        return 1;
  
    if ( a == b )
        return 0;

    for (; *a && *b; a++, b++) {
        if (*a != *b && ascii_toupper(*a) != ascii_toupper(*b))
            break;         
    }
  
    return *a == *b? 0 : (ascii_toupper (*a) - ascii_toupper (*b));      
} /* ascii_strcasecmp */

static unsigned int
trim_trailing_chars( byte *line, unsigned len, const char *trimchars )
{
    byte *p, *mark;
    unsigned n;

    for ( mark = NULL, p = line, n = 0; n < len; n++, p++ ) {
        if ( strchr(trimchars, *p ) ) {
            if( !mark )
                mark = p;
        }
        else
            mark = NULL;
    }
    if ( mark ) {
        *mark = 0;
        return mark - line; 
    }
  
    return len;
} /* trim_trailing_chars */

/* remove trailing white spaces and return the length of the buffer */
unsigned
trim_trailing_ws( byte *line, unsigned len )
{
    return trim_trailing_chars( line, len, " \t\r\n" ); 
} /* trim_trailing_ws */
