/* opencdk.h - Open Crypto Development Kit (OpenCDK)
 *   Copyright (C) 2001, 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef OPENCDK_H
#define OPENCDK_H

#include <gcrypt.h>

#ifndef HAVE_BYTE_TYPEDEF
  #undef byte
  typedef unsigned char byte;
  #define HAVE_BYTE_TYPEDEF
#endif

#ifndef HAVE_U16_TYPEDEF
  #undef u16
  typedef unsigned short u16;
  #define HAVE_U16_TYPEDEF
#endif

#ifndef HAVE_U32_TYPEDEF
  #undef u32
  typedef unsigned int u32;
  #define HAVE_U32_TYPEDEF
#endif

#define OPENCDK_VERSION "0.2.0"

/*********************
 * Algorithm constants
 *********************/
#define OPENPGP_DEF_CIPHER GCRY_CIPHER_CAST5
#define OPENPGP_DEF_MODE GCRY_CIPHER_MODE_CFB
#define OPENPGP_DEF_MD GCRY_MD_SHA1
#define OPENPGP_DEF_S2K 3

/***************
 * MPI constants
 ***************/
#define MAX_MPI_BITS 8192
#define MAX_MPI_BYTES (MAX_MPI_BITS/8)

/*********************
 * Signature constants
 *********************/
#define IS_UID_SIG(s) ( ( (s)->sig_class & ~3 ) == 0x10 )
#define IS_UID_REV(s) ( (s)->sig_class == 0x30 )

/*******************
 * General contexts
 *******************/
typedef struct cdk_string_list {
    struct cdk_string_list *next;
    unsigned int flags;
    char d[1];  
} *CDK_STRLIST;

typedef struct cdk_bstring_s {
    unsigned int len;
    byte d[1];
} *CDK_BSTRING;

typedef struct {
    u16 bits;
    u16 bytes;
    byte data[1];
} CDK_MPI;

typedef struct {
    int algo;
    int keylen;
    int use_mdc;
    byte key[32]; /* 256-bit */
} CDK_DEK;

typedef struct {
    int mode;
    byte hash_algo;
    byte salt[8];
    u32 count;  
} CDK_STRING2KEY;

typedef struct {
    byte type;
    byte value;
} cdk_prefitem_t;

struct cdk_io_buf_s;
typedef struct cdk_io_buf_s *CDK_IOBUF;

/****************
 * Error routines
 ****************/
typedef enum {
    CDKERR_SUCCESS = 0,
    CDKERR_GENERAL = 1,
    CDKERR_FILE_OPEN = 2,
    CDKERR_FILE_CREATE = 24,
    CDKERR_FILE_READ = 3,
    CDKERR_FILE_WRITE = 4,
    CDKERR_BAD_SIGNATURE = 5,
    CDKERR_INV_PACKET = 6,
    CDKERR_INV_ALGO = 7,
    CDKERR_NOT_IMPLEMENT = 9,
    CDKERR_GCRY = 10,
    CDKERR_ARMOR = 11,
    CDKERR_ARMOR_CRC = 12,
    CDKERR_MPI = 13,
    CDKERR_INV_VALUE = 14,
    CDKERR_NOKEY = 15,
    CDKERR_CHECKSUM = 16,
    CDKERR_EOF = 17,
    CDKERR_TIME_CONFLICT = 18,
    CDKERR_ZLIB = 19,
    CDKERR_WEAK_KEY = 20,
    CDKERR_FILTER = 21,
    CDKERR_OUT_OF_CORE = 22,
    CDKERR_WRONG_SECKEY = 23,
    CDKERR_BADMDC = 25,
    CDKERR_INV_MODE = 26,
    CDKERR_NOKEYRING = 27
} cdk_err_t;

const char* cdk_strerror(int ec);

/*************
 * Preferences
 *************/
typedef enum {
    PREFTYPE_NONE = 0,
    PREFTYPE_SYM = 1,
    PREFTYPE_HASH = 2,
    PREFTYPE_ZIP = 3
} cdk_prefs_t;

/*****************
 * Packet routines
 *****************/
typedef enum {
    PKT_RESERVED = 0,
    PKT_PUBKEY_ENC = 1,
    PKT_SIGNATURE = 2,
    PKT_SYMKEY_ENC = 3,
    PKT_ONEPASS_SIG = 4,
    PKT_SECRET_KEY = 5,
    PKT_PUBLIC_KEY = 6,
    PKT_SECRET_SUBKEY = 7,
    PKT_COMPRESSED = 8,
    PKT_ENCRYPTED = 9,
    PKT_MARKER = 10,
    PKT_PLAINTEXT = 11,
    PKT_RING_TRUST = 12,
    PKT_USER_ID = 13,
    PKT_PUBLIC_SUBKEY = 14,
    PKT_OLD_COMMENT = 16,
    PKT_ATTRIBUTE = 17,
    PKT_ENCRYPTED_MDC = 18,
    PKT_MDC = 19,
    PKT_COMMENT = 61
} cdk_packet_types_t;

struct cdk_subpkt_s {
    struct cdk_subpkt_s *next;
    u32 size;
    byte type;
    byte *data;
};
typedef struct cdk_subpkt_s *CDK_SUBPKT;

typedef struct {
    u32 len;
    int is_primary;
    int is_revoked;
    cdk_prefitem_t *prefs;
    byte *attrib_img; /* TAG 17 if not null */
    size_t attrib_len;
    size_t prefs_size;
    int mdc_feature;
    u32 created;
    char name[1];
} cdkPKT_user_id;

typedef struct {
    byte version;
    byte pubkey_algo;
    u32 keyid[2];
    u32 main_keyid[2];
    u32 timestamp;
    u16 expiredate;
    CDK_MPI *mpi[4];
    int is_revoked;
    int is_invalid;
    int has_expired;
    int pubkey_usage;
    cdkPKT_user_id *uid;
    cdk_prefitem_t *prefs;
    size_t prefs_size;
} cdkPKT_public_key;

typedef struct {
    cdkPKT_public_key *pk;
    u16 expiredate;
    int version;
    int pubkey_algo;
    u32 keyid[2];
    u32 main_keyid[2];
    byte s2k_usage;
    struct {
        byte algo;
        byte sha1chk; /* SHA1 is used instead of a 16 bit checksum */
        CDK_STRING2KEY s2k;
        byte iv[16];
        byte ivlen;
    } protect;
    u16 csum;
    CDK_MPI *mpi[4];
    byte *encdata;
    size_t enclen;
    byte is_protected;
    int is_primary;
    int has_expired;
    int is_revoked;
    int req_usage;
} cdkPKT_secret_key;

typedef struct {
    byte version;
    byte sig_class;
    u32 timestamp;
    u32 expiredate;
    u32 keyid[2];
    byte pubkey_algo;
    byte digest_algo;
    byte digest_start[2];  
    u16 hashed_size;
    CDK_SUBPKT hashed;
    u16 unhashed_size;
    CDK_SUBPKT unhashed;
    CDK_MPI *mpi[2];
    struct {
        unsigned exportable:1;
        unsigned revocable:1;
        unsigned policy_url:1;
        unsigned notation:1;
        unsigned expired:1;
        unsigned checked:1;
        unsigned valid:1;
    } flags;
  
    u32 key[2]; /* only valid for key signatures */
} cdkPKT_signature;

typedef struct {
    byte version;
    u32 keyid[2];
    byte sig_class;
    byte digest_algo;
    byte pubkey_algo;
    byte last; /* a stupid flag */
} cdkPKT_onepass_sig;

typedef struct {
    byte version;
    u32 keyid[2];
    int throw_keyid;
    byte pubkey_algo;
    CDK_MPI *mpi[2];
} cdkPKT_pubkey_enc;

typedef struct {
    byte version;
    byte cipher_algo;
    CDK_STRING2KEY s2k;
    byte seskeylen;
    byte seskey[32];
} cdkPKT_symkey_enc;

typedef struct {
    u32 len;
    int extralen;
    byte mdc_method;
    CDK_IOBUF buf;
} cdkPKT_encrypted;

typedef struct {
    byte hash[20];
} cdkPKT_mdc;

typedef struct {
    int len;
    char data[1];
} cdkPKT_comment;

typedef struct {
    u32 len;
    CDK_IOBUF buf;
    int mode;
    u32 timestamp;
    int namelen;
    char name[1];
} cdkPKT_plaintext;

typedef struct {
    int len;
    int algorithm;
    CDK_IOBUF buf;
} cdkPKT_compressed;

struct cdk_packet_struct {
    int pktlen; /* real packet length */
    int pktsize; /* length with all headers */
    cdk_packet_types_t pkttype;
    union {
        cdkPKT_mdc *mdc;
        cdkPKT_user_id *user_id;
        cdkPKT_public_key *public_key;
        cdkPKT_secret_key *secret_key;
        cdkPKT_signature *signature;
        cdkPKT_pubkey_enc *pubkey_enc;
        cdkPKT_symkey_enc *symkey_enc;
        cdkPKT_compressed *compressed;
        cdkPKT_encrypted *encrypted;
        cdkPKT_comment *comment;
        cdkPKT_plaintext *plaintext;
        cdkPKT_onepass_sig *onepass_sig;
    } pkt;
};
typedef struct cdk_packet_struct CDK_PACKET;

struct cdk_kbnode_struct {
    struct cdk_kbnode_struct *next;
    CDK_PACKET *pkt;
    int flag;
    int private_flag; 
};
typedef struct cdk_kbnode_struct *CDK_KBNODE;

int cdk_pkt_new( CDK_PACKET **r_pkt );
int cdk_pkt_alloc( CDK_PACKET **r_pkt, int id );
void cdk_pkt_free( CDK_PACKET *pkt );
void cdk_pkt_release( CDK_PACKET *pkt );
int cdk_pkt_parse( CDK_IOBUF inp, CDK_PACKET *pkt );
int cdk_pkts_parse( CDK_IOBUF inp, CDK_KBNODE *r_pkt );
int cdk_pkt_build( CDK_IOBUF out, CDK_PACKET *pkt );

/*********************
 * Public key routines
 *********************/
#define is_RSA(a) ((a) == GCRY_PK_RSA \
                   || (a) == GCRY_PK_RSA_E \
                   || (a) == GCRY_PK_RSA_S)
#define is_ELG(a) ((a) == GCRY_PK_ELG || (a) == GCRY_PK_ELG_E)
#define is_DSA(a) ((a) == GCRY_PK_DSA)

int cdk_pk_encrypt( cdkPKT_public_key *pk, cdkPKT_pubkey_enc *pke,
                    GCRY_MPI esk );
int cdk_pk_decrypt( cdkPKT_secret_key *sk, cdkPKT_pubkey_enc *pke,
                    GCRY_MPI *r_sk );
int cdk_pk_sign( cdkPKT_secret_key *sk, cdkPKT_signature *sig,
                 const byte *md );
int cdk_pk_verify( cdkPKT_public_key *pk, cdkPKT_signature *sig,
                   const byte *md );

int cdk_pk_get_nbits( cdkPKT_public_key *pk );
int cdk_pk_get_npkey( int algo );
int cdk_pk_get_nskey( int algo );
int cdk_pk_get_nsig( int algo );
int cdk_pk_get_nenc( int algo );

/*********************
 * Secret key routines
 *********************/
int cdk_seckey_unprotect( cdkPKT_secret_key *sk, char *pw );
CDK_DEK* cdk_passphrase_to_dek( int cipher_algo, CDK_STRING2KEY *s2k,
                                int mode, char *pw );

/*************************
 * Message digest routines
 *************************/
void cdk_hash_public_key( cdkPKT_public_key *pk, GCRY_MD_HD hd );
void cdk_hash_user_id( cdkPKT_user_id *uid, int sig_version, GCRY_MD_HD hd );
void cdk_hash_sig_result( cdkPKT_signature *sig, GCRY_MD_HD hd );

/**********************
 * Compression routines
 **********************/
typedef enum {
    ZIP_UNCOMPRESSED = 0,
    ZIP_ZIP = 1,
    ZIP_ZLIB = 2
} zip_algs_t;

typedef struct {
    int algo; /* compress algo */
    int algo1hack;
} compress_filter_s;

int cdk_compress_filter( compress_filter_s *zfx, int control, CDK_IOBUF buf );

/**********************
 * ASCII armor routines
 **********************/
typedef enum {
    ARMOR_MESSAGE = 0,
    ARMOR_PUBKEY = 1,
    ARMOR_SECKEY = 2,
    ARMOR_SIGNATURE = 3,
    ARMOR_CLEARSIG = 4,
} armor_types_t;

typedef struct {
    const char *le; /* line endings */
    const char *hdrlines;
    u32 crc;
    int crc_okay;
    int idx;
    int idx2;
} armor_filter_s;

typedef struct {
    byte *buffer;
    unsigned buffer_size;
    unsigned buffer_len;
    unsigned buffer_pos;
    int truncated;
    int not_dash_escaped;
    int escape_from;
    GCRY_MD_HD md;
    int pending_lf;
    int pending_esc;
} text_filter_s;

int cdk_armor_filter( armor_filter_s *afx, int control, CDK_IOBUF buf );
int cdk_text_filter( text_filter_s *tfx, int control, CDK_IOBUF buf );
int cdk_armor_file( GCRY_SEXP opts, const char *file, const char *output );
int cdk_armor_filter_use( CDK_IOBUF inp );

/****************
 * IOBUF routines
 ****************/
typedef enum {
    IOBUF_MODE_RD = 1,
    IOBUF_MODE_WR = 2,
    IOBUF_MODE_RDWR = IOBUF_MODE_RD|IOBUF_MODE_WR,
    IOBUF_MODE_EXP = 8,
    /* for the file extensions */
    IOBUF_MODE_OUT = 100,
    IOBUF_MODE_SIG = 101,
    IOBUF_FLAG_TMP = 128
} iobuf_mode_t;

typedef enum {
    IOBUF_CTRL_FLUSH = 3,
    IOBUF_CTRL_UNDERFLOW = 4,
} iobuf_ctrl_t;

int cdk_iobuf_open( CDK_IOBUF *ret_buf, const char *fname, iobuf_mode_t mode );
CDK_IOBUF cdk_iobuf_mopen( const char *fname, int mode, int ext );
int cdk_iobuf_from_mem( CDK_IOBUF *ret_buf, byte *buf, size_t buflen );
int cdk_iobuf_new( CDK_IOBUF *ret_buf, size_t size );
int cdk_iobuf_create( CDK_IOBUF *ret_buf, const char *fname );
int cdk_iobuf_expand( CDK_IOBUF buf, size_t size );
int cdk_iobuf_copy( CDK_IOBUF d, CDK_IOBUF s, int add );
CDK_IOBUF cdk_iobuf_temp( void );
void cdk_iobuf_close( CDK_IOBUF buf );
int cdk_iobuf_store( CDK_IOBUF buf, const char *file, int release );
CDK_BSTRING cdk_iobuf_read_mem( CDK_IOBUF buf, size_t pos );
int cdk_iobuf_write_mem( CDK_IOBUF buf, const byte *d, size_t dlen );
int cdk_iobuf_setmode( CDK_IOBUF buf, int mode );
int cdk_iobuf_eof( CDK_IOBUF buf );
int cdk_iobuf_seek( CDK_IOBUF buf, size_t nbytes );
size_t cdk_iobuf_get_length( CDK_IOBUF buf );
size_t cdk_iobuf_tell( CDK_IOBUF buf );
void cdk_iobuf_rewind( CDK_IOBUF buf );
int cdk_iobuf_skip( CDK_IOBUF buf, size_t nbytes );
int cdk_iobuf_goback( CDK_IOBUF buf, size_t nbytes );
int cdk_iobuf_peek( CDK_IOBUF buf, byte *buffer, size_t buflen );
int cdk_iobuf_read( CDK_IOBUF buf, byte *buffer, size_t buflen, size_t *r_nread );
int cdk_iobuf_read_line( CDK_IOBUF buf, byte *buffer, size_t buflen,
                         size_t *r_nread );
int cdk_iobuf_get_line( CDK_IOBUF buf, byte **buffer, unsigned *length,
                        unsigned *max_length );
int cdk_iobuf_write( CDK_IOBUF buf, const byte *buffer, size_t buflen );
int cdk_iobuf_get( CDK_IOBUF buf );
int cdk_iobuf_put( CDK_IOBUF buf, byte b );

/*********************
 * Key database access
 *********************/
enum {
    KEYDB_SEARCH_EXACT = 1,
    KEYDB_SEARCH_SUBSTR = 2,
    KEYDB_SEARCH_SHORT_KEYID = 3,
    KEYDB_SEARCH_KEYID = 4,
    KEYDB_SEARCH_FPR = 5
};

enum {
    KEYDB_TYPE_KEYRING = 0,
    KEYDB_TYPE_ARMORED = 1,
    KEYDB_TYPE_DATA = 2
};

struct cdk_keydb_handle_s {
    CDK_IOBUF buf; /* NULL if the name item is valid */
    u32 old_offset;
    u32 offset;
    char *name;
    int type;
    int secret;
    int used;
};
typedef struct cdk_keydb_handle_s *CDK_KEYDB_HD;

struct cdk_keydb_search_s {
    union {
        const char *pattern;
        u32 keyid[2];
        u32 short_keyid;
        byte fpr[20];
    } u;
    int type;
};
typedef struct cdk_keydb_search_s CDK_KEYDB_SEARCH;

int cdk_keydb_open( CDK_KEYDB_HD kdb, CDK_IOBUF *ret_kr );     
int cdk_keydb_add_resource( const char *name, int is_secret );
int cdk_keydb_remove_resource( int id );
const char* cdk_keydb_get_name( int id );
int cdk_keydb_is_secret( int id );
int cdk_keydb_find_idx( int is_secret, int *r_pos );
CDK_KEYDB_HD cdk_keydb_get_ctx( int id );
int cdk_keydb_search( CDK_KEYDB_HD kdb, CDK_KEYDB_SEARCH *ks,
                      CDK_KBNODE *r_key );
int cdk_keydb_get_bykeyid( CDK_KEYDB_HD kdb, u32 *keyid, CDK_KBNODE *r_key );
int cdk_keydb_get_byfpr( CDK_KEYDB_HD kdb, const byte *fpr,
                         CDK_KBNODE *r_key );
int cdk_keydb_get_bypattern( CDK_KEYDB_HD kdb, const char *patt,
                             CDK_KBNODE *r_pkt );
int cdk_keydb_get_pk( CDK_KEYDB_HD khd, u32 *keyid,
                      cdkPKT_public_key **ret_pk );
int cdk_keydb_sk_get( CDK_KEYDB_HD khd, u32 *keyid,
                      cdkPKT_secret_key **ret_sk );
int cdk_keydb_get_pk_byname( const char *name, cdkPKT_public_key **ret_pk,
                             int alloced );
int cdk_keydb_get_sk_byname( const char *name, cdkPKT_secret_key **ret_sk,
                             char *pw, int unlock, int alloced );
int cdk_keydb_get_keyblock( CDK_IOBUF inp, CDK_KBNODE *ret_key, int *ret_eof );
int cdk_keydb_enum_keyblocks( CDK_IOBUF inp, CDK_KBNODE *ret_key );
int cdk_keydb_export( CDK_IOBUF out, CDK_STRLIST remusr );    
     
CDK_KBNODE cdk_kbnode_new( CDK_PACKET *pkt );
void cdk_kbnode_release( CDK_KBNODE n );
CDK_KBNODE cdk_kbnode_find( CDK_KBNODE node, int pkttype );
CDK_KBNODE cdk_kbnode_find_prev( CDK_KBNODE root, CDK_KBNODE node,
                                 int pkttype );
CDK_KBNODE cdk_kbnode_find_next( CDK_KBNODE node, int pkttype );

/**************
 * Key routines
 **************/
enum {
    CDK_KEY_VALID = 0,
    CDK_KEY_INVALID = 1, /* missing or wrong self signature */
    CDK_KEY_EXPIRED = 2,
    CDK_KEY_REVOKED = 4
};

int cdk_pk_get_fingerprint( cdkPKT_public_key *pk, byte *fpr );
u32 cdk_pk_fingerprint_get_keyid( const byte *fpr, size_t fprlen, u32 *keyid );
u32 cdk_pk_get_keyid( cdkPKT_public_key *pk, u32 *keyid );
u32 cdk_sk_get_keyid( cdkPKT_secret_key *sk, u32 *keyid );
u32 cdk_sig_get_keyid( cdkPKT_signature *sig, u32 *keyid );

int cdk_key_check_sigs( CDK_KBNODE kb_pk, CDK_KEYDB_HD khd, int *r_status );

/********************
 * Signature routines
 ********************/
int cdk_sign_file( GCRY_SEXP opts, CDK_STRLIST locusr, const char *file,
                   const char *output, int detached );
int cdk_verify_file( const char *file, const char *sig_file,
                     cdkPKT_public_key **r_pk );

/********************
 * Subpacket routines
 ********************/
typedef enum {
    SIGSUBPKT_NONE = 0,
    SIGSUBPKT_SIG_CREATED = 2,
    SIGSUBPKT_SIG_EXPIRE = 3,
    SIGSUBPKT_EXPORTABLE = 4,
    SIGSUBPKT_TRUST = 5,
    SIGSUBPKT_REGEXP = 6,
    SIGSUBPKT_REVOCABLE = 7,
    SIGSUBPKT_KEY_EXPIRE = 9,
    SIGSUBPKT_PREFS_SYM = 11,
    SIGSUBPKT_REV_KEY = 12,
    SIGSUBPKT_ISSUER = 16,
    SIGSUBPKT_NOTATION = 20,
    SIGSUBPKT_PREFS_HASH = 21,
    SIGSUBPKT_PREFS_ZIP = 22,
    SIGSUBPKT_KS_FLAGS = 23,
    SIGSUBPKT_PREF_KS = 24,
    SIGSUBPKT_PRIMARY_UID = 25,
    SIGSUBPKT_POLICY = 26,
    SIGSUBPKT_KEY_FLAGS = 27,
    SIGSUBPKT_SIGNERS_UID = 28,
    SIGSUBPKT_REVOC_REASON = 29,
    SIGSUBPKT_FEATURES = 30
} sig_subpkt_types_t;

int cdk_subpkt_new( CDK_SUBPKT *ctx );
void cdk_subpkt_release( CDK_SUBPKT ctx );
struct cdk_subpkt_s* cdk_subpkt_add( CDK_SUBPKT r_ctx );
int cdk_subpkt_init( CDK_SUBPKT ctx, int type, byte *data, u32 size );
struct cdk_subpkt_s* cdk_subpkt_find( CDK_SUBPKT ctx, int type );

/**********************
 * Key certificate list
 **********************/
typedef struct cdk_key_list_s *CDK_KEYLIST;
struct cdk_key_list_s {
    CDK_KEYLIST next;
    union {
        cdkPKT_public_key *pk;
        cdkPKT_secret_key *sk;
    } key;
    int mark;
    int type;
};

int cdk_pklist_select_algo( CDK_KEYLIST pkl );
int cdk_pklist_use_mdc( CDK_KEYLIST pkl );
int cdk_pklist_build( CDK_STRLIST remusr, CDK_KEYLIST *ret_pkl, int use );
void cdk_pklist_release( CDK_KEYLIST pkl );
int cdk_pklist_encrypt( CDK_KEYLIST pkl, CDK_DEK *dek, CDK_IOBUF outp );

int cdk_sklist_build( CDK_STRLIST locusr, CDK_KEYLIST *ret_skl, int unlock,
                      unsigned int use );
void cdk_sklist_release( CDK_KEYLIST skl );
int cdk_sklist_write( CDK_KEYLIST skl, CDK_IOBUF outp, GCRY_MD_HD hash,
                      int sigclass );
int cdk_sklist_write_onepass( CDK_KEYLIST skl, CDK_IOBUF outp, int sigclass );

/*****************
 * Memory routines
 *****************/
void cdk_set_malloc_hooks( void *(*new_alloc_func)(size_t n),
                           void *(*new_alloc_secure_func)(size_t n),
                           void *(*new_realloc_func)(void *p, size_t n),
                           void *(*new_calloc_func)(size_t m, size_t n),
                           void (*new_free_func)(void*) );
int cdk_malloc_hook_initialized( void );
void cdk_secmem_init( size_t size );
void cdk_secmem_end( void );
void* cdk_alloc( size_t size );
void* cdk_alloc_clear( size_t size );
void* cdk_alloc_secure( size_t size );
void* cdk_alloc_secure_clear( size_t size );
void* cdk_calloc( size_t nmemb, size_t size );
void* cdk_realloc( void *ptr, size_t size );
void* cdk_strdup( const char *ptr );
void cdk_free( void *ptr );

CDK_BSTRING cdk_bstring_new( const byte *buffer, size_t buflen );
CDK_BSTRING cdk_bstring_realloc( CDK_BSTRING src, int length );

/**********************
 * CDK standard filters
 **********************/
typedef struct {
    GCRY_CIPHER_HD cipher_hd;
    GCRY_MD_HD mdc_hash;
    CDK_DEK *dek;
    byte enchash[20];
    u32 datalen;
    CDK_IOBUF plain;
    char defer[20];
    int defer_filled;
    int eof_seen;
} cipher_filter_s;

typedef struct {
    int digest_algo;
    GCRY_MD_HD md;
    GCRY_MD_HD md2;
} md_filter_s;

int cdk_md_filter( md_filter_s *mfx, int control, CDK_IOBUF buf );
int cdk_cipher_filter( cipher_filter_s *cfx, int control, CDK_IOBUF outp );

/*****************
 * Cipher handling
 *****************/
int cdk_encrypt_file( GCRY_SEXP opts, CDK_STRLIST remusr, const char *file,
                      const char *output );
int cdk_sym_encrypt_file( GCRY_SEXP opts, char *pw,
                          const char *file, const char *output );
int cdk_sym_signenc_file( GCRY_SEXP opts, char *pw, CDK_STRLIST locusr,
                          const char *file, const char *output );

/*************************
 * Trust database handling
 *************************/
typedef enum {
    TRUST_UNKNOWN = 0,
    TRUST_EXPIRED = 1,
    TRUST_UNDEFINED = 2,
    TRUST_NEVER = 3,
    TRUST_MARGINAL = 4,
    TRUST_FULLY = 5,
    TRUST_ULTIMATE = 6
} trust_t;

typedef enum {
    TRUST_FLAG_REVOKED = 32,
    TRUST_FLAG_SUB_REVOKED = 64,
    TRUST_FLAG_DISABLED = 128
} trust_flags_t;
  
#define TRUST_MASK 15

int cdk_trustdb_check( const char *file, int req_ver );
int cdk_trustdb_get_validity( CDK_IOBUF buf, cdkPKT_user_id *id, int *r_val );
int cdk_trustdb_get_ownertrust( CDK_IOBUF buf, cdkPKT_public_key *pk,
                                int *r_val, int *r_flags );

/*****************
 * String routines
 *****************/
void cdk_strlist_free( CDK_STRLIST sl );
CDK_STRLIST cdk_strlist_add( CDK_STRLIST *list, const char *string );
CDK_STRLIST cdk_strlist_append( CDK_STRLIST *list, const char *string );

/*************************
 * Charset (UTF8) routines
 *************************/
int cdk_utf8_set_charset( const char *newset );
const char* cdk_utf8_get_charset( void );
char* cdk_utf8_from_native( const char *string );
char* cdk_utf8_to_native( const char *string, size_t length, int delim ); 

/******************
 * Logging routines
 ******************/
enum {
    OPENCDK_LOG_VERBOSE = 1,
    OPENCDK_LOG_DEBUG = 2
};
  
void cdk_set_logging( int mask );
void cdk_printf( const char *fmt, ... );

/******
 * Misc
 ******/
const char* cdk_check_version( const char *req_version );

#endif /* OPENCDK_H */





