/* pk.c - Public key functions with S-EXP interface
 *  Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"

static int key_to_sexp( GCRY_SEXP *r_key, cdkPKT_public_key *pk );
static int skey_to_sexp( GCRY_SEXP *r_skey, cdkPKT_secret_key *sk );
static int esk_to_sexp( GCRY_SEXP *r_sexp, GCRY_MPI esk );
static int sig_to_sexp( GCRY_SEXP *r_sig, cdkPKT_signature *sig );
static int sexp_to_sig( cdkPKT_signature *sig, GCRY_SEXP sexp );
static int md_to_sexp( GCRY_SEXP *r_md, int algo, const byte *md, size_t mdlen );
static int sexp_to_enc( cdkPKT_pubkey_enc *enc, GCRY_SEXP sexp );
static int enc_to_sexp( GCRY_SEXP *r_sexp, cdkPKT_pubkey_enc *enc );

#if 0
void
dump_sexp( GCRY_SEXP sexp )
{
    GCRY_SEXP x;
	
    fprintf( stderr, "DUMP of SEXP:\n" );
    gcry_sexp_dump( sexp );
    x = gcry_sexp_cdr( sexp );
    if ( x ) {
        fprintf( stderr, "DUMP of CDR:\n" );
        gcry_sexp_dump( x );
        gcry_sexp_release( x );
    }
} /* dump_sexp */
#endif

static GCRY_MPI*
bitmpi_to_gcrympi( CDK_MPI *m[4], int ncount )
{
    int i = 0, rc = 0;
    GCRY_MPI *d = NULL;
    size_t nbytes = 0;
  
    if ( !m )
        return NULL;

    d = cdk_calloc( ncount, sizeof *d );
    if ( !d )
        return NULL;
    for ( i = 0; i < ncount && !rc; i++ ) {
        nbytes = m[i]->bytes + 2;
        rc = gcry_mpi_scan( &d[i], GCRYMPI_FMT_PGP, m[i]->data, &nbytes );
        if ( rc )
            rc = CDKERR_GCRY;
    }

    while ( rc && i-- ) {
        gcry_mpi_release( d[i] );
        d[i] = NULL;
    }
    return d;
} /* bitmpi_to_gcrympi */

/* Convert an OpenPGP secret key to a gcry S-exp structure. */
static int
skey_to_sexp( GCRY_SEXP *r_skey, cdkPKT_secret_key *sk )
{
    cdkPKT_public_key *pk = NULL;
    GCRY_SEXP sexp = NULL;
    GCRY_MPI *m_pk = NULL, *m_sk = NULL;
    const char *fmt = NULL;
    int ncount = 0, nscount = 0;
    int rc = 0;

    if (!r_skey || !sk || !sk->pk)
        return CDKERR_INV_VALUE;

    pk = sk->pk;
    ncount = cdk_pk_get_npkey( pk->pubkey_algo );
    m_pk = bitmpi_to_gcrympi( pk->mpi, ncount );
    if ( !m_pk )
        return CDKERR_MPI;
    nscount = cdk_pk_get_nskey( pk->pubkey_algo );
    m_sk = bitmpi_to_gcrympi( sk->mpi, nscount );
    if ( !m_sk ) {
        rc = CDKERR_MPI;
        goto leave;
    }  
    if ( is_RSA(pk->pubkey_algo) ) {
        fmt = "(private-key(rsa(n%m)(e%m)(d%m)(p%m)(q%m)(u%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m_pk[0], m_pk[1],
                              m_sk[0], m_sk[1], m_sk[2], m_sk[3] );
    }
    else if ( is_ELG(pk->pubkey_algo) ) {
        fmt = "(private-key(elg(p%m)(g%m)(y%m)(x%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m_pk[0], m_pk[1],
                              m_pk[2], m_sk[0] );
    }
    else if ( is_DSA(pk->pubkey_algo) ) {
        fmt = "(private-key(dsa(p%m)(q%m)(g%m)(y%m)(x%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m_pk[0], m_pk[1], m_pk[2],
                              m_pk[3], m_sk[0] );
    }
    if ( rc )
        rc = CDKERR_GCRY;
    if ( !fmt )
        rc = CDKERR_INV_ALGO;
leave:
    while ( ncount-- && m_pk[ncount] )
        gcry_mpi_release( m_pk[ncount] );
    while ( nscount-- && m_sk[nscount] )
        gcry_mpi_release( m_sk[nscount] );
    *r_skey = sexp;
  
    return rc;
} /* skey_to_sexp */

/* Convert an OpenPGP public key to a gcrypt S-Exp structure.
   This step is needed to perform internal crypto routines on keys. */
static int
key_to_sexp( GCRY_SEXP *r_key, cdkPKT_public_key *pk )
{
    GCRY_SEXP sexp = NULL;
    GCRY_MPI *m = NULL;
    const char *fmt = NULL;
    int ncount = 0;
    int rc = 0;
  
    if ( !r_key || !pk )
        return CDKERR_INV_VALUE;

    ncount = cdk_pk_get_npkey( pk->pubkey_algo );
    m = bitmpi_to_gcrympi( pk->mpi, ncount );
    if ( !m )
        return CDKERR_MPI;
    if ( is_RSA(pk->pubkey_algo) ) {
        fmt = "(public-key(openpgp-rsa(n%m)(e%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m[0], m[1] );
    }
    else if ( is_ELG(pk->pubkey_algo) ) {
        fmt = "(public-key(openpgp-elg(p%m)(g%m)(y%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m[0], m[1], m[2] );
    }
    else if ( is_DSA(pk->pubkey_algo) ) {
        fmt = "(public-key(openpgp-dsa(p%m)(q%m)(g%m)(y%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m[0], m[1], m[2], m[3] );
    }
    if ( rc )
        rc = CDKERR_GCRY;
    if ( !fmt )
        rc = CDKERR_INV_ALGO;
    while ( ncount-- && m[ncount] )
        gcry_mpi_release( m[ncount] );
    *r_key = sexp;
  
    return rc;  
} /* key_to_sexp */

/* Convert an ESK (encrypted session key) into a S-Exp structure. */
static int
esk_to_sexp( GCRY_SEXP *r_sexp, GCRY_MPI esk )
{
    GCRY_SEXP sexp = NULL;
    int rc = 0;
	
    if ( !r_sexp || !esk )
        return CDKERR_INV_VALUE;
  
    rc = gcry_sexp_build( &sexp, NULL, "%m", esk );
    if ( rc )
        rc = CDKERR_GCRY;
    *r_sexp = sexp;
  
    return rc;
} /* esk_to_sexp */

/* Convert a MD (message digest) into a S-Sexp structure. */
static int
md_to_sexp( GCRY_SEXP *r_md, int algo, const byte *md, size_t mdlen )
{
    GCRY_SEXP sexp = NULL;
    GCRY_MPI m = NULL;
    size_t nbytes = 0;
    int rc = 0;

    if ( !r_md || !md )
        return CDKERR_INV_VALUE;

    nbytes = mdlen? mdlen : gcry_md_get_algo_dlen( algo );
    rc = gcry_mpi_scan( &m, GCRYMPI_FMT_USG, md, &nbytes );
    if ( rc )
        return CDKERR_GCRY;
    rc = gcry_sexp_build( &sexp, NULL, "%m", m );
    if ( rc )
        rc = CDKERR_GCRY;
    if ( !rc )
        *r_md = sexp;
    gcry_mpi_release( m );
  
    return rc;
} /* md_to_sexp */

static int
sexp_to_bitmpi( GCRY_SEXP sexp, const char *val, CDK_MPI **ret_buf )
{
    GCRY_SEXP list = NULL;
    GCRY_MPI m = NULL;
    CDK_MPI *buf = NULL;
    size_t nbits = 0, nbytes = 0;
    int rc = 0;

    if ( !sexp || !val || !ret_buf )
        return CDKERR_INV_VALUE;

    list = gcry_sexp_find_token( sexp, val, 0 );
    if ( !list )
        return CDKERR_GCRY;
    m = gcry_sexp_nth_mpi( list, 1, 0 );
    if ( !m ) {
        rc = CDKERR_GCRY;
        goto leave;
    }
    nbits = gcry_mpi_get_nbits( m ); nbytes = ( nbits + 7 ) / 8;
    buf = cdk_alloc_clear( sizeof *buf + nbytes );
    if ( !buf )
        return CDKERR_OUT_OF_CORE;
    buf->data[0] = nbits >> 8;
    buf->data[1] = nbits     ;
    rc = gcry_mpi_print( GCRYMPI_FMT_USG, NULL, &nbytes, m );
    if ( !rc )
        rc = gcry_mpi_print( GCRYMPI_FMT_USG, buf->data + 2, &nbytes, m );
    if ( rc ) {
        rc = CDKERR_GCRY;
        goto leave;
    }
    buf->bytes = nbytes;
    buf->bits = nbits;
    *ret_buf = buf;
  
leave:
    gcry_mpi_release( m );
    gcry_sexp_release( list );
  
    return rc;
} /* sexp_to_bitmpi */

/* Convert a S-Exp structure to an OpenPGP signature. */
static int
sexp_to_sig( cdkPKT_signature *sig, GCRY_SEXP sexp )
{
    int rc = 0;

    if ( !sig || !sexp )
        return CDKERR_INV_VALUE;
  
    if ( is_RSA(sig->pubkey_algo) )
        return sexp_to_bitmpi( sexp, "s", &sig->mpi[0] );
    else if ( is_DSA(sig->pubkey_algo) || is_ELG(sig->pubkey_algo) ) {
        rc = sexp_to_bitmpi( sexp, "r", &sig->mpi[0] );
        if ( !rc )
            rc = sexp_to_bitmpi( sexp, "s", &sig->mpi[1] );
        return rc;
    }
  
    return CDKERR_INV_ALGO;
} /* sexp_to_sig */

/* Convert an OpenPGP signature to gcrypt S-Exp structure.
   Because all internal crypto routines uses S-Sexp context,
   it's used any time we want to check a signature! */
static int
sig_to_sexp( GCRY_SEXP *r_sig, cdkPKT_signature *sig )
{
    GCRY_SEXP sexp = NULL;  
    GCRY_MPI *m = NULL;
    const char *fmt = NULL;
    int ncount = 0;
    int rc = 0;

    if ( !r_sig || !sig )
        return CDKERR_INV_VALUE;
  
    ncount = cdk_pk_get_nsig( sig->pubkey_algo );
    m = bitmpi_to_gcrympi( sig->mpi, ncount );
    if ( !m )
        return CDKERR_MPI;
    if ( is_RSA(sig->pubkey_algo) ) {
        fmt = "(sig-val(openpgp-rsa(s%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m[0] );
    }
    else if ( is_ELG(sig->pubkey_algo) ) {
        fmt = "(sig-val(openpgp-elg(r%m)(s%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m[0], m[1] );
    }  
    else if ( is_DSA(sig->pubkey_algo) ) {
        fmt = "(sig-val(openpgp-dsa(r%m)(s%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m[0], m[1] );
    }
    if ( rc )
        rc = CDKERR_GCRY;
    if ( !fmt )
        rc = CDKERR_INV_ALGO;
    while ( ncount-- && m[ncount] )
        gcry_mpi_release( m[ncount] );
    *r_sig = sexp;
  
    return rc;  
} /* sig_to_sexp */

static int
sexp_to_enc( cdkPKT_pubkey_enc *enc, GCRY_SEXP sexp )
{
    int rc = 0;
  
    if ( !sexp || !enc )
        return CDKERR_INV_VALUE;

    if ( is_RSA(enc->pubkey_algo) )
        return sexp_to_bitmpi( sexp, "a", &enc->mpi[0] );
    else if ( is_ELG(enc->pubkey_algo) ) {
        rc = sexp_to_bitmpi( sexp, "a", &enc->mpi[0] );
        if ( !rc )
            rc = sexp_to_bitmpi( sexp, "b", &enc->mpi[1] );
        return rc;
    }

    return CDKERR_INV_ALGO;
} /* sexp_to_enc */

static int
enc_to_sexp(GCRY_SEXP *r_sexp, cdkPKT_pubkey_enc *enc)
{
    GCRY_SEXP sexp = NULL; 
    GCRY_MPI *m = NULL;
    const char *fmt = NULL;
    int ncount = 0;
    int rc = 0;

    if ( !r_sexp || !enc )
        return CDKERR_INV_VALUE;

    ncount = cdk_pk_get_nenc( enc->pubkey_algo );
    m = bitmpi_to_gcrympi( enc->mpi, ncount );
    if ( !m )
        return CDKERR_MPI;
    if ( is_RSA(enc->pubkey_algo) ) {
        fmt = "(enc-val(openpgp-rsa((a%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m[0] );
    }
    else if ( is_ELG(enc->pubkey_algo) ) {
        fmt = "(enc-val(openpgp-elg((a%m)(b%m)))";
        rc = gcry_sexp_build( &sexp, NULL, fmt, m[0], m[1] );
    }
    if ( rc )
        rc = CDKERR_GCRY;
    if ( !fmt )
        rc = CDKERR_INV_ALGO;
    while ( ncount-- && m[ncount] )
        gcry_mpi_release( m[ncount] );
    *r_sexp = sexp;
  
    return rc;  
} /* enc_to_sexp */

/**
 * cdk_pk_encrypt - Perform a PK encryption
 * @pk: The public key which should used for encryption.
 * @pke: The packet to store the result.
 * @esk: The session key.
 *
 * Encrypt the session key in @esk with the public key and the
 * result is stored into the public key encryption packet @pke.
 **/
int
cdk_pk_encrypt( cdkPKT_public_key *pk, cdkPKT_pubkey_enc *pke, GCRY_MPI esk )
{
    GCRY_SEXP s_data = NULL, s_pkey = NULL, s_ciph = NULL;
    int rc = 0;
	
    if ( !pk || !esk || !pke )
        return CDKERR_INV_VALUE;

    if ( !KEY_CAN_ENCRYPT( pk->pubkey_algo ) )
        return CDKERR_INV_ALGO;

    rc = esk_to_sexp( &s_data, esk );
    if ( !rc )
        rc = key_to_sexp( &s_pkey, pk );
    if ( !rc )
        rc = gcry_pk_encrypt( &s_ciph, s_data, s_pkey );
    if ( !rc )
        rc = sexp_to_enc( pke, s_ciph );
	
    gcry_sexp_release( s_data );
    gcry_sexp_release( s_pkey );
    gcry_sexp_release( s_ciph );
	
    return rc;
} /* cdk_pk_encrypt */

int
cdk_pk_decrypt( cdkPKT_secret_key *sk, cdkPKT_pubkey_enc *pke, GCRY_MPI *r_sk )
{
    GCRY_SEXP s_data = NULL, s_skey = NULL, s_plain = NULL;
    int rc = 0;
  
    if ( !sk || !r_sk || !pke )
        return CDKERR_INV_VALUE;

    rc = skey_to_sexp( &s_skey, sk );
    if ( !rc )
        rc = enc_to_sexp( &s_data, pke );
    if ( !rc ) {    
        rc = gcry_pk_decrypt( &s_plain, s_data, s_skey );
        if ( rc )
            rc = CDKERR_GCRY;
    }
    if ( !rc )
        *r_sk = gcry_sexp_nth_mpi( s_data, 0, 0 );
  
    gcry_sexp_release( s_data );
    gcry_sexp_release( s_skey );
    gcry_sexp_release( s_plain );
    return rc;
} /* cdk_pk_decrypt */

/**
 * cdk_pk_sign - Perform PK signing
 * @sk: The secret key used for signing.
 * @sig: Signature context to store the result.
 * @md: The message digest to sign.
 *
 * Create a signature of the given hash.
 * The signature must be already include all algorithm information,
 * like the message digest and the asymmetric cipher.
 **/
int
cdk_pk_sign( cdkPKT_secret_key *sk, cdkPKT_signature *sig, const byte *md )
{
    GCRY_SEXP s_skey = NULL, s_sig = NULL, s_hash = NULL;
    byte *encmd = NULL;
    size_t enclen = 0;
    int nbits = 0;
    int rc = 0;
  
    if ( !sk || !sk->pk || !sig || !md )
        return CDKERR_INV_VALUE;

    if ( !KEY_CAN_SIGN( sig->pubkey_algo ) )
        return CDKERR_INV_ALGO;

    nbits = cdk_pk_get_nbits( sk->pk );
    rc = _cdk_encode_digest( &encmd, &enclen, sk->pk->pubkey_algo, md,
                             sig->digest_algo, nbits );
    if ( !rc )
        rc = skey_to_sexp( &s_skey, sk );
    if ( !rc )
        rc = md_to_sexp( &s_hash, sig->digest_algo, encmd, enclen );
    if ( !rc ) {
        rc = gcry_pk_sign( &s_sig, s_hash, s_skey );
        if ( rc )
            rc = CDKERR_GCRY;
    }
    if ( !rc )
        rc = sexp_to_sig( sig, s_sig );
    sig->digest_start[0] = md[0];
    sig->digest_start[1] = md[1];

    gcry_sexp_release( s_skey );
    gcry_sexp_release( s_hash );
    gcry_sexp_release( s_sig );
    cdk_free( encmd );
  
    return rc;
} /* cdk_pk_sign */

/**
 * cdk_pk_verify - Perform PK verfication
 * @pk: Public key which should be used for verifying.
 * @sig: The signatre which contains the data.
 * @md: The message digest to verify.
 *
 * Verify a signature.
 **/
int
cdk_pk_verify( cdkPKT_public_key *pk, cdkPKT_signature *sig, const byte *md )
{
    GCRY_SEXP s_pkey = NULL, s_sig = NULL, s_hash = NULL;
    byte *encmd = NULL;
    size_t enclen = 0;
    int nbits = 0;
    int rc = 0;
  
    if ( !pk || !sig || !md )
        return CDKERR_INV_VALUE;

    nbits = cdk_pk_get_nbits( pk );
    rc = key_to_sexp( &s_pkey, pk );
    if ( !rc )
        rc = sig_to_sexp( &s_sig, sig );
    if ( !rc )
        rc = _cdk_encode_digest( &encmd, &enclen, pk->pubkey_algo, md,
                                 sig->digest_algo, nbits );
    if ( !rc )
        rc = md_to_sexp( &s_hash, sig->digest_algo, encmd, enclen );
    if ( !rc ) {      
        rc = gcry_pk_verify( s_sig, s_hash, s_pkey );
        if ( rc )
            rc = CDKERR_BAD_SIGNATURE;
    }

    gcry_sexp_release( s_sig );
    gcry_sexp_release( s_hash );
    gcry_sexp_release( s_pkey );
    cdk_free( encmd );
	
    return rc;
} /* cdk_pk_verify */

int
cdk_pk_get_nbits( cdkPKT_public_key *pk )
{
    if ( !pk || !pk->mpi[0] )
        return 0;
    return pk->mpi[0]->bits;
} /* cdk_pk_get_nbits */

int
cdk_pk_get_npkey( int algo )
{
    int n = gcry_pk_algo_info( algo, GCRYCTL_GET_ALGO_NPKEY, NULL, 0 );
    return n > 0? n : 0;
} /* cdk_pk_get_npkey */

int
cdk_pk_get_nskey( int algo )
{
    int n = gcry_pk_algo_info( algo, GCRYCTL_GET_ALGO_NSKEY, NULL, 0 );
    n -= cdk_pk_get_npkey( algo );
    return n > 0? n : 0;  
} /* cdk_pk_get_nskey */

int
cdk_pk_get_nsig( int algo )
{
    int n = gcry_pk_algo_info( algo, GCRYCTL_GET_ALGO_NSIGN, NULL, 0 );
    return n > 0? n : 0;  
} /* cdk_pk_get_nsig */

int
cdk_pk_get_nenc( int algo )
{
    int n = gcry_pk_algo_info( algo, GCRYCTL_GET_ALGO_NENCR, NULL, 0 );
    return n > 0? n : 0;  
} /* cdk_pk_get_nenc */
