/* pkencrypt.c - Encryption routines
 *  Copyright (C) 2002 Timo Schulz
 *  Copyright (C) 1998, 1999, 2001, 2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"
#include "iobuf.h"

CDK_IOBUF
_cdk_make_plaintext( CDK_IOBUF inp, const char *filename, int mode )
{
    CDK_IOBUF tmp;
    CDK_PACKET pkt;
    cdkPKT_plaintext *pt = NULL;
    const char *file;
    size_t filelen = 0;
    int rc = 0;

    if ( !filename ) file = "_CONSOLE";
    else file = filename;
    
    rc = cdk_iobuf_create( &tmp, NULL );
    if ( rc )
        return NULL;

    filelen = strlen( file );
    pt = cdk_alloc_clear( sizeof *pt + filelen - 1 );
    if ( !pt ) {
        cdk_iobuf_close( tmp );
        return NULL;
    }
    pt->namelen = filelen;
    memcpy( pt->name, file, pt->namelen );
    pt->name[ pt->namelen ] = '\0';
    pt->timestamp = make_timestamp();
    pt->mode = mode ? 't' : 'b';
    pt->len = cdk_iobuf_get_length( inp );
    pt->buf = inp;
    init_packet( &pkt );
    pkt.pkttype = PKT_PLAINTEXT;
    pkt.pkt.plaintext = pt;
    rc = cdk_pkt_build( tmp, &pkt );
    if ( rc ) {        
        cdk_iobuf_close( tmp );
        tmp = NULL;
    }
    cdk_free( pt );

    return tmp;
} /* _cdk_make_plaintext */

#if 0
int
cdk_cipher_filter( void *ctx, int control, CDK_IOBUF buf )
{
    cipher_filter_s *cfx = ctx;
    CDK_BSTRING bs = NULL;
    byte *data = NULL;
    size_t nsize = 0;
    int c = 0, n = 0;
    int rc = 0;

    if ( !ctx || !buf )
        return CDKERR_INV_VALUE;

    if ( control == IOBUF_CTRL_UNDERFLOW ) {
        if ( cfx->mdc_hash ) {
            if ( cfx->eof_seen )
                return 0; /* Nothing to do */
            nsize = cdk_iobuf_get_length( buf );
            if ( nsize < 40 )
                return CDKERR_INV_PACKET;
            data = cdk_alloc_clear( nsize+1 );
            if ( !data )
                return CDKERR_OUT_OF_CORE;
            for ( n=20; n<40; n++ ) {
                c = cdk_iobuf_get( buf );
                if ( buf->eof == -1 )
                    break;
                data[n] = c;
            }
            if ( n == 40 ) {
                /* we have enough stuff - flush the deferred stuff
                   (we have asserted that the buffer is large enough) */
                if( !cfx->defer_filled ) {
                    /* the first time */
                    memcpy( data, data+20, 20 );
                    n = 20;
                }
                else {
                    memcpy( data, cfx->defer, 20 );
                }
                /* now fill up */
                for( ; n < nsize; n++ ) {
                    c = cdk_iobuf_get( buf );
                    if ( buf->eof == -1 )
                        break;
                    data[n] = c;
                }

                /* move the last 20 bytes back to the defer buffer
                   (okay, we are wasting 20 bytes of supplied buffer) */
                n -= 20;
                memcpy( cfx->defer, data+n, 20 );
                cfx->defer_filled = 1;
            }
            else if ( !cfx->defer_filled ) /* eof seen buf empty defer */ {
                /* this is bad because there is an incomplete hash */
                n -= 20;
                memcpy( data, data+20, n );
                cfx->eof_seen = 2; /* eof with incomplete hash */
            }
            else /* eof seen */{
                memcpy( data, cfx->defer, 20 );
                n -= 20;
                memcpy( cfx->defer, data+n, 20 );
                cfx->eof_seen = 1; /* normal eof */
            }
            if ( n ) {
                gcry_cipher_decrypt( cfx->cipher_hd, data, n, NULL, 0 );
                gcry_md_write( cfx->mdc_hash, data, n );
                cdk_iobuf_new( &cfx->plain, n+1 );
                rc = cdk_iobuf_write( cfx->plain, data, n );
            }
		}
        else {
            /* without MDC */
            size_t pos = 0;
            pos = cdk_iobuf_tell( buf );
            bs = cdk_iobuf_read_mem( buf, pos );
            if ( !bs )
                return CDKERR_INV_VALUE;
            cdk_iobuf_new( &cfx->plain, bs->len+1 );
            gcry_cipher_decrypt( cfx->cipher_hd, bs->d, bs->len, NULL, 0 );
            rc = cdk_iobuf_write( cfx->plain, bs->d, bs->len );
        }
        goto leave;
    }

    if ( !cfx->plain )
        return CDKERR_INV_VALUE;
    bs = cdk_iobuf_read_mem( cfx->plain, 0 );
    if ( !bs )
        return CDKERR_INV_VALUE;
    rc = write_header( cfx, buf );
    if ( rc )
        goto leave;
    if ( cfx->mdc_hash )
        gcry_md_write( cfx->mdc_hash, bs->d, bs->len );
    gcry_cipher_encrypt( cfx->cipher_hd, bs->d, bs->len, NULL, 0 );
    rc = cdk_iobuf_write( buf, bs->d, bs->len );
    if ( rc )
        goto leave;

    if ( cfx->mdc_hash ) {
        byte *hash = NULL;
        byte pktdata[22];
        int hashlen = gcry_md_get_algo_dlen( GCRY_MD_SHA1 );

        if ( hashlen != 20 )
            return CDKERR_INV_ALGO;

        /* we must hash the prefix of the MDC packet here */
        pktdata[0] = 0xd3;
        pktdata[1] = 0x14;
        gcry_md_putc( cfx->mdc_hash, pktdata[0] );
        gcry_md_putc( cfx->mdc_hash, pktdata[1] );
        gcry_md_final( cfx->mdc_hash );
        hash = gcry_md_read( cfx->mdc_hash, 0 );
        memcpy( pktdata+2, hash, 20 );
        gcry_cipher_encrypt( cfx->cipher_hd, pktdata, 22, NULL, 0 );
        gcry_md_close( cfx->mdc_hash );
        cfx->mdc_hash = NULL;
        rc = cdk_iobuf_write( buf, pktdata, 22 );
    }
  
leave:
    cdk_free( bs );
  
    return rc;
} /* cdk_cipher_filter */
#endif

/**
 * cdk_encrypt_file - Encryt a file.
 * @opts: sexp structure with the following format:
 * (pubkey-enc(
 *   (textmode%d)
 *   (compress%d)
 *   (armor%d)
 *  ))
 * @remusr: the list of recipients.
 * @file: the plaintext file.
 * @output: the output file.
 **/
int
cdk_encrypt_file( GCRY_SEXP opts, CDK_STRLIST remusr, const char *file,
                  const char *output )
{
    CDK_KEYLIST pkl = NULL;
    cipher_filter_s cfx;
    armor_filter_s afx;
    CDK_IOBUF inp = NULL, outp = NULL, tmp = NULL;
    int rc = 0;

    rc = cdk_iobuf_open( &inp, file, IOBUF_MODE_RD );
    if ( rc )
        return rc;
  
    rc = cdk_iobuf_create( &outp, output );
    if ( rc )
        goto leave;

    memset( &cfx, 0, sizeof cfx );

    rc = cdk_pklist_build( remusr, &pkl, GCRY_PK_USAGE_ENCR );
    if ( rc )
        goto leave;
  
    cfx.dek = cdk_alloc_secure_clear( sizeof *cfx.dek );
    if ( !cfx.dek ) {
        rc = CDKERR_OUT_OF_CORE;
        goto leave;
    }

    cfx.dek->algo = cdk_pklist_select_algo( pkl );
    cfx.dek->use_mdc = cdk_pklist_use_mdc( pkl );    
    rc = _cdk_dek_create( cfx.dek );
    if ( rc == CDKERR_WEAK_KEY )
        goto leave;
    
    rc = cdk_pklist_encrypt( pkl, cfx.dek, outp );
    if ( rc )
        goto leave;

    tmp = _cdk_make_plaintext( inp, file, 0 );
    if ( !tmp ) {
        rc = CDKERR_INV_VALUE;
        goto leave;
    }

    cfx.plain = tmp;
    cfx.datalen = cdk_iobuf_get_length( tmp );
    rc = cdk_cipher_filter( &cfx, IOBUF_CTRL_FLUSH, outp );
    memset( &cfx.dek, 0, sizeof cfx.dek ); /* burn session key */
    
    cdk_iobuf_close( tmp );
    if ( !rc && openpgp_get_opt_val( opts, OPENPGP_ARMOR ) ) {
        memset( &afx, 0, sizeof afx );
        afx.idx = afx.idx2 = ARMOR_MESSAGE;
        rc = cdk_armor_filter( &afx, IOBUF_CTRL_FLUSH, outp );
    }

leave:
    cdk_pklist_release( pkl );
    cdk_free( cfx.dek );
    cdk_iobuf_close( inp );
    cdk_iobuf_close( outp );

    return rc;
} /* cdk_encrypt_file */

/**
 * cdk_sym_encrypt_file - Symmetric file encryption.
 * @opts: sexp structure with the following format:
 * (symkey-enc(
 *   (textmode%d)
 *   (compress%d)
 *   (armor%d)
 *   (digest%d)
 *   (cipher%d)
 * ))
 * @pw: The passphrase.
 * @file: The input file.
 * @output: The output file.
 *
 * Encrypts the given file symmetrically with the passphrase.
 **/
int
cdk_sym_encrypt_file( GCRY_SEXP opts, char *pw, const char *file,
                      const char *output )
{
    CDK_IOBUF inp = NULL, outp = NULL, tmp = NULL;
    CDK_STRING2KEY *s2k = NULL;
    CDK_KEYLIST skl = NULL;
    CDK_PACKET pkt;
    cipher_filter_s cfx;
    armor_filter_s afx;
    int algo = 0, cipher_algo = 0;
    int rc = 0;
  
    if ( !file || !output )
        return CDKERR_INV_VALUE;

    memset( &cfx, 0, sizeof cfx );
  
    rc = cdk_iobuf_open( &inp, file, IOBUF_MODE_RD );
    if ( rc )
        goto leave;

    rc = cdk_iobuf_create( &outp, output );
    if ( rc )
        goto leave;

    algo = openpgp_get_opt_val( opts, OPENPGP_CIPHER );
    cipher_algo = algo? algo : OPENPGP_DEF_CIPHER;
    s2k = cdk_alloc_clear( sizeof *s2k );
    if ( !s2k ) {
        rc = CDKERR_OUT_OF_CORE;
        goto leave;
    }
    s2k->mode = OPENPGP_DEF_S2K;
    s2k->hash_algo = OPENPGP_DEF_MD;
    cfx.dek = cdk_passphrase_to_dek( cipher_algo, s2k, 2, pw );
    if ( !cfx.dek || !cfx.dek->keylen ) {
        rc = CDKERR_GENERAL; /* fixme */
        goto leave;
    }
    if ( s2k ) {
        cdkPKT_symkey_enc *enc = cdk_alloc_clear( sizeof *enc );
        enc->version = 4;
        enc->cipher_algo = cfx.dek->algo;
        enc->s2k = *s2k;
        init_packet( &pkt );
        pkt.pkttype = PKT_SYMKEY_ENC;
        pkt.pkt.symkey_enc = enc;
        rc = cdk_pkt_build( outp, &pkt );
        cdk_free( enc );
        if ( rc )
            goto leave;
    }

    tmp = _cdk_make_plaintext( inp, file, 0 );
    if ( !tmp ) {
        rc = CDKERR_INV_VALUE;
        goto leave;
    }

    cfx.plain = tmp;
    cfx.datalen = cdk_iobuf_get_length( tmp );
    rc = cdk_cipher_filter( &cfx, IOBUF_CTRL_FLUSH, outp );
    memset( &cfx.dek, 0, sizeof cfx.dek ); /* burn session key */
    
    cdk_iobuf_close( tmp );
    if ( !rc && openpgp_get_opt_val( opts, OPENPGP_ARMOR ) ) {
        memset( &afx, 0, sizeof afx );
        afx.idx = afx.idx2 = ARMOR_MESSAGE;
        rc = cdk_armor_filter( &afx, IOBUF_CTRL_FLUSH, outp );
    }
  
leave:
    cdk_sklist_release( skl );
    cdk_free( s2k );
    cdk_free( cfx.dek );
    cdk_iobuf_close( inp );
    cdk_iobuf_close( outp );
  
    return rc;
} /* cdk_sym_encrypt_file */
