/* seckey.c - Secret key routines
 *  Copyright (C) 2002 Timo Schulz
 *  Copyright (C) 1998, 1999, 2000, 2001, 2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"

static int
checksum_mpi( CDK_MPI *m )
{
    int i;
    u16 chksum = 0;

    if ( m ) {
        chksum = (m->bits >> 8) & 0xff;
        chksum += m->bits & 0xff;
        for ( i = 2; i < m->bytes; i++ )
            chksum += m->data[i];
    }
    return chksum;  
} /* checksum_mpi */

/**
 * cdk_hash_passphrase - Create a DEK from a passphrase hash.
 * @dek: data encryption key.
 * @pw: the passphrase.
 * @s2k: the string-to-key context.
 * @create: create salt.
 *
 * Hashes the given passphrase with the algorithm that is set in
 * the S2K context. If create is used, a new salt IV is created.
 **/
int
cdk_hash_passphrase( CDK_DEK *dek, char *pw, CDK_STRING2KEY *s2k, int create )
{
    GCRY_MD_HD md;
    int pass, i;
    int used = 0;
    int pwlen = 0;

    if ( !dek || !pw || !s2k )
        return CDKERR_INV_VALUE;

    if ( !s2k->hash_algo )
        s2k->hash_algo = OPENPGP_DEF_MD;
    pwlen = strlen( pw );
  
    dek->keylen = gcry_cipher_get_algo_keylen( dek->algo );
    md = gcry_md_open( s2k->hash_algo, GCRY_MD_FLAG_SECURE );
    if ( !md )
        return CDKERR_GCRY;

    for ( pass=0; used < dek->keylen; pass++ ) {
        if ( pass ) {
            gcry_md_reset( md );
            for( i=0; i < pass; i++ ) /* preset the hash context */
                gcry_md_putc( md, 0 ); 
        }
        if ( s2k->mode == 1 || s2k->mode == 3 ) {
            int len2 = pwlen + 8;
            u32 count = len2;
            if ( create && !pass ) {
                gcry_randomize( s2k->salt, 8, 1 );
                if ( s2k->mode == 3 )
                    s2k->count = 96; /* 65536 iterations */
            }
            if ( s2k->mode == 3 ) {
                count = (16ul + (s2k->count & 15)) << ((s2k->count >> 4) + 6);
                if( count < len2 )
                    count = len2;
            }
            /* a little bit complicated because we need a ulong for count */
            while ( count > len2 ) { /* maybe iterated+salted */
                gcry_md_write( md, s2k->salt, 8 );
                gcry_md_write( md, pw, pwlen );
                count -= len2; 
            }
            if ( count < 8 )
                gcry_md_write( md, s2k->salt, count );
            else {
                gcry_md_write( md, s2k->salt, 8 );
                count -= 8;
                gcry_md_write( md, pw, count ); 
            }
        }
        else
            gcry_md_write( md, pw, pwlen );
        gcry_md_final( md );
        i = gcry_md_get_algo_dlen(s2k->hash_algo);
        if ( i > dek->keylen - used )
            i = dek->keylen - used;
        memcpy( dek->key+used, gcry_md_read( md, s2k->hash_algo), i );
        used += i;
    }
    cdk_md_close( &md );

    return 0;
} /* cdk_hash_passphrase */

/**
 * cdk_passphrase_to_dek - Convert a passphrase to a DEK.
 * @cipher_algo: symmetric cipher.
 * @s2k: the string-to-key context.
 * @mode: the string-to-mode mode.
 * @pw: the passphrase.
 *
 * Shortform of the routine above. The passphrase is never saved
 * to any buffers, it's just used for the hash functions.
 **/
CDK_DEK*
cdk_passphrase_to_dek( int cipher_algo, CDK_STRING2KEY *s2k,
                       int mode, char *pw )
{
    CDK_DEK *dek = NULL;

    dek = cdk_alloc_secure_clear( sizeof *dek );
    if ( !dek )
        return NULL;
    dek->algo = cipher_algo;
    if ( !*pw && mode == 2 )
        dek->keylen = 0;
    else
        cdk_hash_passphrase( dek, pw, s2k, mode == 2 );
  
    return dek;
} /* cdk_passphrase_to_dek */

/**
 * cdk_unprotect_key - Unprotect a secret key.
 * @sk: the secret key.
 * @pw: the passphrase.
 *
 * Tries to decrypt the the secret key material with the S2K context
 * that is saved in the secret key.
 **/
int
cdk_seckey_unprotect( cdkPKT_secret_key *sk, char *pw )
{
    GCRY_CIPHER_HD chd;
    CDK_DEK *dek = NULL;
    u16 chksum = 0;
    size_t ndata;
    int j, i, pos = 0;
    byte *data = NULL;
  
    if ( !sk )
        return CDKERR_INV_VALUE;

    if ( sk->is_protected ) {
        dek = cdk_passphrase_to_dek(sk->protect.algo, &sk->protect.s2k, 0, pw);
        chd = gcry_cipher_open( sk->protect.algo,
                                GCRY_CIPHER_MODE_CFB,
                                GCRY_CIPHER_SECURE
                                | GCRY_CIPHER_ENABLE_SYNC );
        if ( !chd || !dek )
            return CDKERR_GCRY;
        gcry_cipher_setkey( chd, dek->key, dek->keylen );
        gcry_cipher_setiv( chd, sk->protect.iv, sk->protect.ivlen );
        cdk_free( dek );
        chksum = 0;
        if ( sk->pk->version >= 4 ) {
            ndata = sk->enclen;
            data = cdk_alloc_secure(ndata);
            if ( !data )
                return CDKERR_OUT_OF_CORE;
            gcry_cipher_decrypt( chd, data, ndata, sk->encdata, ndata );
            if ( sk->protect.sha1chk ) {
                /* This is the new SHA1 checksum method to detect tampering
                   with the key as used by the Klima/Rosa attack */
                sk->csum = 0;
                chksum = 1;
                if ( ndata < 20 )
                    return CDKERR_INV_PACKET;
                else {
                    GCRY_MD_HD mhd = gcry_md_open ( GCRY_MD_SHA1, 1 );
                    if ( !mhd )
                        return CDKERR_GCRY; /* algo not available */
                    gcry_md_write( mhd, data, ndata - 20 );
                    gcry_md_final( mhd );
                    if ( !memcmp( gcry_md_read( mhd, GCRY_MD_SHA1 ),
                                  data + ndata - 20, 20 ) )
                        chksum = 0; /* digest does match */
                    gcry_md_close( mhd );
                }
            }
            else { 
                for (i=0; i<ndata-2; i++)
                    chksum += data[i];
                sk->csum = (data[ndata-2] << 8) | data[ndata-1];
            }
            if ( sk->csum == chksum ) {              
                for (i=0; i<cdk_pk_get_nskey(sk->pk->pubkey_algo); i++) {
                    sk->mpi[i]->bits = (data[pos] << 8) | data[pos+1];
                    sk->mpi[i]->bytes = (sk->mpi[i]->bits+7)/8;
                    for (j=0; j<sk->mpi[i]->bytes; j++)
                        sk->mpi[i]->data[j] = data[pos++];
                }
            }
            cdk_free( data );
        }
        else {
            for (i=0; i<cdk_pk_get_nskey(sk->pk->pubkey_algo); i++) {
                gcry_cipher_decrypt( chd, sk->mpi[i]->data,
                                     sk->mpi[i]->bytes, NULL, 0 );
                chksum += checksum_mpi( sk->mpi[i] );
            }
        }
        gcry_cipher_close( chd );
    }
    else {
        chksum = 0;
        for (i=0; i<cdk_pk_get_nskey(sk->pk->pubkey_algo); i++)
            chksum += checksum_mpi(sk->mpi[i]);
    }
    if ( chksum != sk->csum )
        return CDKERR_CHECKSUM;
  
    return 0;
} /* cdk_seckey_unprotect */

