/* write-packet.c - Write OpenPGP packets
 *  Copyright (C) 2001, 2002 Timo Schulz 
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */ 

#include <string.h>
#include <stdio.h>

#include "opencdk.h"
#include "iobuf.h"
#include "main.h"

int
write_32( CDK_IOBUF out, u32 u )
{
    int rc = 0;
    byte buf[4];
    
    if ( out ) {
        buf[0] = u >> 24;
        buf[1] = u >> 16;
        buf[2] = u >>  8;
        buf[3] = u;
        rc = cdk_iobuf_write( out, buf, 4 );
    }
    return rc;   
} /* write_32 */

int
write_16( CDK_IOBUF out, u16 u )
{
    int rc = 0;
    byte buf[2];

    if ( out ) {
        buf[0] = u >> 8;
        buf[1] = u;
        rc = cdk_iobuf_write( out, buf, 2 );
    }
    return rc;
} /* write_16 */

static size_t
calc_mpi_size( CDK_MPI *mpi[4], int ncount, int force )
{
    int i, add = 0;
    size_t size = 0, pref_size = 0;

    if ( !mpi )
        return 0;
  
    for ( i = 0; i < ncount; i++ ) {
        size += mpi[i]->bits;
        pref_size += 2; /* +2 length prefix */
    }
    if ( force )
        add = 7;
    size = ( size +add ) / 8 + pref_size;
  
    return size;
} /* calc_mpi_size */

static int
write_mpi( CDK_IOBUF out, CDK_MPI *m )
{
    int rc = 0;
  
    if ( !out || !m )
        return CDKERR_INV_VALUE;

    if ( m->bits > MAX_MPI_BITS || m->bits == 0 )
        return CDKERR_MPI;
    if ( !m->bytes )
        m->bytes = ( m->bits + 7 ) / 8;
    rc = write_16( out, m->bits );
    if ( !rc )
        rc = cdk_iobuf_write( out, m->data + 2, m->bytes );
  
    return rc;
} /* write_mpi */

static int
pkt_encode_length( CDK_IOBUF out, size_t pktlen )
{
    int rc = 0;
    
    if ( !out )
        return CDKERR_INV_VALUE;
        
    if (pktlen < 192)
        rc = cdk_iobuf_put( out, (pktlen % 256) );
    else if ( (pktlen > 192) && (pktlen < 8383) ) {
        byte buf[2];
        buf[0] = (pktlen+1-192) / 256 + 192;
        buf[1] = (pktlen+1-192) % 256;
        rc = cdk_iobuf_write( out, buf, 2 );
    }
    else {
        cdk_iobuf_put( out, 255);
        rc = write_32( out, pktlen );
    }
	
    return rc;
} /* pkt_encode_length */

static int
pkt_write_head( CDK_IOBUF out, size_t size, int type )
{
    int rc = 0;

    if ( !out )
        return CDKERR_INV_VALUE;
  
    if ( type < 0 || type > 63 )
        return CDKERR_INV_PACKET;
    rc = cdk_iobuf_put( out, (0xC0 | type) );
    if ( !rc )
        rc = pkt_encode_length( out, size );
  
    return rc;
} /* pkt_write_head */

static int
write_subpkt( CDK_IOBUF out, CDK_SUBPKT ctx )
{
    int rc = 0;
  
    if ( !out || !ctx )
        return CDKERR_INV_VALUE;

    if ( ctx->type ) {
        rc = pkt_encode_length( out, ctx->size+1 );
        if ( !rc )
            rc = cdk_iobuf_put( out, ctx->type );
        if ( !rc )
            rc = cdk_iobuf_write( out, ctx->data, ctx->size );
    }
    return rc;
} /* write_subpkt */

static int
write_encrypted( CDK_IOBUF out, cdkPKT_encrypted *enc )
{
    int nbytes = 0;
    int rc = 0;
  
    if ( !out || !enc )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf( "** write encrypted packet %d bytes\n", enc->len );

    nbytes = enc->len? (enc->len + enc->extralen) : 0;
    rc = pkt_write_head( out, nbytes, PKT_ENCRYPTED );
    /* rest of the packet is ciphertext */
  
    return rc;
} /* write_encrypted */

static int
write_encrypted_mdc( CDK_IOBUF out, cdkPKT_encrypted *enc )
{
    int rc = 0;
    int nbytes = 0;
  
    if ( !out || !enc )
        return CDKERR_INV_VALUE;
  
    if ( enc->mdc_method == 0 )
        return CDKERR_INV_PACKET;

    if ( DEBUG_PKT )
        cdk_printf( "** write encrypted mdc packet %d bytes\n", enc->len );

    nbytes = enc->len? ( enc->len + enc->extralen ) : 0;
    rc = pkt_write_head( out, nbytes, PKT_ENCRYPTED_MDC );
    if ( !rc )
        rc = cdk_iobuf_put( out, 1 ); /* version */
    /* rest of the packet is ciphertext */

    return rc;
} /* write_encrypted_mdc */

static int
write_symkey_enc( CDK_IOBUF out, cdkPKT_symkey_enc *ske )
{
    size_t size = 0, s2k_size = 0;
    int rc = 0;
  
    if ( !out || !ske )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf( "** write symmetric key encrypted packet\n" );
  
    if ( ske->s2k.mode == 1 || ske->s2k.mode == 3 )
        s2k_size = 8;
    if ( ske->s2k.mode == 3 )
        s2k_size++;
    size = 4 + s2k_size + ske->seskeylen;
    rc = pkt_write_head( out, size, PKT_SYMKEY_ENC );
    if ( !rc )
        rc = cdk_iobuf_put( out, ske->version );
    if ( !rc )
        rc = cdk_iobuf_put( out, ske->cipher_algo );
    if ( !rc )
        rc = cdk_iobuf_put( out, ske->s2k.mode );
    if ( !rc )
        rc = cdk_iobuf_put( out, ske->s2k.hash_algo );
    if ( ske->s2k.mode == 1 || ske->s2k.mode == 3 ) {
        rc = cdk_iobuf_write( out, ske->s2k.salt, 8 );
        if ( !rc && ske->s2k.mode == 3 )
            rc = cdk_iobuf_put( out, ske->s2k.count );
    }
    return rc;
} /* write_symkey_enc */

static int
write_pubkey_enc( CDK_IOBUF out, cdkPKT_pubkey_enc *pke )
{
    int i, size, nenc = 0;
    int rc = 0;

    if ( !out || !pke )
        return CDKERR_INV_VALUE;

    if ( !KEY_CAN_ENCRYPT( pke->pubkey_algo ) )
        return CDKERR_INV_ALGO;

    if ( DEBUG_PKT )
        cdk_printf( "** write public key encrypted packet\n" );

    nenc = cdk_pk_get_nenc( pke->pubkey_algo );
    size = 10 + calc_mpi_size( pke->mpi, nenc, 0 );
    rc = pkt_write_head( out, size, PKT_PUBKEY_ENC );
    if ( !rc )
        rc = cdk_iobuf_put( out, 3 );
    if ( !rc )
        rc = write_32( out, pke->keyid[0] );
    if ( !rc )
        rc = write_32( out, pke->keyid[1] );
    if ( !rc )
        rc = cdk_iobuf_put( out, pke->pubkey_algo );
    for ( i = 0; i < nenc && !rc; i++ )
        rc = write_mpi( out, pke->mpi[i] );
  
    return rc;
} /* write_pubkey_enc */

static int
write_mdc( CDK_IOBUF out, cdkPKT_mdc *mdc )
{
    int rc = 0;
  
    if ( !out || !mdc )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf( "** write_mdc\n" );
  
    /* This packet requires a fixed header encoding */
    rc = cdk_iobuf_put( out, 0xd3 ); /* packet ID and 1 byte length */
    if ( !rc )
        rc = cdk_iobuf_put( out, 0x14 );
    if ( !rc )
        rc = cdk_iobuf_write( out, mdc->hash, sizeof(mdc->hash) );
    return rc;
} /* write_mdc */

static int
write_signature( CDK_IOBUF out, cdkPKT_signature *sig )
{
    struct cdk_subpkt_s *s = NULL;
    int i, size, nsig = 0;
    int rc = 0;

    if ( !out || !sig )
        return CDKERR_INV_VALUE;

    if ( !KEY_CAN_SIGN( sig->pubkey_algo ) )
        return CDKERR_INV_ALGO;
    else if ( sig->version < 3 || sig->version > 4 )
        return CDKERR_INV_PACKET;

    if ( DEBUG_PKT )
        cdk_printf( "** write signature packet\n" );
  
    nsig = cdk_pk_get_nsig( sig->pubkey_algo );
    if ( sig->version < 4 ) {
        size = 19 + calc_mpi_size( sig->mpi, nsig, 1 );
        rc = pkt_write_head( out, size, PKT_SIGNATURE );
        if ( !rc )
            rc = cdk_iobuf_put( out, 3 );
        if ( !rc )
            rc = cdk_iobuf_put( out, 5 );
        if ( rc )
            rc = cdk_iobuf_put( out, sig->sig_class );
        if ( !rc )
            rc = write_32( out, sig->timestamp );
        if ( !rc )
            rc = write_32( out, sig->keyid[0] );
        if ( !rc )
            rc = write_32( out, sig->keyid[1] );
        if ( !rc )
            rc = cdk_iobuf_put( out, sig->pubkey_algo );
        if ( !rc )
            rc = cdk_iobuf_put( out, sig->digest_algo );
        if ( !rc )
            rc = cdk_iobuf_put( out, sig->digest_start[0] );
        if ( !rc )
            rc = cdk_iobuf_put( out, sig->digest_start[1] );
    }
    else {
        size = 10+cdk_subpkt_calc_size( sig->hashed )
            + cdk_subpkt_calc_size( sig->unhashed )
            + calc_mpi_size( sig->mpi, nsig, 1 );
        rc = pkt_write_head( out, size, PKT_SIGNATURE );
        if ( !rc )
            rc = cdk_iobuf_put( out, 4);
        if ( !rc )
            rc = cdk_iobuf_put( out, sig->sig_class );
        if ( !rc )
            rc = cdk_iobuf_put( out, sig->pubkey_algo );
        if ( !rc )
            rc = cdk_iobuf_put( out, sig->digest_algo );
        if ( !rc )
            rc = write_16( out, sig->hashed_size );
        /*write_16( out, cdk_subpkt_calc_size(sig->hashed) );*/
        for ( s = sig->hashed; s && !rc; s = s->next )
            rc = write_subpkt( out, s );
        if ( !rc )
            rc = write_16( out, sig->unhashed_size );
        /*write_16( out, cdk_subpkt_calc_size(sig->unhashed) );*/
        for ( s = sig->unhashed; s && !rc; s = s->next )
            rc = write_subpkt( out, s );
        if ( !rc )
            rc = cdk_iobuf_put( out, sig->digest_start[0] );
        if ( !rc )
            rc = cdk_iobuf_put( out, sig->digest_start[1] );
    }
  
    for ( i=0; i<nsig && !rc; i++ )
        rc = write_mpi( out, sig->mpi[i] );
  
    return rc;
} /* write_signature */
		
static int
write_public_key( CDK_IOBUF out, cdkPKT_public_key *pk )
{
    int rc = 0;
    int i, size;
    size_t npkey = 0;
	
    if ( !out || !pk )
        return CDKERR_INV_VALUE;
	    
    if ( pk->version < 3 || pk->version > 4 )
        return CDKERR_INV_PACKET;

    if ( DEBUG_PKT )
        cdk_printf( "** write public key packet\n" );

    npkey = cdk_pk_get_npkey( pk->pubkey_algo );
    if ( pk->version < 4 )
        size = 8+calc_mpi_size( pk->mpi, npkey, 0 );
    else
        size = 6+calc_mpi_size( pk->mpi, npkey, 0 );
    rc = pkt_write_head( out, size, PKT_PUBLIC_KEY );
    if ( !rc )
        rc = cdk_iobuf_put( out,  pk->version );
    if ( !rc )
        rc = write_32( out, pk->timestamp );
    if ( !rc && pk->version < 4 )
        rc = write_16( out, pk->expiredate );
    if ( !rc )
        rc = cdk_iobuf_put( out, pk->pubkey_algo );
    for ( i = 0; i < npkey && !rc; i++ )
        rc = write_mpi( out, pk->mpi[i] );
  
    return rc;
} /* write_public_key */

static int
write_public_subkey( CDK_IOBUF out, cdkPKT_public_key *pk )
{
    if ( !out || !pk )
        return CDKERR_INV_VALUE;

    return write_public_key( out, pk );
} /* write_public_subkey */

static int
write_secret_key( CDK_IOBUF out, cdkPKT_secret_key *sk )
{
    cdkPKT_public_key *pk = NULL;
    size_t size = 0;
    size_t npkey, nskey;
    int i = 0;
    int rc = 0;
  
    if ( !out || !sk || !sk->pk )
        return CDKERR_INV_VALUE;

    pk = sk->pk;
    if ( pk->version < 3 || pk->version > 4 )
        return CDKERR_INV_PACKET;

    if ( DEBUG_PKT )
        cdk_printf( "** write secret key packet\n" );

    npkey = cdk_pk_get_npkey( pk->pubkey_algo );
    nskey = cdk_pk_get_nskey( pk->pubkey_algo );

    rc = pkt_write_head( out, size, PKT_SECRET_KEY );
    if ( !rc )
        rc = cdk_iobuf_put( out, pk->version );
    if ( !rc )
        rc = write_32( out, pk->timestamp );
    if ( !rc && pk->version < 4 ) {
        u16 ndays = 0;
        if ( pk->expiredate )
            ndays = (u16)((pk->expiredate - pk->timestamp) / 86400L);
        rc = write_16( out, ndays );
    }
    if ( !rc )
        rc = cdk_iobuf_put( out, pk->pubkey_algo );
    for ( i = 0; i < npkey && !rc; i++ )
        rc = write_mpi( out, pk->mpi[i] );
    if ( sk->is_protected == 0 )
        rc = cdk_iobuf_put( out, 0 );
    else {
        if ( is_RSA( pk->pubkey_algo ) && pk->version < 4 )
            cdk_iobuf_put( out, sk->protect.algo );
        else {
            rc = cdk_iobuf_put( out, 0xff );
            if ( !rc )
                rc = cdk_iobuf_put( out, sk->protect.s2k.mode );
            if ( !rc )
                rc = cdk_iobuf_put( out, sk->protect.s2k.hash_algo );
            if ( !rc && ( sk->protect.s2k.mode == 1
                          || sk->protect.s2k.mode == 3 ) ) { 
                rc = cdk_iobuf_write( out, sk->protect.s2k.salt, 8 );
                if ( !rc && sk->protect.s2k.mode == 3 )
                    rc = cdk_iobuf_put( out, sk->protect.s2k.count );
            }
        }
        rc = cdk_iobuf_write( out, sk->protect.iv, sk->protect.ivlen );
    }
    if ( !rc && sk->is_protected && pk->version > 4 ) {
        if ( sk->encdata && sk->enclen )
            rc = cdk_iobuf_write( out, sk->encdata, sk->enclen );
    }
    else {
        for ( i = 0; i < nskey && !rc; i++ )
            rc = write_mpi( out, sk->mpi[i] );
        if ( !rc )
            rc = write_16( out, sk->csum );
    }

    return rc;
} /* write_secret_key */

static int
write_secret_subkey( CDK_IOBUF out, cdkPKT_secret_key *sk )
{
    if ( !out || !sk )
        return CDKERR_INV_VALUE;

    return write_secret_key( out, sk );
} /* write_secret_subkey */

static int
write_compressed( CDK_IOBUF out, cdkPKT_compressed *cd )
{
    size_t size = 0;
    CDK_BSTRING a;
    int rc = 0;
  
    if ( !out || !cd || !cd->buf )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf( "** write compressed packet\n" );
  
    size = 1 + cd->len;
    rc = pkt_write_head( out, size, PKT_COMPRESSED );
    if ( !rc )
        rc = cdk_iobuf_put( out, cd->algorithm );

    if ( !rc && cd->len ) {
        a = cdk_iobuf_read_mem( cd->buf, 0 );
        if ( !a )
            return CDKERR_INV_VALUE;
        rc = cdk_iobuf_write( out, a->d, a->len );
        cdk_free( a );
    }

    return rc;
} /* write_compressed */

static int
write_plaintext( CDK_IOBUF out, cdkPKT_plaintext *pt )
{
    size_t size = 0;
    CDK_BSTRING a;
    int rc = 0;
  
    if ( !out || !pt )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf( "** write plaintext packet\n" );

    size = 6 + pt->namelen + pt->len;
    rc = pkt_write_head( out, size, PKT_PLAINTEXT );
    if ( !rc )
        rc = cdk_iobuf_put( out, pt->mode );
    if ( !rc )
        rc = cdk_iobuf_put( out, pt->namelen );
    if ( !rc && pt->namelen )
        rc = cdk_iobuf_write( out, pt->name, pt->namelen );
    if ( !rc )
        rc = write_32( out, pt->timestamp );
    if ( !rc && pt->len ) {
        a = cdk_iobuf_read_mem( pt->buf, 0 );
        if ( !a )
            return CDKERR_INV_VALUE;
        rc = cdk_iobuf_write( out, a->d, a->len );
        cdk_free( a );
    }
  
    return rc;
} /* write_plaintext */

static int
write_onepass_sig( CDK_IOBUF out, cdkPKT_onepass_sig *sig )
{
    size_t size = 0;
    int rc = 0;
  
    if ( !out || !sig )
        return CDKERR_INV_VALUE;

    if ( DEBUG_PKT )
        cdk_printf( "** write one pass signature packet\n" );
  
    size = 15;
    rc = pkt_write_head( out, size, PKT_ONEPASS_SIG );
    if ( !rc && sig->version != 3 )
        return CDKERR_INV_PACKET;
    if ( !rc )
        rc = cdk_iobuf_put( out, sig->version );
    if ( !rc )
        rc = cdk_iobuf_put( out, sig->sig_class );
    if ( !rc )
        rc = cdk_iobuf_put( out, sig->digest_algo );
    if ( !rc )
        rc = cdk_iobuf_put( out, sig->pubkey_algo );
    if ( !rc )
        rc = write_32( out, sig->keyid[0] );
    if ( !rc )
        rc = write_32( out, sig->keyid[1] );
    if ( !rc )
        rc = cdk_iobuf_put( out, sig->last );
    
    return rc;
} /* write_onepass_sig */

static int
write_user_id( CDK_IOBUF out, cdkPKT_user_id *id )
{
    int rc = 0;
    
    if ( !out || !id )
        return CDKERR_INV_VALUE;

    if ( id->attrib_img ) {
        /* todo */
    }
    else {
        rc = pkt_write_head( out, id->len, PKT_USER_ID );
        if ( !rc )
            rc = cdk_iobuf_write( out, id->name, id->len );
    }
    return rc;
} /* write_user_id */

int
cdk_pkt_build( CDK_IOBUF out, CDK_PACKET *pkt )
{
    int rc = 0;
  
    if ( !out || !pkt )
        return CDKERR_INV_VALUE;

    switch ( pkt->pkttype ) {
    case PKT_PLAINTEXT:
        rc = write_plaintext( out, pkt->pkt.plaintext );
        break;

    case PKT_ONEPASS_SIG:
        rc = write_onepass_sig( out, pkt->pkt.onepass_sig );
        break;
      
    case PKT_MDC:
        rc = write_mdc( out, pkt->pkt.mdc );
        break;
      
    case PKT_SYMKEY_ENC:
        rc = write_symkey_enc( out, pkt->pkt.symkey_enc );
        break;

    case PKT_ENCRYPTED:
        rc = write_encrypted( out, pkt->pkt.encrypted );
        break;

    case PKT_ENCRYPTED_MDC:
        rc = write_encrypted_mdc( out, pkt->pkt.encrypted );
        break;

    case PKT_PUBKEY_ENC:
        rc = write_pubkey_enc( out, pkt->pkt.pubkey_enc );
        break;

    case PKT_SIGNATURE:
        rc = write_signature( out, pkt->pkt.signature );
        break;

    case PKT_PUBLIC_KEY:
        rc = write_public_key( out, pkt->pkt.public_key );
        break;

    case PKT_PUBLIC_SUBKEY:
        rc = write_public_subkey( out, pkt->pkt.public_key );
        break;

    case PKT_COMPRESSED:
        rc = write_compressed( out, pkt->pkt.compressed );
        break;

    case PKT_SECRET_KEY:
        rc = write_secret_key( out, pkt->pkt.secret_key );
        break;

    case PKT_SECRET_SUBKEY:
        rc = write_secret_subkey( out, pkt->pkt.secret_key );
        break;

    case PKT_USER_ID:
        rc = write_user_id( out, pkt->pkt.user_id );
        break;

    default:
        rc = CDKERR_INV_PACKET;
        break;
    }

    return rc;
} /* cdk_pkt_build */

