/* t-iobuf.c
 *  Copyright (C) 2001, 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#ifdef HAVE_GCRYPT_H
#include <gcrypt.h>
#endif

#include "opencdk.h"
#include "iobuf.h"
#include "main.h"

void
test_iobuf(void)
{
    CDK_IOBUF a;
    char buf[512];
    size_t n = 0;
    int i;
    int rc = 0;

    rc = cdk_iobuf_open(&a, "Makefile.asc", IOBUF_MODE_RD);
    if ( rc ) {
        printf("`%s': can't open!\n", "Makefile.asc");
        return;
    }
    
    while ( !cdk_iobuf_eof(a) ) {
        rc = cdk_iobuf_read_line(a, buf, sizeof(buf)-1, &n);
        if (rc == -1)
            break;
        printf("[%d]\n", n);
        for (i=0; i<n; i++)
            printf("%c", buf[i]);
        printf("\n");
    }
    
    cdk_iobuf_close(a);
}

void
test_iobuf_armor_filter( void )
{
    const char *s = "Makefile.old";
    armor_filter_s afx;
    compress_filter_s cfx;
    int rc = 0;
  
    CDK_IOBUF a;

    rc = cdk_iobuf_open( &a, s, IOBUF_MODE_RD );
    if ( rc ) {
        printf("can't open file `%s'\n", s);
        return;
    }

    cdk_set_logging( 1 );
    printf(" getsize: %d\n", cdk_iobuf_get_length( a ) );
    cdk_iobuf_setmode( a, IOBUF_MODE_WR );
    memset( &cfx, 0, sizeof cfx );
    rc = cdk_compress_filter( &cfx, IOBUF_CTRL_FLUSH, a );
    if ( rc ) {
        printf(" error `%s'\n", cdk_strerror( rc ) );
        cdk_iobuf_close( a );
        return;
    }
    cdk_iobuf_setmode( a, IOBUF_MODE_RD );
    printf(" getsize (compressed): %d\n", cdk_iobuf_get_length( a ) );

    /*
      memset(&afx, 0, sizeof afx);
      rc = cdk_armor_filter( &afx, IOBUF_CTRL_FLUSH, a );
      if (rc)
      {
      printf(" error `%s'\n", cdk_strerror(rc) );
      cdk_iobuf_close( a );
      return;
      }
      */
  
    {
        size_t nbytes = cdk_iobuf_get_length( a );
        /*fwrite( a->data, 1, nbytes, stdout );*/
        printf("EOF `%d'\n", cdk_iobuf_eof(a) );
    
    }
  
    cdk_iobuf_close( a );
}

void
test_iobuf_compr( void )
{
    compress_filter_s cfx;
    CDK_IOBUF a;
    int i;
    int rc = 0;

    cdk_iobuf_new( &a, 8192 );
    for ( i=0; i<32; i++ )
        cdk_iobuf_write( a, "hallo", 4 );
    printf("\niobuf size (%d)\n", cdk_iobuf_get_length( a ) );

    memset( &cfx, 0, sizeof cfx );
    rc = cdk_compress_filter( &cfx, IOBUF_CTRL_FLUSH, a );
    if ( rc )
        printf("compress filter: `%s'\n", cdk_strerror(rc) );
    if ( !rc )
        printf("new iobuf size (%d)\n", cdk_iobuf_get_length( a ) );

    rc = cdk_iobuf_store( a, "test", 1 );
    printf("store rc=%d\n", rc);
}

void
test_iobuf_basic( void )
{
    CDK_IOBUF a, b;
    CDK_BSTRING bs;
    byte buf[512], *p;
    size_t n = 0;
    int rc = 0;

    a = cdk_iobuf_temp();
    printf( "size `%d'\n", cdk_iobuf_get_length(a) );
    rc = cdk_iobuf_write(a, "TEST", 4);
    printf( "iobuf_write=%d\n", rc);
    printf( "size `%d'\n", cdk_iobuf_get_length(a) );
    rc = cdk_iobuf_read(a, buf, 512, &n);
    printf( "iobuf_read=%d\n", rc);
    printf("n `%d': eof `%d'\n", n, cdk_iobuf_eof(a));
    bs = cdk_iobuf_read_mem( a, 0 );
    n = bs->len;
    printf("n=`%d'\n", n);
    while (n--)
        printf("%c", bs->d[n]);
    printf("\n");
    cdk_free( bs );
    cdk_iobuf_close( a );

    cdk_iobuf_new(&a, 4);
    cdk_iobuf_write(a, "TEST", 4);
    printf( "size `%d'\n", cdk_iobuf_get_length(a) );
    cdk_iobuf_expand(a, 3);
    printf( "size `%d'\n", cdk_iobuf_get_length(a) );
    cdk_iobuf_write(a, "FOO", 3);
    printf( "size `%d'\n", cdk_iobuf_get_length(a) );
    bs = cdk_iobuf_read_mem( a, 0 );
    n = bs->len;
    printf("n=%d\n", n);
    while (n--)
        printf("%c", bs->d[n]);
    printf("\n");
    cdk_free( bs );

    b = cdk_iobuf_temp( );
    printf("\n-- copy data\n");
    cdk_iobuf_copy(b, a, 0);
    printf( "n=%d\n", cdk_iobuf_get_length(b) );
    printf("b->size=%d\n", b->size);
    fwrite( b->data->d, 1, b->size, stdout );
  
    cdk_iobuf_close(a);
}  

void
test_iobuf_basic2( void )
{
    CDK_IOBUF a;
    md_filter_s mfx;
    int rc = 0;

    rc = cdk_iobuf_create( &a, "foo_bar" );
    if ( rc ) {
        printf("iobuf_create `%s'\n", cdk_strerror( rc ) );
        return;
    }
    rc = cdk_iobuf_write( a, "test", 4 );
    printf("iobuf_write rc=%d\n", rc);
    rc = cdk_iobuf_write( a, "testest", 8 );
    printf("iobuf_write rc=%d\n", rc);
    cdk_iobuf_close( a );

    memset( &mfx, 0, sizeof mfx );
    mfx.digest_algo = GCRY_MD_SHA1;
    mfx.md = gcry_md_open( GCRY_MD_SHA1, 0 );
    rc = cdk_iobuf_open( &a, "foo_bar", IOBUF_MODE_RD );
    if ( rc ) {
        printf("iobuf_create `%s'\n", cdk_strerror( rc ) );
        return;   
    }
    rc = cdk_md_filter( &mfx, IOBUF_CTRL_FLUSH, a );
    if ( !rc ) {
        int i, n = gcry_md_get_algo_dlen( mfx.digest_algo );
        byte *digest;
        gcry_md_final( mfx.md );
        digest = gcry_md_read( mfx.md, mfx.digest_algo );
        for ( i = 0; i < n; i++ ) {
            printf("%02X", digest[i]);
        }
        printf("\n");
    }
    gcry_md_close( mfx.md );
    cdk_iobuf_close( a );
}
    

void
filter_test( void )
{
    char tmp[] = "This is a test";
    char tmp2[] =
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\n"
        "Comment: A revocation certificate should follow\n"
        "\n"
        "iE4EIBECAA4FAjytw5EHHQJoYWhhIQAKCRC9VyzczMB8NXSkAJ4iMwnzF8/YmxMb\n"
        "fO/WbeAQhIEyvgCfU502zk/iSilojrnFp4Q/BdcYKQ0=\n"
        "=hyVI\n"
        "-----END PGP PUBLIC KEY BLOCK-----\n";
    CDK_IOBUF a;
    armor_filter_s afx;

    a = cdk_iobuf_temp();
    cdk_iobuf_write( a, tmp2, strlen(tmp2) );

    memset( &afx, 0, sizeof afx );
    cdk_armor_filter( &afx, IOBUF_CTRL_UNDERFLOW, a );
    cdk_iobuf_store( a, "test", 1 );
}
    

void
simple_file_test( const char *file )
{
    CDK_IOBUF inp;
    size_t fsize = 0;
  
    if ( !file )
        return;

    if ( cdk_iobuf_open( &inp, file, IOBUF_MODE_RD ) ) {
        printf("%s: Can't open\n", file );
        return;
    }
    fsize = cdk_iobuf_get_length( inp );
    printf("File size %d\n", fsize);
    cdk_iobuf_seek( inp, 10 );

    fsize = cdk_iobuf_tell( inp );
    printf("File pos %d\n", fsize);
    while ( fsize-- )
        printf("%c", cdk_iobuf_get( inp ) );

    cdk_iobuf_close( inp );
}

void
test_dumphex( const char *file )
{
    CDK_IOBUF inp;
    int rc = 0;

    rc = cdk_iobuf_open( &inp, file, IOBUF_MODE_RD );
    if ( rc )
        return rc;
    _cdk_iobuf_hexdump( inp, 0, 0 );
    cdk_iobuf_close( inp );
}

void
test_iobuf_add( void )
{
    CDK_IOBUF inp, t;
    CDK_BSTRING a;
    int i = 0;

    cdk_iobuf_new( &inp, 1024 );
    cdk_iobuf_write( inp, "test", 4 );
    cdk_iobuf_create( &t, NULL );
    for ( i = 0; i < 15000; i++ )
        cdk_iobuf_write( t, "rastrast", 8 );
    cdk_iobuf_copy( inp, t, 1 );
    
    a = cdk_iobuf_read_mem( inp, 0 );
    if ( a ) {
        printf("%8s\n", a->d );
        cdk_free( a );
    }

    cdk_iobuf_close( t );
    cdk_iobuf_close( inp );
}

int
main(int argc, char **argv)
{
#if 0
    if (argc == 2)
        test_iobuf_filters(argv[1]);
#endif
    /*test_iobuf_compr( );*/
    /*test_iobuf();*/
    /*test_iobuf_basic2( );*/
    test_iobuf_add();
    /*if (argc == 2)
      simple_file_test(argv[1]);*/
    /*test_iobuf_armor_filter();*/
    /*filter_test();*/
    /*test_dumphex("foo");*/
    return 0;
}
    







