/* keyid.c - key ID and fingerprint handling
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"
#include "types.h"
#include "packet.h"


int
cdk_pk_get_fingerprint (cdkPKT_public_key * pk, byte * fpr)
{
  GCRY_MD_HD hd;
  int md_algo;
  int dlen = 0;

  if (!pk || !fpr)
    return CDK_Inv_Value;

  if (pk->version < 4 && is_RSA (pk->pubkey_algo))
    md_algo = GCRY_MD_MD5; /* special */
  else
    md_algo = pk->version < 4 ? GCRY_MD_RMD160 : GCRY_MD_SHA1;
  dlen = gcry_md_get_algo_dlen (md_algo);
  hd = gcry_md_open (md_algo, 0);
  if (!hd)
    return CDK_Gcry_Error;
  _cdk_hash_pubkey (pk, hd);
  gcry_md_final (hd);
  memcpy (fpr, gcry_md_read (hd, md_algo), dlen);
  gcry_md_close (hd);
  if (dlen == 16)
    memset (fpr + 16, 0, 4);
  return 0;
}


static u32
fetch_v3_key_byfpr (const byte * fpr, size_t fprlen, u32 * keyid)
{
  CDK_KEYDB_HD khd;
  CDK_KBNODE knode, pk;
  u32 lowbits = 0;
  int rc;

  khd = cdk_keydb_get_ctx (0, -1);
  if (khd)
    {
      rc = cdk_keydb_get_byfpr (khd, fpr, &knode);
      if (!rc && keyid)
	{
	  pk = cdk_kbnode_find (knode, CDK_PKT_PUBLIC_KEY);
	  if (pk)
	    {
	      cdk_pk_get_keyid (pk->pkt->pkt.public_key, keyid);
	      lowbits = keyid[1];
	    }
	  cdk_kbnode_release (knode);
	}
    }
  return lowbits;
}


u32
cdk_pk_fingerprint_get_keyid (const byte * fpr, size_t fprlen, u32 * keyid)
{
  u32 lowbits = 0;

  /* in this case we say the key is a V3 RSA key and we can't
     use the fingerprint to get the keyid. */
  if (fpr && fprlen != 20)
    lowbits = fetch_v3_key_byfpr (fpr, fprlen, keyid);
  else if (keyid && fpr)
    {
      keyid[0] = buffer_to_u32 (fpr + 12);
      keyid[1] = buffer_to_u32 (fpr + 16);
      lowbits = keyid[1];
    }
  else if (fpr)
    lowbits = buffer_to_u32 (fpr + 16);

  return lowbits;
}


u32
cdk_pk_get_keyid (cdkPKT_public_key * pk, u32 * keyid)
{
  u32 lowbits = 0;
  byte buf[24];

  if (pk && (!pk->keyid[0] || !pk->keyid[1]))
    {
      if (pk->version < 4 && is_RSA (pk->pubkey_algo))
	{
	  size_t n = pk->mpi[0]->bytes;
	  const byte * p = pk->mpi[0]->data + 2;
	  pk->keyid[0] = p[n-8] << 24 | p[n-7] << 16 | p[n-6] << 8 | p[n-5];
	  pk->keyid[1] = p[n-4] << 24 | p[n-3] << 16 | p[n-2] << 8 | p[n-1];
	}
      else if (pk->version == 4)
	{
	  cdk_pk_get_fingerprint (pk, buf);
	  pk->keyid[0] = buffer_to_u32 (buf + 12);
	  pk->keyid[1] = buffer_to_u32 (buf + 16);
	}
    }
  lowbits = pk ? pk->keyid[1] : 0;
  if (keyid && pk)
    {
      keyid[0] = pk->keyid[0];
      keyid[1] = pk->keyid[1];
    }
  return lowbits;
}


u32
cdk_sk_get_keyid (cdkPKT_secret_key * sk, u32 * keyid)
{
  return sk && sk->pk? cdk_pk_get_keyid (sk->pk, keyid) : 0;
}


u32
cdk_sig_get_keyid (cdkPKT_signature * sig, u32 * keyid)
{
  u32 lowbits = sig ? sig->keyid[1] : 0;
  if (keyid && sig)
    {
      keyid[0] = sig->keyid[0];
      keyid[1] = sig->keyid[1];
    }
  return lowbits;
}


u32
_cdk_keyid_get (CDK_PACKET * pkt, u32 * keyid)
{
  u32 lowbits;
  
  switch (pkt->pkttype)
    {
    case CDK_PKT_PUBLIC_KEY:
    case CDK_PKT_PUBLIC_SUBKEY:
      lowbits = cdk_pk_get_keyid (pkt->pkt.public_key, keyid);
      break;
      
    case CDK_PKT_SECRET_KEY:
    case CDK_PKT_SECRET_SUBKEY:
      lowbits = cdk_sk_get_keyid (pkt->pkt.secret_key, keyid);
      break;

    case CDK_PKT_SIGNATURE:
      lowbits = cdk_sig_get_keyid (pkt->pkt.signature, keyid);
      break;
      
    default:
      lowbits = 0;
    }
  return lowbits;
}


int
_cdk_fingerprint_get (CDK_PACKET * pkt, byte * fpr)
{
  switch (pkt->pkttype)
    {
    case CDK_PKT_PUBLIC_KEY:
    case CDK_PKT_PUBLIC_SUBKEY:
      return cdk_pk_get_fingerprint (pkt->pkt.public_key, fpr);

    case CDK_PKT_SECRET_KEY:
    case CDK_PKT_SECRET_SUBKEY:
      return cdk_pk_get_fingerprint (pkt->pkt.secret_key->pk, fpr);

    default:
      return CDK_Inv_Packet;
    }

  return 0;
}
