/* new-packet.c - General packet handling (freeing, copying, ...)
 *       Copyright (C) 2001, 2002 Timo Schulz 
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <string.h>
#include <stdio.h>
#include <assert.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"


void
_cdk_free_mpi_array (size_t n, GCRY_MPI * array)
{
  while (n--)
    {
      gcry_mpi_release (array[n]);
      array[n] = NULL;
    }
}


int
cdk_pkt_new (CDK_PACKET ** r_pkt)
{
  CDK_PACKET *pkt;

  pkt = cdk_calloc (1, sizeof *pkt);
  if (!pkt)
    return CDK_Out_Of_Core;
  pkt->pkttype = 0;
  *r_pkt = pkt;

  return 0;
}


static void
free_pubkey_enc (cdkPKT_pubkey_enc *enc)
{
  int nenc;
  if (enc)
    {
      nenc = cdk_pk_get_nenc (enc->pubkey_algo);
      while (enc->mpi && nenc--)
        {
          cdk_free (enc->mpi[nenc]);
          enc->mpi[nenc] = NULL;
        }
      cdk_free (enc);
    }
}


static void
free_plaintext (cdkPKT_plaintext *pt)
{
  cdk_free (pt);
}


void
_cdk_free_userid (cdkPKT_user_id * uid)
{
  if (uid)
    {
      cdk_free (uid->prefs);
      uid->prefs = NULL;
      cdk_free (uid->attrib_img);
      uid->attrib_img = NULL;      
      cdk_free (uid);
    }
}


void
_cdk_free_signature (cdkPKT_signature * sig)
{
  int nsig;
  if (sig)
    {
      nsig = cdk_pk_get_nsig (sig->pubkey_algo);
      while (sig->mpi && nsig--)
        {
          cdk_free (sig->mpi[nsig]);
          sig->mpi[nsig] = NULL;
        }
      cdk_subpkt_free (sig->hashed);
      sig->hashed = NULL;
      cdk_subpkt_free (sig->unhashed);
      sig->unhashed = NULL;
      cdk_free (sig);
    }
}


void
_cdk_free_pubkey (cdkPKT_public_key * pk)
{
  int npkey;
  
  if (pk)
    {
      npkey = cdk_pk_get_npkey (pk->pubkey_algo);
      _cdk_free_userid (pk->uid);
      pk->uid = NULL;
      cdk_free (pk->prefs);
      pk->prefs = NULL;
      while (pk->mpi && npkey--)
        {
          cdk_free (pk->mpi[npkey]);
          pk->mpi[npkey] = NULL; 
        }
      cdk_free (pk);
    }
}


void
_cdk_free_seckey (cdkPKT_secret_key * sk)
{
  int nskey;
  
  if (sk)
    {
      nskey = cdk_pk_get_nskey (sk->pubkey_algo);
      while (sk->mpi && nskey--)
        {
          cdk_free (sk->mpi[nskey]);
          sk->mpi[nskey] = NULL; 
        }
      cdk_free (sk->encdata);
      sk->encdata = NULL;
      _cdk_free_pubkey (sk->pk);
      sk->pk = NULL;
      cdk_free (sk);
    }
}


static void
free_encrypted (cdkPKT_encrypted *enc)
{
  if (enc)
    {
      cdk_stream_close (enc->buf);
      enc->buf = NULL;
      cdk_free (enc);
    }
}


void
cdk_pkt_free (CDK_PACKET * pkt)
{
  if (!pkt)
    return;

  switch (pkt->pkttype)
    {
    case CDK_PKT_ATTRIBUTE    :
    case CDK_PKT_USER_ID      : _cdk_free_userid (pkt->pkt.user_id); break;
    case CDK_PKT_PUBLIC_KEY   :
    case CDK_PKT_PUBLIC_SUBKEY: _cdk_free_pubkey (pkt->pkt.public_key); break;
    case CDK_PKT_SECRET_KEY   :
    case CDK_PKT_SECRET_SUBKEY: _cdk_free_seckey (pkt->pkt.secret_key); break;
    case CDK_PKT_SIGNATURE    : _cdk_free_signature (pkt->pkt.signature); break;
    case CDK_PKT_PUBKEY_ENC   : free_pubkey_enc (pkt->pkt.pubkey_enc); break;
    case CDK_PKT_SYMKEY_ENC   : cdk_free (pkt->pkt.symkey_enc); break;
    case CDK_PKT_MDC          : cdk_free (pkt->pkt.mdc); break;
    case CDK_PKT_ENCRYPTED    :
    case CDK_PKT_ENCRYPTED_MDC: free_encrypted (pkt->pkt.encrypted); break;
    case CDK_PKT_ONEPASS_SIG  : cdk_free (pkt->pkt.onepass_sig); break;
    case CDK_PKT_PLAINTEXT    : free_plaintext (pkt->pkt.plaintext); break;
    case CDK_PKT_COMPRESSED   : cdk_free (pkt->pkt.compressed); break;
    default               : break;
    }
}


void
cdk_pkt_release (CDK_PACKET * pkt)
{
  if (pkt)
    {
      cdk_pkt_free (pkt);
      cdk_free (pkt);
    }
}


int
cdk_pkt_alloc (CDK_PACKET ** r_pkt, int id)
{
  CDK_PACKET *pkt;
  int rc = 0;

  rc = cdk_pkt_new (&pkt);
  if (rc)
    return rc;

  switch (id)
    {
    case CDK_PKT_USER_ID:
      pkt->pkt.user_id = cdk_calloc (1, sizeof *pkt->pkt.user_id);
      if (!pkt->pkt.user_id)
	return CDK_Out_Of_Core;
      break;

    case CDK_PKT_PUBLIC_KEY:
    case CDK_PKT_PUBLIC_SUBKEY:
      pkt->pkt.public_key = cdk_calloc (1, sizeof *pkt->pkt.public_key);
      if (!pkt->pkt.public_key)
	return CDK_Out_Of_Core;
      break;

    case CDK_PKT_SECRET_KEY:
    case CDK_PKT_SECRET_SUBKEY:
      pkt->pkt.secret_key = cdk_calloc (1, sizeof *pkt->pkt.secret_key);
      pkt->pkt.secret_key->pk =
	cdk_calloc (1, sizeof *pkt->pkt.secret_key->pk);
      if (!pkt->pkt.secret_key || !pkt->pkt.secret_key->pk)
	return CDK_Out_Of_Core;
      break;

    case CDK_PKT_SIGNATURE:
      pkt->pkt.signature = cdk_calloc (1, sizeof *pkt->pkt.signature);
      if (!pkt->pkt.signature)
	return CDK_Out_Of_Core;
      break;

    case CDK_PKT_PUBKEY_ENC:
      pkt->pkt.pubkey_enc = cdk_calloc (1, sizeof *pkt->pkt.pubkey_enc);
      if (!pkt->pkt.pubkey_enc)
	return CDK_Out_Of_Core;
      break;

    case CDK_PKT_MDC:
      pkt->pkt.mdc = cdk_calloc (1, sizeof *pkt->pkt.mdc);
      if (!pkt->pkt.mdc)
	return CDK_Out_Of_Core;
      break;

    case CDK_PKT_ENCRYPTED_MDC:
    case CDK_PKT_ENCRYPTED:
      pkt->pkt.symkey_enc = cdk_calloc (1, sizeof *pkt->pkt.symkey_enc);
      if (!pkt->pkt.symkey_enc)
	return CDK_Out_Of_Core;
      break;

    case CDK_PKT_PLAINTEXT:
      pkt->pkt.plaintext = cdk_calloc (1, sizeof *pkt->pkt.plaintext);
      if (!pkt->pkt.plaintext)
	return CDK_Out_Of_Core;
      break;
    }
  pkt->pkttype = id;
  *r_pkt = pkt;

  return 0;
}


struct cdk_prefitem_s *
_cdk_copy_prefs (const struct cdk_prefitem_s * prefs)
{
  size_t n = 0;
  struct cdk_prefitem_s *new_prefs;
  
  if (!prefs)
    return NULL;

  for (n = 0; prefs[n].type; n++)
    ;
  new_prefs = cdk_calloc (1, sizeof *new_prefs * (n + 1));
  if (!new_prefs)
    return NULL;
  for (n = 0; prefs[n].type; n++)
    {
      new_prefs[n].type = prefs[n].type;
      new_prefs[n].value = prefs[n].value;
    }
  new_prefs[n].type = PREFTYPE_NONE;
  new_prefs[n].value = 0;

  return new_prefs;
}


int
_cdk_copy_userid (cdkPKT_user_id ** dst, cdkPKT_user_id * src)
{
  cdkPKT_user_id *u;

  if (!dst || !src)
    return CDK_Inv_Value;

  u = cdk_calloc (1, sizeof *u + strlen (src->name) + 1);
  if (!u)
    return CDK_Out_Of_Core;
  memcpy (u, src, sizeof *u);
  memcpy (u->name, src->name, strlen (src->name));
  u->prefs = _cdk_copy_prefs (src->prefs);
  *dst = u;

  return 0;
}


int
_cdk_copy_pubkey (cdkPKT_public_key ** dst, cdkPKT_public_key * src)
{
  cdkPKT_public_key *k;
  int i;

  if (!dst || !src)
    return CDK_Inv_Value;

  k = cdk_calloc (1, sizeof *k);
  if (!k)
    return CDK_Out_Of_Core;
  memcpy (k, src, sizeof *k);
  if (src->uid)
    _cdk_copy_userid (&k->uid, src->uid);
  if (src->prefs)
    k->prefs = _cdk_copy_prefs (src->prefs);
  for (i = 0; i < cdk_pk_get_npkey (src->pubkey_algo); i++)
    {
      k->mpi[i] = cdk_calloc (1, sizeof **k->mpi + src->mpi[i]->bytes + 2);
      if (!k->mpi[i])
	return CDK_Out_Of_Core;
      k->mpi[i]->bits = src->mpi[i]->bits;
      k->mpi[i]->bytes = src->mpi[i]->bytes;
      /* copy 2 extra bytes (prefix) */
      memcpy (k->mpi[i]->data, src->mpi[i]->data, src->mpi[i]->bytes + 2);
    }
  *dst = k;

  return 0;
}


int
_cdk_copy_seckey (cdkPKT_secret_key ** dst, cdkPKT_secret_key * src)
{
  cdkPKT_secret_key *k;
  int i;

  if (!dst || !src)
    return CDK_Inv_Value;

  k = cdk_calloc (1, sizeof *k);
  if (!k)
    return CDK_Out_Of_Core;
  memcpy (k, src, sizeof *k);
  _cdk_copy_pubkey (&k->pk, src->pk);

  for (i = 0; i < cdk_pk_get_nskey (src->pubkey_algo); i++)
    {
      k->mpi[i] = cdk_calloc (1, sizeof **k->mpi + src->mpi[i]->bytes + 2);
      if (!k->mpi[i])
	return CDK_Out_Of_Core;
      k->mpi[i]->bits = src->mpi[i]->bits;
      k->mpi[i]->bytes = src->mpi[i]->bytes;
      /* copy 2 extra bytes (prefix) */
      memcpy (k->mpi[i]->data, src->mpi[i]->data, src->mpi[i]->bytes + 2);
    }
  *dst = k;

  return 0;
}


int
_cdk_copy_pk_to_sk (cdkPKT_public_key * pk, cdkPKT_secret_key * sk)
{
  if (!pk || !sk)
    return CDK_Inv_Value;

  sk->version = pk->version;
  sk->expiredate = pk->expiredate;
  sk->pubkey_algo = pk->pubkey_algo;
  sk->has_expired = pk->has_expired;
  sk->is_revoked = pk->is_revoked;
  sk->main_keyid[0] = pk->main_keyid[0];
  sk->main_keyid[1] = pk->main_keyid[1];
  sk->keyid[0] = pk->keyid[0];
  sk->keyid[1] = pk->keyid[1];

  return 0;
}


int
_cdk_copy_signature (cdkPKT_signature ** dst, cdkPKT_signature * src)
{
  cdkPKT_signature *s = NULL;
  struct cdk_subpkt_s *res = NULL;

  if (!dst || !src)
    return CDK_Inv_Value;

  s = cdk_calloc (1, sizeof *s);
  if (!s)
    return CDK_Out_Of_Core;
  memcpy (s, src, sizeof *src);

  _cdk_subpkt_copy (&res, src->hashed);
  _cdk_subpkt_copy (&s->hashed, res);
  cdk_subpkt_free (res);
  res = NULL;
  _cdk_subpkt_copy (&res, src->unhashed);
  _cdk_subpkt_copy (&s->unhashed, res);
  cdk_subpkt_free (res);
  res = NULL;
  *dst = s;

  return 0;
}


int
_cdk_pubkey_compare (cdkPKT_public_key * a, cdkPKT_public_key * b)
{
  int na, nb, i;

  if (a->timestamp != b->timestamp
      || a->pubkey_algo != b->pubkey_algo)
    return -1;
  if (a->version < 4 && a->expiredate != b->expiredate)
    return -1;
  na = cdk_pk_get_npkey (a->pubkey_algo);
  nb = cdk_pk_get_npkey (b->pubkey_algo);
  if (na != nb)
    return -1;
  
  for (i = 0; i < na; i++)
    {
      if (memcmp (a->mpi[i]->data, b->mpi[i]->data, a->mpi[i]->bytes))
	return -1;
    }

  return 0;
}


void
cdk_subpkt_free (CDK_SUBPKT ctx)
{
  struct cdk_subpkt_s *s;

  while (ctx)
    {
      s = ctx->next;
      cdk_free (ctx);
      ctx = s;
    }
}


struct cdk_subpkt_s *
cdk_subpkt_find (CDK_SUBPKT ctx, int type)
{
  struct cdk_subpkt_s *s;

  for (s = ctx; s; s = s->next)
    {
      if (s->type == type)
	return s;
    }

  return NULL;
}


CDK_SUBPKT
cdk_subpkt_new (size_t size)
{
  CDK_SUBPKT s;

  s = cdk_calloc (1, sizeof *s + size + 1);
  if (!s)
    return NULL;

  return s;
}


const byte *
cdk_subpkt_get_buf (CDK_SUBPKT ctx, int * r_type, size_t * r_nbytes)
{
  if (!ctx)
    return NULL;
  if (r_type)
    *r_type = ctx->type;
  if (r_nbytes)
    *r_nbytes = ctx->size;
  return ctx->d;
}
  
  
int
_cdk_subpkt_add (CDK_SUBPKT root, CDK_SUBPKT node)
{
  CDK_SUBPKT n1;

  if (!root)
    return CDK_Inv_Value;

  for (n1 = root; n1->next; n1 = n1->next)
    ;
  n1->next = node;
  return 0;
}


byte *
_cdk_subpkt_get_array (CDK_SUBPKT s, int count, size_t *r_nbytes)
{
  CDK_SUBPKT list;
  byte *buf;
  int n, nbytes;
  
  if (!s)
    return NULL;

  buf = cdk_calloc (1, 4096);
  if (!buf)
    return NULL;

  n = 0;
  for (list = s; list; list = list->next)
    {
      nbytes = 1 + list->size; /* type */
      if (nbytes < 192)
        buf[n++] = nbytes;
      else if (nbytes < 8384)
        {
          buf[n++] = nbytes / 256 + 192;
          buf[n++] = nbytes % 256;
        }
      else
        {
          buf[n++] = nbytes >> 24;
          buf[n++] = nbytes >> 16;
          buf[n++] = nbytes >>  8;
          buf[n++] = nbytes;
        }
      buf[n++] = list->type;
      memcpy (buf + n, list->d, list->size);
      n += list->size;
      assert (n < 4096);
    }
  if (count)
    {
      cdk_free (buf);
      buf = NULL;
    }
  if (r_nbytes)
    *r_nbytes = n;
  return buf;
}


int
_cdk_subpkt_copy (struct cdk_subpkt_s ** r_dst, struct cdk_subpkt_s * src)
{
  struct cdk_subpkt_s * root, * p, * node;

  if (!src || !r_dst)
    return CDK_Inv_Value;
  
  root = NULL;
  for (p = src; p; p = p->next)
    {
      node = cdk_subpkt_new (p->size);
      if (node)
        {
          memcpy (node->d, p->d, p->size);
          node->type = p->type;
          node->size = p->size;
        }
      if (!root)
        root = node;
      else
        _cdk_subpkt_add (root, node);
    }
  *r_dst = root;

  return 0;
}


int
_cdk_subpkt_hash (CDK_SUBPKT hashed, size_t *r_nbytes, GCRY_MD_HD hd)
{
  byte *p, buf[2];
  size_t nbytes;

  p = _cdk_subpkt_get_array (hashed, 0, &nbytes);
  if (!p)
    return CDK_Out_Of_Core;
  if (nbytes > 65535)
    return CDK_Inv_Value;
  buf[0] = nbytes >> 8;
  buf[1] = nbytes;
  gcry_md_write (hd, buf, 2);
  gcry_md_write (hd, p, nbytes);
  if (r_nbytes)
    *r_nbytes = nbytes;
  
  return 0;
}


