/* sign.c - Signing routines
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <time.h>
#include <string.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"
#include "filters.h"
#include "stream.h"


int
_cdk_sig_hash_for (int pubkey_algo, int pktver)
{
  if (is_DSA (pubkey_algo))
    return GCRY_MD_SHA1;
  else if (is_RSA (pubkey_algo) && pktver < 4)
    return GCRY_MD_MD5;
  return GCRY_MD_SHA1;	/* default message digest */
}


int
_cdk_sig_create (cdkPKT_public_key * pk, cdkPKT_signature * sig)
{
  byte buf[8];
  size_t nbytes;
  struct cdk_subpkt_s * node;

  if (!sig)
    return CDK_Inv_Value;

  if (pk)
    {
      sig->version = pk->version;
      sig->pubkey_algo = pk->pubkey_algo;
      sig->digest_algo = _cdk_sig_hash_for (pk->pubkey_algo, pk->version);
      cdk_pk_get_keyid (pk, sig->keyid);
    }
  sig->timestamp = make_timestamp ();
  sig->hashed = sig->unhashed = NULL;

  u32_to_buffer (sig->keyid[0], buf);
  u32_to_buffer (sig->keyid[1], buf + 4);
  node = cdk_subpkt_new (8);
  if (node)
    {
      node->type = SIGSUBPKT_ISSUER;
      memcpy (node->d, buf, 8);
      node->size = 8;
    }
  sig->unhashed = node;
  
  u32_to_buffer (sig->timestamp, buf);
  node = cdk_subpkt_new (4);
  if (node)
    {
      node->type = SIGSUBPKT_SIG_CREATED;
      memcpy (node->d, buf, 4);
      node->size = 4;
    }
  sig->hashed = node;
  
  if (sig->expiredate)
    {
      u32 u = sig->expiredate - sig->timestamp;
      u32_to_buffer (u, buf);
      node = cdk_subpkt_new (4);
      if (node)
        {
          node->type = SIGSUBPKT_SIG_EXPIRE;
          memcpy (node->d, buf, 4);
          node->size = 4;
        }
      _cdk_subpkt_add (sig->hashed, node);
    }

  if (sig->hashed)
    {
      _cdk_subpkt_get_array (sig->hashed, 1, &nbytes);
      sig->hashed_size = nbytes;
    }
  if (sig->unhashed)
    {
      _cdk_subpkt_get_array (sig->unhashed, 1, &nbytes);
      sig->unhashed_size = nbytes;
    }
  return 0;
}


/**
 * cdk_file_sign: Sign a file.
 * @locusr: List of userid which should be used for signing
 * @file: Name of the input file
 * @output: Name of the output file
 * @detached: Detach the signature?
 *
 **/
int
cdk_file_sign (CDK_STRLIST locusr, const char * file, const char * output,
               int detached)
{
  CDK_STREAM inp = NULL, outp = NULL;
  CDK_KEYLIST list;
  cdkPKT_secret_key * sk;
  int rc = 0, digest_algo;
  md_filter_s * mfx;

  if (!file || !output)
    return CDK_Inv_Value;
  if (detached != 1)
    return CDK_Not_Implemented;

  rc = cdk_stream_open (file, &inp);
  if (rc)
    return rc;

  rc = cdk_stream_new (output, &outp);
  if (rc)
    {
      cdk_stream_close (inp);
      return rc;
    }

  rc = cdk_sklist_build (locusr, &list, 1, GCRY_PK_USAGE_SIGN);
  if (rc)
    goto leave;

  sk = list->key.sk;
  digest_algo = _cdk_sig_hash_for (sk->pubkey_algo, sk->version);

  /* kick off the filter */
  cdk_stream_set_hash_flag (inp, digest_algo);
  cdk_stream_read (inp, NULL, 0);
  mfx = _cdk_stream_get_opaque (inp, 4);
  if (mfx && mfx->md)
    {
      rc = cdk_sklist_write (list, outp, mfx->md, 0x00);
      gcry_md_close (mfx->md);
    }

leave:  
  cdk_stream_close (inp);
  cdk_stream_close (outp);
  return rc;
}


static void
put_hash_line (CDK_STREAM out, int digest_algo)
{
  const char * s = NULL;
  
  switch (digest_algo)
    {
    case GCRY_MD_MD5   : s = "Hash: MD5\n\n"; break;
    case GCRY_MD_SHA1  : s = "Hash: SHA1\n\n"; break;
    case GCRY_MD_RMD160: s = "Hash: RIPEMD160\n\n"; break;
    default            : s = "Hash: SHA1\n\n"; break;
    }
  cdk_stream_write (out, s, strlen (s));
}


/* FIXME: this construct is ugly and we need to convert it in something
   like enarmor_tmp_stream() soon...*/
static int
clearsign_filter (CDK_STREAM s)
{
  armor_filter_s afx;
  FILE * in, * out;
  char buf[1024];
  int nread, rc;

  memset (&afx, 0, sizeof afx);
  afx.idx = afx.idx2 = CDK_ARMOR_SIGNATURE;

  cdk_stream_seek (s, 0);
  in = s->fp;
  out = tmpfile ();

  rc = armor_encode (&afx, in, out);
  fseek (out, 0, SEEK_SET);
  cdk_stream_seek (s, 0);

  while (!feof (out))
    {
      nread = fread (buf, 1, sizeof buf-1, out);
      if (!nread)
        break;
      buf[nread] = '\0';
      fwrite (buf, 1, nread, s->fp);
    }
  fclose (out);
  return 0;
}


void
_cdk_trim_string (char * s, int canon)
{
  while (s && *s && (   s[strlen (s)-1] == '\t'
                     || s[strlen (s)-1] == '\r'
                     || s[strlen (s)-1] == '\n'
                     || s[strlen (s)-1] == ' '))
    s[strlen (s) -1] = '\0';
  if (canon)
    strcat (s, "\r\n");
}


/**
 * cdk_file_clearsign: Make a cleartext signature of a file.
 * @hd: Handle
 * @locusr: List of userIDs
 * @file: Name of the intput file
 * @output: Name of the output file
 *
 **/
int
cdk_file_clearsign (CDK_HD hd, CDK_STRLIST locusr,
                    const char * file, const char * output)
{
  GCRY_MD_HD md = NULL;
  CDK_STREAM inp = NULL, out = NULL, tmp;
  CDK_KEYLIST list;
  cdkPKT_secret_key * sk;
  const char * s;
  int digest_algo;
  char buf[1024+2];
  int rc, nread = 0;
  
  if (!locusr || !file || !output)
    return CDK_Inv_Value;

  rc = cdk_stream_open (file, &inp);
  if (rc)
    return rc;
  
  rc = cdk_stream_new (output, &out);
  if (rc)
    {
      cdk_stream_close (inp);
      return rc;
    }

  rc = cdk_sklist_build (locusr, &list, 1, GCRY_PK_USAGE_SIGN);
  if (rc)
    goto leave;

  sk = list->key.sk;
  digest_algo = _cdk_sig_hash_for (sk->pubkey_algo, sk->version);
  md = gcry_md_open (digest_algo, 0);
  if (!md)
    {
      rc = CDK_Gcry_Error;
      goto leave;
    }

  s = "-----BEGIN PGP SIGNED MESSAGE-----\n";
  cdk_stream_write (out, s, strlen (s));
  put_hash_line (out, digest_algo);
  
  while (!cdk_stream_eof (inp))
    {
      nread = _cdk_stream_gets (inp, buf, sizeof buf-1);
      if (!nread)
        break;
      _cdk_trim_string (buf, 1);
      gcry_md_write (md, buf, strlen (buf));
      if (buf[0] == '-')
        {
          memmove (&buf[2], buf, nread + 1);
          buf[1] = ' ';
        }
      buf[strlen (buf) - 1] = '\0';
      buf[strlen (buf) - 1] = '\n';
      cdk_stream_write (out, buf, strlen (buf));
    }
  cdk_stream_putc (out, '\n');
  tmp = cdk_stream_tmp ();
  if (!tmp)
    {
      rc = CDK_Out_Of_Core;
      goto leave;
    }
  rc = cdk_sklist_write (list, tmp, md, 0x01);
  if (rc)
    {
      cdk_stream_close (tmp);
      goto leave;
    }
  clearsign_filter (tmp);
  cdk_stream_seek (tmp, 0);

  while (!cdk_stream_eof (tmp))
    {
      nread = cdk_stream_read (tmp, buf, sizeof buf-1);
      if (!nread)
        break;
      cdk_stream_write (out, buf, nread);
    }
  cdk_stream_close (tmp);
  
 leave:
  gcry_md_close (md);
  cdk_stream_close (inp);
  cdk_stream_close (out);
  return rc;
}

