/* pubkey.c
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"

static int key_to_sexp (GCRY_SEXP * r_key, cdkPKT_public_key * pk);
static int skey_to_sexp (GCRY_SEXP * r_skey, cdkPKT_secret_key * sk);
static int esk_to_sexp (GCRY_SEXP * r_sexp, GCRY_MPI esk);
static int sig_to_sexp (GCRY_SEXP * r_sig, cdkPKT_signature * sig);
static int sexp_to_sig (cdkPKT_signature * sig, GCRY_SEXP sexp);
static int md_to_sexp (GCRY_SEXP * r_md, int algo, const byte * md, size_t);
static int sexp_to_enc (cdkPKT_pubkey_enc * enc, GCRY_SEXP sexp);
static int enc_to_sexp (GCRY_SEXP * r_sexp, cdkPKT_pubkey_enc * enc);


#ifdef DEBUG
void
dump_sexp (GCRY_SEXP sexp)
{
  GCRY_SEXP x;

  fprintf (stderr, "DUMP of SEXP:\n");
  gcry_sexp_dump (sexp);
  x = gcry_sexp_cdr (sexp);
  if (x)
    {
      fprintf (stderr, "DUMP of CDR:\n");
      gcry_sexp_dump (x);
      gcry_sexp_release (x);
    }
}
#endif


static GCRY_MPI *
bitmpi_to_gcrympi (CDK_MPI m[4], int ncount)
{
  int i = 0, rc = 0;
  GCRY_MPI * d = NULL;
  size_t nbytes = 0;

  if (!m)
    return NULL;

  d = cdk_calloc (ncount, sizeof *d);
  if (!d)
    return NULL;
  for (i = 0; i < ncount; i++)
    {
      nbytes = m[i]->bytes + 2;
      rc = gcry_mpi_scan (&d[i], GCRYMPI_FMT_PGP, m[i]->data, &nbytes);
      if (rc)
	{
	  rc = CDK_Gcry_Error;
	  break;
	}
    }
  if (rc)
    _cdk_free_mpi_array (i, d);
  return d;
}


static int
skey_to_sexp (GCRY_SEXP * r_skey, cdkPKT_secret_key * sk)
{
  cdkPKT_public_key *pk = NULL;
  GCRY_SEXP sexp = NULL;
  GCRY_MPI * m_pk = NULL, * m_sk = NULL;
  const char * fmt = NULL;
  int ncount = 0, nscount = 0;
  int rc = 0;

  if (!r_skey || !sk || !sk->pk)
    return CDK_Inv_Value;

  pk = sk->pk;
  ncount = cdk_pk_get_npkey (pk->pubkey_algo);
  m_pk = bitmpi_to_gcrympi (pk->mpi, ncount);
  if (!m_pk)
    return CDK_MPI_Error;
  nscount = cdk_pk_get_nskey (sk->pubkey_algo);
  m_sk = bitmpi_to_gcrympi (sk->mpi, nscount);
  if (!m_sk)
    {
      rc = CDK_MPI_Error;
      goto leave;
    }
  if (is_RSA (sk->pubkey_algo))
    {
      fmt = "(private-key(openpgp-rsa(n%m)(e%m)(d%m)(p%m)(q%m)(u%m)))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m_pk[0], m_pk[1],
			    m_sk[0], m_sk[1], m_sk[2], m_sk[3]);
    }
  else if (is_ELG (sk->pubkey_algo))
    {
      fmt = "(private-key(openpgp-elg(p%m)(g%m)(y%m)(x%m)))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m_pk[0], m_pk[1],
			    m_pk[2], m_sk[0]);
    }
  else if (is_DSA (sk->pubkey_algo))
    {
      fmt = "(private-key(openpgp-dsa(p%m)(q%m)(g%m)(y%m)(x%m)))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m_pk[0], m_pk[1], m_pk[2],
			    m_pk[3], m_sk[0]);
    }
  if (rc)
    rc = CDK_Gcry_Error;
  if (!fmt)
    rc = CDK_Inv_Algo;
 leave:
  if (rc)
    {
      _cdk_free_mpi_array (ncount, m_pk);
      _cdk_free_mpi_array (nscount, m_sk);
    }
  *r_skey = sexp;
  return rc;
}


static int
key_to_sexp (GCRY_SEXP * r_key, cdkPKT_public_key * pk)
{
  GCRY_SEXP sexp = NULL;
  GCRY_MPI * m = NULL;
  const char * fmt = NULL;
  int ncount = 0;
  int rc = 0;

  if (!r_key || !pk)
    return CDK_Inv_Value;

  ncount = cdk_pk_get_npkey (pk->pubkey_algo);
  m = bitmpi_to_gcrympi (pk->mpi, ncount);
  if (!m)
    return CDK_MPI_Error;
  if (is_RSA (pk->pubkey_algo))
    {
      fmt = "(public-key(openpgp-rsa(n%m)(e%m)))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m[0], m[1]);
    }
  else if (is_ELG (pk->pubkey_algo))
    {
      fmt = "(public-key(openpgp-elg(p%m)(g%m)(y%m)))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m[0], m[1], m[2]);
    }
  else if (is_DSA (pk->pubkey_algo))
    {
      fmt = "(public-key(openpgp-dsa(p%m)(q%m)(g%m)(y%m)))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m[0], m[1], m[2], m[3]);
    }
  if (rc)
    rc = CDK_Gcry_Error;
  if (!fmt)
    rc = CDK_Inv_Algo;
  _cdk_free_mpi_array (ncount, m);
  *r_key = sexp;
  return rc;
}


static int
esk_to_sexp (GCRY_SEXP * r_sexp, GCRY_MPI esk)
{
  GCRY_SEXP sexp = NULL;
  int rc = 0;

  if (!r_sexp || !esk)
    return CDK_Inv_Value;

  rc = gcry_sexp_build (&sexp, NULL, "%m", esk);
  if (rc)
    rc = CDK_Gcry_Error;
  *r_sexp = sexp;
  return rc;
}


static int
md_to_sexp (GCRY_SEXP * r_md, int algo, const byte * md, size_t mdlen)
{
  GCRY_SEXP sexp = NULL;
  GCRY_MPI m = NULL;
  size_t nbytes = 0;
  int rc = 0;

  if (!r_md || !md)
    return CDK_Inv_Value;

  nbytes = mdlen ? mdlen : gcry_md_get_algo_dlen (algo);
  rc = gcry_mpi_scan (&m, GCRYMPI_FMT_USG, md, &nbytes);
  if (rc)
    return CDK_Gcry_Error;
  rc = gcry_sexp_build (&sexp, NULL, "%m", m);
  if (rc)
    rc = CDK_Gcry_Error;
  if (!rc)
    *r_md = sexp;
  gcry_mpi_release (m);
  return rc;
}


static int
sexp_to_bitmpi (GCRY_SEXP sexp, const char * val, CDK_MPI * ret_buf)
{
  GCRY_SEXP list = NULL;
  GCRY_MPI m = NULL;
  CDK_MPI buf = NULL;
  size_t nbits = 0, nbytes = 0;
  int rc = 0;

  if (!sexp || !val || !ret_buf)
    return CDK_Inv_Value;

  list = gcry_sexp_find_token (sexp, val, 0);
  if (!list)
    return CDK_Gcry_Error;
  m = gcry_sexp_nth_mpi (list, 1, 0);
  if (!m)
    {
      rc = CDK_Gcry_Error;
      goto leave;
    }
  nbits = gcry_mpi_get_nbits (m);
  nbytes = (nbits + 7) / 8;
  buf = cdk_calloc (1, sizeof *buf + nbytes);
  if (!buf)
    return CDK_Out_Of_Core;
  buf->data[0] = nbits >> 8;
  buf->data[1] = nbits;
  rc = gcry_mpi_print (GCRYMPI_FMT_USG, NULL, &nbytes, m);
  if (!rc)
    rc = gcry_mpi_print (GCRYMPI_FMT_USG, buf->data + 2, &nbytes, m);
  if (rc)
    {
      rc = CDK_Gcry_Error;
      goto leave;
    }
  buf->bytes = nbytes;
  buf->bits = nbits;
  *ret_buf = buf;

leave:
  gcry_mpi_release (m);
  gcry_sexp_release (list);
  return rc;
}


static int
sexp_to_sig (cdkPKT_signature * sig, GCRY_SEXP sexp)
{
  int rc = 0;

  if (!sig || !sexp)
    return CDK_Inv_Value;

  if (is_RSA (sig->pubkey_algo))
    return sexp_to_bitmpi (sexp, "s", &sig->mpi[0]);
  else if (is_DSA (sig->pubkey_algo) || is_ELG (sig->pubkey_algo))
    {
      rc = sexp_to_bitmpi (sexp, "r", &sig->mpi[0]);
      if (!rc)
	rc = sexp_to_bitmpi (sexp, "s", &sig->mpi[1]);
      return rc;
    }
  return CDK_Inv_Algo;
}


static int
sig_to_sexp (GCRY_SEXP * r_sig, cdkPKT_signature * sig)
{
  GCRY_SEXP sexp = NULL;
  GCRY_MPI * m = NULL;
  const char * fmt = NULL;
  int ncount = 0;
  int rc = 0;

  if (!r_sig || !sig)
    return CDK_Inv_Value;

  ncount = cdk_pk_get_nsig (sig->pubkey_algo);
  m = bitmpi_to_gcrympi (sig->mpi, ncount);
  if (!m)
    return CDK_MPI_Error;
  if (is_RSA (sig->pubkey_algo))
    {
      fmt = "(sig-val(openpgp-rsa(s%m)))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m[0]);
    }
  else if (is_ELG (sig->pubkey_algo))
    {
      fmt = "(sig-val(openpgp-elg(r%m)(s%m)))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m[0], m[1]);
    }
  else if (is_DSA (sig->pubkey_algo))
    {
      fmt = "(sig-val(openpgp-dsa(r%m)(s%m)))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m[0], m[1]);
    }
  if (rc)
    rc = CDK_Gcry_Error;
  if (!fmt)
    rc = CDK_Inv_Algo;
  
  _cdk_free_mpi_array (ncount, m);
  *r_sig = sexp;
  return rc;
}


static int
sexp_to_enc (cdkPKT_pubkey_enc * enc, GCRY_SEXP sexp)
{
  int rc;

  if (!sexp || !enc)
    return CDK_Inv_Value;

  if (is_RSA (enc->pubkey_algo))
    return sexp_to_bitmpi (sexp, "a", &enc->mpi[0]);
  else if (is_ELG (enc->pubkey_algo))
    {
      rc = sexp_to_bitmpi (sexp, "a", &enc->mpi[0]);
      if (!rc)
	rc = sexp_to_bitmpi (sexp, "b", &enc->mpi[1]);
      return rc;
    }
  return CDK_Inv_Algo;
}


static int
enc_to_sexp (GCRY_SEXP * r_sexp, cdkPKT_pubkey_enc * enc)
{
  GCRY_SEXP sexp = NULL;
  GCRY_MPI * m = NULL;
  const char * fmt = NULL;
  int ncount;
  int rc = 0;

  if (!r_sexp || !enc)
    return CDK_Inv_Value;

  ncount = cdk_pk_get_nenc (enc->pubkey_algo);
  m = bitmpi_to_gcrympi (enc->mpi, ncount);
  if (!m)
    return CDK_MPI_Error;
  if (is_RSA (enc->pubkey_algo))
    {
      fmt = "(enc-val(openpgp-rsa((a%m))))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m[0]);
    }
  else if (is_ELG (enc->pubkey_algo))
    {
      fmt = "(enc-val(openpgp-elg((a%m)(b%m))))";
      rc = gcry_sexp_build (&sexp, NULL, fmt, m[0], m[1]);
    }
  if (rc)
    rc = CDK_Gcry_Error;
  if (!fmt)
    rc = CDK_Inv_Algo;
  
  _cdk_free_mpi_array (ncount, m);
  *r_sexp = sexp;
  return rc;
}


static int
is_unprotected (cdkPKT_secret_key * sk)
{
  if (sk->is_protected && !sk->mpi[0])
    return 0;
  return 1;
}


int
cdk_pk_encrypt (cdkPKT_public_key * pk, cdkPKT_pubkey_enc * pke, GCRY_MPI esk)
{
  GCRY_SEXP s_data = NULL, s_pkey = NULL, s_ciph = NULL;
  int rc = 0;

  if (!pk || !esk || !pke)
    return CDK_Inv_Value;

  if (!KEY_CAN_ENCRYPT (pk->pubkey_algo))
    return CDK_Inv_Algo;

  rc = esk_to_sexp (&s_data, esk);
  if (!rc)
    rc = key_to_sexp (&s_pkey, pk);
  if (!rc)
    rc = gcry_pk_encrypt (&s_ciph, s_data, s_pkey);
  if (!rc)
    rc = sexp_to_enc (pke, s_ciph);

  gcry_sexp_release (s_data);
  gcry_sexp_release (s_pkey);
  gcry_sexp_release (s_ciph);

  return rc;
}


int
cdk_pk_decrypt (cdkPKT_secret_key * sk, cdkPKT_pubkey_enc * pke,
                GCRY_MPI * r_sk)
{
  GCRY_SEXP s_data = NULL, s_skey = NULL, s_plain = NULL;
  int rc = 0;

  if (!sk || !r_sk || !pke)
    return CDK_Inv_Value;

  if (!is_unprotected (sk))
    return CDK_Inv_Mode;
  
  rc = skey_to_sexp (&s_skey, sk);
  if (!rc)
    rc = enc_to_sexp (&s_data, pke);
  if (!rc)
    {
      rc = gcry_pk_decrypt (&s_plain, s_data, s_skey);
      if (rc)
	rc = CDK_Gcry_Error;
    }
  if (!rc)
    *r_sk = gcry_sexp_nth_mpi (s_plain, 0, 0);

  gcry_sexp_release (s_data);
  gcry_sexp_release (s_skey);
  gcry_sexp_release (s_plain);
  
  return rc;
}


int
cdk_pk_sign (cdkPKT_secret_key * sk, cdkPKT_signature * sig, const byte * md)
{
  GCRY_SEXP s_skey = NULL, s_sig = NULL, s_hash = NULL;
  byte * encmd = NULL;
  size_t enclen = 0;
  int nbits = 0;
  int rc = 0;

  if (!sk || !sk->pk || !sig || !md)
    return CDK_Inv_Value;

  if (!is_unprotected (sk))
    return CDK_Inv_Mode;
  
  if (!KEY_CAN_SIGN (sig->pubkey_algo))
    return CDK_Inv_Algo;

  nbits = cdk_pk_get_nbits (sk->pk);
  rc = _cdk_pkcs1_digest (&encmd, &enclen, sk->pk->pubkey_algo, md,
                          sig->digest_algo, nbits);
  if (!rc)
    rc = skey_to_sexp (&s_skey, sk);
  if (!rc)
    rc = md_to_sexp (&s_hash, sig->digest_algo, encmd, enclen);
  if (!rc)
    {
      rc = gcry_pk_sign (&s_sig, s_hash, s_skey);
      if (rc)
	rc = CDK_Gcry_Error;
    }
  if (!rc)
    rc = sexp_to_sig (sig, s_sig);
  sig->digest_start[0] = md[0];
  sig->digest_start[1] = md[1];

  gcry_sexp_release (s_skey);
  gcry_sexp_release (s_hash);
  gcry_sexp_release (s_sig);
  cdk_free (encmd);

  return rc;
}


int
cdk_pk_verify (cdkPKT_public_key * pk, cdkPKT_signature * sig, const byte * md)
{
  GCRY_SEXP s_pkey = NULL, s_sig = NULL, s_hash = NULL;
  byte * encmd = NULL;
  size_t enclen = 0;
  int nbits = 0;
  int rc = 0;

  if (!pk || !sig || !md)
    return CDK_Inv_Value;

  nbits = cdk_pk_get_nbits (pk);
  rc = key_to_sexp (&s_pkey, pk);
  if (!rc)
    rc = sig_to_sexp (&s_sig, sig);
  if (!rc)
    rc = _cdk_pkcs1_digest (&encmd, &enclen, pk->pubkey_algo, md,
                            sig->digest_algo, nbits);
  if (!rc)
    rc = md_to_sexp (&s_hash, sig->digest_algo, encmd, enclen);
  if (!rc)
    {
      rc = gcry_pk_verify (s_sig, s_hash, s_pkey);
      if (rc)
	rc = CDK_Bad_Sig;
    }

  gcry_sexp_release (s_sig);
  gcry_sexp_release (s_hash);
  gcry_sexp_release (s_pkey);
  cdk_free (encmd);

  return rc;
}


int
cdk_pk_get_nbits (cdkPKT_public_key * pk)
{
  if (!pk || !pk->mpi[0])
    return 0;
  return pk->mpi[0]->bits;
}


int
cdk_pk_get_npkey (int algo)
{
  int n = gcry_pk_algo_info (algo, GCRYCTL_GET_ALGO_NPKEY, NULL, 0);
  return n > 0 ? n : 0;
}


int
cdk_pk_get_nskey (int algo)
{
  
  int n = gcry_pk_algo_info (algo, GCRYCTL_GET_ALGO_NSKEY, NULL, 0);
  n -= cdk_pk_get_npkey (algo);
  return n > 0 ? n : 0;
}


int
cdk_pk_get_nsig (int algo)
{
  int n = gcry_pk_algo_info (algo, GCRYCTL_GET_ALGO_NSIGN, NULL, 0);
  return n > 0 ? n : 0;
}


int
cdk_pk_get_nenc (int algo)
{
  int n = gcry_pk_algo_info (algo, GCRYCTL_GET_ALGO_NENCR, NULL, 0);
  return n > 0 ? n : 0;
}


int
_cdk_pk_algo_usage (int algo)
{
  int usage = 0;

  switch (algo)
    {
    case GCRY_PK_RSA  : usage = GCRY_PK_USAGE_SIGN | GCRY_PK_USAGE_ENCR; break;
    case GCRY_PK_RSA_E: usage = GCRY_PK_USAGE_ENCR; break;
    case GCRY_PK_RSA_S: usage = GCRY_PK_USAGE_SIGN; break;
    case GCRY_PK_ELG  : usage = GCRY_PK_USAGE_SIGN | GCRY_PK_USAGE_ENCR; break;
    case GCRY_PK_ELG_E: usage = GCRY_PK_USAGE_ENCR; break;
    case GCRY_PK_DSA  : usage = GCRY_PK_USAGE_SIGN; break; 
    }

  return usage;  
}


int
_cdk_pk_test_algo (int algo, unsigned int usage_flags)
{
  size_t n = usage_flags;
  
  if (algo < 0 || algo > 110)
    return GCRYERR_INV_PK_ALGO;
  return gcry_pk_algo_info (algo, GCRYCTL_TEST_ALGO, NULL, &n);    
}


static int
read_mpi (CDK_MPI a, byte * buf, size_t * r_count, size_t * r_nbits)
{
  if (!a || !buf || !r_count)
    return CDK_Inv_Value;
  
  if (a->bytes + 2 > *r_count)
    return CDK_General_Error;
  *r_count = a->bytes + 2;
  memcpy (buf, a->data, *r_count);
  if (r_nbits)
    *r_nbits = a->bits;
  return 0;
}

  
int
cdk_pk_get_mpi (cdkPKT_public_key * pk, int idx,
                byte * buf, size_t * r_count, size_t * r_nbits)
{
  if (!pk || idx < 0 || !r_count)
    return CDK_Inv_Value;
  if (idx > cdk_pk_get_npkey (pk->pubkey_algo))
    return CDK_Inv_Value;
  return read_mpi (pk->mpi[idx], buf, r_count, r_nbits);
}


int
cdk_sk_get_mpi (cdkPKT_secret_key * sk, int idx,
                byte * buf, size_t * r_count, size_t * r_nbits)
{
  if (!sk || idx < 0 || !r_count)
    return CDK_Inv_Value;
  if (idx > cdk_pk_get_nskey (sk->pubkey_algo))
    return CDK_Inv_Value;
  return read_mpi (sk->mpi[idx], buf, r_count, r_nbits);
}


static u16
checksum_mpi (CDK_MPI m)
{
  int i;
  u16 chksum = 0;

  if (m)
    {
      for (i = 0; i < m->bytes + 2; i++)
        chksum += m->data[i];
    }
  return chksum;
}


int
cdk_seckey_unprotect (cdkPKT_secret_key * sk, char * pw)
{
  GCRY_CIPHER_HD hd;
  CDK_DEK dek = NULL;
  CDK_MPI a;
  u16 chksum = 0;
  size_t ndata, nbits;
  int j, i, dlen, pos = 0, nskey;
  byte * data = NULL;

  if (!sk)
    return CDK_Inv_Value;

  nskey = cdk_pk_get_nskey (sk->pubkey_algo);
  if (sk->is_protected)
    {
      dek = cdk_passphrase_to_dek (sk->protect.algo, sk->protect.s2k, 0, pw);
      if (!dek)
        return CDK_General_Error;
      hd = gcry_cipher_open (sk->protect.algo,
			      GCRY_CIPHER_MODE_CFB,
			      GCRY_CIPHER_SECURE | GCRY_CIPHER_ENABLE_SYNC);
      if (!hd)
        {
          cdk_free (dek);
          return CDK_Gcry_Error;
        }
      gcry_cipher_setkey (hd, dek->key, dek->keylen);
      gcry_cipher_setiv (hd, sk->protect.iv, sk->protect.ivlen);

      wipemem (dek, sizeof dek);
      cdk_free (dek);
      chksum = 0;
      if (sk->version == 4)
	{
	  ndata = sk->enclen;
	  data = cdk_salloc (ndata);
	  if (!data)
	    return CDK_Out_Of_Core;
	  gcry_cipher_decrypt (hd, data, ndata, sk->encdata, ndata);
	  if (sk->protect.sha1chk)
	    {
	      /* This is the new SHA1 checksum method to detect tampering
	         with the key as used by the Klima/Rosa attack */
	      sk->csum = 0;
	      chksum = 1;
              dlen = gcry_md_get_algo_dlen (GCRY_MD_SHA1);
	      if (ndata < dlen)
                {
                  cdk_free (data);
                  return CDK_Inv_Packet;
                }
	      else
		{
		  GCRY_MD_HD md = gcry_md_open (GCRY_MD_SHA1, 1);
		  if (!md)
		    return CDK_Gcry_Error;
		  gcry_md_write (md, data, ndata - dlen);
		  gcry_md_final (md);
		  if (!memcmp (gcry_md_read (md, GCRY_MD_SHA1),
			       data + ndata - dlen, dlen))
		    chksum = 0;	/* digest does match */
		  gcry_md_close (md);
		}
	    }
	  else
	    {
	      for (i = 0; i < ndata - 2; i++)
		chksum += data[i];
	      sk->csum = data[ndata - 2] << 8 | data[ndata - 1];
	    }
	  if (sk->csum == chksum)
	    {
	      for (i = 0; i < nskey; i++)
		{
                  nbits = data[pos] << 8 | data[pos + 1];
                  ndata = (nbits + 7) / 8;
                  a = sk->mpi[i] = cdk_scalloc (sizeof *a + ndata + 2);
                  if (!a)
                    {
                      cdk_free (data);
                      return CDK_Out_Of_Core;
                    }
		  a->bits = nbits;
		  a->bytes = ndata;
		  for (j = 0; j < ndata + 2; j++)
                    a->data[j] = data[pos++];
		}
	    }
          wipemem (data, sk->enclen);
	  cdk_free (data);
	}
      else
	{
          chksum = 0;
	  for (i = 0; i < nskey; i++)
	    {
              a = sk->mpi[i];
              gcry_cipher_sync (hd);
	      gcry_cipher_decrypt (hd, a->data + 2, a->bytes, NULL, 0);
	      chksum += checksum_mpi (a);
	    }
	}
      gcry_cipher_close (hd);
    }
  else
    {
      chksum = 0;
      for (i = 0; i < nskey; i++)
	chksum += checksum_mpi (sk->mpi[i]);
    }

  if (chksum != sk->csum)
    return CDK_Chksum_Error;

  return 0;
}


int
_cdk_seckey_get_csum (cdkPKT_secret_key * sk)
{
  u16 csum = 0, i;

  for (i = 0; i < cdk_pk_get_nskey (sk->pubkey_algo); i++)
    csum += checksum_mpi (sk->mpi[i]);
  return csum;
}


int
cdk_pk_get_fingerprint (cdkPKT_public_key * pk, byte * fpr)
{
  GCRY_MD_HD hd;
  int md_algo;
  int dlen = 0;

  if (!pk || !fpr)
    return CDK_Inv_Value;

  if (pk->version < 4 && is_RSA (pk->pubkey_algo))
    md_algo = GCRY_MD_MD5; /* special */
  else
    md_algo = pk->version < 4 ? GCRY_MD_RMD160 : GCRY_MD_SHA1;
  dlen = gcry_md_get_algo_dlen (md_algo);
  hd = gcry_md_open (md_algo, 0);
  if (!hd)
    return CDK_Gcry_Error;
  _cdk_hash_pubkey (pk, hd, 1);
  gcry_md_final (hd);
  memcpy (fpr, gcry_md_read (hd, md_algo), dlen);
  gcry_md_close (hd);
  if (dlen == 16)
    memset (fpr + 16, 0, 4);
  return 0;
}


u32
cdk_pk_fingerprint_get_keyid (const byte * fpr, size_t fprlen, u32 * keyid)
{
  u32 lowbits = 0;

  /* in this case we say the key is a V3 RSA key and we can't
     use the fingerprint to get the keyid. */
  if (fpr && fprlen != 20)
    return 0;
  else if (keyid && fpr)
    {
      keyid[0] = _cdk_buftou32 (fpr + 12);
      keyid[1] = _cdk_buftou32 (fpr + 16);
      lowbits = keyid[1];
    }
  else if (fpr)
    lowbits = _cdk_buftou32 (fpr + 16);

  return lowbits;
}


u32
cdk_pk_get_keyid (cdkPKT_public_key * pk, u32 * keyid)
{
  u32 lowbits = 0;
  byte buf[24];

  if (pk && (!pk->keyid[0] || !pk->keyid[1]))
    {
      if (pk->version < 4 && is_RSA (pk->pubkey_algo))
	{
	  size_t n = pk->mpi[0]->bytes;
	  const byte * p = pk->mpi[0]->data + 2;
	  pk->keyid[0] = p[n-8] << 24 | p[n-7] << 16 | p[n-6] << 8 | p[n-5];
	  pk->keyid[1] = p[n-4] << 24 | p[n-3] << 16 | p[n-2] << 8 | p[n-1];
	}
      else if (pk->version == 4)
	{
	  cdk_pk_get_fingerprint (pk, buf);
	  pk->keyid[0] = _cdk_buftou32 (buf + 12);
	  pk->keyid[1] = _cdk_buftou32 (buf + 16);
	}
    }
  lowbits = pk ? pk->keyid[1] : 0;
  if (keyid && pk)
    {
      keyid[0] = pk->keyid[0];
      keyid[1] = pk->keyid[1];
    }
  return lowbits;
}


u32
cdk_sk_get_keyid (cdkPKT_secret_key * sk, u32 * keyid)
{
  return sk && sk->pk? cdk_pk_get_keyid (sk->pk, keyid) : 0;
}


u32
cdk_sig_get_keyid (cdkPKT_signature * sig, u32 * keyid)
{
  u32 lowbits = sig ? sig->keyid[1] : 0;
  
  if (keyid && sig)
    {
      keyid[0] = sig->keyid[0];
      keyid[1] = sig->keyid[1];
    }
  return lowbits;
}


u32
_cdk_keyid_get (CDK_PACKET * pkt, u32 * keyid)
{
  u32 lowbits;
  
  switch (pkt->pkttype)
    {
    case CDK_PKT_PUBLIC_KEY:
    case CDK_PKT_PUBLIC_SUBKEY:
      lowbits = cdk_pk_get_keyid (pkt->pkt.public_key, keyid);
      break;
      
    case CDK_PKT_SECRET_KEY:
    case CDK_PKT_SECRET_SUBKEY:
      lowbits = cdk_sk_get_keyid (pkt->pkt.secret_key, keyid);
      break;

    case CDK_PKT_SIGNATURE:
      lowbits = cdk_sig_get_keyid (pkt->pkt.signature, keyid);
      break;
      
    default:
      lowbits = 0;
    }
  return lowbits;
}


int
_cdk_fingerprint_get (CDK_PACKET * pkt, byte * fpr)
{
  switch (pkt->pkttype)
    {
    case CDK_PKT_PUBLIC_KEY:
    case CDK_PKT_PUBLIC_SUBKEY:
      return cdk_pk_get_fingerprint (pkt->pkt.public_key, fpr);

    case CDK_PKT_SECRET_KEY:
    case CDK_PKT_SECRET_SUBKEY:
      return cdk_pk_get_fingerprint (pkt->pkt.secret_key->pk, fpr);

    default:
      return CDK_Inv_Packet;
    }

  return 0;
}
