/* -*- Mode: C; c-file-style: "bsd" -*- */
/* verify.c - Verify signatures
 *        Copyright (C) 2001, 2002, 2003 Timo Schulz 
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "opencdk.h"
#include "main.h"
#include "filters.h"
#include "packet.h"


struct {
    const char *name;
    int algo;
} digest_table[] = {
    {"MD5",       GCRY_MD_MD5},
    {"SHA1",      GCRY_MD_SHA1},
    {"RIPEMD160", GCRY_MD_RMD160},
    {"MD2",       GCRY_MD_MD2},
    {"TIGER192",  GCRY_MD_TIGER},
    {"SHA256",    GCRY_MD_SHA256},
    {NULL, 0}
};


static int file_verify_clearsign( CDK_HD, const char *, const char * );


static int
stream_peek( CDK_STREAM inp, char *s, size_t count )
{
    unsigned off;
    int nbytes;
    
    off = cdk_stream_tell( inp );
    nbytes = _cdk_stream_gets( inp, s, count );
    cdk_stream_seek( inp, off );
    return nbytes;
}


int
cdk_stream_verify( CDK_HD hd, CDK_STREAM inp, CDK_STREAM out )
{
    if( cdk_armor_filter_use( inp ) ) {
        cdk_stream_set_armor_flag( inp, 0 );
        cdk_stream_control( inp, CDK_STREAMCTL_RESET, 1 );
    }
    return _cdk_proc_packets( hd, inp, NULL, NULL, NULL );
}
    

/**
 * cdk_file_verify:
 * @hd: the session handle
 * @file: the input file
 * @output: the output file
 *
 * Verify a signature.
 **/
int
cdk_file_verify( CDK_HD hd, const char * file, const char * output )
{
    CDK_STREAM inp;
    char buf[1024];
    int rc, n;

    if( !hd || !file )
        return CDK_Inv_Value;
    if( output && !hd->opt.overwrite && _cdk_check_file( output ) )
        return CDK_Inv_Mode;
  
    rc = cdk_stream_open ( file, &inp );
    if( rc )
        return rc;
    if( cdk_armor_filter_use( inp ) ) {
        n = stream_peek( inp, buf, sizeof buf-1 );
        if( !n )
            return CDK_EOF;
        buf[n] = '\0';
        if( strstr( buf, "BEGIN PGP SIGNED MESSAGE" ) ) {
            cdk_stream_close( inp );
            rc = file_verify_clearsign( hd, file, output );
            return rc;
        }
        cdk_stream_set_armor_flag( inp, 0 );
        cdk_stream_control( inp, CDK_STREAMCTL_RESET, 1 );
    }
    rc = _cdk_proc_packets( hd, inp, NULL, NULL, NULL );
    cdk_stream_close( inp );
    return rc;
}


void
_cdk_result_verify_free( Verify_Result res )
{
    if( res ) {
        cdk_free( res->sig_data );
        cdk_free( res->notation );
        cdk_free( res );
    }
}


Verify_Result
_cdk_result_verify_new( void )
{
    Verify_Result res;

    res = cdk_calloc( 1, sizeof *res );
    if( !res )
        return NULL;
    return res;
}


static char *
sigdata_encode( CDK_HD hd, int fmt )
{
    return "";
}


/**
 * cdk_sig_get_info_as_xml:
 * @hd: session handle
 * @advanced: flag to indicate whether to include extended data
 *
 * Return an XML fragment which contains the status of the verification
 * procedure. If no signature is present, the fragment is empty.
 **/
byte *
cdk_sig_get_info_as_xml( CDK_HD hd, int advanced )
{
    char * p;
    const char * fmt, * dat;
    const u32 * keyid;
    
    if( !hd || !cdk_sig_get_ulong_attr( hd, 0, CDK_ATTR_STATUS ) )
        return cdk_strdup( "<OpenPGP-Signature-Status>"
                           "</OpenPGP-Signature-Status>" );
    fmt = "<OpenPGP-Signature>\n"
          "  <Version>%d</Version>\n"
          "  <Created>%lu</Created>\n"
          "  <Publickey-Algo>%d</Publickey-Algo>\n"
          "  <Digest-Algo>%d</Digest-Algo>\n"
          "  <KeyID>%08lX%08lX</KeyID>\n"
          "  <Expired>%d</Expired>\n"
          "  <Status>%d</Status>\n"
          "  <Data>%s</Data>\n"
          "</OpenPGP-Signature>\n";
    dat = advanced? sigdata_encode( hd, 0 ) : "";
    p = cdk_calloc( 1, strlen( fmt ) + strlen( dat ) + 512 );
    if( !p )
        return NULL;
    keyid = cdk_sig_get_data_attr( hd, 0, CDK_ATTR_KEYID );
    sprintf( p, fmt,
             cdk_sig_get_ulong_attr( hd, 0, CDK_ATTR_VERSION ),
             cdk_sig_get_ulong_attr( hd, 0, CDK_ATTR_CREATED ),
             cdk_sig_get_ulong_attr( hd, 0, CDK_ATTR_ALGO_PK ),
             cdk_sig_get_ulong_attr( hd, 0, CDK_ATTR_ALGO_MD ),
             keyid[0], keyid[1],
             cdk_sig_get_ulong_attr( hd, 0, CDK_ATTR_FLAGS )
             & CDK_FLAG_SIG_EXPIRED? 1 : 0,
             cdk_sig_get_ulong_attr( hd, 0, CDK_ATTR_STATUS ),
             dat
        );
    return p;
}


/**
 * cdk_sig_get_ulong_attr:
 * @hd: session handle
 * @idx: index of the signature
 * @what: attribute id
 *
 * Extract the requested attribute of the signature. The returned value
 * is always an integer (max. 32-bit).
 **/
u32
cdk_sig_get_ulong_attr (CDK_HD hd, int idx, int what)
{
    Verify_Result res;
    u32 val = 0;
  
    if( !hd || !hd->result.verify )
        return 0;

    assert( idx == 0 );
    res = hd->result.verify;
    switch( what ) {
    case CDK_ATTR_CREATED: val = res->created; break;
    case CDK_ATTR_EXPIRE : val = res->expires; break;
    case CDK_ATTR_KEYID  : val = res->keyid[1]; break;
    case CDK_ATTR_STATUS : val = res->sig_status; break;
    case CDK_ATTR_ALGO_PK: val = res->pubkey_algo; break;
    case CDK_ATTR_ALGO_MD: val = res->digest_algo; break;
    case CDK_ATTR_VERSION: val = res->sig_ver; break;
    case CDK_ATTR_LEN    : val = res->sig_len; break;
    case CDK_ATTR_FLAGS  : val = res->sig_flags; break;
    default              : val = 0; break;
    }
  
    return val;
}


/**
 * cdk_sig_get_data_attr:
 * @hd: session handle
 * @idx: index of the signature
 * @what: attribute id.
 *
 * Extract the requested attribute of the signature. The returned value
 * is always a constant object to the data.
 **/
const void *
cdk_sig_get_data_attr( CDK_HD hd, int idx, int what )
{
    Verify_Result res;
    const void * val;
  
    if( !hd || !hd->result.verify )
        return NULL;

    assert( idx == 0 );
    res = hd->result.verify;
    switch( what ) {
    case CDK_ATTR_KEYID   : val = res->keyid; break;
    case CDK_ATTR_NOTATION: val = res->notation; break;
    case CDK_ATTR_MPI     : val = res->sig_data; break;
    default               : val = NULL;
    }

    return val;
}


static int
file_verify_clearsign (CDK_HD hd, const char * file, const char * output)
{
    CDK_STREAM inp = NULL, out = NULL, tmp = NULL;
    GCRY_MD_HD md = NULL;
    char buf[512], chk[512];
    const char * s;
    int rc = 0;
    int i, is_signed = 0, nbytes;
    int digest_algo = 0;

    if( output ) {
        rc = cdk_stream_create( output, &out );
        if( rc )
            return rc;
    }
  
    rc = cdk_stream_open (file, &inp);
    if (rc)
        return rc;  

    while (!cdk_stream_eof (inp)) {
        nbytes = _cdk_stream_gets (inp, buf, sizeof buf-1);
        if (!nbytes)
            break;
        if (!strncmp (buf, "-----BEGIN PGP SIGNED MESSAGE-----", 35)) {
            is_signed = 1;
            break;
        }
    }
    if (cdk_stream_eof (inp) && !is_signed) {
        rc = CDK_Armor_Error;
        goto leave;
    }
  
    while (!cdk_stream_eof (inp)) {
        nbytes = _cdk_stream_gets (inp, buf, sizeof buf-1);
        if (!nbytes)
            break;
        if (nbytes == 1) /* empty line */
            break;
        else if (!strncmp (buf, "Hash: ", 6)) {
            for (i = 0; (s = digest_table[i].name); i++) {
                if (!strcmp (buf + 6, s)) {
                    digest_algo = digest_table[i].algo;
                    break;
                }
            }
        }
    }

    if( digest_algo && _cdk_md_test_algo( digest_algo ) ) {
        rc = CDK_Inv_Algo;
        goto leave;
    }
    if( !digest_algo )
        digest_algo = GCRY_MD_MD5;
    md = gcry_md_open( digest_algo, 0 );
    if( !md ) {
        rc = CDK_Gcry_Error;
        goto leave;
    }

    s = "-----BEGIN PGP SIGNATURE-----";
    while (!cdk_stream_eof (inp)) {
        nbytes = _cdk_stream_gets (inp, buf, sizeof buf-1);
        if (!nbytes)
            break;
        if (!strncmp (buf, s, strlen (s)))
            break;
        else {
            stream_peek (inp, chk, sizeof chk-1);
            i = strncmp (chk, s, strlen (s));
            if (strlen (buf) == 0 && i == 0)
                continue; /* skip last '\n' */
            _cdk_trim_string (buf, i == 0? 0 : 1);
            gcry_md_write (md, buf, strlen (buf));
        }
        if (!strncmp (buf, "- ", 2))
            memmove (buf, buf + 2, nbytes - 2);
        if (out) {
            buf[strlen (buf) - 1] = 0;
            buf[strlen (buf) - 1] = '\n';
            cdk_stream_write (out, buf, strlen (buf));
        }
    }

    tmp = cdk_stream_tmp ();
    if (!tmp) {
        rc = CDK_Out_Of_Core;
        goto leave;
    }
  
    s = "-----BEGIN PGP SIGNATURE-----\n";
    _cdk_stream_puts( tmp, s );
    while (!cdk_stream_eof (inp)) {
        nbytes = _cdk_stream_gets (inp, buf, sizeof buf-1);
        if (!nbytes)
            break;
        if (nbytes < (sizeof buf -3)) {
            buf[nbytes-1] = '\n';
            buf[nbytes] = '\0';
        }
        cdk_stream_write (tmp, buf, nbytes);
    }
    cdk_stream_tmp_set_mode (tmp, 0); /* switch to read-mode */
    cdk_stream_seek (tmp, 0);
    cdk_stream_control (tmp, CDK_STREAMCTL_RESET, 1);
    cdk_stream_set_armor_flag (tmp, 0);
    cdk_stream_read (tmp, NULL, 0);

    rc = _cdk_proc_packets( hd, tmp, NULL, NULL, md );

 leave:
    cdk_stream_close (out);
    cdk_stream_close (tmp);
    cdk_stream_close (inp);
    return rc;
}
