/* ipsec_input.c */

/* This file contains the part of the core IPsec engine that processes
   and decapsulates inbound IPsec packets */


#include <linux/types.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/in.h>
#include <linux/in6.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/if_arp.h>
#include <linux/mroute.h>
#include <linux/random.h>

#include <net/datalink.h>
#include <net/sock.h>
#include <net/ip.h>
#include <net/protocol.h>
#include <linux/udp.h>

#include "sadb.h"
#include "ipsec.h"
#include "transform.h"
#include <syslog.h>

static char addr_s1[18],addr_s2[18];

struct sk_buff *ipsec_input(struct sk_buff *skb)
{
	struct iphdr *iph=(struct iphdr *)skb->data;
  	struct net_addr ipsec_addr;
	unsigned char *data=0;
  	unsigned long spi,sn;
  	struct sadb_dst_node *d=0;
	struct ipsec_transform *transform=0;

#ifdef MODULE
	ipsec_inc_mod_count();
#endif

	if ((iph->protocol == IPPROTO_ESP) || (iph->protocol == IPPROTO_AH))
	{
	  	ipsec_addr.addr_family = AF_INET;
  		ipsec_addr.addr_union.ip_addr.s_addr = iph->saddr;
		data = skb->data + (iph->ihl << 2);
  		spi = (iph->protocol == IPPROTO_ESP)?ntohl(*(unsigned long *)data):ntohl(((struct ah_header *)data)->spi);

  		if (spi < 256){
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
      			printk(KERN_NOTICE "IPSEC: Inbound: SPI out of range: src %s dst %s spi %lx \n", addr_s1, addr_s2, spi);
      			goto ipsec_in_error;
  		}

/* Obtain SADB Information */
		if ((d = (struct sadb_dst_node *)get_sadb_node(ipsec_addr,iph->protocol,spi,0,1)) == 0){
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
      			printk(KERN_NOTICE "IPSEC: Inbound: No SA found: src %s dst %s spi %lx \n", addr_s1,addr_s2,spi);
      			goto ipsec_in_error;
  		}

		if (d->sa_info.flags & IPSEC_CHECK_REPLAY_FLAG)
		{
			if (d->sa_info.lastsn == MAX_RP_SN)
			{
					strcpy(addr_s1,ntoa(iph->saddr));
					strcpy(addr_s2,ntoa(iph->daddr));
      					printk(KERN_NOTICE "IPSEC: Inbound: Sequence Number Wrapped: src %s dst %s spi %lx \n", addr_s1,addr_s2,spi);
					sadb_delete(d);
      					goto ipsec_in_error;
			}
			else 
			{
				if (d->sa_info.lastsn >= MAX_RP_SN-RP_SN_WARN)
				{
						strcpy(addr_s1,ntoa(iph->saddr));
						strcpy(addr_s2,ntoa(iph->daddr));
      						printk(KERN_NOTICE "IPSEC: Inbound: Sequence Number About to Wrap: src %s dst %s spi %lx \n", addr_s1,addr_s2,spi);
				}
  		                sn = (iph->protocol == IPPROTO_ESP)?ntohl(*(unsigned long *)(data+sizeof(spi))):ntohl(*(unsigned long *)(data + sizeof(struct ah_header)));
				if (!check_replay_window(sn,d))
				{
					strcpy(addr_s1,ntoa(iph->saddr));
					strcpy(addr_s2,ntoa(iph->daddr));
       					printk(KERN_NOTICE "IPSEC: Inbound: Bad Sequence Number: src %s dst %s spi %lx seq# %lx \n", addr_s1,addr_s2,spi,sn);
      					goto ipsec_in_error;
				}
			}
		}
	if (d->sa_info.lifetime.bytes_remaining > 0)
		d->sa_info.lifetime.bytes_remaining = IPSEC_MAX(0, d->sa_info.lifetime.bytes_remaining-(long)((d->sa_info.protocol == IPPROTO_ESP) ? skb->len-(iph->ihl<<2) : skb->len));
#ifdef SI_TESTER
       	printk(KERN_NOTICE "IPSEC[%lu]: Inbound: Pre-decapsulation:src %s:dst %s:prot %d:spi %lx:", si_cnt++,
 		strcpy(addr_s1,ntoa(iph->saddr)), strcpy(addr_s2,ntoa(iph->daddr)), iph->protocol, spi);
	si_bprintk(skb->data,skb->len);
	printk("\n");
#endif
		switch (d->sa_info.protocol)
		{
		case IPPROTO_ESP:

			if (d->sa_info.alg.esp.auth_alg_id != NO_ALG)
			{
				if ((transform = ipsec_get_transform(d->sa_info.alg.esp.auth_alg_id, AUTH_TRANSFORM)) == NULL){
					strcpy(addr_s1,ntoa(iph->saddr));
					strcpy(addr_s2,ntoa(iph->daddr));
					printk(KERN_NOTICE  "IPSEC_ESP: Inbound: Bad authentication alg: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      					goto ipsec_in_error;
				}

				if ((skb = transform->input_handler(skb,d,transform)) == NULL){
					strcpy(addr_s1,ntoa(iph->saddr));
					strcpy(addr_s2,ntoa(iph->daddr));
					printk(KERN_NOTICE  "IPSEC_ESP: Inbound: authentication failure: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      					goto ipsec_in_error;
				}
			}
			
			if ((transform = ipsec_get_transform(d->sa_info.alg.esp.crypto_alg_id, CIPHER_TRANSFORM))==NULL){
				strcpy(addr_s1,ntoa(iph->saddr));
				strcpy(addr_s2,ntoa(iph->daddr));
				printk(KERN_NOTICE  "IPSEC_ESP: Inbound: Bad encryption alg: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      				goto ipsec_in_error;
			}
		break;
		case IPPROTO_AH:

			if ((transform = ipsec_get_transform(d->sa_info.alg.ah.auth_alg_id, AUTH_TRANSFORM))==NULL){
				strcpy(addr_s1,ntoa(iph->saddr));
				strcpy(addr_s2,ntoa(iph->daddr));
				printk(KERN_NOTICE  "IPSEC_AH: Inbound: Bad authenticator alg: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      				goto ipsec_in_error;
			}
		break;
		}

		if (!(skb = transform->input_handler(skb,d,transform))) goto ipsec_in_error;

		iph = (struct iphdr *)skb->data;

		if ((d->sa_info.flags & IPSEC_TUNNEL_FLAG) && (iph->protocol != IPPROTO_IPIP))
		{
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
			printk(KERN_NOTICE  "IPSEC: Inbound: SA specifies tunnel, no tunnel in packet : src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      			goto ipsec_in_error;
		}

/* This tunnel code was taken from /usr/src/linux/net/ipip.c with slight modifications */
		if (iph->protocol == IPPROTO_IPIP)
		{
			skb->security |= SKB_SECURITY_IPSEC;
			skb->mac.raw = skb->nh.raw;
        		skb->nh.raw = skb_pull(skb, skb->h.raw - skb->data);
       			memset(&(IPCB(skb)->opt), 0, sizeof(struct ip_options));
			skb->pkt_type = PACKET_HOST;
			if (!(skb->dev->flags & IFF_LOOPBACK))
			{
				dst_release(skb->dst);
				skb->dst = NULL;
			}
			skb->protocol = __constant_htons(ETH_P_IP);
			skb->ip_summed = 0;
#ifdef SI_TESTER
	iph=(struct iphdr *)skb->data;
       	printk(KERN_NOTICE "IPSEC[%lu]: Inbound: Post-decapsulation:src %s:dst %s:prot %d:spi %lx:",  si_cnt++,
 		strcpy(addr_s1,ntoa(iph->saddr)), strcpy(addr_s2,ntoa(iph->daddr)), iph->protocol, spi);
	si_bprintk(skb->data,skb->len);
	printk("\n");
#endif
			netif_rx(skb);
#ifdef MODULE
			ipsec_dec_mod_count();
#endif
			return NULL;
		}
#ifdef SI_TESTER
       	printk(KERN_NOTICE "IPSEC[%lu]: Inbound: Post-decapsulation:src %s:dst %s:prot %d:spi %lx:", si_cnt++, 
 		strcpy(addr_s1,ntoa(iph->saddr)), strcpy(addr_s2,ntoa(iph->daddr)), iph->protocol, spi);
	si_bprintk(skb->data,skb->len);
	printk("\n");
#endif
#ifdef MODULE
		ipsec_dec_mod_count();
#endif
		if ((iph->protocol == IPPROTO_ESP) || (iph->protocol == IPPROTO_AH))
		{
			skb = ipsec_input(skb);
			return skb;
		} else
		{
			return skb;
		}
	}
/* policy checks */
/* more will be done above when/if combined mode is figured out */
	else 
	{

		if (skb->security & SKB_SECURITY_IPSEC)
		{
			skb->security ^= SKB_SECURITY_IPSEC;
#ifdef MODULE
			ipsec_dec_mod_count();
#endif
			return skb;
		}

		if (ipsec_sys_policy.inbound.prot_flag&NON_PROTECTED)
		{
#ifdef MODULE
			ipsec_dec_mod_count();
#endif
			return skb;
		}

  		ipsec_addr.addr_family = AF_INET;
  		ipsec_addr.addr_union.ip_addr.s_addr = iph->saddr;

		/* Obtain SADB Information */
		if ((null_inbound_sa_found(ipsec_addr)) && (ipsec_sys_policy.inbound.prot_flag&NULL_ASSOCIATION))
		{
#ifdef MODULE
			ipsec_dec_mod_count();
#endif
			return skb;
		}

		if (iph->protocol == IPPROTO_UDP)
		{
                    if (registered_km_port == ntohs(UDP_DEST_PORT((unsigned char *)iph+(iph->ihl<<2))))
		    {
			if (local_addr(skb->dev, iph->daddr))
			{
#ifdef MODULE
				ipsec_dec_mod_count();
#endif
				return skb;
			}
		    }
		}

		strcpy(addr_s1,ntoa(iph->saddr));
		strcpy(addr_s2,ntoa(iph->daddr));
      		printk(KERN_INFO "IPSEC: inbound: Security required, none in packet: src %s dst %s \n", addr_s1,addr_s2);
      		goto ipsec_in_error;
  	}


ipsec_in_error:

	if (skb) kfree_skb(skb);
#ifdef MODULE
	ipsec_dec_mod_count();
#endif
	ip_statistics.IpInDiscards++;
	return NULL;
}
