/* sadb_ui.c - Security Association Data Base User Interface routines */


#include <linux/autoconf.h>
#include <netinet/in.h>
#include <asm/types.h>
#include <stdio.h>
#include <errno.h>
#include <netdb.h>
#include <sys/time.h>
#include <asm/fcntl.h>
#include <linux/netlink.h>
#include <ipsec/sadb.h>
#include <ipsec/ipsec.h>

#define	K_SET	1
#define	K_CHANGE	3
#define	K_DELETE	4
#define	K_FLUSH	5
#define	K_GET	6

#define min(a,b) a<b?a:b

struct keytab {
	char	*kt_cp;
	int	kt_i;
} keywords[] = {
	{"set", K_SET},
	{"add", K_SET},
	{"change", K_CHANGE},
	{"delete", K_DELETE},
	{"get", K_GET},
	{"flush", K_FLUSH},
{0, 0}
};

char buffer[sizeof(struct sadb_msg)+1];

struct msghdr msghdr;

struct sadb_msg m_sadbmsg;
struct sockaddr_nl nladdr;

int	pid, uid;
int	s;
int	keyword();
int	verbose,name;
int 	heading= 1;
int	smallout = 0;
/*  inbound/outbound for one-way SA's                           */
int     inbound, outbound;
void	entry();

char * t_str(char * s)
{
	char * str;

printf("1%s2\n",s);
	if (!s) return s;
	str = s;
	*(s+strlen(s)-1)=0;
printf("1%s2\n",str);
	return str;
}

static void usage(cp)
	char *cp;
{
    if (verbose){
	(void) fprintf(stderr, "usage: sadb [ -nsv ] get \n");
	(void) fprintf(stderr, "                              [p[rot]\n" );
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "                           OR\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "                              [d[est]\n");
        (void) fprintf(stderr, "                                [<dest name|address|prefix>\n");
	(void) fprintf(stderr, "                                local-spi(hex)\n");
	(void) fprintf(stderr, "                                protocol]\n");
	(void) fprintf(stderr,"                                   50 - ESP\n");
	(void) fprintf(stderr,"                                   51 - AH\n");
	(void) fprintf(stderr,"                                   2 - NULL_SA\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "  OR\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "usage: sadb [ -nsv ] [add|set] \n");
	(void) fprintf(stderr, "                              [p[rot] <0-3> <0-3>]\n" );
	(void) fprintf(stderr, "                                0 - protected traffic only\n");
	(void) fprintf(stderr, "                                1 - Allow unprotected traffic\n");
	(void) fprintf(stderr, "                                2 - Allow assoc. w/ no security\n");
	(void) fprintf(stderr, "                                3 - 1 & 2\n");
	(void) fprintf(stderr, "                                (inbound followed by outbound)\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "                           OR\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "                              [d[est] \n");
        (void) fprintf(stderr, "                                <dest name|address|prefix>\n");
	(void) fprintf(stderr, "                                prefix_len(32 if not prefix)\n");
	(void) fprintf(stderr, "                                local-spi(hex)\n");
	(void) fprintf(stderr, "                                <peer name|address>\n");
	(void) fprintf(stderr, "                                peer-spi(hex)\n");
	(void) fprintf(stderr, "                                tunnel-flag\n");
	(void) fprintf(stderr, "                                check-replay-flag\n");
	(void) fprintf(stderr, "                                initial-sequence-number\n");
	(void) fprintf(stderr, "                                protocol\n");
	(void) fprintf(stderr,"                                   2 - NULL_SA\n");
	(void) fprintf(stderr,"                                   50 - ESP\n");
	(void) fprintf(stderr,"                                   51 - AH\n");
	(void) fprintf(stderr,"                                 linked-protocol\n");
	(void) fprintf(stderr,"                                   0 - no link\n");
	(void) fprintf(stderr,"                                   50 - ESP\n");
	(void) fprintf(stderr,"                                   51 - AH\n");
	(void) fprintf(stderr,"                                 linked-local-spi(hex)\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr,"                               AND one of the following sets of parameters\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr,"                                 (ESP Specific Parameters)\n");
	(void) fprintf(stderr, "                                crypto-alg-id \n");
	(void) fprintf(stderr,"                                   2 -   DES/CBC\n");
	(void) fprintf(stderr,"                                   3 -   3DES/CBC \n");
	(void) fprintf(stderr,"                                   4 -   RC5/CBC\n");
	(void) fprintf(stderr,"                                   5 -   Blowfish/CBC\n");
	(void) fprintf(stderr,"                                   6 -   IDEA/CBC\n");
	(void) fprintf(stderr,"                                   7 -   ESP_NULL \n");
	(void) fprintf(stderr,"                                   8 -   ESP_MARS (New)\n");
	(void) fprintf(stderr,"                                   9 -   ESP_RC6 (New)\n");
	(void) fprintf(stderr,"                                   10 -  ESP_RIJNDAEL (New)\n");
	(void) fprintf(stderr,"                                   11 -  ESP_SERPENT (New)\n");
	(void) fprintf(stderr,"                                   12 -  ESP_TWOFISH (New)\n");
	(void) fprintf(stderr,"                                   252 - Cipher Test\n");
	(void) fprintf(stderr, "                                 crypto-iv-length \n");
	(void) fprintf(stderr, "                                 crypto-key length (# bytes) \n");
	(void) fprintf(stderr, "                                 crypto-key \n");
	(void) fprintf(stderr, "                                 auth-alg-id \n");
	(void) fprintf(stderr,"                                    0 -   none\n"); 
	(void) fprintf(stderr,"                                    132 - HMAC-MD5-96\n");
	(void) fprintf(stderr,"                                    133 - HMAC-SHA-1-96\n");
	(void) fprintf(stderr,"                                    253 - Authenticator Test\n");
	(void) fprintf(stderr, "                                 auth-iv-length \n");
	(void) fprintf(stderr, "                                 auth-key length (# bytes) \n");
	(void) fprintf(stderr, "                                 auth-key (empty if length=0)\n");
	(void) fprintf(stderr, "                                 auth-data-length\n" );
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "                                 OR\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr,"                                  (AH Specific Parameters)\n");
	(void) fprintf(stderr, "                                 auth-alg-id \n");
	(void) fprintf(stderr,"                                    132 - HMAC-MD5-96\n");
	(void) fprintf(stderr,"                                    133 - HMAC-SHA-1-96\n");
	(void) fprintf(stderr,"                                    253 - Authenticator Test\n");
	(void) fprintf(stderr, "                                 auth-iv-length \n");
	(void) fprintf(stderr, "                                 auth-key length (# bytes) \n");
	(void) fprintf(stderr, "                                 auth-key \n");
	(void) fprintf(stderr, "                                 auth-data-length\n" );
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "  OR\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "usage: sadb [ -nsv ] delete d[est]\n");
        (void) fprintf(stderr, "                                <dest name|address|prefix>\n");
	(void) fprintf(stderr, "                                local-spi(hex)\n");
	(void) fprintf(stderr, "                                protocol\n");
	(void) fprintf(stderr,"                                   2 - NULL_SA\n");
	(void) fprintf(stderr,"                                   50 - ESP\n");
	(void) fprintf(stderr,"                                   51 - AH\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "  OR\n");
	(void) fprintf(stderr, "\n");
	(void) fprintf(stderr, "usage: sadb [ -nsv ] flush\n");
    }

	if (cp){
		fprintf(stderr,"sadb usage error: %s\n",cp);
	        if (!verbose) fprintf(stderr,"     -v for more info\n");
	}
	exit(1);
}

void
quit(s)
	char *s;
{
	int sverrno = errno;

	(void) fprintf(stderr, "sadb: ");
	if (s)
		(void) fprintf(stderr, "%s: ", s);
	(void) fprintf(stderr, "%s\n", strerror(sverrno));
	exit(1);
	/* NOTREACHED */
}

int getaddr(s,paddr)
char *s;
struct net_addr *paddr;
{
	int success;
	struct hostent *h;
	struct netent *n;
	u_long **haddr;
	if (!strcmp(s,"0")) return 0;

	if (h=gethostbyname(s)){
		paddr->addr_family = h->h_addrtype;

		switch (h->h_addrtype){
		case AF_INET:
			haddr = (u_long **)h->h_addr_list;
			paddr->addr_union.ip_addr.s_addr = (*(struct in_addr **)haddr)->s_addr;
			return 1;
		break;
		default:
			return 0;
		break;
		}

	}

	if (n=getnetbyname(s))
	{
		paddr->addr_family = h->h_addrtype;

		switch (h->h_addrtype){
		case AF_INET:
			paddr->addr_union.ip_addr.s_addr = n->n_net;
			return 1;
		break;
		default:
			return 0;
		break;
		}
	}

	return 0;
}
void print_family(fam)
unsigned short fam;
{
     switch (fam){
     case AF_INET:
       printf("INET");
     break;
     default:
     break;
  }
}

int print_name(addr)
struct net_addr addr;
{
	struct hostent *h;
	struct netent *n;


    if (addr.addr_family == AF_INET){ 

	if (h=gethostbyaddr((char *)&(addr.addr_union.ip_addr),sizeof(addr.addr_union.ip_addr),addr.addr_family)){
		printf("%s",h->h_name);
		return 1;
	}
	
	if (n=getnetbyaddr(addr.addr_union.ip_addr.s_addr,addr.addr_family)){
		printf("%s",n->n_name);
		return 1;
	}
    }
	return 0;
}

void print_key(key)
key_type key;
{
	register short i;

	for (i=0;i<key.key_len;i++) printf("%02x",key.key_data[i]);
}

void print_addr(addr)
struct net_addr addr;
{
	if (addr.addr_family == AF_INET)
           printf("%s",inet_ntoa(addr.addr_union.ip_addr));

}

void print_flags(unsigned short f)
{
	printf(" [");
	if (f & IPSEC_MANUAL_KM_FLAG)
		printf("M");
	if (f & IPSEC_CHECK_REPLAY_FLAG)
		printf("C");
	if (f & IPSEC_TUNNEL_FLAG)
		printf("T");
	if (f & IPSEC_PARTIAL_FLAG)
		printf("P");
	if (f & IPSEC_NEG_KM_FLAG)
		printf("N");
	if (f & IPSEC_KM_KICKED_FLAG)
		printf("K");
	if (f & IPSEC_LIFETIME_PERMANENT_FLAG)
		printf("L");
	if (f & IPSEC_INBOUND_FLAG)
		printf("I");
	if (f & IPSEC_OUTBOUND_FLAG)
		printf("O");
	printf("]");
}

void print_alg(unsigned short id)
{
	/* Alg_ID */
	switch (id){
	case NO_ALG:
		printf("None");
	break;
	case ESP_DES_CBC:
		printf("DES/CBC(%u)",id);
	break;
	case ESP_RC5_CBC:
		printf("RC5/CBC(%u)",id);
	break;
	case ESP_BLOWFISH_CBC:
		printf("Blowfish/CBC(%u)",id);
	break;
	case ESP_IDEA_CBC:
		printf("IDEA/CBC(%u)",id);
	break;
	case ESP_3DES_CBC:
		printf("3DES/CBC(%u)",id);
	break;
	case HMACMD596:
		printf("HMAC_MD5_96(%u)",id);
	break;
	case HMACSHA96:
		printf("HMAC_SHA1_96(%u)",id);
	break;
	case CIPHER_TEST_ALG:
		printf("Cipher Test Alg.(%u)",id);
	break;
	case AUTH_TEST_ALG:
		printf("Authenticator Testing Alg.(%u)",id);
	break;
	case ESP_NULL:
		printf("ESP_NULL(%u)",id);
	break;
	case ESP_MARS_CBC:
		printf("ESP_MARS_CBC(%u)",id);
	break;
	case ESP_RC6_CBC:
		printf("ESP_RC6_CBC(%u)",id);
	break;
	case ESP_RIJNDAEL_CBC:
		printf("ESP_RIJNDAEL_CBC(%u)",id);
	break;
	case ESP_SERPENT_CBC:
		printf("ESP_SERPENT_CBC(%u)",id);
	break;
	case ESP_TWOFISH_CBC:
		printf("ESP_TWOFISH_CBC(%u)",id);
	break;
	default:
		printf("Unsupported(%u)",id);
	break;
	}
}

void print_dst(struct sadb_dst_node dst)
{

	if (dst.sa_info.spi == 0)
		 return;

	if (heading)
	{
		printf("Local-SPI Destination Prefix_length IPSEC-Peer     SPI Flags Seq.# Protocol Linked-Protocol Linked-SPI\n"); 
		switch (dst.sa_info.protocol)
		{
		case IPPROTO_ESP:
        		printf("       Crypto\n");
        		printf("       Alg_ID     IVEC_Length\n");
        		printf("                  Crypto Key: Length Key  \n");
		case IPPROTO_AH:
        		printf("       Auth\n");
        		printf("       Alg_ID     IVEC_Length Auth_Data_Length\n");
        		printf("                  Auth Key:   Length Key  \n");
			printf("--------------------------------------------------------\n");
		break;
		case PROTO_NULL_SA:
		default:
#if 0
        		printf("       Port #\n");
#endif
			printf("--------------------------------------------------------\n");
		break;
		}
		heading = 0; 
	}

	if (dst.dst.addr_family)
	{
	/* SPI */
	  printf("%lx    ",dst.sa_info.spi);
	  print_family(dst.dst.addr_family);
	  printf(":");
	/* Destination */
	   if(name){
		if (!print_name(dst.dst))
			 print_addr(dst.dst);
	   } else
			 print_addr(dst.dst);

	 printf(" %3u ",dst.prefix_len);

	/* Peer */
	   if(name){
		if (!print_name(dst.sa_info.peer_addr))
			 print_addr(dst.sa_info.peer_addr);
	   }
	   else
			 print_addr(dst.sa_info.peer_addr);

	/* Peer_SPI */
		printf(" %lx",dst.sa_info.peer_spi);  

	/* Flags */
		print_flags(dst.sa_info.flags);
	/* Sequence Number */
		printf(" %lu",dst.sa_info.sn); 
	/* Protocol */
		switch (dst.sa_info.protocol){
		case IPPROTO_ESP:
	/* Crypto Alg */
			printf(" ESP(%u)   ",dst.sa_info.protocol);
			if (dst.sa_info.next_encap.protocol == IPPROTO_ESP)
				printf(" ESP(%u)   ",dst.sa_info.next_encap.protocol);
			else if (dst.sa_info.next_encap.protocol == IPPROTO_AH)
				printf(" AH(%u)   ",dst.sa_info.next_encap.protocol);
			else
				printf(" %u       ",dst.sa_info.next_encap.protocol);
	  		printf("%lx\n",dst.sa_info.next_encap.spi);
			print_alg(dst.sa_info.alg.esp.crypto_alg_id);
	/* IVEC_length */
			if ((dst.sa_info.alg.esp.crypto_alg_id == ESP_DES_CBC)&&(!dst.sa_info.alg.esp.crypto_ivec_length))
				printf("(RFC1829-COMPAT)");
			printf("  %3u",dst.sa_info.alg.esp.crypto_ivec_length);
	/* KEY */
			printf("\n           ");
	                printf("%3u  ",dst.sa_info.alg.esp.outbound_crypto_key.key_len);
			print_key(dst.sa_info.alg.esp.outbound_crypto_key);
			printf("\n           ");

	/* Auth Alg */
			print_alg(dst.sa_info.alg.esp.auth_alg_id);
	/* IVEC_length */
			printf("  %3u",dst.sa_info.alg.esp.auth_ivec_length);
	/* ICV_Length */
			printf(" %3u",dst.sa_info.alg.esp.auth_data_length);
	/* KEY */
			printf("\n           ");
	                printf("%3u  ",dst.sa_info.alg.esp.outbound_auth_key.key_len);
			print_key(dst.sa_info.alg.esp.outbound_auth_key);
		break;
		case IPPROTO_AH:
			printf(" AH(%u)    ",dst.sa_info.protocol);
			if (dst.sa_info.next_encap.protocol == IPPROTO_ESP)
				printf(" ESP(%u)   ",dst.sa_info.next_encap.protocol);
			else if (dst.sa_info.next_encap.protocol == IPPROTO_AH)
				printf(" AH(%u)   ",dst.sa_info.next_encap.protocol);
			else
				printf(" %u   ",dst.sa_info.next_encap.protocol);
	  		printf("%lx\n",dst.sa_info.next_encap.spi);
			print_alg(dst.sa_info.alg.ah.auth_alg_id);
	/* IVEC_length */
			printf("  %3u",dst.sa_info.alg.ah.auth_ivec_length);
	/* ICV_Length */
			printf(" %3u",dst.sa_info.alg.ah.auth_data_length);
	/* KEYs */
			printf("\n           ");
	                printf("%3u  ",dst.sa_info.alg.ah.outbound_auth_key.key_len);
			print_key(dst.sa_info.alg.ah.outbound_auth_key);
		break;
		case PROTO_NULL_SA:
			printf(" NULL_SA(%u)\n",dst.sa_info.protocol);
		break;
		default:
			printf(" unknown(%u)\n",dst.sa_info.protocol);
		break;
		}

		printf("\n\n");
	}
	else printf("\n");
}

void print_smalldst(struct sadb_dst_node dst)
{

	struct timeval tv;
	struct timezone tz;
	struct tm *tm;

	if (dst.sa_info.spi == 0)
		 return;

	if (dst.dst.addr_family)
	{
	/* SPI */
	  printf("\nlocal_spi         %lx\naddress_family    ",dst.sa_info.spi);
  
	  print_family(dst.dst.addr_family);
          printf("\ndestination       ");
	/* Destination */
	   if(name){
		if (!print_name(dst.dst))
			 print_addr(dst.dst);
	   } else
			 print_addr(dst.dst);

	 printf("\naddr_prefix       %u\n",dst.prefix_len);
	 printf("ipsec_peer        ");

	/* Peer */
	   if(name){
		if (!print_name(dst.sa_info.peer_addr))
			 print_addr(dst.sa_info.peer_addr);
	   }
	   else
			 print_addr(dst.sa_info.peer_addr);

	/* Peer_SPI */
		printf("\npeer_spi          %lx\nflags            ",dst.sa_info.peer_spi);  

	/* Flags */
		print_flags(dst.sa_info.flags);
	/* Sequence Number */
		printf("\nsequence_number   %lu\n",dst.sa_info.sn); 
	/* Lifetime Fields */
	/* Bytes Remaining */
		if (!(dst.sa_info.flags & IPSEC_LIFETIME_PERMANENT_FLAG))
		{
			if (dst.sa_info.lifetime.bytes_remaining <= SA_BYTES_EXPIRED)
				printf("lifetime-B        EXPIRED\n");
			else if (dst.sa_info.lifetime.bytes_remaining <= SA_BYTES_EXPIRED_WARNING)
				printf("lifetime-B        %lu bytes remaining (WARNING)\n",dst.sa_info.lifetime.bytes_remaining);
			else
				printf("lifetime-B        %lu bytes remaining\n",dst.sa_info.lifetime.bytes_remaining);

			gettimeofday(&tv,&tz);
			tm = (struct tm *)localtime(&dst.sa_info.lifetime.time_expired);

			if (dst.sa_info.lifetime.time_expired-SA_TIME_EXPIRED <= tv.tv_sec)
				printf("lifetime-T        EXPIRED\n");
			else if (dst.sa_info.lifetime.time_expired-SA_TIME_EXPIRED_WARNING <= tv.tv_sec)
				printf("lifetime-T        %02d/%02d/%02d %02d:%02d:%02d (%ds) (WARNING)\n",tm->tm_mon,tm->tm_mday,tm->tm_year-((int)(tm->tm_year/100)*100),tm->tm_hour,tm->tm_min, tm->tm_sec, (dst.sa_info.lifetime.time_expired - tv.tv_sec) - SA_TIME_EXPIRED);
			else
				printf("lifetime-T        %02d/%02d/%02d %02d:%02d:%02d (%ds)\n",tm->tm_mon,tm->tm_mday,tm->tm_year-((int)(tm->tm_year/100)*100),tm->tm_hour,tm->tm_min, tm->tm_sec, (dst.sa_info.lifetime.time_expired - tv.tv_sec) - SA_TIME_EXPIRED);
		}
		else
		{
			printf("lifetime-B        N/A\n",dst.sa_info.lifetime.bytes_remaining);
			printf("lifetime-T        N/A\n");
		}
	/* Protocol */
		switch (dst.sa_info.protocol){
		case IPPROTO_ESP:
	/* Crypto Alg */
			printf("protocol          ESP(%u)\n",dst.sa_info.protocol);
			if (dst.sa_info.next_encap.protocol == IPPROTO_ESP)
				printf("linked-protocol   ESP(%u)\n",dst.sa_info.next_encap.protocol);
			else if (dst.sa_info.next_encap.protocol == IPPROTO_AH)
				printf("linked-protocol   AH(%u)\n",dst.sa_info.next_encap.protocol);
			else
				printf("linked-protocol   %u\n",dst.sa_info.next_encap.protocol);
	  		printf("linked-spi        %lx\n",dst.sa_info.next_encap.spi);
			printf("algorithm         ");
			print_alg(dst.sa_info.alg.esp.crypto_alg_id);
			if ((dst.sa_info.alg.esp.crypto_alg_id == ESP_DES_CBC)&&(!dst.sa_info.alg.esp.crypto_ivec_length))
				printf("(RFC1829-COMPAT)");
	/* IVEC_length */
			printf("\nIVEC_length       %u\n",dst.sa_info.alg.esp.crypto_ivec_length);
	/* KEY */
	                printf("key_len           %u\n",dst.sa_info.alg.esp.outbound_crypto_key.key_len);
	                if (dst.sa_info.alg.esp.outbound_crypto_key.key_len)
			{
				printf("key               ");
				print_key(dst.sa_info.alg.esp.outbound_crypto_key);
				printf("\n");
			}
			printf("algorithm         ");

	/* Auth Alg */
			print_alg(dst.sa_info.alg.esp.auth_alg_id);
	/* IVEC_length */
			printf("\nIVEC_length       %u\n",dst.sa_info.alg.esp.auth_ivec_length);
	/* ICV_Length */
			printf("auth_data_length  %u\n",dst.sa_info.alg.esp.auth_data_length);
	/* KEY */
	                printf("key_len           %u\n",dst.sa_info.alg.esp.outbound_auth_key.key_len);
	                if (dst.sa_info.alg.esp.outbound_auth_key.key_len)
			{
				printf("key               ");
				print_key(dst.sa_info.alg.esp.outbound_auth_key);
				printf("\n");
			}
		break;
		case IPPROTO_AH:
			printf("protocol          AH(%u)\n",dst.sa_info.protocol);
			if (dst.sa_info.next_encap.protocol == IPPROTO_ESP)
				printf("linked-protocol   ESP(%u)\n",dst.sa_info.next_encap.protocol);
			else if (dst.sa_info.next_encap.protocol == IPPROTO_AH)
				printf("linked-protocol   AH(%u)\n",dst.sa_info.next_encap.protocol);
			else
				printf("linked-protocol   %u\n",dst.sa_info.next_encap.protocol);
	  		printf("linked-spi        %lx\n",dst.sa_info.next_encap.spi);
			printf("algorithm         ");
			print_alg(dst.sa_info.alg.ah.auth_alg_id);
	/* IVEC_length */
			printf("\nIVEC_length       %u\n",dst.sa_info.alg.ah.auth_ivec_length);
	/* ICV_Length */
			printf("auth_data_length  %u\n",dst.sa_info.alg.ah.auth_data_length);
	/* KEY */
	                printf("key_len           %u\n",dst.sa_info.alg.ah.outbound_auth_key.key_len);
	                if (dst.sa_info.alg.ah.outbound_auth_key.key_len)
			{
				printf("key               ");
				print_key(dst.sa_info.alg.ah.outbound_auth_key);
				printf("\n");
			}
		break;
		case PROTO_NULL_SA:
			printf("protocol          NULL_SA(%u)\n",dst.sa_info.protocol);
		break;
		default:
			printf("protocol          unknown(%u)\n",dst.sa_info.protocol);
		break;
		}

	}
	else printf("\n");
}
void print_getmsg(m)
struct sadb_msg *m;
{
	switch(m->request_type){
	case SADB_SYSPOL:
		printf("System Protection Level\n");
		printf("------ ---------- -----\n");
		printf("Inbound: ");
		printf("<Protected Traffic");
		if (m->sadb_info.sys_pol.inbound.prot_flag&NON_PROTECTED)
		    printf("|Non-Protected Traffic");
		if (m->sadb_info.sys_pol.inbound.prot_flag&NULL_ASSOCIATION)
		    printf("|Empty Security Associations");
		if (m->sadb_info.sys_pol.inbound.prot_flag==PROTECTED)
		    printf(" Only");
		printf(">\n");
		printf("Outbound:");
		printf("<Protected Traffic");
		if (m->sadb_info.sys_pol.outbound.prot_flag&NON_PROTECTED)
		    printf("|Non-Protected Traffic");
		if (m->sadb_info.sys_pol.outbound.prot_flag&NULL_ASSOCIATION)
		    printf("|Empty Security Associations");
		if (m->sadb_info.sys_pol.outbound.prot_flag==PROTECTED)
		    printf(" Only");
		printf(">\n");
	break;
	case SADB_DST:
		if (smallout)
			print_smalldst(m->sadb_info.dst);
		else
			print_dst(m->sadb_info.dst);
	break;
	default:
		printf("\n");
	break;
	}
}


int sadbmsg(argv, flags)
	register char **argv;
	int flags;
{
	static int seq;
	int sadblen,arg_index;
	int nflags;
	register int l,i;

	errno = 0;
	bzero((char *)&m_sadbmsg, sizeof(m_sadbmsg));
        if (*argv[0] != 'g')
            shutdown(s, 0); /* Don't want to read back our messages */

	switch (*argv[0]){
	case 's':
	case 'a':
		m_sadbmsg.m_sadbm.sadbm_type = SADBM_SET;
		if (!argv[1]) usage("Missing <source|dest|prot>");
		switch (*argv[1]){
		case 'p':
			m_sadbmsg.request_type = SADB_SYSPOL;
			if (!argv[2]) usage("Missing inbound prot parameter");
			switch (*argv[2]){
			case '0':
				m_sadbmsg.sadb_info.sys_pol.inbound.prot_flag = PROTECTED;
			break;
			case '1':
				m_sadbmsg.sadb_info.sys_pol.inbound.prot_flag = PROTECTED|NON_PROTECTED;
			break;
			case '2':
				m_sadbmsg.sadb_info.sys_pol.inbound.prot_flag = PROTECTED|NULL_ASSOCIATION;
			break;
			case '3':
				m_sadbmsg.sadb_info.sys_pol.inbound.prot_flag = PROTECTED|NON_PROTECTED|NULL_ASSOCIATION;
			break;
			default:usage("Inbound Protection Level Error");
			break;
			}
			if (!argv[3]) usage("Missing outbound prot parameter");
			switch (*argv[3]){
			case '0':
				m_sadbmsg.sadb_info.sys_pol.outbound.prot_flag = PROTECTED;
			break;
			case '1':
				m_sadbmsg.sadb_info.sys_pol.outbound.prot_flag = PROTECTED|NON_PROTECTED;
			break;
			case '2':
				m_sadbmsg.sadb_info.sys_pol.outbound.prot_flag = PROTECTED|NULL_ASSOCIATION;
			break;
			case '3':
				m_sadbmsg.sadb_info.sys_pol.outbound.prot_flag = PROTECTED|NON_PROTECTED|NULL_ASSOCIATION;
			break;
			default:usage("Outbound Protection Level Error");
			break;
			}
		break;
		case 'd':
			m_sadbmsg.request_type = SADB_DST;
			/* Destination */
                        arg_index = 2;
			if (!argv[arg_index++]) usage("Missing <name|address>");
			if (!getaddr(argv[arg_index-1],&m_sadbmsg.sadb_info.dst.dst))
				usage(argv[arg_index-1]);
			/* prefix length */
			if (!argv[arg_index++]) usage("Missing prefix length");
			m_sadbmsg.sadb_info.dst.prefix_len = atoi(argv[arg_index-1]);

			/* SPI */
			if (!argv[arg_index++]) usage("Missing local-spi");
			sscanf(argv[arg_index-1],"%lx", &m_sadbmsg.sadb_info.dst.sa_info.spi);
			/* UID */
			m_sadbmsg.sadb_info.dst.sa_info.uid = 0;
			/* Peer */
			if (!argv[arg_index++]) usage("Missing <name|address>");
			if (!getaddr(argv[arg_index-1],&m_sadbmsg.sadb_info.dst.sa_info.peer_addr))
				usage(argv[arg_index-1]);
			/* Peer_SPI */
			if (!argv[arg_index++]) usage("Missing peer-spi");
			sscanf(argv[arg_index-1],"%lx", &m_sadbmsg.sadb_info.dst.sa_info.peer_spi);

			m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_LIFETIME_PERMANENT_FLAG; 

			/* Transport_Mechanism */
			if (!argv[arg_index++]) usage("Missing tunnel flag");
				if (atoi(argv[arg_index-1]))
					m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_TUNNEL_FLAG;
			if (!argv[arg_index++]) usage("Missing check replay flag");
				if (atoi(argv[arg_index-1]))
					 m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_CHECK_REPLAY_FLAG;
			m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_MANUAL_KM_FLAG;
/*  Request inbound only/outbound only/both inbound and outbound processing  */
			if (inbound)
				m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_INBOUND_FLAG;
			if (outbound)
				m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_OUTBOUND_FLAG;
			if (!argv[arg_index++]) usage("Missing initial-sequence-number");
			sscanf(argv[arg_index-1],"%lx", &m_sadbmsg.sadb_info.dst.sa_info.sn);

			/* Protocol */
			if (!argv[arg_index++]) usage("Missing protocol");
			m_sadbmsg.sadb_info.dst.sa_info.protocol = atoi(argv[arg_index-1]);

			/* Linked-Protocol */
			if (!argv[arg_index++]) usage("Missing linked-protocol");
			m_sadbmsg.sadb_info.dst.sa_info.next_encap.protocol = atoi(argv[arg_index-1]);

			/* Linked-SPI */
			if (!argv[arg_index++]) usage("Missing linked-spi");
			sscanf(argv[arg_index-1],"%lx", &m_sadbmsg.sadb_info.dst.sa_info.next_encap.spi);
			/* UID */

			switch (m_sadbmsg.sadb_info.dst.sa_info.protocol)
			{
			case IPPROTO_ESP:

			/* Alg_ID */
			if (!argv[arg_index++]) usage("Missing crypto alg-id");
			if ((m_sadbmsg.sadb_info.dst.sa_info.alg.esp.crypto_alg_id = atoi(argv[arg_index-1]))==NO_ALG) usage("Missing crypto alg-id");
			/* Ivec_Length */
			if (!argv[arg_index++]) usage("Missing crypto ivec length");
			m_sadbmsg.sadb_info.dst.sa_info.alg.esp.crypto_ivec_length = atoi(argv[arg_index-1]);
			/* Keys */
			if (!argv[arg_index++]) usage("Missing crypto key length");
			m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_len = atoi(argv[arg_index-1]);
                        if (m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_len){
			    if (!argv[arg_index++]) usage("Missing crypto key");
			    for (i=0;i<m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_len;i++)
			        sscanf(&argv[arg_index-1][i*2],"%2x",
			           (u_char *)&(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_data[i]));
                        }
			
			m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_crypto_key.key_len = 
				m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_len;
			memcpy(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_crypto_key.key_data,
				m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_data, 
				m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_crypto_key.key_len);
			/* Alg_ID */
			if (!argv[arg_index++]) usage("Missing auth alg-id");
			if ((m_sadbmsg.sadb_info.dst.sa_info.alg.esp.auth_alg_id = atoi(argv[arg_index-1])) == NO_ALG) break;
			/* Ivec_Length */
			if (!argv[arg_index++]) usage("Missing auth ivec length");
			m_sadbmsg.sadb_info.dst.sa_info.alg.esp.auth_ivec_length = atoi(argv[arg_index-1]);

			if (!argv[arg_index++]) usage("Missing auth key length");
			m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_len = atoi(argv[arg_index-1]);
                       	if (m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_len){
		    	if (!argv[arg_index++]) usage("Missing auth key");
		    	for (i=0;i<m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_len;i++)
		        	sscanf(&argv[arg_index-1][i*2],"%2x",
		           	(u_char *)&(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_data[i]));
                       	}
			m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_auth_key.key_len = 
				m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_len;
			memcpy(m_sadbmsg.sadb_info.dst.sa_info.alg.esp.inbound_auth_key.key_data,
				m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_data,
				m_sadbmsg.sadb_info.dst.sa_info.alg.esp.outbound_auth_key.key_len);
			/* ICV_Length */
			if (!argv[arg_index++]) usage("Missing icv-length");
			m_sadbmsg.sadb_info.dst.sa_info.alg.esp.auth_data_length = atoi(argv[arg_index-1]);
			break;
			case IPPROTO_AH:

			/* Alg_ID */
			if (!argv[arg_index++]) usage("Missing auth alg-id");
			if ((m_sadbmsg.sadb_info.dst.sa_info.alg.ah.auth_alg_id = atoi(argv[arg_index-1]))==NO_ALG) usage("Missing auth alg-id");
			/* Ivec_Length */
			if (!argv[arg_index++]) usage("Missing auth ivec length");
			m_sadbmsg.sadb_info.dst.sa_info.alg.ah.auth_ivec_length = atoi(argv[arg_index-1]);

			if (!argv[arg_index++]) usage("Missing auth key length");
			m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_len = atoi(argv[arg_index-1]);
                        if (m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_len){
			    if (!argv[arg_index++]) usage("Missing auth key");
			    for (i=0;i<m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_len;i++)
			        sscanf(&argv[arg_index-1][i*2],"%2x",
			           (u_char *)&(m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_data[i]));
                        }
			m_sadbmsg.sadb_info.dst.sa_info.alg.ah.inbound_auth_key.key_len = 
				m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_len;
			
			memcpy(m_sadbmsg.sadb_info.dst.sa_info.alg.ah.inbound_auth_key.key_data,
				m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_data,
				m_sadbmsg.sadb_info.dst.sa_info.alg.ah.outbound_auth_key.key_len);
			/* ICV_Length */
			if (!argv[arg_index++]) usage("Missing icv-length");
			m_sadbmsg.sadb_info.dst.sa_info.alg.ah.auth_data_length = atoi(argv[arg_index-1]);
			break;
			case PROTO_NULL_SA:
/* XXXX If null SA can be either manual or negotiated, need to set flags correctly 
        here, rather than in kernel - while we're at it, might as well set sn to 0 */
			    nflags = m_sadbmsg.sadb_info.dst.sa_info.flags;
/* 			   Make sure null SA's TUNNEL & CHECK REPLAY FLAGS are OFF */
			    if (nflags & IPSEC_TUNNEL_FLAG)
			        m_sadbmsg.sadb_info.dst.sa_info.flags ^= IPSEC_TUNNEL_FLAG;
			    if (nflags & IPSEC_CHECK_REPLAY_FLAG)
			        m_sadbmsg.sadb_info.dst.sa_info.flags ^= IPSEC_CHECK_REPLAY_FLAG;
/* 			   Make sure null SA's sn is 0 */
			    m_sadbmsg.sadb_info.dst.sa_info.sn = 0;
			break;
			default: usage("Unknown ipsec protocol");
			break;
			}
		break;
		default:
			usage("Not <dest|prot>");
		break;
		}
		break;
	case 'c':
		m_sadbmsg.m_sadbm.sadbm_type = SADBM_CHANGE;
		break;
	case 'f':
		m_sadbmsg.m_sadbm.sadbm_type = SADBM_FLUSH;
		break;
	case 'g':
		m_sadbmsg.m_sadbm.sadbm_type = SADBM_GET;
		if (!argv[1]) usage("Missing <source|dest|prot>");
		switch (*argv[1]){
		case 'p':
			m_sadbmsg.request_type = SADB_SYSPOL;
		break;
		case 'd':
			m_sadbmsg.request_type = SADB_DST;
		        if (argv[2]){
				if (!getaddr(argv[2],&m_sadbmsg.sadb_info.dst.dst)) usage(argv[2]);
				if (!argv[3]) usage("Missing spi");
			        sscanf(argv[3],"%lx", &m_sadbmsg.sadb_info.dst.sa_info.spi);
				/* Protocol */
				if (!argv[4]) usage("Missing protocol");
				m_sadbmsg.sadb_info.dst.sa_info.protocol = atoi(argv[4]);
			}
/*  Request inbound only/outbound only/both inbound and outbound processing  */
			if (inbound)
				m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_INBOUND_FLAG;
			if (outbound)
				m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_OUTBOUND_FLAG;

		break;
		default:
			usage("Not <source|dest|prot>");
		break;
		}
		break;
	case 'd':
		m_sadbmsg.m_sadbm.sadbm_type = SADBM_DELETE;
		if (!argv[1]) usage("Missing <source|dest|prot>");
		switch (*argv[1]){
		case 'd':
			m_sadbmsg.request_type = SADB_DST;
			/* Destination */
			if (!argv[2]) usage("Missing <name|address>");
			if (!getaddr(argv[2],&m_sadbmsg.sadb_info.dst.dst))
				usage(argv[2]);
			/* SPI */
			if (!argv[3]) usage("Missing spi");
			sscanf(argv[3],"%lx", &m_sadbmsg.sadb_info.dst.sa_info.spi);
			/* Protocol */
			if (!argv[4]) usage("Missing protocol");
			m_sadbmsg.sadb_info.dst.sa_info.protocol = atoi(argv[4]);
/*  Request inbound only/outbound only/both inbound and outbound processing  */
			if (inbound)
				m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_INBOUND_FLAG;
			if (outbound)
				m_sadbmsg.sadb_info.dst.sa_info.flags |= IPSEC_OUTBOUND_FLAG;
		break;
		default:
			usage("Not <source|dest|prot>");
		break;
		}
		break;
	default:
		fprintf(stderr,"Command not supported\n");
		exit(-1);
	}
	m_sadbmsg.m_sadbm.sadbm_flags = flags;
	m_sadbmsg.m_sadbm.sadbm_version = SADBM_VERSION;
	m_sadbmsg.m_sadbm.sadbm_msglen = l = sizeof(struct sadb_msg);

	if ((sadblen = write(s, (char *)&m_sadbmsg, l,MSG_DONTWAIT)) < 0) {
		perror("writing to SADB socket");
		return (-1);
	}

	if (m_sadbmsg.m_sadbm.sadbm_type == SADBM_GET) {
		heading = 1;
		while (1)
		{
			struct sadb_msg *sm;

			sadblen = read(s, (char *)buffer, sizeof(struct sadb_msg),MSG_NOSIGNAL);
			sm = (struct sadb_msg *)buffer;
			print_getmsg(sm);
			if (sm->m_sadbm.sadbm_flags & SADBM_DONE)
				break;
		}

	}
	return (0);
}

void entry(argc, argv)
	int argc;
	register char **argv;
{
	char *err;
	int ishost = 0, ret, attempts, oerrno, flags = 0;
	int key;
	struct hostent *hp = 0;
	union sadb_union sadb_info;

#ifndef SI_TESTER
	if (uid) {
		errno = EACCES;
		quit("must be root to query security association database");
	}
#endif
	flags = SADBM_MANUAL;
	for (attempts = 1; ; attempts++) {
		errno = 0;
		if ((ret = sadbmsg(argv,flags)) == 0)
			break;
	}
	exit(0);
}



int keyword(cp)
	char *cp;
{
	register struct keytab *kt = keywords;

	while (kt->kt_cp && strcmp(kt->kt_cp, cp))
		kt++;
	return kt->kt_i;
}

int main(argc, argv)
	int argc;
	char **argv;
{
	extern int optind;
	int k,ch,errno;
	

	verbose = 0;
	name = 1;
	inbound = 0;
	outbound = 0;
	while ((ch = getopt(argc, argv, "iovpns")) != EOF)
		switch(ch) {
/*   -i (inbound)/ -o (outbound) for one-way SA's                   */
		case 'i':
			inbound = 1;
			break;
		case 'o':
			outbound = 1;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'n':
			name = 0;
			break;
		case 's':
			smallout = 1;
			break;
		case '?':
		default:
			usage("Unknown Option");
		}

	if (argc < 2)
	   usage("Missing Parameters");
	argc -= optind;
	argv += optind;

        if ((!inbound)&&(!outbound)){
		inbound = 1;
		outbound = 1;
	}
	if (*argv){

		pid = getpid();
		uid = getuid();
		
		/* Open /dev/ipsec for kernel communications - 
		if busy, sleep and try again */
		for (k = 0; k < 5; k++) 
		{
        		if ((s = open("/dev/ipsec", O_RDWR)) > 0)
                		break;
        		if ((errno != EBUSY) || (k == 4))
				usage("bad device /dev/ipsec");
        		sleep(3);
    		}

		switch (keyword(*argv)) {
		case K_GET:
			/* FALLTHROUGH */
		case K_CHANGE:
			/* FALLTHROUGH */
		case K_FLUSH:
			/* FALLTHROUGH */
		case K_SET:
			/* FALLTHROUGH */
		case K_DELETE:
			/* FALLTHROUGH */
			entry(argc, argv);
			break;
		default:
			usage("Erroneous Command");
			break;
		}
	}
	else
		usage("Missing Command");
	
	exit(0);
}

