/* $FreeBSD: src/tools/tools/crypto/cryptokeytest.c,v 1.1 2003/01/06 22:11:56 sam Exp $ */
/*
 * The big num stuff is a bit broken at the moment and I've not yet fixed it.
 * The symtom is that odd size big nums will fail.  Test code below (it only
 * uses modexp currently).
 * 
 * --Jason L. Wright
 */
#include <sys/types.h>
#include <sys/ioctl.h>
#ifdef linux
#include <endian.h>
#else
#include <sys/endian.h>
#endif
#include <sys/time.h>
#include <opencrypto/cryptodev.h>
#include <openssl/bn.h>
#include <fcntl.h>
#include <err.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>

static int crypto_fd = -1;
static char devicename[CRYPTO_NAME_LEN] = "";
static int errs;
static int verbose = 0;

/*
 * Convert a little endian byte string in 'p' that
 * is 'plen' bytes long to a BIGNUM. If 'dst' is NULL,
 * a new BIGNUM is allocated.  Returns NULL on failure.
 *
 * XXX there has got to be a more efficient way to do
 * this, but I haven't figured out enough of the OpenSSL
 * magic.
 */
BIGNUM *
le_to_bignum(BIGNUM *dst, u_int8_t *p, int plen)
{
	u_int8_t *pd;
	int i;

	if (plen == 0)
		return (NULL);

	if ((pd = (u_int8_t *)malloc(plen)) == NULL)
		return (NULL);

	for (i = 0; i < plen; i++)
		pd[i] = p[plen - i - 1];

	dst = BN_bin2bn(pd, plen, dst);
	free(pd);
	return (dst);
}

/*
 * Convert a BIGNUM to a little endian byte string.
 * If 'rd' is NULL, allocate space for it, otherwise
 * 'rd' is assumed to have room for BN_num_bytes(n)
 * bytes.  Returns NULL on failure.
 */
u_int8_t *
bignum_to_le(BIGNUM *n, u_int8_t *rd)
{
	int i, j, k;
	int blen = BN_num_bytes(n);

	if (blen == 0)
		return (NULL);
	if (rd == NULL)
		rd = (u_int8_t *)malloc(blen);
	if (rd == NULL)
		return (NULL);

	for (i = 0, j = 0; i < n->top; i++) {
		for (k = 0; k < BN_BITS2 / 8; k++) {
			if ((j + k) >= blen)
				goto out;
			rd[j + k] = n->d[i] >> (k * 8);
		}
		j += BN_BITS2 / 8;
	}
out:
	return (rd);
}

/*
 * assumes that the group has been initialized, that is, that the
 * str_modulus has been copied and turned into an MP_INT already.
 *
 *
 * This algorithm will not work if the top bits of the modulus are
 * 0x80000, but all of the IETF moduli have been chosen such that the
 * top bits are 0xffffffff 0xffffffff.
 *
 */
void calc_reciprocal(BIG_NUM *rec, BIG_NUM *modp)
{
	mpz_t n, one,reciprocal;
	int nlen = 2*group->bytes*BITS_PER_BYTE-1;
	int tries=1000000;

	mpz_init(one);
	mpz_init(reciprocal);
	mpz_init(n);

	mpz_set_ui(one, 1);

	/* calculate 1 followed by 2 times number of bits, minus 1 */
	mpz_mul_2exp(n, one, nlen);

	/* now reciprocal is n divied by group */
	mpz_tdiv_q(reciprocal, n, group->modulus);

	mpz_mul(n, group->modulus, reciprocal);

	/* make sure that result has a 1 bit in highest position */
	while(mpz_tstbit(n, nlen) == 1 && tries-->0) {

		mpz_sub_ui(reciprocal, reciprocal, 1);
		mpz_mul(n, group->modulus, reciprocal);
	}

	fprintf(stdout, "Group %d Tries: %d\nModulus: ", group->group, tries);
	mpz_out_str(stdout, 16, group->modulus);

	fprintf(stdout, "\nReciprocal: ");
	mpz_out_str(stdout, 16, reciprocal);

	fprintf(stdout, "\nProduct: ");

	mpz_out_str(stdout, 16, n);
	fprintf(stdout, "\n\n");

	mpz_clear(n);
	mpz_clear(reciprocal);
	mpz_clear(one);
}


int
UB_mod_exp(BIGNUM *res,
	   BIGNUM *a,
	   BIGNUM *b,
	   BIGNUM *c,         /* modulus */
	   BN_CTX *ctx)
{
	struct crypt_kop kop;
	u_int8_t *ale, *ble, *cle, *rle;
	BIGNUM recip;

	if (crypto_fd == -1) {
		int fd, fdc = open("/dev/crypto", O_RDONLY);

		if (fdc == -1)
			err(1, "/dev/crypto");
		if (ioctl(fdc, CRIOGET, &fd) == -1)
			err(1, "CRIOGET");
		close(fdc);
		crypto_fd = fd;
	}

	calc_reciprocal(&recip, cle);

	if ((ale = bignum_to_le(a, NULL)) == NULL)
		err(1, "bignum_to_le, a");
	if ((ble = bignum_to_le(b, NULL)) == NULL)
		err(1, "bignum_to_le, b");
	if ((cle = bignum_to_le(c, NULL)) == NULL)
		err(1, "bignum_to_le, c");
	if ((rle = bignum_to_le(&recip, NULL)) == NULL)
		err(1, "bignum_to_le, recip");

	bzero(&kop, sizeof(kop));
	kop.crk_op = CRK_MOD_EXP;
	kop.crk_iparams = 4;
	kop.crk_oparams = 1;
	kop.crk_param[0].crp_p = (char *)ale;
	kop.crk_param[0].crp_nbits = BN_num_bytes(a) * 8;
	kop.crk_param[1].crp_p = (char *)ble;
	kop.crk_param[1].crp_nbits = BN_num_bytes(b) * 8;
	kop.crk_param[2].crp_p = (char *)cle;
	kop.crk_param[2].crp_nbits = BN_num_bytes(c) * 8;
	kop.crk_param[3].crp_p = (char *)rle;
	kop.crk_param[3].crp_nbits = BN_num_bytes(recip) * 8;
	strncpy (kop.crypto_device_name, devicename, sizeof (kop.crypto_device_name));

	if (ioctl(crypto_fd, CIOCKEY, &kop) == -1) {
		printf("ioctl with %08x\n", CIOCKEY);
		if(errno == EINVAL) {
			err(44, "function not implemented\n");
		}
		warn("CIOCKEY");
	}

	bzero(ale, BN_num_bytes(a));
	free(ale);
	bzero(ble, BN_num_bytes(b));
	free(ble);
	bzero(rle, BN_num_bytes(rle));
	free(rle);

	if (kop.crk_status != 0) {
		printf("error %d\n", kop.crk_status);
		bzero(cle, BN_num_bytes(c));
		free(cle);
		return (-1);
	} else {
		res = le_to_bignum(res, cle, BN_num_bytes(c));
		bzero(cle, BN_num_bytes(c));
		free(cle);
		if (res == NULL)
			err(1, "le_to_bignum");
		return (0);
	}
	return (0);
}

void
show_result(a, b, c, sw, hw)
BIGNUM *a, *b, *c, *sw, *hw;
{
	printf("\n");

	printf("A = ");
	BN_print_fp(stdout, a);
	printf("\n");

	printf("B = ");
	BN_print_fp(stdout, b);
	printf("\n");

	printf("C = ");
	BN_print_fp(stdout, c);
	printf("\n");

	printf("sw= ");
	BN_print_fp(stdout, sw);
	printf("\n");

	printf("hw= ");
	BN_print_fp(stdout, hw);
	printf("\n");

	printf("\n");
}

void
testit(void)
{
	BIGNUM *a, *b, *c, *r1, *r2;
	BN_CTX *ctx;

	ctx = BN_CTX_new();

	a = BN_new();
	b = BN_new();
	c = BN_new();    /* the modulus */
	r1 = BN_new();
	r2 = BN_new();

	BN_pseudo_rand(a, 1023, 0, 0);
	BN_pseudo_rand(b, 1023, 0, 0);
	do {
		BN_pseudo_rand(c, 1024, 0, 0);
	} while (!BN_is_odd(c));
	BN_set_bit(c, 1023);

	if (BN_cmp(a, c) > 0) {
		BIGNUM *rem = BN_new();

		BN_mod(rem, a, c, ctx);
		UB_mod_exp(r2, rem, b, c, ctx);
		BN_free(rem);
	} else {
		UB_mod_exp(r2, a, b, c, ctx);
	}
	BN_mod_exp(r1, a, b, c, ctx);

	if (BN_cmp(r1, r2) != 0) {
		errs++;
		show_result(a, b, c, r1, r2);
	}

	BN_free(r2);
	BN_free(r1);
	BN_free(c);
	BN_free(b);
	BN_free(a);
	BN_CTX_free(ctx);
}

static void
usage(const char* cmd)
{
	printf ("usage: %s [-v] [-d devicename] [count]\n", cmd);
	printf ("where devicename is one of:\n"
		"    anydevice   - any device,\n"
		"    anyhardware - any hardware device,\n"
		"    anysoftware - any software device,\n"
		"    software    - use cryptosoft device;\n"
		"    ... or any hardware device name.\n"
		"\n"
		"default count is 1000\n"
		"\n"
		"-v be verbose (can be specified multiple times)\n");
	exit (-1);
}

int
main(int argc, char **argv)
{
	char ch;
	int i, count = 1000;

	while ((ch = getopt (argc, argv, "vd:")) != -1) {
		switch (ch) {
		case 'd':
			strncpy (devicename, optarg, CRYPTO_NAME_LEN);
			break;
		case 'v':
			verbose++;
			break;
		default:
			usage(argv[0]);
		}
	}
	if (argc == (optind+1))
		count = atoi(argv[optind]);
	else if (argc>optind)
		usage(argv[0]);

	for (i = 0; i < count; i++) {
		fprintf(stderr, "test %d, errs=%d\n", i, errs);
		testit();
	}
	return (0);
}
