/*-
 * Copyright (c) 2006 Michael Richardson <mcr@xelerance.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Effort sponsored in part by Hifn inc.
 */

/*
 * this program maps the hifn vulcan Public Key engine and runs various
 * test case on it.
 *
 */

#include <stdio.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <fcntl.h>
#include <dev/hifn/hifn7751reg.h>

#include "/i/hifn/osw-ocf/programs/pluto/vulcan/vulcanpk_funcs.c"

void sanitycheckpk(unsigned char *mapping)
{
	u_int32_t stat;

	printf("sanitycheck: %p size=%d\n", mapping, HIFN_1_PUB_MEMEND);

	vulcanpk_init(mapping);

	printf("setting to PK mode=%08x\n", PUB_WORD(HIFN_1_PUB_MODE));

	//hexdump(mapping+0x300, 256);

	//PUB_WORD(HIFN_1_PUB_BASE) = 

	/* clear out public key memory */
	memset(mapping+HIFN_1_PUB_MEM, 0, HIFN_1_PUB_MEMEND-HIFN_1_PUB_MEM);

	/* verify that memory cleared */
	{
		int i;
		for(i=HIFN_1_PUB_MEM; i< HIFN_1_PUB_MEMEND; i++) {
			if(mapping[i]!=0) {
				printf("failed to clear pubkey memory at %d (value=%02x)\n", i, mapping[i]);
				exit(4);
			}
		}
	}
	printf("memory cleared properly\n");

	print_status(PUB_WORD(HIFN_1_PUB_STATUS));

	{
		struct pkprogram pk1;
		unsigned char aValue[PKVALUE_LEN];
		unsigned char bValue[PKVALUE_LEN];
		
		memset(&pk1, 0, sizeof(pk1));
		memset(aValue, 0, sizeof(aValue));
		memset(bValue, 0, sizeof(bValue));

		pk1.valuesLittleEndian=1;
		aValue[9]=10;
		bValue[9]=30;

		pk1.aValues[0]=aValue;
		pk1.aValueLen[0]=10;
		pk1.aValues[1]=bValue;
		pk1.aValueLen[1]=10;
		pk1.oOffset = 2;
		pk1.pk_program[0]=
			/* sizes are ModLen=96(*32=3072), EXP_len=1, RED_len=0 */
			(0<<24)|(1<<8)|(96);
		pk1.pk_program[1]=
			/* opcode 0001(ADD), with A=0, B=1 (6) */
			(1<<24)|(6<<8)|(0<<0);
		
		pk1.pk_proglen = 1;
		pk_verbose_execute=1;

		execute_pkprogram(mapping, &pk1);
	}
}

void do_diffiehellman(unsigned char *mapping)
{
    struct pkprogram pk1;

    /* this id modp1536, from draft-ietf-ike-01.txt */
    unsigned char aValue[] = {
#include "modp1536.h"
    };
    unsigned char bValue[PKVALUE_LEN];
    unsigned char oValue[PKVALUE_LEN];
    int expLen;
    int modLen;

    printf("\n\n Calculate reciprocal\n");

    hexdump(aValue, sizeof(aValue));
    memset(&pk1, 0, sizeof(pk1));
    
    /* 2. set preliminary reciprocol to zero */
    memset(bValue, 0, sizeof(bValue));
    
    /* 3. set top chunk (word,byte) of preliminary reciprocal to
     * bit inverse of modulus */
    bValue[0] = aValue[0] ^ 0xff;
    
    /* 4. set top bit of preliminary reciprocal to 1 */
    bValue[0] |= 0x80;
    
    expLen = 12;
    modLen = sizeof(aValue)/4;

    pk1.valuesLittleEndian=1;
    pk1.aValues[0]=aValue;
    pk1.aValueLen[0]=sizeof(aValue);
    pk1.aValues[1]=bValue;
    pk1.aValueLen[1]=sizeof(aValue);
    
    pk1.pk_program[0]=
	/* sizes are ModLen=96(*32=3072), EXP_len=1, RED_len=0 */
	(modLen<<24)|(expLen<<8)|(96);
    pk1.pk_program[1]=
	/* opcode 1110(Calc rec), with A=0, B=1 (6) */
	(1<<24)|(6<<8)|(0<<0);
    pk1.pk_proglen = 1;

    pk1.oOffset = 1;
    pk1.oValue  = oValue;
    pk1.oValueLen = PKVALUE_LEN;
    
    execute_pkprogram(mapping, &pk1);
}

void do_rsasignature(unsigned char *mapping)
{
}

void do_rsaverify(unsigned char *mapping)
{
}



int main(int argc, char *argv[])
{
	unsigned char *vulcanpkreg;

	vulcanpkreg = mapvulcanpk();
	if(vulcanpkreg == NULL) exit(5);

	sanitycheckpk(vulcanpkreg);

	do_diffiehellman(vulcanpkreg);
	do_rsasignature(vulcanpkreg);
	do_rsaverify(vulcanpkreg);

	exit(0);
}
	
/*
 * Local Variables:
 * c-basic-offset:4
 * c-style: pluto
 * End:
 */
