/*
 *	MainFrame.java		2002/11/17
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.swingui;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import com.alistairphipps.jsilc.core.*;

/** 
 * The main frame for the application
 * @author Alistair K Phipps
 * @version 2002/11/17
 */
public class MainFrame extends JFrame
{
	/** uihandler that owns us - reference kept for callback */
	private SwingUIHandler _uih;
	/** menu item for Connect - kept so we can gray it out */
	private JMenuItem _miConnect;
	/** menu item for Disconnect - kept so we can gray it out */
	private JMenuItem _miDisconnect;
	/** map to look up a chat window for a user */
	private Map _mapEntityToChatFrame = Collections.synchronizedMap( new HashMap() );
	
	/** Inner class to handle Connect menu option */
	private class ConnectL implements ActionListener
	{
		public void actionPerformed( ActionEvent e )
		{
			ConnectDlg dlg = new ConnectDlg( getThis() );	// pass this so it can call us back with results
			dlg.show();	// should block until Connect/Cancel selected
		}
	}

	/** Get MainFrame "this" for ConnectL */
	private MainFrame getThis()
	{
		return this;
	}

	/** Inner class to handle Disconnect menu option */
	private class DisconnectL implements ActionListener
	{
		public void actionPerformed( ActionEvent e )
		{
			disconnect();
		}
	}
	
	/** Inner class to handle Exit menu option */
	private class ExitL implements ActionListener
	{
		public void actionPerformed( ActionEvent e )
		{
			System.exit( 0 );
		}
	}

	/** Inner class to handle Tile menu option */
	private class TileL implements ActionListener
	{
		public void actionPerformed( ActionEvent e )
		{ // TODO
		}
	}
	
	/** Inner class to handle Cascade menu option */
	private class CascadeL implements ActionListener
	{
		public void actionPerformed( ActionEvent e )
		{ // TODO
		}
	}
	
	/** Inner class to handle About menu option */
	private class AboutL implements ActionListener
	{
		public void actionPerformed( ActionEvent e )
		{
			AboutDlg dlg = new AboutDlg( null ); // XXX maybe should specify frame as the parent - doesn't seem to matter though
			dlg.show();
		}
	}

	/** helper method to perform a connect - called from ConnectDlg */
	void connect( String strUsername, String strHost, int iPort )
	{
		_miConnect.setEnabled( false );
		_miDisconnect.setEnabled( true );
		_uih.connect( strUsername, strHost, iPort );
	}

	/** helper method to perform a disconnect */
	private void disconnect()
	{
		_miConnect.setEnabled( true );
		_miDisconnect.setEnabled( false );
		_uih.disconnect();
		// close all chat frames
		Iterator i = _mapEntityToChatFrame.values().iterator();
		while( i.hasNext() )
			( (ChatFrame)i.next() ).hide();
		_mapEntityToChatFrame = Collections.synchronizedMap( new HashMap() );
	}
	
	/** output message to chat frame */
	public void outputMessage( Entity eSource, String strMessage )
	{
		ChatFrame c;
		
		// if we don't have a frame for this user already, create one
		if( ( c = (ChatFrame)_mapEntityToChatFrame.get( eSource ) ) == null )
		{
			c = new ChatFrame( this, eSource );
			_mapEntityToChatFrame.put( eSource, c );
			getContentPane().add( c );
		}
 
		c.outputMessage( strMessage );
	}

	/** send messageto remote user (called by chatframe) */
	void sendMessage( Entity eDest, String strMessage )
	{
		_uih.sendMessage( eDest, strMessage );
	}
	
	/** Helper method to set up the menus
	 * @return MenuBar containing info on the menus
	 */
	public JMenuBar createMenus()
	{
		// set up the menu
		JMenuItem[] file = { new JMenuItem( "Connect...", KeyEvent.VK_C ), new JMenuItem( "Disconnect", KeyEvent.VK_D ), new JMenuItem( "Exit", KeyEvent.VK_X ) };
		_miConnect = file[0]; _miDisconnect = file[1]; // we need these elsewhere to grey them out
		_miDisconnect.setEnabled( false );	// initially disconnected
		ActionListener[] fileL = { new ConnectL(), new DisconnectL(), new ExitL() };
		JMenuItem[] window = { new JMenuItem( "Tile", KeyEvent.VK_T ), new JMenuItem( "Cascade", KeyEvent.VK_C ) };
		ActionListener[] windowL = { new TileL(), new CascadeL() };
		JMenuItem[] help = { new JMenuItem( "About jSILC", KeyEvent.VK_A ) };
		ActionListener[] helpL = { new AboutL() };

		JMenu f = new JMenu( "File" );
		f.setMnemonic( KeyEvent.VK_F );
		for( int i = 0; i < file.length; i++ )
		{
			file[i].addActionListener( fileL[i] );
			f.add( file[i] );
		}
		JMenu w = new JMenu( "Window" );
		w.setMnemonic( KeyEvent.VK_W );
		for( int i = 0; i < window.length; i++ )
		{
			window[i].addActionListener( windowL[i] );
			w.add( window[i] );
		}
		JMenu h = new JMenu( "Help" );
		h.setMnemonic( KeyEvent.VK_H );
		for( int i = 0; i < help.length; i++ )
		{
			help[i].addActionListener( helpL[i] );
			h.add( help[i] );
		}

		JMenuBar mb = new JMenuBar();
		mb.add( f );
		mb.add( w );
		mb.add( h );
		
		return mb;
	}
	
	/** Default ctor - sets up the frame
	 */
	public MainFrame( SwingUIHandler uih )
	{
		super( "jSILC" );
		try
		{	// try to set cross-platform java look and feel
			UIManager.setLookAndFeel( UIManager.getCrossPlatformLookAndFeelClassName() );
        } 
		catch (Exception e)
		{}	// doesn't matter if it fails
		
		// set member var so we can call back to our UI handler
		_uih = uih;
		
		// setup desktop for internal frames
		JDesktopPane desktop = new JDesktopPane();
		setContentPane( desktop );
		
		// set up menu
		setJMenuBar( createMenus() );
		
		//Make the frame be indented 50 pixels from each edge of the screen.
        int inset = 50;
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        setBounds(inset, inset, screenSize.width - inset*2, screenSize.height - inset*2 );


		// make app exit when user closes main window
		addWindowListener(
			new WindowAdapter()
			{
           		public void windowClosing( WindowEvent e )
				{
                	System.exit(0);
            	}
        	}
		);

		// display the main window
		setVisible( true );
	}
}
