/*
 *	SILCHeader.java		2002/11/05
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.silcprotocol;

import com.alistairphipps.util.Hex;
import java.nio.ByteBuffer;

/** Encapsulates information about a SILC packet header, and allows conversion to/from the unencrypted binary representation of the header.
 *
 * @author Alistair K Phipps
 * @version 20021108
 */
public class SILCHeader
{
	/** length of packet, not including padding */
	private short _sPayloadLength;

	/** whether packet includes private message encrypted using private key set by client */
	private boolean _bPrivateMessageKey;
	/** whether packet consists of list of packet payloads indicated by the Packet Type field */
	private boolean _bList;
	/** whether packet was broadcast (from a router server) */
	private boolean _bBroadcast;
	/** whether payload of packet is compressed */
	private boolean _bCompressed;
	
	/** packet type */
	private byte _yType;
	/** length of padding */
	private byte _yPadLength;
	/** source ID type */
	private byte _ySrcIdType;
	/** source ID */
	private byte[] _ylSrcId;
	/** dest ID type */
	private byte _yDestIdType;
	/** dest ID */
	private byte[] _ylDestId;

	/** Constructor to initialise the various data fields using specified values
	 * @param yType packet type
	 * @param sPayloadLength length of packet (header+payload+mac) excluding padding
	 * @param bPrivateMessageKey whether includes priv msg encrypted using private key set by client
	 * @param bList whether packet has list of packet payloads
	 * @param bBroadcast whether packet was broadcast from router
	 * @param bCompressed whether payload is compressed
	 * @param ySrcIdType type of source ID
	 * @param ylSrcId source ID
	 * @param yDestIdType type of destination ID
	 * @param ylDestId destination ID
	 * @param yPadLength length of padding that is going to be inserted into packet
	 */
	public SILCHeader( byte yType, short sPayloadLength, byte ySrcIdType, byte[] ylSrcId, byte yDestIdType, byte[] ylDestId, byte yPadLength, boolean bList, boolean bPrivateMessageKey, boolean bBroadcast, boolean bCompressed )
	{
		_yType = yType;
		_sPayloadLength = sPayloadLength;
		_bPrivateMessageKey = bPrivateMessageKey;
		_bList = bList;
		_bBroadcast = bBroadcast;
		_bCompressed = bCompressed;
		_ySrcIdType = ySrcIdType;
		_ylSrcId = ylSrcId;
		_yDestIdType = yDestIdType;
		_ylDestId = ylDestId;
		_yPadLength = yPadLength;
	}

	/** Constructor to initialise the various data fields from a received (decrypted) packet header in binary form
	 * @param yl binary data forming a packet header
	 */
	public SILCHeader( byte[] yl )
	{
		ByteBuffer yb = ByteBuffer.wrap( yl );
		_sPayloadLength = yb.getShort();
		byte yFlags = yb.get();
		_bPrivateMessageKey = ( yFlags & 0x01 ) != 0;
		_bList = ( yFlags & 0x02 ) != 0;
		_bBroadcast = ( yFlags & 0x04 ) != 0;
		_bCompressed = ( yFlags & 0x08 ) != 0;
		_yType = yb.get();
		_yPadLength = yb.get();
		assert( yb.get() == 0 ); // reserved
		_ylSrcId = new byte[ yb.get() & 0xFF ];
		_ylDestId = new byte[ yb.get() & 0xFF ];
		_ySrcIdType = yb.get();
		yb.get( _ylSrcId );
		_yDestIdType = yb.get();
		yb.get( _ylDestId );
	}
	
	/** Convert the data in this packet header to the form that needs to be sent over the network.
	 * @return byte list to be sent over network
	 */
	public byte[] toByteList()
	{
		assert( _ylSrcId != null );
		assert( _ylDestId != null );
		ByteBuffer yb = ByteBuffer.allocate( getEmptyLength() + _ylSrcId.length + _ylDestId.length );
		yb.putShort( _sPayloadLength );
 		// setup the flags byte
		yb.put( (byte)( ( _bPrivateMessageKey ? 0x01 : 0x00 ) | ( _bList ? 0x02 : 0x00 ) | ( _bBroadcast ? 0x04 : 0x00 ) | ( _bCompressed ? 0x08 : 0x00 ) ) );
		yb.put( _yType );
		yb.put( _yPadLength );
		yb.put( (byte)0 ); // reserved
		yb.put( (byte)_ylSrcId.length );
		yb.put( (byte)_ylDestId.length );
		yb.put( _ySrcIdType );
		yb.put( _ylSrcId );
		yb.put( _yDestIdType );
		yb.put( _ylDestId );
		return yb.array();
	}

	// doc inherited
	public String toString()
	{
		String strRet;
		strRet = "Payload length: " + Integer.toString( _sPayloadLength & 0xFFFF );
		strRet += "\nPrivate Message Key: " + Boolean.toString( _bPrivateMessageKey );
		strRet += "\nList: " + Boolean.toString( _bList );
		strRet += "\nBroadcast: " + Boolean.toString( _bBroadcast );
		strRet += "\nCompressed: " + Boolean.toString( _bCompressed );
		strRet += "\nType: " + SILCPacketType.toString( _yType );
		strRet += "\nPad length: " + Integer.toString( _yPadLength & 0xFF );
		strRet += "\nSource ID type: " + SILCIdType.toString( _ySrcIdType );
		strRet += "\nSource ID: " + Hex.toString( _ylSrcId );
		strRet += "\nDestination ID type: " + SILCIdType.toString( _yDestIdType );
		strRet += "\nDestination ID: " + Hex.toString( _ylDestId );
		return strRet;
	}

	/** Get the length of a packet header without any source / destination id fields
	 * @return packet header length
	 */
	public static short getEmptyLength()
	{
		return 10;
	}

	public byte getType()
	{
		return _yType;
	}
	
	public short getPayloadLength()
	{
		return _sPayloadLength;
	}
	
	public byte getPadLength()
	{
		return _yPadLength;
	}

	public byte getSourceIdType()
	{
		return _ySrcIdType;
	}

	public byte[] getSourceId()
	{
		return _ylSrcId;
	}

	public byte getDestIdType()
	{
		return _yDestIdType;
	}

	public byte[] getDestId()
	{
		return _ylDestId;
	}
}
