/*
 *	SILCKeyExchange2Payload.java		2002/11/13
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.silcprotocol;

import java.lang.String;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.Arrays;
import org.bouncycastle.crypto.*;
import org.bouncycastle.crypto.agreement.*;
import org.bouncycastle.crypto.digests.*;
import org.bouncycastle.crypto.generators.*;
import org.bouncycastle.crypto.params.*;
import org.bouncycastle.crypto.engines.*;
import org.bouncycastle.crypto.encodings.*;
import com.alistairphipps.util.Hex;
import com.alistairphipps.util.UBigInteger;

/** Class to implement a Key Exchange payload as received by client in a KEY_EXCHANGE_2 packet.
 * @author Alistair Phipps
 * @version 20021113
 */
public class SILCKeyExchange2Payload extends SILCKeyExchangePayload
{
	byte[] _ylDHKey;
	byte[] _ylHash;
	
	/** Constructor to set up the payload information from a bytelist - used to set up received KE_2 payload.
	 * @param yl List of bytes forming received payload
	 */
	public SILCKeyExchange2Payload( byte[] yl )
	{
		ByteBuffer yb = ByteBuffer.wrap( yl );
		short sPubKeyLength = yb.getShort();
		short sPubKeyType = yb.getShort();
		assert( sPubKeyType == SILCPubKeyType.SILC );	// we only support SILC style pub key for now - but simple to add support for other types
		byte[] ylPubKey = new byte[ sPubKeyLength & 0xFFFF ];
		yb.get( ylPubKey );
		_pubkey = new SILCPubKey( ylPubKey );
		short sPubDataLength = yb.getShort();
		_ylPubData = new byte[ sPubDataLength & 0xFFFF ];
		yb.get( _ylPubData );
		short sSigLength = yb.getShort();
		_ylSigData = new byte[ sSigLength & 0xFFFF ];
		yb.get( _ylSigData );
	}
	
	public boolean isValid( byte[] ylKES, DHPrivateKeyParameters dhpv_i, SILCPubKey pubkey_i, byte[] ylPubData_i ) throws InvalidCipherTextException
	{
		// is valid if calculated hash = signature decrypted with provided public key
		calculateDHKeyAndHash( ylKES, dhpv_i, pubkey_i, ylPubData_i );
		return Arrays.equals( calculateSig( _pubkey.toRSAKeyParameters(), _ylSigData ), _ylHash );
	}
	
	/** Calculate hash for KE_2: hash( KES payload | pub key | initiator's pub key | e | f | KEY ) where KEY is agreed D-H key
	 * @param ylKES KES Payload
	 * @param dhpv D-H private key generated for KE_1 packet
	 * @param pubkey_i Public key of initiator (SILC RSA key) - one we sent in KE_1 packet
	 * @param ylPubData_i D-H public key generated for KE_1 packet
	 */
	private void calculateDHKeyAndHash( byte[] ylKES, DHPrivateKeyParameters dhpv, SILCPubKey pubkey_i, byte[] ylPubData_i )
	{
		// calculate DH key from our KE1 private key and the public key received in the pubdata field of the KE2 packet
		DHBasicAgreement e = new DHBasicAgreement();
		e.init( dhpv );
		BigInteger key = e.calculateAgreement( new DHPublicKeyParameters( UBigInteger.fromByteArray( _ylPubData ), new DHParameters( s_biP, s_biG ) ) );
		_ylDHKey = UBigInteger.toByteArray( key );

		//System.out.println( "***** KEY *****: " + com.alistairphipps.util.Hex.toString( _ylDHKey ) );
		
		ByteBuffer yb = ByteBuffer.allocate( ylKES.length + _pubkey.toByteList().length + pubkey_i.toByteList().length + _ylPubData.length  + ylPubData_i.length + _ylDHKey.length );

		yb.put( ylKES ); // KES
		yb.put( _pubkey.toByteList() ); // pub key
		yb.put( pubkey_i.toByteList() ); // initiator's pub key
		yb.put( ylPubData_i ); // e
		yb.put( _ylPubData ); // f
		yb.put( _ylDHKey );	// key

		//System.out.println( "HASH DATA: " + com.alistairphipps.util.Hex.toString( yb.array() ) );
		Digest digest = new SHA1Digest();
		digest.update( yb.array(), 0, yb.array().length );
		_ylHash = new byte[ digest.getDigestSize() ];
		digest.doFinal( _ylHash, 0 );
		//System.out.println( "***** HASH *****: " + com.alistairphipps.util.Hex.toString( ylHash ) );
	}

	public byte[] getDHKey()
	{
		return _ylDHKey;
	}

	public byte[] getHash()
	{
		return _ylHash;
	}
}

