/*
 *	SILCPayload.java		2002/11/06
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.silcprotocol;

import java.nio.charset.Charset;
import java.lang.String;
import java.nio.ByteBuffer;

/** Class that SILC Payload classes must inherit - provides base interface and common functionality.
 * @author Alistair K Phipps
 * @version 20021108
 */
public abstract class SILCPayload
{
	/** charset used for encoding/decoding characters in packets - shared across all instances for efficiency */
	static protected Charset s_csUTF8 = Charset.forName( "UTF-8" );

	/** Convert variables in this payload into the form that needs to be sent over the network (but not encrypted).
	 * @return byte list to be sent over network
	 */
	public abstract byte[] toByteList();
	
	/** Return human-readable string representation of the payload (primarily for debugging).
	 * @return string representing state
	 */
	public abstract String toString();

	/** Return true if the payload actually contains a list of payloads.
	 * @return boolean saying if payload is actually list of payloads
	 */
	public abstract boolean isList();

	/** Helper method to read a Short giving a length, then that number of bytes from a ByteBuffer, treating those bytes as UTF-8 encoded characters and returning them in the form of a String.  Position afterwards is set to that after string.
	 * @param yb ByteBuffer with the bytes representing UTF-8 encoded characters
	 * @return String with the characters from the byte buffer
	 */
	protected String readShortAndString( ByteBuffer yb )
	{
		String str;
		int iLimit = yb.limit();
		// retrieve a short from the yb, and use it for the length of the string
		yb.limit( ( yb.getShort() & 0xFFFF ) + yb.position() ); // note getShort affects position
		// retrieve UTF8 string
		str = s_csUTF8.decode( yb ).toString();
		// reset limit
		yb.limit( iLimit );
		return str;
	}

	/** Helper method to convert a String into a UTF-8 encoded sequence of bytes return a bytebuffer with a short giving the length and the bytes.  XXX Could use some optimisation.
	 * @param str String to be encoded
	 * @return ByteBuffer with short and string
	 */
	protected ByteBuffer writeShortAndString( String str )
	{
		ByteBuffer yb = s_csUTF8.encode( str );
		ByteBuffer yb2 = ByteBuffer.allocate( yb.limit() + 2 );
		yb2.putShort( (short)yb.limit() );
		yb2.put( yb );
		yb2.position( 0 );	// reset to start
		return yb2;
	}

	/** Helper method to read an Int giving a length, then that number of bytes from a ByteBuffer, treating those bytes as UTF-8 encoded characters and returning them in the form of a String.  FIXME: Note that a String has a max length of an int, and this tries to read up to an unsigned int number of characters.
	 * @param yb ByteBuffer with the bytes representing UTF-8 encoded characters.  Position afterwards is set to that after string.
	 * @return String with the characters from the byte buffer
	 */
	protected String readIntAndString( ByteBuffer yb )
	{
		String str;
		int iLimit = yb.limit();
		// retrieve an int from the yb, and use it for the length of the string
		yb.limit( ( yb.getInt() & 0xFFFFFFFF ) + yb.position() ); // note getShort affects position
		// retrieve UTF8 string
		str = s_csUTF8.decode( yb ).toString();
		// reset limit
		yb.limit( iLimit );
		return str;
	}

	/** Helper method to convert a String into a UTF-8 encoded sequence of bytes return a bytebuffer with an int giving the length and the bytes.  XXX Could use some optimisation.  FIXME: Note that a String has a max length of an int, and this tries to write up to an unsigned int number of characters.
	 * @param str String to be encoded
	 * @return ByteBuffer with int and string
	 */
	protected ByteBuffer writeIntAndString( String str )
	{
		ByteBuffer yb = s_csUTF8.encode( str );
		ByteBuffer yb2 = ByteBuffer.allocate( yb.limit() + 4 );
		yb2.putInt( (int)yb.limit() );
		yb2.put( yb );
		yb2.position( 0 );	// reset to start
		return yb2;
	}

}
