#
#  srt.m4
#
#  Author: Pekka Riikonen <priikone@silcnet.org>
#
#  Copyright (C) 2007 - 2008 Pekka Riikonen
#
#  The contents of this file are subject to one of the Licenses specified 
#  in the COPYING file;  You may not use this file except in compliance 
#  with the License.
#
#  The software distributed under the License is distributed on an "AS IS"
#  basis, in the hope that it will be useful, but WITHOUT WARRANTY OF ANY
#  KIND, either expressed or implied.  See the COPYING file for more
#  information.
#

# Function to check if system has SMP kernel.
#
# Usage: SILC_SYSTEM_IS_SMP([ACTION-IF-FOUND] [, ACTION-IF-NOT-FOUND]
#                           [, ACTION-IF-NOT-DETECTED])
#
# The ACTION-IF-NOT-DETECTED is called if we could not detect whether or
# not the system is SMP.
#
# x_is_smp variable is set to true or false as a result for calling this
# function.  Caller may use the variable to check for the result in the
# code.
#
AC_DEFUN([SILC_SYSTEM_IS_SMP],
[
  AC_MSG_CHECKING(whether system has SMP kernel)
  x_is_smp=false

  case "$target" in
    *-*-linux*)
      # Take data from Linux /proc
      if test -f /proc/stat; then
        cpucount=`grep "^cpu" /proc/stat -c 2> /dev/null`
        if test $cpucount -gt 1; then
          AC_DEFINE([SILC_SMP], [], [SILC_SMP])
          AC_MSG_RESULT(yes)
          x_is_smp=true
          ifelse([$1], , :, [$1])
        else
          AC_MSG_RESULT(no)
          ifelse([$2], , :, [$2])
        fi
      else
        AC_MSG_RESULT(no)
        ifelse([$2], , :, [$2])
      fi
      ;;

    *-*-*bsd*)
      # BSDs can have SMP info in sysctl 'kern.smp.cpus' variable
      sysctl="sysctl -n kern.smp.cpus"
      cpucount=`(/sbin/$sysctl 2> /dev/null || \
                 /usr/sbin/$sysctl 2> /dev/null || echo -n 0)`
      if test $cpucount -gt 1; then
        AC_DEFINE([SILC_SMP], [], [SILC_SMP])
        AC_MSG_RESULT(yes)
         x_is_smp=true
        ifelse([$1], , :, [$1])
      else
        AC_MSG_RESULT(no)
        ifelse([$2], , :, [$2])
      fi
      ;;

    *)
      AC_MSG_RESULT(cannot detect on this system)
      ifelse([$3], , :, [$3])
      ;;
  esac
])

# Function to check for CPU feature flags.
#
# Usage: SILC_CPU_FLAG(flag, enable, [, ACTION-IF-FOUND]
#                      [, ACTION-IF-NOT-FOUND])
#
# x_have_cpu_<flag> variable is set to true or false value as a result for
# calling this function for the <flag>.  Caller may use the variable to
# check the result in the code.  If <enable> is true this will add
# AC_ARG_ENABLE option for the <flag>.
#
AC_DEFUN([SILC_CPU_FLAG],
[
  AC_MSG_CHECKING(whether CPU supports $1)
  x_have_cpu_$1=false

  if test x$2 = xtrue; then
    AC_ARG_ENABLE($1,
      [  --enable-$1 ],
      [
        AC_MSG_RESULT(yes - enabled by --enable-$1)
        x_have_cpu_$1=true
      ])
  fi

  if test x$x_have_cpu_$1 = xfalse; then
    case "$target" in
      *-*-linux*)
        # Take data from Linux /proc
        if test -f /proc/cpuinfo; then
          cpuflags=`grep "^flags.*$1 " /proc/cpuinfo 2> /dev/null`
          if test $? != 0; then
            AC_MSG_RESULT(no)
            x_have_cpu_$1=false
            ifelse([$4], , :, [$4])
          else
            AC_MSG_RESULT(yes)
            x_have_cpu_$1=true
            ifelse([$3], , :, [$3])
          fi
        else
          AC_MSG_RESULT(no)
          x_have_cpu_$1=false
          ifelse([$4], , :, [$4])
        fi
        ;;

      *-*-*bsd*)
        # BSDs have some flags in sysctl 'machdep' variable
        cpuflags=`/sbin/sysctl machdep 2> /dev/null | grep "\.$1.*.1"`
        if test $? != 0; then
          AC_MSG_RESULT(no)
           x_have_cpu_$1=false
          ifelse([$4], , :, [$4])
        else
          AC_MSG_RESULT(yes)
          x_have_cpu_$1=true
          ifelse([$3], , :, [$3])
        fi
        ;;

      *)
        AC_MSG_RESULT(no - cannot detect on this system)
        x_have_cpu_$1=false
        ifelse([$4], , :, [$4])
        ;;
    esac
  fi
])

# Function to check if compiler option works with the compiler.  If you
# want the option added to some other than CFLAGS variable use the
# SILC_ADD_CC_FLAGS which supports to specifiable destination variable.
#
# Usage: SILC_ADD_CFLAGS(FLAGS, [ACTION-IF-FAILED])
#
AC_DEFUN([SILC_ADD_CFLAGS],
[ tmp_CFLAGS="$CFLAGS"
  CFLAGS="$CFLAGS $1"
  AC_MSG_CHECKING(whether $CC accepts $1 flag)
  AC_TRY_LINK([], [], [AC_MSG_RESULT(yes)], [AC_MSG_RESULT(no)
				       CFLAGS="$tmp_CFLAGS"
				       $2])
  unset tmp_CFLAGS
])

# Function to check if compiler option works with the compiler,
# destination variable specifiable
#
# Usage: SILC_ADD_CC_FLAGS(VAR, FLAGS, [ACTION-IF-FAILED])
#
AC_DEFUN([SILC_ADD_CC_FLAGS],
[ tmp_CFLAGS="$1_CFLAGS"
  $1_CFLAGS="${$1_CFLAGS} $2"
  AC_MSG_CHECKING(whether $CC accepts $2 flag)
  AC_TRY_LINK([], [], [AC_MSG_RESULT(yes)], [AC_MSG_RESULT(no)
				       $1_CFLAGS="$tmp_CFLAGS"
				       $3])
  unset tmp_CFLAGS
])
