/*

  silcdll.h

  Author: Pekka Riikonen <priikone@silcnet.org>

  Copyright (C) 2007 - 2008 Pekka Riikonen

  The contents of this file are subject to one of the Licenses specified 
  in the COPYING file;  You may not use this file except in compliance 
  with the License.

  The software distributed under the License is distributed on an "AS IS"
  basis, in the hope that it will be useful, but WITHOUT WARRANTY OF ANY
  KIND, either expressed or implied.  See the COPYING file for more
  information.

*/

/****h* silcutil/Shared Object Interface
 *
 * DESCRIPTION
 *
 * Platform independent iterface for loading and using shared objects and
 * dynamically linked libraries (DLLs).
 *
 * EXAMPLE
 *
 * SilcDll dll;
 * SilcFuncCb function;
 *
 * dll = silc_dll_load("/path/to/my.so");
 * function = silc_dll_getsym(dll, "my_function");
 *
 * // Call the funtion
 * function(arg1, arg2);
 *
 * silc_dll_close(dll);
 *
 ***/

#ifndef SILCDLL_H
#define SILCDLL_H

/****s* silcutil/SilcDll
 *
 * NAME
 *
 *    typedef void *SilcDll;
 *
 * DESCRIPTION
 *
 *    This context represents the shared object and it is allocated by
 *    silc_dll_load and is destroyed with silc_dll_close functions.
 *    The context is given as argument to all funtions in this interface.
 *
 ***/
#ifdef SILC_UNIX
typedef void *SilcDll;
#elif SILC_WIN32
typedef HMODULE SilcDll;
#else
typedef void *SilcDll;
#endif /* SILC_UNIX */

/****f* silcutil/silc_dll_load
 *
 * SYNOPSIS
 *
 *    SilcDll silc_dll_load(const char *object_path);
 *
 * DESCRIPTION
 *
 *    Load shared object or DLL indicated by the `object_path'.  The path
 *    must include the absolute path to the object and the object name.
 *    Returns the SilcDll context or NULL on error.  The actual error
 *    message may be available by calling silc_dll_error function.  Symbols
 *    may be retrieved from the returned context by calling silc_dll_getsym.
 *
 ***/
SilcDll silc_dll_load(const char *object_path);

/****f* silcutil/silc_dll_close
 *
 * SYNOPSIS
 *
 *    void silc_dll_close(SilcDll dll);
 *
 * DESCRIPTION
 *
 *    Closes the shared object indicated by `dll'.  Any symbol retrieved
 *    from the `dll' with silc_dll_getsym will become invalid and cannot
 *    be used anymore.
 *
 ***/
void silc_dll_close(SilcDll dll);

/****f* silcutil/silc_dll_getsym
 *
 * SYNOPSIS
 *
 *    void *silc_dll_getsym(SilcDll dll, const char *symbol);
 *
 * DESCRIPTION
 *
 *    Returns the memory address of the symbol indicated by `symbol' from
 *    the shared object indicated by `dll'.  If such symbol does not exist
 *    this returns NULL.
 *
 ***/
void *silc_dll_getsym(SilcDll dll, const char *symbol);

/****f* silcutil/silc_dll_error
 *
 * SYNOPSIS
 *
 *    const char *silc_dll_error(SilcDll dll);
 *
 * DESCRIPTION
 *
 *    This routine may return error string after an error has occured with
 *    the shared object indicated by `dll'.  If error string is not available
 *    this will return NULL.
 *
 ***/
const char *silc_dll_error(SilcDll dll);

#endif /* SILCDLL_H */
