/*

  silcrand.h

  Author: Pekka Riikonen <priikone@silcnet.org>

  Copyright (C) 2008 Pekka Riikonen

  The contents of this file are subject to one of the Licenses specified 
  in the COPYING file;  You may not use this file except in compliance 
  with the License.

  The software distributed under the License is distributed on an "AS IS"
  basis, in the hope that it will be useful, but WITHOUT WARRANTY OF ANY
  KIND, either expressed or implied.  See the COPYING file for more
  information.

*/

/****h* silcutil/Random Number Interface
 *
 * DESCRIPTION
 *
 * Interface for generating pseudo-random numbers.  This random number
 * generator must not be used in applications needing cryptographically
 * strong random numbers.  For that purpose the SILC Crypto Toolkit should
 * be used.
 *
 * The implementation uses Mersenne Twister random number generator.
 *
 * The SILC Rand API is thread-safe.  Each thread can use the API at the
 * same time without affecting the random state of other threads.
 *
 * EXAMPLE
 *
 * silc_rand_seed(seed);
 * printf("Random number: %lu", silc_rand());
 *
 ***/

#ifndef SILCRAND_H
#define SILCRAND_H

/****f* silcutil/silc_rand_seed
 *
 * SYNOPSIS
 *
 *    void silc_rand_seed(SilcUInt32 seed);
 *
 * DESCRIPTION
 *
 *    Seed the random number generator with the value `seed'.  This should
 *    be called before starting to retrieve random numbers and the `seed'
 *    itself should be random or non-determinable.  It is also wise to reseed
 *    the random number generator periodically if random numbers are used
 *    a lot.
 *
 ***/
void silc_rand_seed(SilcUInt32 seed);

/****f* silcutil/silc_rand
 *
 * SYNOPSIS
 *
 *    SilcUInt32 silc_rand(void);
 *
 * DESCRIPTION
 *
 *    Returns 32-bit random number.
 *
 ***/
SilcUInt32 silc_rand(void);

/****f* silcutil/silc_rand64
 *
 * SYNOPSIS
 *
 *    SilcUInt64 silc_rand64(void)
 *
 * DESCRIPTION
 *
 *    Returns 64-bit random number.
 *
 ***/
SilcUInt64 silc_rand64(void);

#endif /* SILCRAND_H */
