/*
    encr.c -- everything that deals with encryption
    Copyright (C) 1998,99 Ivo Timmermans <zarq@iname.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/socket.h>

#ifdef HAVE_GMP_H
# include <gmp.h>
#else
# ifdef HAVE_GMP2_GMP_H
#  include <gmp2/gmp.h>
# endif
#endif

#include <xalloc.h>

#include <cipher.h>

#include "conf.h"
#include "encr.h"
#include "net.h"

#define ENCR_GENERATOR "0xd"
#define ENCR_PRIME "0x7fffffffffffffffffffffffffffffff" /* Mersenne :) */

char text_key[1000];
char *my_public_key_base36;
int key_inited = 0, encryption_keylen;
mpz_t my_private_key, my_public_key, generator, shared_prime;

static char* mypassphrase;
static int mypassphraselen;

int char_hex_to_bin(int c)
{
  if(isdigit(c))
    return c - '0';
  else
    return tolower(c) - 'a' + 10;
}

int str_hex_to_bin(unsigned char *bin, unsigned char *hex)
{
  int i = 0, j = 0, l = strlen(hex);

  if(l&1)
    {
      i = j = 1;
      bin[0] = char_hex_to_bin(hex[0]);
    }
  for(; i < l; i+=2, j++)
    bin[j] = (char_hex_to_bin(hex[i]) << 4) + char_hex_to_bin(hex[i+1]);

  return j&1?j+1:j;
}

int read_passphrase(char *which, char **out)
{
  FILE *f;
  config_t const *cfg;
  char *filename;
  int size;
  extern char *confbase;
  char *pp;

  if((cfg = get_config_val(passphrasesdir)) == NULL)
    {
      filename = xmalloc(strlen(confbase)+13+strlen(which));
      sprintf(filename, "%spassphrases/%s", confbase, which);
    }
  else
    {
      filename = xmalloc(strlen(cfg->data.ptr)+2+strlen(which));
      sprintf(filename, "%s/%s", (char*)cfg->data.ptr, which);
    }

  if((f = fopen(filename, "rb")) == NULL)
    {
      syslog(LOG_ERR, "Could not open %s: %m", filename);
      return -1;
    }

  fscanf(f, "%d ", &size);
  size >>= 2; /* nibbles->bits */
  pp = xmalloc(size+2);
  fgets(pp, size+1, f);
  fclose(f);

  *out = xmalloc(size);
  return str_hex_to_bin(*out, pp);
}

int read_my_passphrase(void)
{
  if((mypassphraselen = read_passphrase("local", &mypassphrase)) < 0)
    return -1;

  return 0;
}

int generate_private_key(void)
{
  FILE *f;
  int i;
  char *s;

  if(debug_lvl > 0)
    syslog(LOG_NOTICE, "Generating %d bits keys.", PRIVATE_KEY_BITS);

  if((f = fopen("/dev/urandom", "r")) == NULL)
    {
      syslog(LOG_ERR, "Opening /dev/urandom failed: %m");
      return -1;
    }

  s = xmalloc((2 * PRIVATE_KEY_LENGTH) + 1);

  for(i = 0; i < PRIVATE_KEY_LENGTH; i++)
    sprintf(&s[i << 1], "%02x", fgetc(f));

  s[2 * PRIVATE_KEY_LENGTH] = '\0';

  mpz_init_set_str(my_private_key, s, 16);

  return 0;
}

void calculate_public_key(void)
{
  mpz_powm(my_public_key, generator, my_private_key, shared_prime);
  my_public_key_base36 = mpz_get_str(NULL, 36, my_public_key);
}

unsigned char static_key[] = { 0x9c, 0xbf, 0x36, 0xa9, 0xce, 0x20, 0x1b, 0x8b, 0x67, 0x56, 0x21, 0x5d, 0x27, 0x1b, 0xd8, 0x7a };

int security_init(void)
{
  mpz_init(my_public_key);
  mpz_init_set_str(shared_prime, ENCR_PRIME, 0);
  mpz_init_set_str(generator, ENCR_GENERATOR, 0);

  if(read_my_passphrase() < 0)
    return -1;
  if(generate_private_key() < 0)
    return -1;

  if(cypher_init(CYPHER_BLOWFISH) < 0)
    return -1;

  calculate_public_key();

  return 0;
}

void set_shared_key(char *almost_key)
{
  char *tmp;
  int len;
  mpz_t ak, our_shared_key;

  mpz_init_set_str(ak, almost_key, 36);
  mpz_init(our_shared_key);
  mpz_powm(our_shared_key, ak, my_private_key, shared_prime);

  tmp = mpz_get_str(NULL, 16, our_shared_key);
  len = str_hex_to_bin(text_key, tmp);

  cypher_set_key(&encryption_key, len, &text_key[0]);
  key_inited = 1;
  encryption_keylen = len;

  if(debug_lvl > 2)
    syslog(LOG_INFO, "Encryption key set to %s", tmp);

  free(tmp);
  mpz_clear(ak);
  mpz_clear(our_shared_key);
}


void encrypt_passphrase(passphrase_t *pp)
{
  char key[1000];
  char tmp[1000];
  int len;
  BF_KEY bf_key;
  
  mpz_get_str(&tmp[0], 16, my_public_key);
  len = str_hex_to_bin(key, tmp);

  cypher_set_key(&bf_key, len, &key[0]);

  low_crypt_key(mypassphrase, pp->phrase, &bf_key, mypassphraselen, BF_ENCRYPT);
  pp->len = ((mypassphraselen - 1) | 7) + 5;

  if(key_inited)
    cypher_set_key(&encryption_key, encryption_keylen, &text_key[0]);
}

int verify_passphrase(conn_list_t *cl, unsigned char *his_pubkey)
{
  char key[1000];
  char tmp[1000];
  int len;
  mpz_t pk;
  unsigned char *out;
  BF_KEY bf_key;
  char which[sizeof("123.123.123.123")+1];
  char *meuk;

  mpz_init_set_str(pk, his_pubkey, 36);
  mpz_get_str(&tmp[0], 16, pk);
  len = str_hex_to_bin(key, tmp);
  out = xmalloc(cl->pp->len+3);

  cypher_set_key(&bf_key, len, &key[0]);
  low_crypt_key(cl->pp->phrase, out, &bf_key, cl->pp->len, BF_DECRYPT);
  if(key_inited)
    cypher_set_key(&encryption_key, encryption_keylen, &text_key[0]);

  sprintf(&which[0], IP_ADDR_S, IP_ADDR_V(cl->vpn_ip));
  if((len = read_passphrase(which, &meuk)) < 0)
    return -1;

  if(memcmp(meuk, out, len))
    return -1;

  return 0;
}

char *powerize(char *k1, char *k2)
{
  mpz_t tmp1, tmp2, res;
  char *r;

  mpz_init_set_str(tmp1, k1, 36);
  mpz_init_set_str(tmp2, k2, 36);
  mpz_init(res);
  mpz_powm(res, tmp1, tmp2, shared_prime);

  r = mpz_get_str(NULL, 36, res);

  mpz_clear(res);
  mpz_clear(tmp1);
  mpz_clear(tmp2);

  return r;
}
