/*
    net.c -- most of the network code
    Copyright (C) 1998,99 Ivo Timmermans <zarq@iname.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/signal.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>
#include <syslog.h>
#include <unistd.h>

#include <cipher.h>
#include <xalloc.h>

#include "conf.h"
#include "encr.h"
#include "net.h"
#include "protocol.h"

int tap_fd = -1;

int total_tap_in = 0;
int total_tap_out = 0;
int total_socket_in = 0;
int total_socket_out = 0;

/* The global list of existing connections */
conn_list_t *conn_list = NULL;
conn_list_t *myself = NULL;

/*
  look for a connection associated with the given vpn ip,
  return its connection structure
*/
const conn_list_t *lookup_conn(ip_t ip)
{
  conn_list_t *p = conn_list;

  for(p = conn_list; p != NULL; p = p->next)
    if((ip & p->vpn_mask) == (p->vpn_ip & p->vpn_mask))
      return p;

  return NULL;
}

/*
  free a conn_list_t element and all its pointers
*/
void free_conn_element(conn_list_t *p)
{
  if(p->hostname)
    free(p->hostname);
  if(p->pp)
    free(p->pp);
  free(p);
}

/*
  remove all marked connections
*/
void prune_conn_list(void)
{
  conn_list_t *p, *q = NULL;

  for(p = conn_list; p != NULL; )
    {
      if(p->status.remove)
	{
	  if(q)
	    {
	      q->next = p->next;
	      free_conn_element(p);
	      p = q->next;
	    }
	  else
	    {
	      conn_list = q = p->next;
	      free_conn_element(p);
	      p = q;
	    }
	}
      else
	p = p->next;
    }
}

/*
  creates new conn_list element, and initializes it
*/
conn_list_t *new_conn_list(void)
{
  conn_list_t *p = xmalloc(sizeof(conn_list_t));

  memset(p, '\0', sizeof(conn_list_t));
  return p;
}

void destroy_conn_list(void)
{
  conn_list_t *p, *q;

  for(p = conn_list; p != NULL; )
    {
      if(p->hostname)
	free(p->hostname);
      if(p->pp)
	free(p->pp);
      q = p;
      p = p->next;
      free(q);
    }

  conn_list = NULL;
}

/*
  strip off the MAC adresses of an ethernet frame
*/
void strip_mac_addresses(vpn_packet_t *p)
{
  unsigned char tmp[MAXSIZE];

  memcpy(tmp, p->data, p->len);
  p->len -= 12;
  memcpy(p->data, &tmp[12], p->len);
}

/*
  reassemble MAC addresses
*/
void add_mac_addresses(vpn_packet_t *p)
{
  unsigned char tmp[MAXSIZE];

  memcpy(&tmp[12], p->data, p->len);
  p->len += 12;
  tmp[0] = tmp[6] = 0xfe;
  tmp[1] = tmp[7] = 0xfd;
  *((ip_t*)(&tmp[2])) = *((ip_t*)(&tmp[26]));
  *((ip_t*)(&tmp[8])) = *((ip_t*)(&tmp[30]));
  memcpy(p->data, &tmp[0], p->len);
}

/*
  send a packet to the given vpn ip.
*/
int send_packet(ip_t to, real_packet_t *packet)
{
  int r;
  conn_list_t const *cl;

  if((cl = lookup_conn(to)) == NULL)
    {
      syslog(LOG_NOTICE, "trying to look up " IP_ADDR_S " in connection list failed.",
	     IP_ADDR_V(to));
      return -1;
    }

  if(!cl->status.active)
    {
      if(debug_lvl > 1)
	syslog(LOG_INFO, IP_ADDR_S " is not ready, dropping packet.", IP_ADDR_V(cl->vpn_ip));
      return 0; /* We don't want to mess up, do we? */
    }

  if((r = send(cl->socket, packet, packet->len, 0)) < 0)
    {
      syslog(LOG_ERR, "Error sending data: %m");
      return -1;
    }

  total_socket_out += r;

  return r;
}

int send_broadcast(conn_list_t *cl, real_packet_t *packet)
{
  conn_list_t *p;

  for(p = cl; p != NULL; p = p->next)
    if(send_packet(p->real_ip, packet) < 0)
      {
	syslog(LOG_ERR, "%d: Could not send a broadcast packet to %08lx (%08lx): %m",
	       __LINE__,
	       p->vpn_ip, p->real_ip);
	break; /* FIXME: should retry later, and send a ping over the metaconnection. */
      }

  return 0;
}

char *hostlookup(unsigned long addr)
{
  char *name;
  struct hostent *host = NULL;
  struct in_addr in;

  in.s_addr = addr;

  host = gethostbyaddr((char *)&in, sizeof(in), AF_INET);

  if(host)
    {
      name = xmalloc(strlen(host->h_name)+20);
      sprintf(name, "%s (%s)", host->h_name, inet_ntoa(in));
    }
  else
    {
      name = xmalloc(20);
      sprintf(name, "%s", inet_ntoa(in));
    }

  return name;
}

/*
  Turn a string into an IP addy with netmask
  return NULL on failure
*/
ip_mask_t *
strtoip(const char *str)
{
  ip_mask_t *ip;
  int masker;
  char *q, *p;
  struct hostent *h;

  p = str;
  if((q = strchr(p, '/')))
    {
      *q = '\0';
      q++; /* q now points to netmask part, or NULL if no mask */
    }

  if(!(h = gethostbyname(p)))
    {
      fprintf(stderr, "Error looking up `%s': %s\n", p, sys_errlist[h_errno]);
      return NULL;
    }

  masker = 0;
  if(q)
    {
      masker = strtol(q, &p, 10);
      if(q == p || (*p))
	return NULL;
    }

  ip = xmalloc(sizeof(ip_mask_t));
  ip->ip = ntohl(*((ip_t*)(h->h_addr_list[0])));
  ip->mask = ~((1 << (32 - masker)) - 1);

  return ip;
}

int setup_tap_fd(void)
{
  int nfd;
  const char *tapfname;
  config_t const *cfg;
  
  if((cfg = get_config_val(tapdevice)) == NULL)
    tapfname = "/dev/tap0";
  else
    tapfname = cfg->data.ptr;

  if((nfd = open(tapfname, O_RDWR | O_NONBLOCK)) < 0)
    {
      syslog(LOG_ERR, "Could not open %s: %m", tapfname);
      return -1;
    }

  tap_fd = nfd;
  return 0;
}

int setup_listen_meta_socket(int port)
{
  int nfd, flags;
  struct sockaddr_in a;
  const int one = 1;

  if((nfd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
    {
      syslog(LOG_ERR, "Creating metasocket failed: %m");
      return -1;
    }

  if(setsockopt(nfd, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one)))
    {
      syslog(LOG_ERR, "setsockopt: %m");
      return -1;
    }

  flags = fcntl(nfd, F_GETFL);
  if(fcntl(nfd, F_SETFL, flags | O_NONBLOCK) < 0)
    {
      syslog(LOG_ERR, "fcntl: %m");
      return -1;
    }

  memset(&a, 0, sizeof(a));
  a.sin_family = AF_INET;
  a.sin_port = htons(port);
  a.sin_addr.s_addr = htonl(INADDR_ANY);

  if(bind(nfd, (struct sockaddr *)&a, sizeof(struct sockaddr)))
    {
      syslog(LOG_ERR, "bind: %m");
      return -1;
    }

  if(listen(nfd, 3))
    {
      syslog(LOG_ERR, "listen: %m");
      return -1;
    }

  return nfd;
}

int setup_vpn_in_socket(int port)
{
  int nfd, flags;
  struct sockaddr_in a;
  const int one = 1;

  if((nfd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0)
    {
      syslog(LOG_ERR, "Creating socket failed: %m");
      return -1;
    }

  if(setsockopt(nfd, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one)))
    {
      syslog(LOG_ERR, "setsockopt: %m");
      return -1;
    }

  flags = fcntl(nfd, F_GETFL);
  if(fcntl(nfd, F_SETFL, flags | O_NONBLOCK) < 0)
    {
      syslog(LOG_ERR, "fcntl: %m");
      return -1;
    }

  memset(&a, 0, sizeof(a));
  a.sin_family = AF_INET;
  a.sin_port = htons(port);
  a.sin_addr.s_addr = htonl(INADDR_ANY);

  if(bind(nfd, (struct sockaddr *)&a, sizeof(struct sockaddr)))
    {
      syslog(LOG_ERR, "bind: %m");
      return -1;
    }

  return nfd;
}

int setup_outgoing_meta_socket(conn_list_t *cl)
{
  int flags;
  struct sockaddr_in a;
  config_t const *cfg;

  if((cfg = get_config_val(upstreamport)) == NULL)
    cl->port = 24;
  else
    cl->port = cfg->data.val;

  cl->meta_socket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
  if(cl->meta_socket == -1)
    {
      syslog(LOG_ERR, "Creating socket failed: %m");
      return -1;
    }

  a.sin_family = AF_INET;
  a.sin_port = htons(cl->port);
  a.sin_addr.s_addr = htonl(cl->real_ip);

  if(connect(cl->meta_socket, (struct sockaddr *)&a, sizeof(a)) == -1)
    {
      syslog(LOG_ERR, IP_ADDR_S ":%d: %m", IP_ADDR_V(cl->real_ip), cl->port);
      return -1;
    }

  flags = fcntl(cl->meta_socket, F_GETFL);
  if(fcntl(cl->meta_socket, F_SETFL, flags | O_NONBLOCK) < 0)
    {
      syslog(LOG_ERR, "fcntl: %m");
      return -1;
    }

  cl->hostname = hostlookup(htonl(cl->real_ip));

  syslog(LOG_INFO, "Connected to %s:%hd" , cl->hostname, cl->port);

  return 0;
}

int setup_outgoing_connection(ip_t ip)
{
  conn_list_t *ncn;

  ncn = new_conn_list();
  ncn->real_ip = ip;

  if(setup_outgoing_meta_socket(ncn) < 0)
    {
      syslog(LOG_ERR, "Could not set up a meta connection.");
      free_conn_element(ncn);
      return -1;
    }

  ncn->status.meta = 1;
  ncn->status.outgoing = 1;
  ncn->next = conn_list;
  conn_list = ncn;

  return 0;
}

int setup_myself(void)
{
  config_t const *cfg;

  myself = new_conn_list();

  if(!(cfg = get_config_val(myvpnip)))
    {
      syslog(LOG_ERR, "No value for my VPN IP given");
      return -1;
    }

  myself->vpn_ip = cfg->data.ip->ip;
  myself->vpn_mask = cfg->data.ip->mask;

  if(!(cfg = get_config_val(listenport)))
    myself->port = 24;
  else
    myself->port = cfg->data.val;

  if((myself->meta_socket = setup_listen_meta_socket(myself->port)) < 0)
    {
      syslog(LOG_ERR, "Unable to set up a listening socket");
      return -1;
    }

  if((myself->socket = setup_vpn_in_socket(myself->port)) < 0)
    {
      syslog(LOG_ERR, "Unable to set up an incoming vpn data socket");
      close(myself->meta_socket);
      return -1;
    }

  myself->status.active = 1;

  syslog(LOG_NOTICE, "Ready: listening on port %d.", myself->port);

  return 0;
}

int setup_network_connections(void)
{
  config_t const *cfg;

  if((cfg = get_config_val(pingtimeout)) == NULL)
    timeout = 10;
  else
    timeout = cfg->data.val;

  if(setup_tap_fd() < 0)
    return -1;

  if(setup_myself() < 0)
    return -1;

  if((cfg = get_config_val(upstreamip)) == NULL)
    /* No upstream IP given, we're listen only. */
    return 0;

  if(setup_outgoing_connection(cfg->data.ip->ip))
    return -1;

  return 0;
}

RETSIGTYPE
sigalrm_handler(int a)
{
  config_t const *cfg;

  if((cfg = get_config_val(upstreamip)) == NULL)
    {
      syslog(LOG_ERR, "Serious programming error. line %d of %s.", __LINE__, __FILE__);
      return;
    }

  if(!setup_outgoing_connection(cfg->data.ip->ip))
    signal(SIGALRM, SIG_IGN);
  else
    {
      signal(SIGALRM, sigalrm_handler);
      alarm(5);
      syslog(LOG_ERR, "Still failed to connect to other. Will retry in 5 seconds.");
    }
}

void close_network_connections(void)
{
  conn_list_t *p;

  for(p = conn_list; p != NULL; p = p->next)
    {
      send_termreq(p);
      
      if(p->status.active)
	{
	  shutdown(p->socket, 0); /* No more receptions */
	  close(p->socket);
	}
      if(p->status.meta)
	{
	  shutdown(p->socket, 0); /* No more receptions */
          close(p->meta_socket);
        }
    }

  if(myself)
    if(myself->status.active)
      {
	close(myself->meta_socket);
	close(myself->socket);
      }

  close(tap_fd);
  destroy_conn_list();

  syslog(LOG_NOTICE, "Terminating.");
  return;
}

int setup_vpn_connection(conn_list_t *cl)
{
  int nfd, flags;
  struct sockaddr_in a;

  nfd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if(nfd == -1)
    {
      syslog(LOG_ERR, "%d: Creating socket failed: %m", __LINE__);
      return -1;
    }

  a.sin_family = AF_INET;
  a.sin_port = htons(cl->port);
  a.sin_addr.s_addr = htonl(cl->real_ip);

  if(connect(nfd, (struct sockaddr *)&a, sizeof(a)) == -1)
    {
      syslog(LOG_ERR, "%d: %08lx:%d: %m", __LINE__, ntohs(cl->real_ip),
	     cl->port);
      return -1;
    }

  flags = fcntl(nfd, F_GETFL);
  if(fcntl(nfd, F_SETFL, flags | O_NONBLOCK) < 0)
    {
      syslog(LOG_ERR, "%d: fcntl: %m", __LINE__);
      return -1;
    }

  cl->socket = nfd;

  return 0;
}

conn_list_t *create_new_connection(int sfd)
{
  conn_list_t *p;
  struct sockaddr_in ci;
  int len = sizeof(ci);

  p = new_conn_list();

  if(getpeername(sfd, &ci, &len) < 0)
    {
      syslog(LOG_ERR, "Error: getpeername: %m");
      return NULL;
    }

  p->hostname = hostlookup(ci.sin_addr.s_addr);
  p->real_ip = ntohl(ci.sin_addr.s_addr);
  p->meta_socket = sfd;
  p->status.meta = 1;

  syslog(LOG_NOTICE, "Connection from %s:%d", p->hostname, htons(ci.sin_port));

  if(send_basic_info(p) < 0)
    {
      free(p);
      return NULL;
    }

  return p;
}

void build_fdset(fd_set *fs)
{
  conn_list_t *p;

  FD_ZERO(fs);

  for(p = conn_list; p != NULL; p = p->next)
    {
      if(p->status.meta)
	FD_SET(p->meta_socket, fs);
      if(p->status.active)
	FD_SET(p->socket, fs);
    }

  FD_SET(myself->meta_socket, fs);
  FD_SET(myself->socket, fs);
  FD_SET(tap_fd, fs);
}

int handle_incoming_vpn_data(conn_list_t *cl)
{
  real_packet_t rp;
  vpn_packet_t vp;
  int lenin;
  int x, l = sizeof(x);

  if(getsockopt(cl->socket, SOL_SOCKET, SO_ERROR, &x, &l) < 0)
    {
      syslog(LOG_ERR, "%d: getsockopt failed: %m", __LINE__);
      return -1;
    }
  if(x)
    {
      syslog(LOG_ERR, "%d: Socket error: %s", __LINE__, sys_errlist[x]);
      return -1;
    }

  rp.len = -1;
  lenin = recvfrom(cl->socket, &rp, MTU, 0, NULL, NULL);
  if(lenin <= 0) /* Waarschijnlijk is de andere kant er mee gekapt */
    {
      syslog(LOG_ERR, "%d: Receive: %m.\n", __LINE__);
      return -1;
    }
  total_socket_in += lenin;
  if(rp.len >= 0)
    {
      do_decrypt(&rp, &vp);
      add_mac_addresses(&vp);
      if((lenin = write(tap_fd, &vp, vp.len + 2)) < 0)
	syslog(LOG_ERR, "%d: write: %m", __LINE__);
      else
	total_tap_out += lenin;
    }

  return 0;
}

void terminate_connection(conn_list_t *cl)
{
  if(cl->status.remove)
    return;

  if(debug_lvl > 0)
    syslog(LOG_NOTICE, "Closing connection with %s.", cl->hostname);

  if(cl->status.timeout)
    send_timeout(cl);
  else if(!cl->status.termreq)
    send_termreq(cl);

  close(cl->socket);
  if(cl->status.meta)
    close(cl->meta_socket);

  if(cl->status.outgoing)
    {
      alarm(5);
      signal(SIGALRM, sigalrm_handler);
      syslog(LOG_NOTICE, "Try to re-establish outgoing connection in 5 seconds.");
    }
  
  cl->status.remove = 1;
}

int send_broadcast_ping(void)
{
  conn_list_t *p;

  for(p = conn_list; p != NULL; p = p->next)
    {
      if(p->status.remove)
	continue;
      if(p->status.active && p->status.meta)
	{
	  if(send_ping(p))
	    terminate_connection(p);
	  else
	    {
	      p->status.pinged = 1;
	      p->status.got_pong = 0;
	    }
	}
    }

  return 0;
}

int check_dead_connections(void)
{
  conn_list_t *p;

  for(p = conn_list; p != NULL; p = p->next)
    {
      if(p->status.remove)
	continue;
      if(p->status.active && p->status.meta && p->status.pinged && !p->status.got_pong)
	{
	  syslog(LOG_INFO, "%s (" IP_ADDR_S ") didn't respond to ping",
		 p->hostname, IP_ADDR_V(p->vpn_ip));
	  p->status.timeout = 1;
	  terminate_connection(p);
	}
    }

  return 0;
}

int handle_new_meta_connection(conn_list_t *cl)
{
  conn_list_t *ncn;
  struct sockaddr client;
  int nfd, len = sizeof(struct sockaddr);

  if((nfd = accept(cl->meta_socket, &client, &len)) < 0)
    {
      syslog(LOG_ERR, "accept: %m");
      return -1;
    }

  if((ncn = create_new_connection(nfd)) == NULL)
    {
      shutdown(nfd, 2);
      close(nfd);
      syslog(LOG_NOTICE, "Closed attempted connection.");
      return 0;
    }

  ncn->status.meta = 1;
  ncn->next = conn_list;
  conn_list = ncn;

  return 0;
}

int handle_incoming_meta_data(conn_list_t *cl)
{
  int x, l = sizeof(x), lenin;
  unsigned char tmp[1600];
  int request;
  
  if(getsockopt(cl->meta_socket, SOL_SOCKET, SO_ERROR, &x, &l) < 0)
    {
      syslog(LOG_ERR, "%d: getsockopt failed: %m", __LINE__);
      return -1;
    }
  if(x)
    {
      syslog(LOG_ERR, "Socket error: %s", __LINE__, sys_errlist[x]);
      return -1;
    }

  if((lenin = recv(cl->meta_socket, &tmp, sizeof(tmp), 0)) <= 0)
    {
      syslog(LOG_ERR, "%d: receive: %m", __LINE__);
      return -1;
    }

  request = (int)(tmp[0]);

  if(debug_lvl > 3)
    syslog(LOG_DEBUG, "got request %d", request);

  if(request_handlers[request] == NULL)
    {
      if(debug_lvl > 0)
	syslog(LOG_ERR, "Unknown request %d.", request);
    }
  else
    if(request_handlers[request](cl, tmp, lenin) < 0)
      return -1;
  
  return 0;
}

void check_network_activity(fd_set *f)
{
  conn_list_t *p;
  int x, l = sizeof(x);

  for(p = conn_list; p != NULL; p = p->next)
    {
      if(p->status.remove)
	continue;
      if(p->status.active)
	if(FD_ISSET(p->socket, f))
	  {
	    /*
	      The only thing that can happen to get us here is apparently an
	      error on this outgoing(!) UDP socket that isn't immediate (i.e.
	      something that will not trigger an error directly on send()).
	      I've once got here when it said `No route to host'.
	    */
	    getsockopt(p->socket, SOL_SOCKET, SO_ERROR, &x, &l);
	    syslog(LOG_ERR, "%d: Data socket error: %s", __LINE__, sys_errlist[x]);
	    terminate_connection(p);
	    return;
	  }  
      if(p->status.meta)
	if(FD_ISSET(p->meta_socket, f))
	  if(handle_incoming_meta_data(p) < 0)
	    {
	      terminate_connection(p);
	      return;
	    } 
    }
  
  if(FD_ISSET(myself->socket, f))
    handle_incoming_vpn_data(myself);
  if(FD_ISSET(myself->meta_socket, f))
    handle_new_meta_connection(myself);
}

void handle_tap_input(void)
{
  real_packet_t rp;
  vpn_packet_t vp;
  ip_t from, to;
  int ether_type, lenin;
  
  memset(&vp, 0, sizeof(vp));
  if((lenin = read(tap_fd, &vp, MTU)) <= 0)
    {
      syslog(LOG_ERR, "Error while reading from tapdevice: %m");
      return;
    }
  
  total_tap_in += lenin;
  
  ether_type = ntohs(*((unsigned short*)(&vp.data[12])));
  if(ether_type != 0x0800)
    {
      if(debug_lvl >= 1)
	syslog(LOG_INFO, "Non-IP ethernet frame %04x from " MAC_ADDR_S,
	       ether_type, MAC_ADDR_V(vp.data[6]));
      return;
    }
  
  from = ntohl(*((unsigned long*)(&vp.data[26])));
  to = ntohl(*((unsigned long*)(&vp.data[30])));
  
  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "An IP packet (%04x) for " IP_ADDR_S " from " IP_ADDR_S,
	   ether_type, IP_ADDR_V(to), IP_ADDR_V(from));
  if(debug_lvl > 2)
    syslog(LOG_DEBUG, MAC_ADDR_S " to " MAC_ADDR_S,
	   MAC_ADDR_V(vp.data[0]), MAC_ADDR_V(vp.data[6]));
  
  vp.len = (length_t)lenin - 2;
  strip_mac_addresses(&vp);
  do_encrypt(&vp, &rp);
  send_packet(to, &rp);
}

void main_loop(void)
{
  fd_set fset;
  struct timeval tv;
  int r;

  for(;;)
    {
      tv.tv_sec = timeout;
      tv.tv_usec = 0;

      prune_conn_list();
      build_fdset(&fset);

      if((r = select(FD_SETSIZE, &fset, NULL, NULL, &tv)) < 0)
        {
	  if(errno == EINTR) /* because of alarm */
	    continue;
          syslog(LOG_ERR, "Error while waiting for input: %m\n");
          return;
        }
      if(r == 0)     /* Timeout... hm... something might be wrong. */
	{
	  check_dead_connections();
	  send_broadcast_ping();
	  continue;
	}

      check_network_activity(&fset);

      /* local tap data */
      if(FD_ISSET(tap_fd, &fset))
	handle_tap_input();
    }
}
