/*
    protocol.c -- handle the meta-protocol
    Copyright (C) 1999 Ivo Timmermans <zarq@iname.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <string.h>
#include <syslog.h>
#include <sys/socket.h>

#include <xalloc.h>

#include "conf.h"
#include "encr.h"
#include "net.h"
#include "protocol.h"

int send_ack(conn_list_t *cl)
{
  unsigned char tmp = ACK;

  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "Send ACK to %s", cl->hostname);

  if((send(cl->meta_socket, &tmp, sizeof(tmp), 0)) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int send_termreq(conn_list_t *cl)
{
  termreq_t tmp;

  tmp.type = TERMREQ;
  tmp.vpn_ip = myself->vpn_ip;

  if(debug_lvl > 2)
    syslog(LOG_DEBUG, "Send TERMREQ(" IP_ADDR_S ") to " IP_ADDR_S, IP_ADDR_V(tmp.vpn_ip),
	   IP_ADDR_V(cl->vpn_ip));

  if((send(cl->meta_socket, &tmp, sizeof(tmp), 0)) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int send_timeout(conn_list_t *cl)
{
  termreq_t tmp;

  tmp.type = PINGTIMEOUT;
  tmp.vpn_ip = myself->vpn_ip;

  if(debug_lvl > 2)
    syslog(LOG_DEBUG, "Send TIMEOUT(" IP_ADDR_S ") to " IP_ADDR_S, IP_ADDR_V(tmp.vpn_ip),
	   IP_ADDR_V(cl->vpn_ip));

  if((send(cl->meta_socket, &tmp, sizeof(tmp), 0)) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int send_ping(conn_list_t *cl)
{
  unsigned char tmp = PING;

  if(debug_lvl > 3)
    syslog(LOG_DEBUG, "pinging " IP_ADDR_S, IP_ADDR_V(cl->vpn_ip));

  if((send(cl->meta_socket, &tmp, sizeof(tmp), 0)) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int send_pong(conn_list_t *cl)
{
  unsigned char tmp = PONG;

  if((send(cl->meta_socket, &tmp, sizeof(tmp), 0)) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int send_add_host(conn_list_t *cl, conn_list_t *new_host)
{
  add_host_t tmp;

  tmp.type = ADD_HOST;
  tmp.real_ip = new_host->real_ip;
  tmp.vpn_ip = new_host->vpn_ip;
  tmp.vpn_mask = new_host->vpn_mask;
  tmp.portnr = new_host->port;

  if((send(cl->meta_socket, (unsigned char*)&tmp, sizeof(add_host_t), 0)) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int send_basic_info(conn_list_t *cl)
{
  basic_info_t tmp;

  tmp.type = BASIC_INFO;
  tmp.protocol = PROT_CURRENT;

  tmp.portnr = myself->port;
  tmp.vpn_ip = myself->vpn_ip;
  tmp.vpn_mask = myself->vpn_mask;

  if(debug_lvl > 2)
    syslog(LOG_DEBUG, "Send BASIC_INFO(%d,%hd," IP_ADDR_S "," IP_ADDR_S ") to " IP_ADDR_S,
	   tmp.protocol, tmp.portnr, IP_ADDR_V(tmp.vpn_ip), IP_ADDR_V(tmp.vpn_mask),
	   IP_ADDR_V(cl->real_ip));

  if((send(cl->meta_socket, &tmp, sizeof(tmp), 0)) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int send_passphrase(conn_list_t *cl)
{
  passphrase_t tmp;

  tmp.type = PASSPHRASE;
  encrypt_passphrase(&tmp);

  if(debug_lvl > 2)
    syslog(LOG_DEBUG, "Send PASSPHRASE(%hd,%s) to " IP_ADDR_S, tmp.len, tmp.phrase,
	   IP_ADDR_V(cl->vpn_ip));

  if((send(cl->meta_socket, &tmp, tmp.len+3, 0)) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int send_public_key(conn_list_t *cl)
{
  public_key_t *tmp;

  tmp = (public_key_t*)xmalloc(strlen(my_public_key_base36)+sizeof(public_key_t));
  tmp->type = PUBLIC_KEY;
  tmp->len = strlen(my_public_key_base36);
  strcpy(&tmp->key, my_public_key_base36);

  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "Send PUBLIC_KEY(%hd,%s) to " IP_ADDR_S, tmp->len, &tmp->key,
	   IP_ADDR_V(cl->vpn_ip));

  if((send(cl->meta_socket, tmp, tmp->len+sizeof(public_key_t), 0)) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int send_calculate(conn_list_t *cl, char *k)
{
  calculate_t *tmp;

  tmp = xmalloc(strlen(k)+sizeof(calculate_t));
  tmp->type = CALCULATE;
  tmp->len = strlen(k);
  strcpy(&tmp->key, k);

  if(send(cl->meta_socket, tmp, tmp->len+4, 0) < 0)
    {
      syslog(LOG_ERR, "send: %m");
      return -1;
    }

  return 0;
}

int notify_others(conn_list_t *new)
{
  conn_list_t *p;

  for(p = conn_list; p != NULL; p = p->next)
    {
      if(p != new && p != myself)
	send_add_host(p, new);
    }

  return 0;
}

int basic_info_h(conn_list_t *cl, unsigned char *d, int len)
{
  basic_info_t *tmp = (basic_info_t*)d;

  cl->protocol_version = tmp->protocol;
  cl->port = tmp->portnr;
  cl->vpn_ip = tmp->vpn_ip;
  cl->vpn_mask = tmp->vpn_mask;

  if(cl->protocol_version <= PROT_TOO_OLD)
    {
      syslog(LOG_ERR, "Peer uses protocol version %d which is too old.",
	     cl->protocol_version);
      return -1;
    }

  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "got BASIC_INFO(%hd," IP_ADDR_S "," IP_ADDR_S ")", cl->port,
	   IP_ADDR_V(cl->vpn_ip), IP_ADDR_V(cl->vpn_mask));
  if(debug_lvl > 0)
    syslog(LOG_DEBUG, "Peer uses protocol version %d",
	   cl->protocol_version);

  if(cl->status.outgoing)
    {
      if(setup_vpn_connection(cl) < 0)
	return -1;
      send_basic_info(cl);
    }
  else
    {
      if(setup_vpn_connection(cl) < 0)
	return -1;
      send_passphrase(cl);
    }

  cl->status.active = 0;

  return 0;
}

int passphrase_h(conn_list_t *cl, unsigned char *d, int len)
{
  passphrase_t *tmp = (passphrase_t*)d;

  cl->pp = xmalloc(tmp->len+3);
  memcpy(cl->pp, tmp, tmp->len+3);

  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "got PASSPHRASE(%hd,%s)", cl->pp->len, cl->pp->phrase);

  if(cl->status.outgoing)
    send_passphrase(cl);
  else
    send_public_key(cl);

  return 0;
}

int public_key_h(conn_list_t *cl, unsigned char *d, int len)
{
  int i;
  conn_list_t *p;
  char *g_n;
  public_key_t *tmp = (public_key_t*)d;

  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "got PUBLIC_KEY(%hd,%s)", tmp->len, &tmp->key);

  g_n = xmalloc(tmp->len+1);
  strcpy(g_n, &tmp->key);

  if(verify_passphrase(cl, g_n))
    {
      /* intruder! */
      syslog(LOG_ERR, "Intruder: passphrase does not match.");
      return -1;
    }

  if(debug_lvl > 1)
    syslog(LOG_INFO, "Passphrase OK");

  for(p = conn_list, i = 0; p != NULL; p = p->next)
    if(p->status.meta && p->status.active)
      p->nr = ++i;

  if(i == 0) /* only one connection: the one we're dealing with here. */
    {
      if(debug_lvl > 2)
	syslog(LOG_INFO, "setting shared key; almost_key = %s", g_n);
      set_shared_key(g_n);
      if(!cl->status.outgoing)
	{
	  cl->status.active = 1;
	  syslog(LOG_NOTICE, "Connection with %s activated.", cl->hostname);
	  send_ack(cl);
	  return 0;
	}
    }

  for(p = conn_list; p != NULL; p = p->next)
    if(p->status.meta && p->status.active)
      {
	if(send_calculate(p, my_public_key_base36) < 0)
	  {
	    syslog(LOG_ERR, "send failed.. dammit, WHY?");
	    return -1;
	  }
	if(send_calculate(p, g_n) < 0)
	  {
	    syslog(LOG_ERR, "send failed.. dammit, WHY?");
            return -1;
          }
      }

  if(cl->status.outgoing)
    send_public_key(cl);

  return 0;
}

int ack_h(conn_list_t *cl, unsigned char *d, int len)
{
  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "got ACK");
  
  cl->status.active = 1;
  syslog(LOG_NOTICE, "Connection with %s activated.", cl->hostname);

  return 0;
}

int termreq_h(conn_list_t *cl, unsigned char *d, int len)
{
  syslog(LOG_NOTICE, IP_ADDR_S " wants to quit", IP_ADDR_V(cl->vpn_ip));
  cl->status.termreq = 1;
  terminate_connection(cl);

  return 0;
}

int timeout_h(conn_list_t *cl, unsigned char *d, int len)
{
  syslog(LOG_NOTICE, IP_ADDR_S " says it's gotten a timeout from us", IP_ADDR_V(cl->vpn_ip));
  cl->status.termreq = 1;
  terminate_connection(cl);

  return 0;
}

int ping_h(conn_list_t *cl, unsigned char *d, int len)
{
  if(debug_lvl > 3)
    syslog(LOG_DEBUG, "responding to ping from " IP_ADDR_S, IP_ADDR_V(cl->vpn_ip));
  cl->status.pinged = 0;
  cl->status.got_pong = 1;

  send_pong(cl);
  
  return 0;
}

int pong_h(conn_list_t *cl, unsigned char *d, int len)
{
  if(debug_lvl > 3)
    syslog(LOG_DEBUG, "ok, got pong from " IP_ADDR_S, IP_ADDR_V(cl->vpn_ip));
  cl->status.got_pong = 1;

  return 0;
}

int add_host_h(conn_list_t *cl, unsigned char *d, int len)
{
  add_host_t *tmp = (add_host_t*)d;

  if(debug_lvl > 0)
    syslog(LOG_DEBUG, "Add host request from " IP_ADDR_S, IP_ADDR_V(cl->vpn_ip));
  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "got ADD_HOST(" IP_ADDR_S "," IP_ADDR_S ",%hd)",
	   IP_ADDR_V(tmp->vpn_ip), IP_ADDR_V(tmp->vpn_mask), tmp->portnr);
  syslog(LOG_NOTICE, "Ignoring add host request");

  return 0;
}

int calculate_h(conn_list_t *cl, unsigned char *d, int len)
{
  unsigned char tmp[100];

  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "got CALCULATE(%s)", &d[1]);

  memset(tmp, 0, sizeof(tmp));
  powerize(&tmp[1], &d[1]);
  tmp[0] = CALC_RES;

  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "Send CALC_RES(%s)", &tmp[1]);

  if((send(cl->meta_socket, &tmp, strlen(tmp), 0)) < 0)
  {
      syslog(LOG_ERR, "send: %m");
      return -1;
  }

  return 0;
}

int calc_res_h(conn_list_t *cl, unsigned char *d, int len)
{
  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "got CALC_RES(%s)", &d[1]);

    {
      cl->status.active = 1;
      syslog(LOG_NOTICE, "Connection with %s activated.", cl->hostname);
      send_ack(cl);
    }

  return 0;
}

int almost_key_h(conn_list_t *cl, unsigned char *d, int len)
{
  if(debug_lvl > 1)
    syslog(LOG_DEBUG, "got ALMOST_KEY(%s)", &d[1]);

  return 0;
}

int (*request_handlers[256])(conn_list_t*, unsigned char*, int) = {
  0, ack_h, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  termreq_h, timeout_h, 0, 0, 0, 0, 0, 0, 0, 0,
  ping_h, pong_h, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  add_host_h, basic_info_h, passphrase_h, public_key_h, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  calculate_h, calc_res_h, almost_key_h, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0
};

