/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* the big league: data pump                                                  */
/*============================================================================*/

/*
 * sequencer
 *
 * input: anchor - pointer to vpnd global data
 *
 * This procedure performs all necessary processing to transfer data
 * between the local slip interface and the slip interface of the peer.
 */

void sequencer(VPN *anchor)
{
	time_t current;		/* current time in seconds		*/
	time_t next_slip;	/* next slip retry in seconds		*/
	time_t next_peer;	/* next peer connect retry in seconds	*/
	time_t next_ping;	/* next time to idle ping in seconds	*/
	time_t next_server;	/* next time create srv sk in seconds	*/
	time_t next_key;	/* next key create time in seconds	*/
	int result;		/* general purpose function call result	*/
	int i;			/* general purpose usage		*/
	WORD08 *lptr;		/* local input rover			*/
	WORD08 *rptr;		/* peer input rover			*/
	WORD32 localdata;	/* data bytes in local input buffer	*/
	WORD32 remotedata;	/* data bytes in peer input buffer	*/
	WORD08 sendkey;		/* flag that key change is required	*/
	WORD08 enabled;		/* peer link suspend flag		*/
	WORD08 slipstate;	/* slip link state and error code	*/
	WORD08 peerstate;	/* peer link state and error code	*/
	WORD16 leaky;		/* leaky bucket count			*/
	WORD16 chklink;		/* flag for link check required		*/
	WORD16 dosuspend;	/* flag for link suspend required	*/
	fd_set *r;		/* read set for select			*/
	fd_set *w;		/* write set for select			*/
	fd_set *e;		/* exception set for select		*/
	struct timeval t;	/* timeout for select			*/
	struct sockaddr_in a;	/* peer connection data			*/


	/* debug message */

	ENTER("sequencer");

	/* set up buffer and file descriptor set pointers */

	lptr=anchor->lbfr;
	rptr=anchor->rbfr;
	r=&anchor->r;
	w=&anchor->w;
	e=&anchor->e;

	/* if plaintext mode on serial interface */

	if(!anchor->cipher&&anchor->serialmode)
	{
		/* never exit this loop, until a term event occurs */

		while(1)
		{
			/* done in case of termination event */

			if(terminate)goto doexit;

			/* try to get serial device, handle errors */

			if(!serialhandler(anchor))
			{
				logmsg(DEVFAIL,0,anchor);
				goto perr1;
			}

			/* run modem init chat, handle errors */

			if((result=chathandler(3,anchor))!=0)
			{
				if(result!=256)logmsg(CHATFAIL,result,anchor);
				goto perr2;
			}

			/* create slip interface, handle errors */

			if((result=sliphandler(1,anchor))!=0)
			{
				logmsg(SLIPFAIL,result,anchor);
				goto perr2;
			}

			/* print message */

			logmsg(DCONNECT,0,anchor);

			/* execute linkup process, if there is any */

			doexec(anchor->linkup);

			/* while select returns no serial line exception */

			while(1)
			{
				/* done in case of termination event */

				if(terminate)goto doexit;

				/* set serial line exception bit */

				FD_ZERO(e);
				FD_SET(anchor->serial,e);

				/* wait until select is interrupted or an
				   exception occurs */

				if(select(anchor->serial+1,NULL,NULL,e,NULL)
					==-1)
				{
					/* debug message */

					DEBUG1(3,"select interrupted\n");

					/* ignore interrupts */

					if(errno==EINTR||errno==EAGAIN)continue;

					/* print info and terminate */

					logmsg(SELECTFAIL,0,anchor);
					goto doexit;
				}

				/* exception on serial line */

				if(FD_ISSET(anchor->serial,e))
				{
					/* print log message, drop slip
					   interface, close serial line,
					   wait and retry */

					logmsg(PEERFAIL,111,anchor);

					/* execute linkdown process,
					   if there is any */

					doexec(anchor->linkdown);
					goto perr3;
				}
			}

			/* error handling */

perr3:			sliphandler(3,anchor);
perr2:			serialhandler(anchor);
perr1:			sleep(anchor->retry);
		}
	}

	/* do only, if not in extended and if not in plaintext mode */

	if(!anchor->extended&&anchor->cipher)
	{
		/* read the master key from file, if this fails print a log
		   message and terminate */

		if(!readmaster(anchor))
		{
			logmsg(KEYFAIL,0,anchor);
			goto doexit;
		}

		/* fill whitening buffer */

		genwhite(anchor->secret,anchor);
	}

	/* set link suspend mode according to configuration data */

	if(anchor->suspend)enabled=0;
	else enabled=1;

	/* reset leaky bucket count */

	leaky=anchor->noresponse;

	/* set peer link state and slip link state to down */

	peerstate=slipstate=0x80;

	/* reset link suspend count, link test count, key change flag and 
	   buffer contents length variables */

	dosuspend=chklink=next_key=sendkey=localdata=remotedata=0;

	/* force connection to peer, creation of slip link and idle pings */

	next_ping=next_peer=next_slip=current=next_server=time(NULL)+1;

	/* do, until a fatal error or a termination event occurs */

	while(1)
	{
		/* done in case of termination event */

		if(terminate)goto doexit;

		/* if the slip link is down and it's time to retry */

		if(slipstate==0x80&&next_slip<=current)
		{
			/* if link creation is successful print info */

			if(!(slipstate=sliphandler(0,anchor)))
				logmsg(SLIPUP,0,anchor);
		}

		/* handle slip link errors */

		if((result=slipstate&0x7f)!=0)
		{
			/* write info */

			logmsg(SLIPFAIL,result,anchor);

			/* drop slip link if it is up */

			if(anchor->slip)sliphandler(2,anchor);

			/* set up slip link retry */

			next_slip=current+anchor->retry;

			/* flush data buffers */

			remotedata=localdata=0;

			/* set slip state to down */

			slipstate=0x80;
		}
		
		/* handle peer link errors */

		if((result=peerstate&0x7f)!=0)
		{
			/* print proper message according to mode, in
			   client mode retry and take care of suspend */

			if(!anchor->mode&&!anchor->serialmode)
			{
				logmsg(SPEERFAIL,result,anchor);
				next_server=current+1;
			}
			else
			{
				logmsg(PEERFAIL,result,anchor);
				next_peer=current+anchor->retry;
				if(anchor->mode&&anchor->suspend)enabled=0;
			}

			/* close peer connection */

			disconnect(0,anchor);

			/* flush data buffers */

			remotedata=localdata=0;

			/* set slip state to down */

			peerstate=0x80;
		}

		/* if not in serial line mode */

		if(!anchor->serialmode)
		{
			/* if the peer link is not established */

			if(peerstate)
			{
				/* if we're the server, the server socket does
				   not exist and it is time to create the
				   server socket */

				if(!anchor->mode&&anchor->server==-1&&
					next_server<=current)
				{
					/* get server socket, handle errors */

					if((result=tcpsocket(anchor))!=0)
					{
						/* give info */

						logmsg(SRVSFAIL,result,anchor);

						/* retry socket creation */

						next_server=current+
							anchor->retry;
					}

					/* in case of successful socket creation
					   print info */

					else logmsg(SRVSLSTN,0,anchor);
				}

				/* if we're the client, it is time to create the
				   link and link creation is not suspended */

				if(anchor->mode&&enabled&&next_peer<=current)
				{
					/* if peer link creation fails */

					if((result=tcpsocket(anchor))!=0)
					{
						/* print info */

						logmsg(PEERFAIL,result,anchor);

						/* retry link creation */

						next_peer=current+anchor->retry;
					}

					/* if crypto setup fails in crypto
					   mode */

					else if((result=anchor->cipher?
						cryptosetup(anchor):0)!=0)
					{
						/* print info */

						logmsg(CRYPTFAIL,result,anchor);

						/* close peer connection */

						disconnect(0,anchor);

						/* retry link creation */

						next_peer=current+anchor->retry;
					}

					/* in case of successful link
					   creation */

					else 
					{
						/* print info */

						logmsg(CCONNECT,0,anchor);

						/* set up time for key change,
						   reset link check count, reset
						   link suspend count, mark peer
						   link as established */

						next_key=current+anchor->keyttl;
						peerstate=dosuspend=chklink=
							sendkey=0;

						/* execute linkup process,
						   if there is any */

						doexec(anchor->linkup);
					}
				}
			}
		}

		/* if in serial line mode if serial line is closed and
		   no child process is running and it is time for the
		   next try */

		else if(anchor->serial==-1&&next_peer<=current&&!anchor->child)
		{
			/* if accessing serial line device fails */

			if(!serialhandler(anchor))
			{
				/* print info */

				logmsg(DEVFAIL,0,anchor);

				/* retry serial device access */

				next_peer=current+anchor->retry;
			}

			/* if serial line is now open and configured,
			   start init chat process, handle errors */

			else if(chathandler(1,anchor))
			{
				/* close serial device */

				serialhandler(anchor);

				/* retry fork */

				next_peer=current+anchor->retry;
			}
		}

		/* get current time in seconds */

		current=time(NULL);

		/* reset highest file descriptor number */

		i=0;

		/* reset sets for select call */

		FD_ZERO(r);
		FD_ZERO(w);
		FD_ZERO(e);

		/* if the slip link is up */

		if(!slipstate)
		{
			/* check for read possible if empty local buffer */

			if(!localdata)FD_SET(anchor->pty,r);

			/* check for write possible if peer data available */

			if(remotedata)FD_SET(anchor->pty,w);

			/* always check for exceptions */

			FD_SET(anchor->pty,e);

			/* set new highest file descriptor if pty is it */

			if(anchor->pty>i)i=anchor->pty;
		}

		/* if the peer link is up */

		if(!peerstate)
		{
			/* signal key change if not disabled and
			   time has come for it */

			if(anchor->keyttl)if(current>=next_key)sendkey=1;

			/* check for read possible if empty peer buffer */

			if(!remotedata)FD_SET(anchor->peer,r);

			/* check for write possible if local data available */

			if(localdata||sendkey)FD_SET(anchor->peer,w);

			/* always check for exceptions */

			FD_SET(anchor->peer,e);

			/* set new highest file descriptor if peer is it */

			if(anchor->peer>i)i=anchor->peer;
		}

		/* if the peer link is down in server mode and if the
		   sever socket exists */

		else if(!anchor->mode&&anchor->server!=-1)
		{
			/* check for pending connect request */

			FD_SET(anchor->server,r);

			/* always check for exceptions */

			FD_SET(anchor->server,e);

			/* set new highest file descriptor if server is it */

			if(anchor->server>i)i=anchor->server;
		}

		/* set 1 second timeout for select call */

		t.tv_sec=1;
		t.tv_usec=0;

		/* check descriptor sets, handle fatal errors and signals */

		if((result=select(i+1,r,w,e,&t))<0)
		{
			/* debug message */

			DEBUG1(3,"select interrupted\n");

			/* ignore interrupts */

			if(errno==EINTR||errno==EAGAIN)continue;

			/* print info and terminate */

			logmsg(SELECTFAIL,0,anchor);
			goto doexit;
		}

		/* if there's a serial line init chat process */

		if(anchor->child)
		{
			/* debug message */

			DEBUG1(3,"child running\n");

			/* if the process did terminate */

			if((i=chathandler(4,anchor))!=257)
			{
				/* if line initialization failed */

				if(i)
				{
					/* print info */

					logmsg(CHATFAIL,i,anchor);

					/* close serial line */

					serialhandler(anchor);

					/* retry serial line */

					next_peer=current+anchor->retry;
				}

				/* if line initialization is successful */

				else
				{
					/* reset leaky bucket count */

					leaky=anchor->noresponse;

					/* set up link handle to remote */

					anchor->peer=anchor->serial;

					/* if not in plaintext mode do
					   crypto setup */

					if(anchor->cipher)i=cryptosetup(anchor);

					/* if crypto setup fails */

					if(i)
					{
						/* print info */

						logmsg(CRYPTFAIL,i,anchor);

						/* disconnect from peer */

						disconnect(0,anchor);

						/* retry connection */

						next_peer=current+anchor->retry;
					}

					/* if crypto setup was ok */

					else
					{
						/* print info */

						logmsg(DCONNECT,0,anchor);

						/* set up next key change,
						   reset link check count,
						   reset link suspend count,
						   mark peer link as up */

						next_key=current+anchor->keyttl;
						peerstate=dosuspend=chklink=
							sendkey=0;

						/* execute linkup process
						   if there is any */

						doexec(anchor->linkup);
					}
				}
			}
		}

		/* if keepalive and both interfaces enabled
		    dummy read echo replies, if at least one
		    reply reset leaky bucket count */

		if(anchor->keepalive&&!slipstate&&!peerstate)
			if(recvicmp(anchor))
		{
			leaky=anchor->noresponse;
			DEBUG1(3,"icmp received\n");
		}

		/* if nothing to do */

		if(!result)
		{
			/* debug message */

			DEBUG2(3,"idle, current=%08lx\n",current);

			/* if link check is required and the peer link
			   exists */

			if(anchor->linktest&&!peerstate)
			{
				/* if limit not reached increase
				   link check count */

				if(chklink<anchor->linktest)chklink++;
			}

			/* if link suspend is requested and the peer
			   link exists */

			if(anchor->suspend&&!peerstate)
			{
				/* increase suspend timer, done if
				   timeout not yet reached */

				if(dosuspend<anchor->suspend)dosuspend++;

				/* disconnect from peer and set
				   suspend mode if timeout reached */

				else
				{
					disconnect(0,anchor);
					peerstate=0x80;
					enabled=0;
				}
			}

			/* if idle pings are requested and links are up
			   and it is time for the next ping */

			if(anchor->keepalive&&!slipstate&&!peerstate)
			{
				/* done if it's not time to ping */

				if(next_ping>current)continue;

				/* send echo request */

				sendicmp(anchor);

				/* debug message */

				DEBUG1(3,"icmp sent\n");

				/* set up time for next ping */

				next_ping=current+anchor->keepalive;

				/* if serial mode if empty leaky bucket
				   set peer error code */

				if(anchor->serialmode)if(!--leaky)peerstate=229;
			}

			/* make sure idle pings are made when interfaces
			   are up and idle pings are requested */

			else next_ping=current+anchor->keepalive;

			/* repeat whole process */

			continue;
		}

		/* set time for next idle ping */

		next_ping=current+anchor->keepalive;

		/* if the peer connection exists and there's a peer exception */

		if(!peerstate&&FD_ISSET(anchor->peer,e))
		{
			/* if the exception was caused by pending out of band 
			   data retry */

			if(oobrecv(anchor))continue;

			/* set peer error code */

			peerstate=230;
		}

		/* is the peer connection doesn't exist but server socket does
		   exist */

		else if(peerstate&&!anchor->mode&&anchor->server!=-1)
		{
			/* in case of a server socket exception */

			if(FD_ISSET(anchor->server,e))
			{
				/* close server socket */

				disconnect(1,anchor);

				/* print info */

				logmsg(SRVSFAIL,101,anchor);

				/* set server socket retry time */

				next_server=current+anchor->retry;
			}

			/* is the peer connection doesn't exist but the server
			   socket does exist and if there's a connection
			   pending */

			else if(FD_ISSET(anchor->server,r))
			{
				/* successful connection */

				if(!(i=accept_socket(&a,anchor)))
				{
					/* print connection info */

					logmsg(SCONNECT,a.sin_addr.s_addr,
						anchor);

					/* if crypto init fails in crypto
					   mode */

					if((result=anchor->cipher?
						cryptosetup(anchor):0)!=0)
					{
						/* print info */

						logmsg(SCRYPT,result,anchor);

						/* close peer connection */

						disconnect(0,anchor);
					}

					/* if crypto init is successful */

					else
					{
						/* reset leaky bucket count */

						leaky=anchor->noresponse;

						/* set next key change time */

						next_key=current+anchor->keyttl;

						/* reset new key required,
						   link check count,
						   link suspend count,
						   mark peer link as up */

						peerstate=dosuspend=chklink=
							sendkey=0;

						/* close server socket */

						disconnect(1,anchor);

						/* execute linkup process,
						   if there is any */

						doexec(anchor->linkup);
					}
				}

				/* handle accept errors */

				else logmsg(i==1?ACPTERR:ILLEGAL,
					a.sin_addr.s_addr,anchor);
			}
		}

		/* if the slip link exists and there's an exception pending
		   set slip error code */

		if(!slipstate&&FD_ISSET(anchor->pty,e))slipstate=229;

		/* if the peer link exists and send is possible */

		if(!peerstate&&FD_ISSET(anchor->peer,w))
		{
			/* if it's time to check the link */

			if(anchor->linktest&&chklink==anchor->linktest)
			{
				/* reset link check count */

				chklink=0;

				/* send linkcheck request, handle errors */

				if(!oobsend(PING,anchor))i=MAXWAIT;

				/* in case of successful linkcheck request send
				   wait for reply for a limited time */

				else for(i=0;i<MAXWAIT;i++)
				{
					/* try to receive out of band data,
					   done, if linkcheck response
					   received */

					if(oobrecv(anchor)==2)break;

					/* wait for 100 msec, then retry */

					usleep(100000);
				}

				/* if no linkcheck response received set peer
				   error */

				if(i==MAXWAIT)peerstate=231;

				/* restart loop as writing to peer may not yet
				   be possible again */

				continue;
			}

			/* if it's time to send a new key */

			else if(sendkey)
			{
				/* reset key send required flag */

				sendkey=0;

				/* if not in plaintext mode */

				if(anchor->cipher)
				{
					/* if send of new key fails set peer
					   error */

					if(!datasend(NULL,0,anchor))
						peerstate=232;

					/* in case of success */

					else
					{
						/* set next key change time */

						next_key=current+anchor->keyttl;

						/* reset suspend and linktest
						   counts */

						dosuspend=chklink=0;
					}

					/* restart loop as writing to peer may
					   not yet be possible again */

					continue;
				}
			}
		}

		/* if peer data are available and if there's no data from
		   peer in the buffer */

		if(!peerstate&&!remotedata&&FD_ISSET(anchor->peer,r))
		{
			/* reset buffer rover */

			rptr=anchor->rbfr;

			/* in case of successful read set new buffer length */

			if((i=datarecv(rptr,MAXBLOCK,anchor))>0)
			{
				/* store new received data length 
				   if slip link is up */

				if(!slipstate)remotedata=i-1;

				/* change key/iv if new key from peer */

				if(i==1&&anchor->mode)sendkey=1;

				/* reset suspend and link test count */

				dosuspend=chklink=0;
			}

			/* in case of read failure set peer error */

			else peerstate=233;
		}

		/* if there's data from the peer in the buffer and if we can
		   write to the slip interface */

		if(!slipstate&&remotedata&&FD_ISSET(anchor->pty,w))
		{

			/* if write to slip is successful */

			if((i=write(anchor->pty,rptr,remotedata))>0)
			{
				/* adjust data pointer */

				rptr+=i;

				/* adjust data length */

				remotedata-=i;
			}

			/* in case of write failure  set slip error code */

			else slipstate=230;
		}

		/* if data from slip are available and if there's no data
		   from slip in the buffer */

		if(!slipstate&&!localdata&&FD_ISSET(anchor->pty,r))
		{
			/* if suspended */

			if(!enabled)
			{
				/* reset suspend mode */

				enabled=1;

				/* restart loop to initate connect to peer */

				continue;
			}

			/* reset buffer rover */

			lptr=anchor->lbfr;

			/* in case of read failure set slip error code */

			if(!(i=read(anchor->pty,lptr,MAXBLOCK)))slipstate=231;

			/* in case of successful read adjust buffer length
			   if peer link exists */

			else if(!peerstate)localdata=i;
		}

		/* if there is data from the slip interface available and
		   if data can be written to peer */

		if(!peerstate&&localdata&&FD_ISSET(anchor->peer,w))
		{
			/* flush buffer in case of successful send, reset
			   suspend and link test counters */

			if(datasend(lptr,localdata,anchor))
				dosuspend=chklink=localdata=0;

			/* in case of send failure set peer error */

			else peerstate=234;
		}
	}

	/* debug message */

doexit:	LEAVE("sequencer");
}
