/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* good-ole-line: serial interface stuff                                      */
/*============================================================================*/

/*
 * serialhandler
 *
 * input: anchor - pointer to vpnd global data
 *
 * return: 1 if line is open, else 0
 *
 * This procedure opens a serial line and sets it to
 * raw state at the requested speed (speed, 8N1) or
 * it closes the serial line, if it is open.
 */

int serialhandler(VPN *anchor)
{
	struct termios t;	/* tty settings		*/
	int i;			/* general purpose	*/


	/* debug message */

	ENTER("serialhandler");

	/* if line is open, close it */

	if(anchor->serial!=-1)goto delete;

	/* open serial device, react to errors */

	if((anchor->serial=
	    open(anchor->serialdev,O_RDWR|O_NONBLOCK|O_NOCTTY))==-1)
		JUMP("open",err1);

	/* save tty device attributes, react to errors */

	if(tcgetattr(anchor->serial,&(anchor->line))<0)JUMP("tcgetattr",err2);

	/* change file protection for privacy, handle errors */

	if(fchmod(anchor->serial,0))JUMP("fchmod",err2);

	/* set dtr line, react to errors */

	if(ioctl(anchor->serial,TIOCMGET,&i)<0)JUMP("ioctl(TIOCMGET)",err3);
	i|=TIOCM_DTR;
	if(ioctl(anchor->serial,TIOCMSET,&i)<0)JUMP("ioctl(TIOCMSET)",err3);

	/* set tty device attributes, react to errors */

	t.c_cflag=CS8|CREAD|HUPCL|(anchor->rtscts?CRTSCTS:0)|
		(anchor->localline?CLOCAL:0);
	t.c_iflag=IGNBRK|IGNPAR;
	t.c_oflag=0;
	t.c_lflag=0;
	for(i=0;i<NCCS;i++)t.c_cc[i]=0;
	t.c_cc[VMIN]=1;
	t.c_cc[VTIME]=0;
	cfsetispeed(&t,anchor->speed);
	cfsetospeed(&t,anchor->speed);
	if(tcsetattr(anchor->serial,TCSAFLUSH,&t)<0)JUMP("tcsetattr",err4);

	/* signal success */

	goto err1;

	/* flush buffers, restore line settings */

delete:	if(tcsetattr(anchor->serial,TCSAFLUSH,&(anchor->line))<0)
		ERRNO("tcsetattr");

	/* drop dtr line */

err4:	if(ioctl(anchor->serial,TIOCMGET,&i)>=0)
	{
		i&=~TIOCM_DTR;
		if(ioctl(anchor->serial,TIOCMSET,&i)<0)ERRNO("ioctl(TIOCMSET)");
	}
	else ERRNO("ioctl(TIOCMGET)");

	/* restore previous file permissions */

err3:	fchmod(anchor->serial,anchor->normalmode);

	/* close device file */

err2:	if(close(anchor->serial))ERRNO("close");

	/* mark device file as closed */

	anchor->serial=-1;

	/* debug message */

err1:	LEAVE("serialhandler");

	/* signal success or error */

	return (anchor->serial!=-1?1:0);
}
