/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* shhhhhhh! the crypto stuff (blowfish itself is resident in its own file)   */
/*============================================================================*/

/*
 * getrandom
 *
 * input:  len    - the amount of random data to collect
 *	   anchor - pointer to vpnd global data
 *
 * output: out - 64 bytes of random data
 *
 * return: 0 - random data collection failed
 *	   1 - random data collection successful
 *
 * This procedure gathers and returns 64 bytes of random data.
 */

int getrandom(WORD08 *out,WORD32 len,VPN *anchor)
{
	int fd;	/* file descriptor of random device */


	/* debug message */

	ENTER("getrandom");

	/* open random device (no block if insufficient entropy),
	   handle errors */

	if((fd=open(anchor->randomdev,O_RDONLY))<0)RETURN("open",0);

	/* read random data, handle errors */

	if(read(fd,out,len)!=len)
	{
		close(fd);
		RET("read",0);
	}

	/* close random device */

	if(close(fd))ERRNO("close");

	/* debug message */

	LEAVE("getrandom");

	/* signal success */

	return 1;
}

/*
 * checksum
 *
 * input:  data   - the data the checksum is to be created for
 *	   length - the length of the data
 *	   unused - unused, for hmac compatability
 *
 * output: out - the location the 2 byte checksum is to be stored at
 *
 * This procedure creates a checksum of given data and stores it to the
 * given location.
 */

#ifndef CRYPTOX86

void checksum(register WORD08 *data,register WORD32 length,WORD08 *out,void *unused)
{
register WORD32 sum=0;	/* checksum creation buffer */


/* create checksum */

while(length--)sum=((sum<<1)|((sum>>15)&1))^*data++;

/* return checksum to caller */

*out++=(WORD08)(sum);
*out=(WORD08)(sum>>8);
}

#endif

/*
 * procedure encrypt_cfb_64_8
 *
 * input: total    - the amount of data elements to be processed
 *	  anchor   - pointer to vpnd global data
 *
 * inout: data     - the data to be processed
 *
 * This procedure encodes n sets of data bytes using
 * 8 and 64 bit cipher feedback (CFB) mode.
 */

#ifndef CRYPTOX86

void encrypt_cfb_64_8(WORD08 *data,WORD32 total,VPN *anchor)
{
	register WORD32 i;		/* general purpose usage	*/
	register WORD08 *ptr;		/* buffer fast access		*/
	register WORD08 mem;		/* general purpose memory	*/
	WORD08 bfr[BLOWFISH_BLOCKSIZE];	/* encryption buffer		*/


	/* debug message */

	ENTER("encrypt_cfb_64_8");

	/* copy iv to buffer */

	for(i=0,ptr=bfr;i<BLOWFISH_BLOCKSIZE;i++)*ptr++=anchor->local_iv[i];

	/* increase iv */

	for(mem=1,i=0,ptr=anchor->local_iv;mem&&i<BLOWFISH_BLOCKSIZE;i++)
	{
		*ptr+=mem;
		mem=*ptr++?0:1;
	}

	/* for all large blocks use CFB64 */

	while(total>=BLOWFISH_BLOCKSIZE)
	{
		/* encrypt buffer */

		anchor->encrypt(bfr,anchor->local);

		/* XOR data bytes with buffer, memorize new iv
		   and do whitening */

		for(i=0,ptr=bfr;i<BLOWFISH_BLOCKSIZE;i++)
		{
			*data^=*ptr;
			*ptr++=*data;
			*data++^=anchor->white[anchor->wwhite++];
		}

		/* adjust data count */

		total-=BLOWFISH_BLOCKSIZE;
	}

	/* for all remaining data bytes use CFB8 */

	while(total--)
	{
		/* encrypt buffer */

		anchor->encrypt(bfr,anchor->local);

		/* XOR data byte with buffer */

		*data^=*bfr;

		/* shift iv left and append cipher byte and do whitening */

		for(i=0,ptr=bfr+1;i<BLOWFISH_BLOCKSIZE-1;i++)bfr[i]=*ptr++;
		bfr[i]=*data;
		*data++^=anchor->white[anchor->wwhite++];
	}

	/* debug message */

	LEAVE("encrypt_cfb_64_8");
}

#endif

/*
 * procedure decrypt_cfb_64_8
 *
 * input: total    - the amount of data elements to be processed
 *	  anchor   - pointer to vpnd global data
 *
 * inout: data     - the data to be processed
 *
 * This procedure decodes n sets of data bytes using
 * 8 and 64 bit cipher feedback (CFB) mode.
 */

#ifndef CRYPTOX86

void decrypt_cfb_64_8(WORD08 *data,WORD32 total,VPN *anchor)
{
	register WORD32 i;		/* general purpose usage	*/
	register WORD08 *ptr;		/* buffer fast access		*/
	register WORD08 mem;		/* general purpose memory	*/
	WORD08 bfr[BLOWFISH_BLOCKSIZE];	/* encryption buffer		*/


	/* debug message */

	ENTER("decrypt_cfb_64_8");

	/* copy iv to buffer */

	for(i=0,ptr=bfr;i<BLOWFISH_BLOCKSIZE;i++)*ptr++=anchor->remote_iv[i];

	/* increase iv */

	for(mem=1,i=0,ptr=anchor->remote_iv;mem&&i<BLOWFISH_BLOCKSIZE;i++)
	{
		*ptr+=mem;
		mem=*ptr++?0:1;
	}

	/* for all large blocks use CFB64 */

	while(total>=BLOWFISH_BLOCKSIZE)
	{
		/* encrypt buffer */

		anchor->encrypt(bfr,anchor->remote);

		/* unwhite, memorize new iv and then XOR data bytes 
		   with buffer */

		for(i=0,ptr=bfr;i<BLOWFISH_BLOCKSIZE;i++)
		{
			*data^=anchor->white[anchor->rwhite++];
			mem=*data;
			*data++^=*ptr;
			*ptr++=mem;
		}

		/* adjust data count */

		total-=BLOWFISH_BLOCKSIZE;
	}

	/* for all remaining data bytes use CFB8 */

	while(total--)
	{
		/* encrypt buffer */

		anchor->encrypt(bfr,anchor->remote);

		/* unwhite, memorize and then XOR data byte with buffer */

		*data^=anchor->white[anchor->rwhite++];
		mem=*data;
		*data++^=*bfr;

		/* shift iv left and append cipher MSB */

		for(i=0,ptr=bfr+1;i<BLOWFISH_BLOCKSIZE-1;i++)bfr[i]=*ptr++;
		bfr[i]=mem;
	}

	/* debug message */

	LEAVE("decrypt_cfb_64_8");
}

#endif

/*
 * datasend
 *
 * input:  length - the length of the data to be sent
 *	   anchor - pointer to vpnd global data
 *
 * output: data - the data to be sent
 *
 * return: 1 - success
 *	   0 - failure
 *
 * This procedure sends data encypted to the remote side.
 * It wraps the data with integrity checking information.
 * A maximum of BLOCKSIZE bytes can be sent. If the length of
 * the data to be sent is 0, new key exchange is done.
 */

#ifndef CRYPTOX86

int datasend(register WORD08 *data,register WORD32 length,VPN *anchor)
{
	register WORD32 i;			/* counter/index	*/
	WORD08 bfr[22+MAXBLOCK+(MAXBLOCK/2)];	/* work buffer		*/
	WORD08 *hdr=bfr+20-anchor->mac;		/* data header		*/
	WORD08 *tmp=bfr+22;			/* compression/data bfr	*/
						/* new key and iv memory*/
#ifdef COMPRESS
	WORD32 tmplen;				/* used for compression	*/
#endif


	/* debug message */

	ENTER("datasend");

	/* key change requested */

	if(!length)
	{
		/* error if in plaintext mode */

		if(!anchor->cipher)RET("key change in plaintext mode",0);

		/* debug message */

		DEBUG1(1,"key change\n");

		/* create length header */

		hdr[anchor->mac]=hdr[anchor->mac+1]=0;

		/* server */

		if(!anchor->mode)
		{
			/* get 80 random bytes */

			if(!getrandom(tmp,BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY,
				anchor))GOTO("getrandom",keyerr);

			/* encrypt 80 bytes in ECB mode (you never know) */

			for(i=0;i<BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY;
				i+=BLOWFISH_BLOCKSIZE)
				blowfish_encrypt(tmp+i,anchor->local);

			/* create checksum header */

			(anchor->macproc)(tmp,BLOWFISH_BLOCKSIZE
					+BLOWFISH_MAXKEY,
					hdr,(void *)(&(anchor->hmkey)));

			/* memorize new key and iv */

			for(i=0;i<BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY;i++)
			    tmp[i+BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY]=tmp[i];

			/* encrypt header */

			encrypt_cfb_64_8(hdr,anchor->mac+2,anchor);

			/* encrypt data */

			encrypt_cfb_64_8(tmp,
				BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY,anchor);

			/* send data to peer, handle errors */

			if(!dosend(hdr,BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY+
				anchor->mac+2,
				anchor))GOTO("dosend key",keyerr);

			/* toggle local key schedule */

			if(anchor->local==anchor->u1.sched)
				anchor->local=anchor->u2.sched;
			else anchor->local=anchor->u1.sched;

			/* generate schedule from new key */

			blowfish_key_schedule(tmp+BLOWFISH_BLOCKSIZE
				+BLOWFISH_MAXKEY+BLOWFISH_BLOCKSIZE,
				anchor->keysize,anchor->local);
		}

		/* client */

		else
		{
			/* get 8 random bytes */

			if(!getrandom(tmp,BLOWFISH_BLOCKSIZE,anchor))
				GOTO("getrandom",keyerr);

			/* encrypt 8 bytes in ECB mode (you never know) */

			blowfish_encrypt(tmp,anchor->local);

			/* create checksum header */

			(anchor->macproc)(tmp,BLOWFISH_BLOCKSIZE,hdr,
					(void *)(&(anchor->hmkey)));

			/* memorize new iv */

			for(i=0;i<BLOWFISH_BLOCKSIZE;i++)
			    tmp[i+BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY]=tmp[i];

			/* encrypt header */

			encrypt_cfb_64_8(hdr,anchor->mac+2,anchor);

			/* encrypt data */

			encrypt_cfb_64_8(tmp,BLOWFISH_BLOCKSIZE,anchor);

			/* send data to peer, handle errors */

			if(!dosend(hdr,BLOWFISH_BLOCKSIZE+anchor->mac+2,anchor))
				GOTO("dosend key",keyerr);

			/* from now on use key schedule of key received
			   from server */

			anchor->local=anchor->remote;
		}

		/* setup new iv */

		for(i=0;i<BLOWFISH_BLOCKSIZE;i++)
		  anchor->local_iv[i]=tmp[i+BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY];

		/* clean up */

		for(i=0;i<BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY;i++)
			tmp[i]=tmp[i+BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY]=0;

		/* debug message */

		LEAVE("datasend");

		/* signal success */

		return 1;

		/* error handling */

keyerr:		for(i=0;i<BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY;i++)
			tmp[i]=tmp[i+BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY]=0;
		return 0;
	}

	/* signal error if maximum length exceeded */

	if(length>MAXBLOCK)RET("illegal length",0);

	/* debug only */

#ifdef DEBUG
	if(debug>5)
	{
		int l;
		printf("data: ");
		for(l=0;l<length;l++)printf("%02x ",data[l]);
		printf("\n");
	}
#endif

#ifdef COMPRESS

	/* set up size of compression buffer */

	tmplen=sizeof(tmp);

	/* no compression if peer doesn't support it or
	   if disabled by configuration */

	if(anchor->peercomp||anchor->nocompress)tmplen=0;

	/* no compression if data is anyway too small */

	else if(length<anchor->threshold)tmplen=0;

	/* if compression fails signal no compression */

	else if(compress2(tmp,&tmplen,data,length,9))
	{
		DEBUG1(3,"compress err\n");
		tmplen=0;
	}

	/* if compressed result >= original size signal no compression */

	else if(tmplen>=length)tmplen=0;

	/* in case of compression */

	if(tmplen)
	{
		/* debug message */

		DEBUG1(3,"compressed data\n");

		/* change data length to compressed data length */

		length=tmplen;

		/* create length header with MSB set */

		hdr[anchor->mac]=(WORD08)(length);
		hdr[anchor->mac+1]=((WORD08)(length>>8))|0x80;
	}

	/* in case of no compression */

	else
	{
		/* copy data to buffer */

		memcpy(tmp,data,length);

		/* create regular length header */

		hdr[anchor->mac]=(WORD08)(length);
		hdr[anchor->mac+1]=(WORD08)(length>>8);
	}
#else
	/* copy data to buffer */

	memcpy(tmp,data,length);

	/* create length header */

	hdr[anchor->mac]=(WORD08)(length);
	hdr[anchor->mac+1]=(WORD08)(length>>8);
#endif

	/* create checksum header */

	(anchor->macproc)(tmp,length,hdr,(void *)(&(anchor->hmkey)));

	/* debug message (only the first four bytes of checksum for hmac) */

	DEBUG3(4,"length=%04lx chksum=%04lx\n",
		((WORD32)(hdr[anchor->mac+1])<<8)|(WORD32)(hdr[anchor->mac]),
		(anchor->hmacmode?((WORD32)(hdr[3])<<24)|
		(WORD32)(hdr[2]<<16):0)|
		((WORD32)(hdr[1])<<8)|(WORD32)(hdr[0]));

	/* encrypt header if not in plaintext mode */

	encrypt_cfb_64_8(hdr,anchor->mac+2,anchor);

	/* encrypt data if not in plaintext mode */

	encrypt_cfb_64_8(tmp,length,anchor);

	/* send header and data to peer, handle errors */

	if(!dosend(hdr,length+anchor->mac+2,anchor))GOTO("dosend data",keyerr);

	/* debug message */

	LEAVE("datasend");

	/* signal success */

	return 1;
}

#endif

/*
 * datarecv
 *
 * input:  max    - the length of the data buffer
 *	   anchor - pointer to vpnd global data
 *
 * output: data - the data to be received
 *
 * return: >0 - success, the amount of data received + 1
 *	   0  - failure
 *
 * This procedure receives encypted data from the remote side.
 * It performs basic data integrity checking. Furthermore
 * it performes special handling when a new key is received,
 * i.e. it sets up the new key schedule and iv from the
 * received key.
 */

#ifndef CRYPTOX86

int datarecv(register WORD08 *data,WORD32 max,VPN *anchor)
{
	WORD08 hdr[22];				/* data header		*/
	WORD08 sum[20];				/* calculated checksum	*/
	register WORD32 i;			/* counter/index	*/
	register WORD32 length;			/* data length		*/
#ifdef COMPRESS
	WORD32 tmplen=max;			/* used for compression	*/
	WORD08 tmp[MAXBLOCK+(MAXBLOCK/2)];	/* compression buffer	*/
						/* new key and iv 	*/
	WORD08 *newkey=tmp;
#else
						/* new key and iv 	*/
	WORD08 newkey[BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY];
#endif


	/* debug message */

	ENTER("datarecv");

	/* read header from peer, signal errors */

	if(!dorecv(hdr,anchor->mac+2,anchor))RET("dorecv hdr",0);

	/* decrypt received data if not in plaintext mode */

	decrypt_cfb_64_8(hdr,anchor->mac+2,anchor);

	/* debug message (only first four bytes for hmac) */

	DEBUG3(4,"length=%04lx chksum=%04lx\n",
		((WORD32)(hdr[anchor->mac+1])<<8)|(WORD32)(hdr[anchor->mac]),
		(anchor->hmacmode?((WORD32)(hdr[3])<<24)|(WORD32)(hdr[2]<<16):0)|
		((WORD32)(hdr[1])<<8)|(WORD32)(hdr[0]));

	/* new key from peer */

	if(!hdr[anchor->mac]&&!hdr[anchor->mac+1])
	{
		/* illegal, if in plaintext mode */

		if(!anchor->cipher)RET("key in plaintext mode",0);

		/* debug message */

		DEBUG1(1,"new key from peer\n");

		/* server */

		if(!anchor->mode)
		{
			/* read expected amount of data,signal errors */

			if(!dorecv(newkey,BLOWFISH_BLOCKSIZE,anchor))
				GOTO("dorecv newkey",keyerr);

			/* decrypt received data */

			decrypt_cfb_64_8(newkey,BLOWFISH_BLOCKSIZE,anchor);

			/* calculate checksum of read data */

			(anchor->macproc)(newkey,BLOWFISH_BLOCKSIZE,sum,
				(void *)(&(anchor->hmkey)));

			/* debug message */

			DEBUG3(4,"length=%04lx chksum=%04lx\n",
				((WORD32)(hdr[anchor->mac+1])<<8)|
					(WORD32)(hdr[anchor->mac]),
				(anchor->hmacmode?((WORD32)(hdr[3])<<24)|
					(WORD32)(hdr[2]<<16):0)|
					((WORD32)(hdr[1])<<8)|(WORD32)(hdr[0]));

			/* in case of checksum mismatch signal error */

			for(i=0;i<anchor->mac;i++)if(hdr[i]!=sum[i])
				GOTO("chksum error",keyerr);

			/* from now on use key schedule 
			   of shared new key */

			anchor->remote=anchor->local;
		}
		else
		{
			/* read expected amount of data,signal errors */

			if(!dorecv(newkey,BLOWFISH_BLOCKSIZE+
				BLOWFISH_MAXKEY,anchor))
				GOTO("dorecv newkey",keyerr);

			/* decrypt received data */

			decrypt_cfb_64_8(newkey,
				BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY,anchor);

			/* calculate checksum of read data */

			(anchor->macproc)(newkey,BLOWFISH_BLOCKSIZE+
				BLOWFISH_MAXKEY,sum,(void *)(&(anchor->hmkey)));

			/* debug message */

			DEBUG3(4,"length=%04lx chksum=%04lx\n",
				((WORD32)(hdr[anchor->mac+1])<<8)|
					(WORD32)(hdr[anchor->mac]),
				(anchor->hmacmode?((WORD32)(hdr[3])<<24)|
					(WORD32)(hdr[2])<<16:0)|
					((WORD32)(hdr[1])<<8)|(WORD32)(hdr[0]));

			/* in case of checksum mismatch signal error */

			for(i=0;i<anchor->mac;i++)if(hdr[i]!=sum[i])
				GOTO("chksum error",keyerr);

			/* toggle remote key schedule */

			if(anchor->remote==anchor->u1.sched)
				anchor->remote=anchor->u2.sched;
			else anchor->remote=anchor->u1.sched;

			/* generate schedule from new key */

			blowfish_key_schedule(newkey+BLOWFISH_BLOCKSIZE,
				anchor->keysize,anchor->remote);
		}

		/* setup new iv */

		for(i=0;i<BLOWFISH_BLOCKSIZE;i++)anchor->remote_iv[i]=newkey[i];

		/* clean up */

		for(i=0;i<BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY;i++)
			newkey[i]=0;

		/* debug message */

		LEAVE("datarecv");

		/* signal success */

		return 1;

		/* error handling */

keyerr:		for(i=0;i<BLOWFISH_BLOCKSIZE+BLOWFISH_MAXKEY;i++)
			newkey[i]=0;
		return 0;
	}

	/* restore data length */

#ifdef COMPRESS
	length=(((WORD32)((hdr[anchor->mac+1]&0x7f)))<<8)|((WORD32)(hdr[anchor->mac]));
#else
	length=(((WORD32)(hdr[anchor->mac+1]))<<8)|((WORD32)(hdr[anchor->mac]));
#endif
	/* signal error if length exceeds buffer size */

	if(length>max)RET("illegal length",0);

#ifdef COMPRESS

	/* in case of compressed data */

	if(hdr[anchor->mac+1]&0x80)
	{
		/* debug message */

		DEBUG1(3,"compressed data\n");

		/* read expected amount of data, signal errors */

		if(!dorecv(tmp,length,anchor))RET("dorecv data",0);

		/* decrypt received data if not in plaintext mode */

		decrypt_cfb_64_8(tmp,length,anchor);

		/* calculate checksum of read data */

		(anchor->macproc)(tmp,length,sum,
			(void *)(&(anchor->hmkey)));

		/* decompress data, signal errors */

		if(uncompress(data,&tmplen,tmp,length))
			RET("uncompress err",0);

		/* set up actual data length */

		length=tmplen;
	}

	/* in case of uncompressed data */

	else
	{
		/* read expected amount of data, signal errors */

		if(!dorecv(data,length,anchor))RET("dorecv data",0);

		/* decrypt received data if not in plaintext mode */

		decrypt_cfb_64_8(data,length,anchor);

		/* calculate checksum of read data */

		(anchor->macproc)(data,length,sum,
			(void *)(&(anchor->hmkey)));
	}
#else
	/* read expected amount of data, signal errors */

	if(!dorecv(data,length,anchor))RET("dorecv data",0);

	/* decrypt received data if not in plaintext mode */

	decrypt_cfb_64_8(data,length,anchor);

	/* calculate checksum of read data */

	(anchor->macproc)(data,length,sum,(void *)(&(anchor->hmkey)));
#endif
	/* debug message */

#ifdef DEBUG
	if(debug>5)
	{
		int l;
		printf("data: ");
		for(l=0;l<length;l++)printf("%02x ",data[l]);
		printf("\n");
	}
	if(debug>3)
	{
		int l;
		printf("chksum=");
		for(l=0;l<anchor->mac;l++)printf("%02x",hdr[l]);
		printf(" expected=");
		for(l=0;l<anchor->mac;l++)printf("%02x",sum[l]);
		printf("\n");
	}
#endif

	/* in case of checksum mismatch signal error */

	for(i=0;i<anchor->mac;i++)if(hdr[i]!=sum[i])RET("chksum error",0);

	/* debug message */

	LEAVE("datarecv");

	/* signal success returning data length+1 */

	return length+1;
}

#endif

/*
 * encrypt_cbc
 *
 * input:  schedule - the key schedule
 *	   len      - the length of the data to be encrypted
 *	   iv       - the initialization vector
 *
 * inout:  data     - the data to be encrypted (the buffer this variable
 *	              points to must have a size that is a multiple of 8)
 *
 * This procedure encrypts data in CBC (Cipher Block Chaining) mode.
 */

#ifndef CRYPTOX86

void encrypt_cbc(register WORD08 *data,WORD32 len,WORD32 *schedule,WORD08 *iv)
{
	register WORD32 i;	/* counter/index	*/
	register WORD32 j;	/* counter/index	*/
	register WORD08 *ptr;	/* next iv		*/


	/* encrypt block 1 */

	for(i=len>>3,j=0,ptr=data;j<8;j++)*data++^=iv[j];
	blowfish_encrypt(ptr,schedule);

	/* encrypt block 2 to n */

	for(i--;i;i--)
	{
		for(j=0;j<8;j++)*data++^=*ptr++;
		blowfish_encrypt(ptr,schedule);
	}
}

#endif

/*
 * decrypt_cbc
 *
 * input:  schedule - the key schedule
 *	   len      - the length of the data to be decrypted
 *	   iv       - the initialization vector
 *
 * inout:  data     - the data to be decrypted (the buffer this variable
 *	              points to must have a size that is a multiple of 8)
 *
 * This procedure decrypts data in CBC (Cipher Block Chaining) mode.
 */

#ifndef CRYPTOX86

void decrypt_cbc(register WORD08 *data,WORD32 len,WORD32 *schedule,WORD08 *iv)
{
	register WORD32 i;	/* counter/index	*/
	register WORD32 j;	/* counter/index	*/
	register WORD08 *m;	/* buffer fast access	*/
	WORD08 bfr1[8];		/* next iv buffer	*/
	WORD08 bfr2[8];		/* next iv buffer	*/


	/* copy iv to next iv buffer */

	memcpy(bfr2,iv,8);

	/* for all blocks minus one */

	for(i=len>>3;i>1;i--)
	{
		/* decrypt block and get next iv */

		memcpy(bfr1,bfr2,8);
		memcpy(bfr2,data,8);
		blowfish_decrypt(data,schedule);
		for(j=0,m=bfr1;j<8;j++)*data++^=*m++;
	}

	/* decrypt last block */

	blowfish_decrypt(data,schedule);
	for(j=0,m=bfr2;j<8;j++)*data++^=*m++;
}

#endif

/*
 * genwhite
 *
 * input: src    - a pointer to the master secret
 *	  anchor - pointer to vpnd global data
 *
 * This procedure prepares the whitening buffer.
 */

#ifndef CRYPTOX86

void genwhite(WORD08 *src,VPN *anchor)
{
	int i;				/* counter/index	*/
	BLOWFISH_SCHEDULE(sched);	/* key schedule		*/


	/* generate key schedule from master key */

	blowfish_key_schedule(src,anchor->keysize,sched);

	/* preset the whitening buffer with well known values */

	for(i=0;i<256;i++)anchor->white[i]=(WORD08)(i);

	/* encrypt the buffer four times */

	for(i=0;i<256;i+=BLOWFISH_BLOCKSIZE)
	{
		blowfish_encrypt(anchor->white+i,sched);
		blowfish_encrypt(anchor->white+i,sched);
		blowfish_encrypt(anchor->white+i,sched);
		blowfish_encrypt(anchor->white+i,sched);
	}

	/* clean up */

	for(i=0;i<BLOWFISH_SCHEDULESIZE;i++)sched[i]=0;
}

#endif

/*
 * readmaster
 *
 * input: anchor - pointer to vpnd global data
 *
 * return: 1 in case of success, else 0
 *
 * This procedure reads the master secret from file to memory.
 */

int readmaster(VPN *anchor)
{
	int f;		/* file handle		*/
	int i;		/* counter/index	*/


	/* debug message */

	ENTER("readmaster");

	/* open key file, handle errors */

	if((f=open(anchor->keyfile,O_RDONLY))==-1)RETURN("open",0);

	/* read key file, handle errors */

	i=anchor->extended?XKEYSIZE:BLOWFISH_MAXKEY;
	if(read(f,anchor->secret,i)!=i)
	{
		ERRNO("read");
		close(f);
		return 0;
	}

	/* close key file, handle errors */

	if(close(f))
	{
		for(i=0;i<XKEYSIZE;i++)anchor->secret[i]=0;
		RETURN("close",0);
	}

	/* debug message */

	LEAVE("readmaster");

	/* signal success */

	return 1;
}

/*
 * writemaster
 *
 * input:  sync   - 0 = asynchronous write, 1 = synchronous write
 *	   anchor - pointer to vpnd global data
 *
 * return: 1 in case of success, else 0
 *
 * This procedure saves the master secret from memory to file.
 */

int writemaster(int sync,VPN *anchor)
{
	int f;		/* file handle		*/
	int i;		/* counter/index	*/


	/* debug message */

	ENTER("writemaster");

	/* open key file, handle errors */

	if((f=open(anchor->keyfile,
		sync?(O_WRONLY|O_CREAT|O_EXCL|O_SYNC):O_WRONLY,0400))==-1)
		RETURN("open",0);

	/* write key file, handle errors: note that so synchronized
	   access or fsync() would be required here we anyway write
	   only one physical block and the protocol is a bit crash
	   resistant, so we ignore safety here for the sake of speed */

	i=anchor->extended?XKEYSIZE:BLOWFISH_MAXKEY;
	if(write(f,anchor->secret,i)!=i)
	{
		ERRNO("write");
		if(sync)unlink(anchor->keyfile);
		close(f);
		return 0;
	}

	/* close key file, handle errors */

	if(close(f))
	{
		ERRNO("close");
		if(sync)unlink(anchor->keyfile);
		return 0;
	}

	/* debug message */

	LEAVE("writemaster");

	/* signal success */

	return 1;
}

/*
 * genmaster
 *
 * input:  anchor - pointer to vpnd global data
 *
 * return: 0 - success
 *	   1 - user/group not root
 *	   2 - getting random data failed
 *	   3 - changing working directory to target directory failed
 *	   4 - writing (local or remote) master key file failed
 *
 * This procedure creates a new master key and stores it
 * to the given file(s) in basic or extended key file format.
 */

int genmaster(VPN *anchor)
{
	time_t t;			/* current time			*/
	clock_t uptime;			/* uptime in clock ticks	*/
	struct tms unused;		/* required but unused		*/
	int i;				/* counter/index		*/
	WORD08 *rkey;			/* remote master key file	*/
	WORD08 *secret;			/* local master key file	*/
	WORD08 *master0;		/* extended mode secret 0	*/
	WORD08 *master1;		/* extended mode secret 1	*/
	WORD32 *sched;			/* key schedule buffer		*/
	WORD08 *ptr;			/* general purpose pointer	*/


	/* debug message */

	ENTER("genmaster");

	/* set up buffer pointers */

	rkey=(WORD08 *)(anchor->u2.bfr);
	sched=anchor->u1.sched;
	secret=anchor->secret;
	master0=secret+MASTER0;
	master1=secret+MASTER1;
	ptr=anchor->extended?master0:secret;

	/* reset memory */

	memset(secret,0,XKEYSIZE);
	memset(rkey,0,XKEYSIZE);

	/* get current time */

	t=time(NULL);

	/* check thet real user/group id is root */

	if(getuid()||getgid())
	{
		printf("You must be root and group root to create a key\n");
		return 1;
	}

	/* get 72 random bytes */

	if(!getrandom(ptr,BLOWFISH_MAXKEY,anchor))goto rnderr;

	/* generate key schedule from random data */

	blowfish_key_schedule(ptr,BLOWFISH_MAXKEY,sched);

	/* get another 72 random bytes */

	if(!getrandom(master1,BLOWFISH_MAXKEY,anchor))goto rnderr;

	/* encrypt 72 bytes in ECB mode */

	for(i=0;i<BLOWFISH_MAXKEY;i+=BLOWFISH_BLOCKSIZE)
		blowfish_encrypt(master1+i,sched);

	/* generate key schedule from encrypted data */

	blowfish_key_schedule(master1,BLOWFISH_MAXKEY,sched);

	/* encrypt 72 bytes in ECB mode */

	for(i=0;i<BLOWFISH_MAXKEY;i+=BLOWFISH_BLOCKSIZE)
		blowfish_encrypt(ptr+i,sched);

	/* make new key schedule from time */

	master1[0]=(WORD08)(t);
	master1[1]=(WORD08)(t>>8);
	master1[2]=(WORD08)(t>>16);
	master1[3]=(WORD08)(t>>24);

	/* generate key schedule from time and previous data */

	blowfish_key_schedule(master1,BLOWFISH_MAXKEY,sched);

	/* encrypt 72 bytes in ECB mode */

	for(i=0;i<BLOWFISH_MAXKEY;i+=BLOWFISH_BLOCKSIZE)
		blowfish_encrypt(ptr+i,sched);

	/* only in case of extended master key file format */

	if(anchor->extended)
	{
		/* copy local key data to remote key data */

		memcpy(rkey,secret,XKEYSIZE);

		/* get local iv */

		if(!getrandom(rkey+IV0,BLOWFISH_BLOCKSIZE,anchor))goto rnderr;

		/* get remote iv */

		if(!getrandom(secret+IV0,BLOWFISH_BLOCKSIZE,anchor))goto rnderr;

		/* duplicate ivs */

		memcpy(secret+IV1,secret+IV0,BLOWFISH_BLOCKSIZE);
		memcpy(rkey+IV1,rkey+IV0,BLOWFISH_BLOCKSIZE);

		/* generate key schedule from default password */

		blowfish_key_schedule("vpnd",4,sched);

		/* encrypt local master secret in CBC mode and duplicate it */

		encrypt_cbc(master0,BLOWFISH_MAXKEY,sched,rkey+IV0);
		memcpy(master1,master0,BLOWFISH_MAXKEY);

		/* encrypt remote master secret in CBC mode and duplicate it */

		encrypt_cbc(rkey+MASTER0,BLOWFISH_MAXKEY,sched,secret+IV0);
		memcpy(rkey+MASTER1,rkey+MASTER0,BLOWFISH_MAXKEY);

		/* get system uptime in clock ticks */

		uptime=times(&unused);

		/* generate first four bytes of key identifier from uptime */

		ptr=secret+KEYID;
		ptr[0]=(WORD08)(uptime);
		ptr[1]=(WORD08)(uptime>>8);
		ptr[2]=(WORD08)(uptime>>16);
		ptr[3]=(WORD08)(uptime>>24);

		/* get remaining key identifier */

		if(!getrandom(ptr+4,20,anchor))
		{
rnderr:			printf("Failed to gather random data\n");
			return 2;
		}

		/* duplicate key identifier */

		memcpy(rkey+KEYID,ptr,24);

		/* change working directory to target directory */

		if(chdir(anchor->keyfile))
		{
			ERRNO("chdir");
			printf("%s: no such directory\n",anchor->keyfile);
			return 3;
		}

		/* write local key to file, handle errors */

		anchor->keyfile=anchor->lkeyfile;
		if(!writemaster(1,anchor))goto crterr;

		/* print message */

		printf("New key file %s/%s created.\n",getcwd(NULL,0),
			anchor->keyfile);

		/* prepare writing of remote key to file */

		memcpy(secret,rkey,XKEYSIZE);
		anchor->keyfile=anchor->rkeyfile;
	}

	/* write key to file, handle errors */

	if(!writemaster(1,anchor))
	{
crterr:		printf("Can't create %s\n",anchor->keyfile);
		return 4;
	}

	/* reset memory */

	memset(rkey,0,XKEYSIZE);

	/* print message */

	printf("New key file %s/%s created.\n",getcwd(NULL,0),anchor->keyfile);

	/* debug message */

	LEAVE("genmaster");

	/* signal success */

	return 0;
}

/*
 * cryptosetup
 *
 * input:  anchor - pointer to vpnd global data
 *
 * return: 0 - success
 *	   1 - send/receive of new key and server iv failed
 *	   2 - send/receive of iv failed
 *	   3 - master key file read error
 *	   4 - master key file version not supported
 *	   5 - extended information receive failed
 *	   6 - extended information send failed
 *	   7 - key id mismatch error
 *	   8 - transaction id mismatch
 *	   9 - sending of remote iv to peer failed
 *	  10 - receive of local iv from peer failed
 *	  11 - creating new local iv failed
 *	  12 - sending new local iv failed
 *	  13 - receiving new remote iv failed
 *	  14 - challenge send error
 *	  15 - challenge receive error
 *	  16 - challenge comparison error
 *	  17 - master key file save error
 *	  18 - peer doesn't support hmac
 *
 * This procedure establishes encrypted communication with
 * the remote system either in server or in client mode.
 */

int cryptosetup(VPN *anchor)
{
	int i;				/* counter/index		*/
	WORD32 src;			/* numeric of local ip		*/
	WORD32 dst;			/* numeric of remote ip		*/
	WORD08 xbfr[32];		/* extended mode data buffer	*/
	WORD32 id=0;			/* selected transaction id	*/
	WORD08 *secret;			/* shared secret memory area	*/
	WORD08 *master;			/* current master secret	*/
	WORD08 *iv;			/* current master secret iv	*/


	/* debug message */

	ENTER("cryptosetup");

	/* set up pointers */

	master=secret=anchor->secret;

	/* only in extended mode */

	if(anchor->extended)
	{
		/* read master key file, handle errors */

		if(!readmaster(anchor))RET("readmaster",3);

		/* check vmaster key file version number */

		if(secret[XKEYVER])RET("version error",4);

		/* in client mode */

		if(anchor->mode)
		{
			/* copy local transaction id to transfer buffer */

			xbfr[0]=secret[TRANSACTID];

			/* reset capabilities bytes of transfer buffer */

			memset(xbfr+1,0,7);

			/* set compression request bit if compiled in
			   and possible */
#ifdef COMPRESS
			if(!anchor->nocompress)xbfr[7]|=1;
#endif
			/* set up hmac capabilities */

			xbfr[7]|=2|((WORD08)((anchor->hmacmode)<<2));

			/* copy key id to transfer buffer */

			memcpy(xbfr+8,secret+KEYID,24);

			/* send buffer to peer, handle errors */

			if(!dosend(xbfr,32,anchor))RET("dosend",6);

			/* read data from peer, handle errors */

			if(!dorecv(xbfr,8,anchor))RET("dorecv",5);
		}

		/* in server mode */

		else
		{
			/* read data from peer, handle errors */

			if(!dorecv(xbfr,32,anchor))RET("dorecv",5);

			/* check key id, handle errors */

			if(memcmp(xbfr+8,secret+KEYID,24))
				RET("key id mismatch",7);

			/* copy local transaction id to transfer buffer */

			xbfr[8]=secret[TRANSACTID];

			/* reset capabilities bytes of transfer buffer */

			memset(xbfr+9,0,7);

			/* set compression request bit if compiled in
			   and possible */
#ifdef COMPRESS
			if(!anchor->nocompress)xbfr[15]|=1;
#endif
			/* set up hmac capabilities */

			xbfr[15]|=2|((WORD08)((anchor->hmacmode)<<2));

			/* send buffer to peer, handle errors */

			if(!dosend(xbfr+8,8,anchor))RET("dosend",6);
		}

		/* process transaction id */

		if(*xbfr==secret[TRANSACTID])id=*xbfr;
		else if(*xbfr+1==secret[TRANSACTID])
		{
			logmsg(FALLBACK,0,anchor);
			id=*xbfr;
		}
		else if(*xbfr==secret[TRANSACTID]+1)
		{
			logmsg(FALLBACK,0,anchor);
			id=secret[TRANSACTID];
		}
		else RET("transaction id error",8);

		/* set up pointers */

		iv=secret+((id&1)?IV1:IV0);
		master=secret+((id&1)?MASTER1:MASTER0);

		/* disable compression if peer doesn't support it */
#ifdef COMPRESS
		if(xbfr[7]&1)anchor->peercomp=0;
		else anchor->peercomp=1;
#endif

		/* handle hmac options */

		if(!(xbfr[7]&2))
		{
			if(anchor->hmacopt==2)
			{
				RET("peer doesn't support hmac",18);
			}
			else i=0;
		}
		else
		{
			i=(xbfr[7]&0xc)>>2;
			i=(i>=anchor->hmacmode?i:anchor->hmacmode);
		}

		anchor->mackey=macinit[i];
		anchor->macproc=mactab[i];
		anchor->mac=maclen[i];

		/* send remote iv to peer, handle errors */

		if(!dosend(iv,BLOWFISH_BLOCKSIZE,anchor))RET("dosend",9);

		/* create key schedule from default password */

		blowfish_key_schedule("vpnd",4,anchor->xsched);

		/* receive local iv from peer, handle errors */

		if(!dorecv(xbfr,BLOWFISH_BLOCKSIZE,anchor))RET("dorecv",10);

		/* decrypt both master secrets with user password
		   if there is any */

		if(anchor->passwd)decrypt_cbc(secret+MASTER0,
			BLOWFISH_MAXKEY*2,anchor->u3.sched,anchor->piv);

		/* decrypt master secret */

		decrypt_cbc(master,BLOWFISH_MAXKEY,anchor->xsched,xbfr);

		/* fill whitening buffer */

		genwhite(master,anchor);
	}

	/* reset whitening indexes */

	anchor->rwhite=anchor->wwhite=0;

	/* convert local and remote ip to numbers */

	src=(WORD32)ntohl(getaddr(anchor->local_ip));
	dst=(WORD32)ntohl(getaddr(anchor->remote_ip));

	/* copy local and remote ips to initial key according to client
	   or server mode */

	if(anchor->mode)
	{
		for(i=0;i<4;i++)
		{
			anchor->local_iv[7-i]=anchor->remote_iv[7-i]=
				(WORD08)(src>>(i<<3));
			anchor->local_iv[3-i]=anchor->remote_iv[3-i]=
				(WORD08)(dst>>(i<<3));
		}
	}
	else
	{
		for(i=0;i<4;i++)
		{
			anchor->local_iv[3-i]=anchor->remote_iv[3-i]=
				(WORD08)(src>>(i<<3));
			anchor->local_iv[7-i]=anchor->remote_iv[7-i]=
				(WORD08)(dst>>(i<<3));
		}
	}

	/* setup pointers to key schedule */

	anchor->local=anchor->remote=anchor->u1.sched;

	/* create key schedule from master secret */

	blowfish_key_schedule(master,anchor->keysize,anchor->u1.sched);

	/* create preprocessed hmac key in hmac mode */

	if(anchor->mackey)(anchor->mackey)(master,BLOWFISH_MAXKEY,
		(void *)(&(anchor->hmkey)));

	/* in client mode */

	if(anchor->mode)
	{
		/* receive new key and server iv, signal error if this fails */

		if(datarecv(NULL,0,anchor)!=1)RET("datarecv",1);

		/* send new client iv, signal error in case of failure */

		if(!datasend(NULL,0,anchor))RET("datasend",2);
	}

	/* in server mode */

	else
	{
		/* send new key and server iv, signal error in case of error */

		if(!datasend(NULL,0,anchor))RET("datasend",1);

		/* receive client iv, signal error in case of failure */

		if(datarecv(NULL,0,anchor)!=1)RET("datarecv",2);
	}

	/* only in extended mode */

	if(anchor->extended)
	{
		/* copy decrypted master secret to other master secret */

		memcpy(secret+((id&1)?MASTER0:MASTER1),master,BLOWFISH_MAXKEY);

		/* encrypt current master secret again */

		encrypt_cbc(master,BLOWFISH_MAXKEY,anchor->xsched,xbfr);

		/* increment transaction count */

		id++;

		/* set up pointers */

		iv=secret+((id&1)?IV1:IV0);
		master=secret+((id&1)?MASTER1:MASTER0);

		/* get new random local iv */

		if(!getrandom(xbfr,BLOWFISH_BLOCKSIZE,anchor))
			RET("getrandom",11);

		/* encrypt current master secret */

		encrypt_cbc(master,BLOWFISH_MAXKEY,anchor->xsched,xbfr);

		/* encrypt both master secrets with user password
		   if there is any */

		if(anchor->passwd)encrypt_cbc(secret+MASTER0,
			BLOWFISH_MAXKEY*2,anchor->u3.sched,anchor->piv);

		/* save local iv */

		memcpy(xbfr+BLOWFISH_BLOCKSIZE,xbfr,BLOWFISH_BLOCKSIZE);

		/* send new local iv to peer, handle errors */

		if(!datasend(xbfr,BLOWFISH_BLOCKSIZE,anchor))RET("datasend",12);

		/* receive new remote iv from peer, handle errors */

		if(!datarecv(xbfr,BLOWFISH_BLOCKSIZE,anchor))RET("datarecv",13);

		/* store remote iv */

		memcpy(iv,xbfr,BLOWFISH_BLOCKSIZE);

		/* xor local and remote iv, keep copy of result */

		for(i=0;i<BLOWFISH_BLOCKSIZE;i++)
		{
			xbfr[i]^=xbfr[BLOWFISH_BLOCKSIZE+i];
			xbfr[BLOWFISH_BLOCKSIZE+i]=xbfr[i];
		}

		/* send result to peer, handle errors */

		if(!datasend(xbfr,BLOWFISH_BLOCKSIZE,anchor))RET("datasend",14);

		/* receive result of peer, handle errors */

		if(!datarecv(xbfr,BLOWFISH_BLOCKSIZE,anchor))RET("datarecv",15);

		/* compare stored and received result, handle errors */

		if(memcmp(xbfr+BLOWFISH_BLOCKSIZE,xbfr,BLOWFISH_BLOCKSIZE))
			RET("challenge comparison error",16);

		/* write new transaction id to master key buffer */

		secret[TRANSACTID]=(WORD08)(id);

		/* save master key file, handle errors */

		if(!writemaster(0,anchor))RET("writemaster",17);
	}

	/* debug message */

	LEAVE("cryptosetup");

	/* signal success */

	return 0;
}

/*
 * chgpasswd
 *
 * input:  anchor - pointer to vpnd global data
 *
 * return: 0 in case of success, 1 in case of error
 *
 * This procedure either sets, changes or removes the user password
 * of an extended master key file.
 */

int chgpasswd(VPN *anchor)
{
	char *pwd;	/* result of getpass()	*/
	char *oldpass;	/* old password memory	*/
	char *newpass;	/* new password memory	*/
	WORD08 *secret;	/* shared key area	*/
	WORD32 *sched;	/* key schedule		*/


	/* debug message */

	ENTER("chgpasswd");

	/* set up buffer pointers */

	oldpass=anchor->u1.bfr;
	newpass=anchor->u2.bfr;
	secret=anchor->secret;
	sched=anchor->u3.sched;

	/* read master key from file, handle errors */

	if(!readmaster(anchor))goto fileerr;

	/* read and verify old user password, handle errors */

	pwd=getpass("Enter old password:  ");
	strcpy(oldpass,pwd);
	memset(pwd,0,strlen(pwd));
	pwd=getpass("Repeat old password: ");
	if(strcmp(oldpass,pwd))goto pwderr;

	/* reset buffer of getpass() function */

	memset(pwd,0,strlen(pwd));

	/* read and verify new user password, handle errors */

	pwd=getpass("Enter new password:  ");
	strcpy(newpass,pwd);
	memset(pwd,0,strlen(pwd));
	pwd=getpass("Repeat new password: ");
	if(strcmp(newpass,pwd))
	{
pwderr:		printf("Passwords don't match\n");
		memset(secret,0,XKEYSIZE);
		return 1;
	}

	/* reset buffer of getpass() function */

	memset(pwd,0,strlen(pwd));

	/* if an old password is given */

	if(*oldpass)
	{
		/* decrypt master secrets with old password */

		blowfish_key_schedule(oldpass,strlen(oldpass),sched);
		memset(oldpass,0,128);
		decrypt_cbc(secret+MASTER0,BLOWFISH_MAXKEY*2,sched,anchor->piv);
		memset(sched,0,BLOWFISH_SCHEDULESIZE);
	}

	/* if a new password is given */

	if(*newpass)
	{
		/* encrypt master secrets with new password */

		blowfish_key_schedule(newpass,strlen(newpass),sched);
		memset(newpass,0,128);
		encrypt_cbc(secret+MASTER0,BLOWFISH_MAXKEY*2,sched,anchor->piv);
		memset(sched,0,BLOWFISH_SCHEDULESIZE);
	}

	/* write master key back to file, handle errors */

	if(!writemaster(0,anchor))
	{
fileerr:	printf("Can't access %s\n",anchor->keyfile);
		memset(secret,0,XKEYSIZE);
		return 1;
	}

	/* reset master key file memory */

	memset(secret,0,XKEYSIZE);

	/* print message */

	printf("User password of %s modified\n",anchor->keyfile);

	/* debug message */

	LEAVE("chgpasswd");

	/* signal success */

	return 0;
}
