/*
 * virtual private network daemon (vpnd)
 *
 * (c) 2005 Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* reduce to the min: common data packet format                               */
/*============================================================================*/

/* crc table required to generate ppp checksums */

static WORD16 crc16[256]=
{
	0x0000,0x1189,0x2312,0x329b,0x4624,0x57ad,0x6536,0x74bf,
	0x8c48,0x9dc1,0xaf5a,0xbed3,0xca6c,0xdbe5,0xe97e,0xf8f7,
	0x1081,0x0108,0x3393,0x221a,0x56a5,0x472c,0x75b7,0x643e,
	0x9cc9,0x8d40,0xbfdb,0xae52,0xdaed,0xcb64,0xf9ff,0xe876,
	0x2102,0x308b,0x0210,0x1399,0x6726,0x76af,0x4434,0x55bd,
	0xad4a,0xbcc3,0x8e58,0x9fd1,0xeb6e,0xfae7,0xc87c,0xd9f5,
	0x3183,0x200a,0x1291,0x0318,0x77a7,0x662e,0x54b5,0x453c,
	0xbdcb,0xac42,0x9ed9,0x8f50,0xfbef,0xea66,0xd8fd,0xc974,
	0x4204,0x538d,0x6116,0x709f,0x0420,0x15a9,0x2732,0x36bb,
	0xce4c,0xdfc5,0xed5e,0xfcd7,0x8868,0x99e1,0xab7a,0xbaf3,
	0x5285,0x430c,0x7197,0x601e,0x14a1,0x0528,0x37b3,0x263a,
	0xdecd,0xcf44,0xfddf,0xec56,0x98e9,0x8960,0xbbfb,0xaa72,
	0x6306,0x728f,0x4014,0x519d,0x2522,0x34ab,0x0630,0x17b9,
	0xef4e,0xfec7,0xcc5c,0xddd5,0xa96a,0xb8e3,0x8a78,0x9bf1,
	0x7387,0x620e,0x5095,0x411c,0x35a3,0x242a,0x16b1,0x0738,
	0xffcf,0xee46,0xdcdd,0xcd54,0xb9eb,0xa862,0x9af9,0x8b70,
	0x8408,0x9581,0xa71a,0xb693,0xc22c,0xd3a5,0xe13e,0xf0b7,
	0x0840,0x19c9,0x2b52,0x3adb,0x4e64,0x5fed,0x6d76,0x7cff,
	0x9489,0x8500,0xb79b,0xa612,0xd2ad,0xc324,0xf1bf,0xe036,
	0x18c1,0x0948,0x3bd3,0x2a5a,0x5ee5,0x4f6c,0x7df7,0x6c7e,
	0xa50a,0xb483,0x8618,0x9791,0xe32e,0xf2a7,0xc03c,0xd1b5,
	0x2942,0x38cb,0x0a50,0x1bd9,0x6f66,0x7eef,0x4c74,0x5dfd,
	0xb58b,0xa402,0x9699,0x8710,0xf3af,0xe226,0xd0bd,0xc134,
	0x39c3,0x284a,0x1ad1,0x0b58,0x7fe7,0x6e6e,0x5cf5,0x4d7c,
	0xc60c,0xd785,0xe51e,0xf497,0x8028,0x91a1,0xa33a,0xb2b3,
	0x4a44,0x5bcd,0x6956,0x78df,0x0c60,0x1de9,0x2f72,0x3efb,
	0xd68d,0xc704,0xf59f,0xe416,0x90a9,0x8120,0xb3bb,0xa232,
	0x5ac5,0x4b4c,0x79d7,0x685e,0x1ce1,0x0d68,0x3ff3,0x2e7a,
	0xe70e,0xf687,0xc41c,0xd595,0xa12a,0xb0a3,0x8238,0x93b1,
	0x6b46,0x7acf,0x4854,0x59dd,0x2d62,0x3ceb,0x0e70,0x1ff9,
	0xf78f,0xe606,0xd49d,0xc514,0xb1ab,0xa022,0x92b9,0x8330,
	0x7bc7,0x6a4e,0x58d5,0x495c,0x3de3,0x2c6a,0x1ef1,0x0f78
};

/*
 * encval
 *
 * input:  hdr - the storage area for the encoded value
 *	   val - the value to be encoded
 *
 * return: the amount of bytes required for the encoded value
 *
 * This procedure encodes a value according to rfc2508.
 */

static int encval(WORD08 *hdr,int val)
{
	/* values from -16384 to -129 */

	if(val<-128)
	{
		/* adjust value for encoding */

		val+=16384;

		/* write encoded value */

		hdr[0]=0xc0;
		hdr[1]=(WORD08)(val>>8);
		hdr[2]=(WORD08)(val);

		/* return amount of bytes required */

		return 3;
	}

	/* values from -128 to -1 */

	else if(val<0)
	{
		/* write adjusted encoded value */

		hdr[0]=0x80;
		hdr[1]=(WORD08)(val+128);

		/* return amount of bytes required */

		return 2;
	}

	/* values from 0 to 127 */

	else if(val<128)
	{
		/* write encoded value */

		hdr[0]=(WORD08)(val);

		/* return amount of bytes required */

		return 1;
	}

	/* values from 128 to 16383 */

	else if(val<16384)
	{
		/* write encoded value */

		hdr[0]=(WORD08)(val>>8)|0x80;
		hdr[1]=(WORD08)(val);

		/* return amount of bytes required */

		return 2;
	}

	/* values from 16384 to 4194303 */

	else
	{
		/* write encoded value */

		hdr[0]=(WORD08)(val>>16)|0xc0;
		hdr[1]=(WORD08)(val>>8);
		hdr[2]=(WORD08)(val);

		/* return amount of bytes required */

		return 3;
	}
}

/*
 * decval
 *
 * input:  hdr - pointer to encoded value
 *	   len - remaining length staring with encoded value
 *
 * output: val - the decoded value
 *
 * return: the consumed heqader bytes or -1 in case of error
 *
 * This procedure decodes a rfc2508 encoded value.
 */

static int decval(WORD08 *hdr,int len,int *val)
{
	/* abort if not sufficient input bytes */

	if(!len)return -1;

	/* act according to the encoding indicator */

	switch(*hdr&0xc0)
	{
	/* two byte encoding */

	case 0x80:
		/* abort if not sufficient input bytes */

		if(len<2)return -1;

		/* set up returned value */

		*val=hdr[0]&0x3f;
		*val<<=8;
		*val|=hdr[1];

		/* adjust returned value if it is negative */

		if(!(*val&0x7f80))*val-=128;

		/* return consumed bytes */

		return 2;

	/* three byte encoding */

	case 0xc0:
		/* abort if not sufficient input bytes */

		if(len<3)return -1;

		/* set up returned value */

		*val=hdr[0]&0x3f;
		*val<<=8;
		*val|=hdr[1];
		*val<<=8;
		*val|=hdr[2];

		/* adjust returned value if it is negative */

		if(!(*val&0x3fc000))*val-=16384;

		/* return consumed bytes */

		return 3;

	/* one byte encoding */

	default:/* set up returned value */

		*val=hdr[0];

		/* return consumed bytes */

		return 1;
	}
}

/*
 * rtpcomp_init
 *
 * input: pkg - pointer to packet work area
 *
 * This procedure initializes the the rfc2508 compressed rtp header
 * sender work area.
 */

void rtpcomp_init(PKG *dta)
{
	int i;	/* counter and index */


	/* for all sender indexes execept the last one */

	for(i=0;i<RTPMAX-1;i++)
	{
		/* set up ring pointer to next element */

		dta->rtpstate[i].next=&dta->rtpstate[i+1];

		/* set up compression index */

		dta->rtpstate[i].id=(WORD08)(i);

		/* reset ip source and destination addresses to invalid value */

		dta->rtpstate[i].hdr.saddr=0;
		dta->rtpstate[i].hdr.daddr=0;
	}

	/* set up last ring pointer to first element */

	dta->rtpstate[RTPMAX-1].next=&dta->rtpstate[0];

	/* set up last compression index */

	dta->rtpstate[RTPMAX-1].id=RTPMAX-1;

	/* reset last ip source and destination addresses to invalid value */

	dta->rtpstate[RTPMAX-1].hdr.saddr=0;
	dta->rtpstate[RTPMAX-1].hdr.daddr=0;

	/* set up compression ring working pointer */

	dta->rtplast=&dta->rtpstate[0];

	/* for all negative cache indexes execept the last one */

	for(i=0;i<RTPNEG-1;i++)
	{
		/* set up ring pointer to next element */

		dta->negstate[i].next=&dta->negstate[i+1];

		/* set up compression skip counter */

		dta->negstate[i].skip=0;
	}

	/* set up last ring pointer to first element */

	dta->negstate[RTPNEG-1].next=&dta->negstate[0];

	/* set up last compression index */

	dta->negstate[RTPNEG-1].skip=0;

	/* set up negative cache ring working pointer */

	dta->neglast=&dta->negstate[0];
}

/*
 * rtp_compress
 *
 * input: pkg - pointer to packet work area
 *
 * This procedure packs a rfc2508 compressed rtp header if possible.
 */

static void rtp_compress(PKG *dta)
{
	RTP *data;	/* datagram ip/udp/rtp header			*/
	RTPSTATE *cs;	/* current compression slot			*/
	NEGSTATE *ns;	/* current negative cache slot			*/
	NEGSTATE *neg;	/* last current negative cache state in ring	*/
	int iddelta;	/* current to previous packet ip id delta	*/
	int seqdelta;	/* current to previous rtp sequence number delta*/
	int timedelta;	/* curent to previous rtp timestamp delta	*/
	int ccflag;	/* cc/csrc state change flag			*/
	int olen;	/* header processing index/compressed length	*/
	int hlen;	/* uncompressed header length			*/


	/* abort if datagram is too short for a rtp datagram */

	if(dta->length<sizeof(RTP)-sizeof(data->csrc)+2)goto fail;

	/* get pointer to datagram headers */

	data=(RTP *)(dta->data+2);

	/* abort if not an ip header without options */

	if(data->verihl!=0x45)goto fail;

	/* abort of any fragmentation bits set (ignore df bit) */

	if(ntohs(data->frag_off)&0xbfff)goto fail;

	/* abort if not udp protocol */

	if(data->protocol!=17)goto fail;

	/* abort if odd udp source port */

	if(ntohs(data->source)&1)goto fail;

	/* abort if odd udp destination port */

	if(ntohs(data->dest)&1)goto fail;

	/* abort if no rtp v2 indicator bits */

	if((data->verpxcc&0xc0)!=0x80)goto fail;

	/* calculate actual rtp header length */

	olen=((data->verpxcc&0x0f)<<2)+
		sizeof(RTP)-sizeof(data->csrc);

	/* abort if datagram is too short for actual rtp header length */

	if(dta->length-2<olen)goto fail;

	/* special case handling if the frame size is larger thane the size
	   specified in the datagram (can happen e.g. with VoIP hardphones
	   with a buggy rtp implementation */

	if(ntohs(data->tot_len)!=dta->length-2||
		ntohs(data->len)!=dta->length-22)
	{
		WORD16 l=ntohs(data->tot_len);	/* datagram total length */


		/* abort if the datagram total length doesn't match the
		   datagram udp length or if it is larger than the frame
		   length */

		if(l!=ntohs(data->len)+20||l>dta->length-2||l<olen)goto fail;

		/* reset frame length to length indicated by datagram */

		dta->length=l+2;
	}

	/* get pointer to last recently used negative cache slot */

	ns=dta->neglast->next;

	/* if the current packet doesn't match the cache previous one */

	if(!ns->skip||data->saddr!=ns->saddr||data->daddr!=ns->daddr||
		*((WORD32 *)(&data->source))!=*((WORD32 *)(&ns->source)))
	{
		/* do until we have walked the complete ring */

		do
		{
			/* set up current last negative cache state */

			neg=ns;

			/* set up pointer to current negative cache element */

			ns=ns->next;

			/* if the current packet matches this one in the
			   negative cache */

			if(ns->skip&&data->saddr==ns->saddr&&
				data->daddr==ns->daddr&&
				*((WORD32 *)(&data->source))==
					*((WORD32 *)(&ns->source)))
			{
				/* decrease packets to skip counter */

				ns->skip--;

				/* abort compression */

				goto fail;
			}
		} while(ns!=dta->neglast);
	}
	else
	{
		/* decrease packets to skip counter */

		ns->skip--;

		/* abort compression */

		goto fail;
	}

	/* get pointer to last recently created context */

	cs=dta->rtplast->next;

	/* if the current rtp packet doesn't match the context previous one */

	if(data->saddr!=cs->hdr.saddr||data->daddr!=cs->hdr.daddr||
		*((WORD32 *)(&data->source))!=
			*((WORD32 *)(&cs->hdr.source))||
		data->ssrc!=cs->hdr.ssrc||data->frag_off!=cs->hdr.frag_off||
		data->ttl!=cs->hdr.ttl||data->tos!=cs->hdr.tos)
	{
		RTPSTATE *last;	/* last current compression state in ring */


		/* do until we have walked the complete ring */

		do
		{
			/* set up current last compression state */

			last=cs;

			/* set up pointer to current compression element */

			cs=cs->next;

			/* continue with header compression if the packet
			   of this element matches the current packet */

			if(data->saddr==cs->hdr.saddr&&
				data->daddr==cs->hdr.daddr&&
				*((WORD32 *)(&data->source))==
					*((WORD32 *)(&cs->hdr.source))&&
				data->ssrc==cs->hdr.ssrc&&
				data->frag_off==cs->hdr.frag_off&&
				data->ttl==cs->hdr.ttl&&data->tos==cs->hdr.tos)
					goto match;
		} while(cs!=dta->rtplast);

		/* set up new ring working pointer */

		dta->rtplast=last;

		/* reset negative caching detection leaky bucket counter */

		cs->negative=0;

		/* use current element for new compression */

		goto uncomp;
	}

match:	/* we got a matching packet and may use compression if the previous
	   and current packet udp checksum usage is identical */

	if(data->chksum&&!cs->hdr.chksum)goto uncomp;
	if(!data->chksum&&cs->hdr.chksum)goto uncomp;

	/* we must use an uncompressed packet if padding or extension
	   bits are changed */

	if((data->verpxcc&0x30)!=(cs->hdr.verpxcc&0x30))goto uncomp;

	/* we must use an uncompressed packet if the payload type changes */

	if((data->mpt&0x7f)!=cs->hdr.mpt)goto uncomp;

	/* we must use an uncompressed datagram if the ip id delta
	   between current and previous packet is too large */

	iddelta=(int)(ntohs(data->id)-ntohs(cs->hdr.id));
	if(iddelta>4194303||iddelta<-16384)goto uncomp;

	/* we must use an uncompressed datagram if the rtp sequence number
	   delta between current and previous packet is too large */

	seqdelta=(int)(ntohs(data->seq)-ntohs(cs->hdr.seq));
	if(seqdelta>4194303||seqdelta<-16384)goto uncomp;

	/* we must use an uncompressed datagram if the rtp timestamp delta
	   between current and previous packet is too large */

	timedelta=(int)(ntohl(data->timestamp)-ntohl(cs->hdr.timestamp));
	if(timedelta>4194303||timedelta<-16384)goto uncomp;

	/* reset negative cacheing detection */

	cs->negative=0;

	/* set up compressed header compression index */

	dta->hdr[0]=cs->id;

	/* set up compressed header flags value (comp. sequence number) */

	dta->hdr[1]=cs->seq=(cs->seq+1)&0xf;

	/* set up compressed header flags value (rtp marker bit) */

	dta->hdr[1]|=data->mpt&0x80;

	/* if the rtp sequence number delta doesn't match the previous delta
	   set up the rtp sequence number bit in the flags value */

	if(seqdelta!=1)dta->hdr[1]|=0x40;

	/* if the rtp timestamp delta doesn't match the previous delta
	   set up the rtp timestamp bit in the flags value */

	if((int)(cs->timestamp)!=timedelta)dta->hdr[1]|=0x20;

	/* if the ip id delta doesn't match the previous delta set up the
	   ip id bit in the flags value */

	if((int)(cs->ipid)!=iddelta)dta->hdr[1]|=0x10;

	/* if the udp checksum is in use */

	if(cs->hdr.chksum)
	{
		/* add the current udp checksum to the compressed header */

		*((WORD16 *)(dta->hdr+2))=data->chksum;

		/* set up current compressed header length */

		hlen=4;
	}

	/* set up current compressed header length for no udp checksum case */

	else hlen=2;

	/* if the current cc value doesn't match the previous one set up
	   the cc changed flag */

	if((data->verpxcc&0x0f)!=(cs->hdr.verpxcc&0x0f))ccflag=1;

	/* if the current cc value matches the previous one */

	else
	{
		int i;	/* counter and index	*/
		int j;	/* rtp cc value		*/


		/* reset the cc changed flag, set it, if one of the
		   rtp csrc fields hads a changed value */

		for(ccflag=0,i=0,j=data->verpxcc&0x0f;i<j;i++)
			if(data->csrc[i]!=cs->hdr.csrc[i])
		{
			ccflag=1;
			break;
		}
	}

	/* if the flag bits of the first flag value are all ones
	   we must pretend that the cc value or an associated
	   csrc field did change */

	if(dta->hdr[1]>=0xf0)ccflag=1;

	/* if any of the csrc values is changed */

	if(ccflag)
	{
		/* set up compressed packet second flags value */

		dta->hdr[hlen]=dta->hdr[1]&0xf0;

		/* adjust compressed packet first flags value */

		dta->hdr[1]|=0xf0;

		/* set up new cc value in second flags value */

		dta->hdr[hlen++]|=data->verpxcc&0x0f;
	}

	/* if the ip id delta did change copy the encoded delta to
	   the compressed header and adjust the header index */

	if((int)(cs->ipid)!=iddelta)hlen+=encval(dta->hdr+hlen,iddelta);

	/* if the rtp sequence number delta did change copy the encoded delta
	   to the compressed header and adjust the header index */

	if(seqdelta!=1)hlen+=encval(dta->hdr+hlen,seqdelta);

	/* if the rtp timestamp delta did change copy the encoded delta to
	   the compressed header and adjust the header index */

	if((int)(cs->timestamp)!=timedelta)
		hlen+=encval(dta->hdr+hlen,timedelta);

	/* if a cc value did change */

	if(ccflag)
	{
		/* copy required csrc values to header */

		memcpy(dta->hdr+hlen,data->csrc,(data->verpxcc&0x0f)<<2);

		/* adjust header index */

		hlen+=(data->verpxcc&0x0f)<<2;
	}

	/* memorize new ip id delta */

	cs->ipid=(WORD16)(iddelta);

	/* memorize new rtp timestamp delta */

	cs->timestamp=(WORD32)(timedelta);

	/* set up new ip id in the stored header */

	cs->hdr.id=data->id;

	/* set up new rtp sequence number in the stored header */

	cs->hdr.seq=data->seq;

	/* set up new rtp timestamp in the stored header */

	cs->hdr.timestamp=data->timestamp;

	/* adjust packet start for compressed header */

	dta->data+=olen-hlen;

	/* copy compressed header to packet */

	memcpy(dta->data+2,dta->hdr,hlen);

	/* adjust length to reflect the now compressed datagram */

	dta->length-=olen-hlen;

	/* change state to compressed rtp packet */

	dta->state=RTP_COMP;

	/* done */

	return;

uncomp:	/* in case of an uncompressed rtp packet if we have at least two
	   consecutive uncompressed packets for the same connection this
	   is probably not a rtp stream */

	if(++cs->negative>1)
	{
		/* set up new ring working pointer */

		dta->neglast=neg;

		/* set up end point information in the negative cache */
		ns->saddr=data->saddr;
		ns->daddr=data->daddr;
		*((WORD32 *)(&ns->source))=*((WORD32 *)(&data->source));

		/* skip the next 250 packet for this connection */

		ns->skip=250;

		/* abort compression */

		goto fail;
	}

	/* save the current ip/udp/rtp headers */

	memcpy(&cs->hdr,data,(data->verpxcc&0x0f)*sizeof(data->csrc[0])+
		sizeof(RTP)-sizeof(data->csrc));

	/* reset the rtp marker bit of the stored header */

	cs->hdr.mpt&=0x7f;

	/* set up one byte geader and compression id */

	data->tot_len=htons(cs->id|0x4000);

	/* set up initial compression sequence number */

	data->len=0;

	/* set up initial local compression sequence number */

	cs->seq=0;

	/* set up initial ip id delta */

	cs->ipid=1;

	/* set up initial rtp timestamp delta */

	cs->timestamp=0;

	/* change state to uncompressed rtp packet */

	dta->state=RTP_UNCOMP;

	/* done */

	return;

	/* in case of compression failure change state to IPv4 */

fail:	dta->state=IP;
}

/*
 * rtp_uncompress
 *
 * input: pkg  - pointer to packet work area
 *	  base - datagram start buffer offset
 *
 * return: 0 in case of success, -1 in case of error
 *
 * This procedure unpacks a rfc2508 compressed rtp header.
 */

static int rtp_uncompress(PKG *dta,int base)
{
	RTP *data;	/* uncompressed rtp datagam headers structure	*/
	RTPSTATE *cs;	/* pointer to current decompression data	*/
	int olen;	/* compressed header processing index/length	*/
	int hlen;	/* uncompressed header length/general purpose	*/
	int cc;		/* new cc value or -1 if value not changed	*/
	int val;	/* general purpose usage			*/
	WORD08 m;	/* current rtp marker bit value			*/
	WORD32 sum;	/* used to restore ip header checksum		*/


	/* in case of an uncompressed rtp header */

	if(dta->state==RTP_UNCOMP)
	{
		/* abort if the datagram is too short for rtp */

		if(dta->processed<sizeof(RTP)-sizeof(data->csrc)+2)
			goto err;

		/* get pointer to datagram headers */

		data=(RTP *)(dta->bfr+base);

		/* get actual 'compressed' header length (+2 for packet hdr) */

		olen=((data->verpxcc&0x0f)<<2)+
			sizeof(RTP)-sizeof(data->csrc);

		/* abort if not a one byte context id */

		if((ntohs(data->tot_len)&0xc000)!=0x4000)goto err;

		/* get compression index from datagram, abort if index is
		   out of bounds */

		if((val=ntohs(data->tot_len)&0xff)>RTPMAX-1)goto err;

		/* get pointer to comression work area from index */

		cs=&dta->rtpstate[val];

		/* copy ip/udp/rtp headers to work area */

		memcpy(&cs->hdr,data,olen);

		/* reset the ip checksum of the stored header */

		cs->hdr.check=0;

		/* clear the rtp marker bit of the stored header */

		cs->hdr.mpt&=0x7f;

		/* store the inital compression sequence number */

		cs->seq=htons(data->len)&0xf;

		/* preset ip id delta value */

		cs->ipid=1;

		/* preset rtp timestamp delta value */

		cs->timestamp=0;

		/* restore ip header total length */

		data->tot_len=htons((WORD16)(dta->total-2));

		/* restore udp datagram length */

		data->len=htons((WORD16)(dta->total-22));

		/* calculate 'uncompressed' data length */

		dta->length=dta->processed-2;

		/* copy 'uncompressed' header and data to output buffer */

		memcpy(dta->hdr,data,dta->length);

		/* set the processing state to IPv4 */

		dta->state=IP;

		/* signal success */

		return 0;
	}

	/* error if not compressed rtp header */

	if(dta->state!=RTP_COMP)goto err;

	/* error, if not at least 2 rtp compressed header bytes available */

	if(dta->processed<4)goto err;

	/* error if compression index is out of bounds */

	if(dta->bfr[base]>RTPMAX-1)goto err;

	/* get pointer to decompression data from compression index */

	cs=&dta->rtpstate[dta->bfr[base]];

	/* adjust and compare local sequence number to remote one, abort on
	   mismatch */

	if((dta->bfr[base+1]&0xf)!=(cs->seq=(cs->seq+1)&0xf))goto err;

	/* if the compressed header must contain a udp checksum */

	if(cs->hdr.chksum)
	{
		/* abort if header doesn't contain the udp checksum */

		if(dta->processed<6)goto err;

		/* copy the current udp checksum to the stored header */

		cs->hdr.chksum=*((WORD16 *)(dta->bfr+base+2));

		/* set up processing index */

		olen=base+4;
	}

	/* no udp checksum in compressed header, set up processing index */

	else olen=base+2;

	/* if the compressed header contains the second status field */

	if(dta->bfr[base+1]>=0xf0)
	{
		/* abort if header doesn't contain the second status field */

		if(dta->processed<olen+1)goto err;

		/* copy the second status field to the first one, adjust
		   processing index */

		dta->bfr[base+1]=dta->bfr[olen++];

		/* get cc value from status field */

		cc=dta->bfr[base+1]&0x0f;
	}

	/* as the second header doesn't contain a second status field the cc
	   value of the rtp header doesn't change */

	else cc=-1;

	/* get rtp marker bit from status field */

	m=dta->bfr[base+1]&0x80;

	/* if the compressed header contains an ip id delta */

	if(dta->bfr[base+1]&0x10)
	{
		/* decode delta value from compressed header, abort if the
		   compressed header is too short */

		if((hlen=decval(dta->bfr+olen,dta->processed-olen,&val))==-1)
			goto err;

		/* adjust processing index */

		olen+=hlen;

		/* store new ip id delta */

		cs->ipid=val;
	}

	/* increase stored ip id value by ip id delta */

	cs->hdr.id=htons((WORD16)((WORD32)(ntohs(cs->hdr.id))+
		(WORD32)(cs->ipid)));

	/* if the compressed header contains a rtp sequence number delta */

	if(dta->bfr[base+1]&0x40)
	{
		/* decode delta value from compressed header, abort if the
		   compressed header is too short */

		if((hlen=decval(dta->bfr+olen,dta->processed-olen,&val))==-1)
			goto err;

		/* adjust processing index */

		olen+=hlen;

		/* increase the stored rtp sequence number by the specified
		   delta value */

		cs->hdr.seq=htons((WORD16)(ntohs(cs->hdr.seq))+((WORD16)(val)));
	}

	/* if the compresed headed doesn't contain a rtp sequence number delta
	   increase the stored rtp sequence number by one */

	else cs->hdr.seq=htons((WORD16)(ntohs(cs->hdr.seq))+((WORD16)(1)));

	/* if the compressed header contains a new rtp timestamp delta */

	if(dta->bfr[base+1]&0x20)
	{
		/* decode delta value from compressed header, abort if the
		   compressed header is too short */

		if((hlen=decval(dta->bfr+olen,dta->processed-olen,&val))==-1)
			goto err;

		/* adjust processing index */

		olen+=hlen;

		/* store new rtp timestamp delta */

		cs->timestamp=val;
	}

	/* increase stored rtp timestamp value by rtp timestamp delta */

	cs->hdr.timestamp=htonl(((WORD32)(ntohl(cs->hdr.timestamp)))+
		((WORD32)(cs->timestamp)));

	/* if the compressed header contains new csrc data */

	if(cc>0)
	{
		/* abort of the compressed header doesn't contain the
		expected amount of csrc data */

		if(dta->processed<olen+(cc<<2))goto err;

		/* copy the new csrc data to the stored header */

		memcpy(cs->hdr.csrc,dta->bfr+olen,cc<<2);

		/* adjust processing index */

		olen+=cc<<2;
	}

	/* if the compressed header did contain a cc value set the cc
	   value of the stored header accordingly */

	if(cc>=0)cs->hdr.verpxcc=((cs->hdr.verpxcc)&0xf0)|(cc&0x0f);

	/* otherwise retrieve cc value from stored header */

	else cc=cs->hdr.verpxcc&0x0f;

	/* calculate size of uncompressed datagram header */

	hlen=sizeof(RTP)-sizeof(data->csrc)+(cc<<2);

	/* copy saved uncompressed header to output buffer */

	memcpy(dta->hdr,&cs->hdr,hlen);

	/* copy remaining datagram data area to output area */

	memcpy(dta->hdr+hlen,dta->bfr+olen,dta->processed+base-2-olen);

	/* calculate actual compressed datagram header size */

	olen-=base-2;

	/* calculate uncompressed data length */

	dta->length=dta->processed+hlen-olen;

	/* get pointer to uncompressed datagram headers */

	data=(RTP *)(dta->hdr);

	/* restore marker bit state */

	data->mpt|=m;

	/* retore ip length and udp length */

	data->tot_len=htons((WORD16)(dta->total+hlen-olen));
	data->len=htons((WORD16)(dta->total-20+hlen-olen));

	/* recalculate and store ip header checksum */

	for(sum=0,cc=0;cc<10;cc++)sum+=ntohs(((WORD16 *)(dta->hdr))[cc]);
	sum=(sum&0xffff)+(sum>>16);
	sum=(sum&0xffff)+(sum>>16);
	data->check=htons((WORD16)(~sum));

	/* change state to IPv4 */

	dta->state=IP;

	/* signal success */

	return 0;

	/* signal error */

err:	return -1;
}

/*
 * pkg_slip_init
 *
 * input: dta - a pointer to the packet work area.
 *
 * This procedure resets a packet work area for use with slip.
 */

void pkg_slip_init(PKG *dta)
{
	/* debug message */

	ENTER("pkg_slip_init");

	/* reset work area */

	dta->state=NONE;
	dta->length=2;
	dta->processed=0;
	dta->chksum=0xffff;
	dta->bfr[0]=0;

	/* debug message */

	LEAVE("pkg_slip_init");
}

/*
 * slip_to_pkg
 *
 * input:  pkg  - a pointer to the packet work area
 *	   in   - a pointer to the input bytes
 *	   ilen - the amount of input bytes available
 *
 * return: the amount of input bytes consumed
 *
 * This procedure converts a slip data stream to packet format.
 */

int slip_to_pkg(PKG *dta,WORD08 *in,int ilen)
{
	int len=0;	/* input processing index */


	/* debug message */

	ENTER("slip_to_pkg");

	/* if we are in error state continue with error processing */

	if(dta->state==ERR)goto err;

	/* process input */

	while(len<ilen)switch(in[len])
	{
		int val;	/* general purpose usage */

	/* end of frame marker */

	case 192:
		/* error if not at least one data byte */

		if(dta->length<3)goto err;

		/* set up packet data pointer */

		dta->data=dta->bfr;

		/* assert that packet payload doesn't exceed MAXBLOCK bytes */

		if(dta->length>MAXBLOCK+2)goto err;

		/* convert van jacobsen compression indicators to
		   packet tcp compression indicators */

		if(dta->bfr[2]&0x80)
		{
			dta->state=VJC_COMP;
			dta->bfr[2]&=0x7f;
		}
		else if(dta->bfr[2]>=0x70)
		{
			dta->state=VJC_UNCOMP;
			dta->bfr[2]&=0x4f;
		}

		/* otherwise packet is plain ip, possibly do rtp compression */

		else rtp_compress(dta);

		/* set up packet header */

		val=dta->length-3;
		dta->data[0]=(WORD08)((dta->state<<4)|(val>>8));
		dta->data[1]=(WORD08)(val);

		/* mark packet as complete */

		dta->state=STREAM;

		/* debug message */

		LEAVE("slip_to_pkg");

		/* return amount of input bytes consumed */

		return len+1;

	/* escape character */

	case 219:
		/* set escape flag */

		dta->bfr[0]=1;

		/* increase input processing index */

		len++;

		/* done */

		break;

	/* all other data bytes */

	default:/* switch to error mode if maximum payload length is exceeded */

		if(dta->length==MAXBLOCK+2)goto err;

		/* if the previous byte was the escape character */

		if(dta->bfr[0])
		{
			/* reset escape flag */

			dta->bfr[0]=0;

			/* unescape character 192 */

			if(in[len]==220)
			{
				/* write out character */

				dta->bfr[dta->length++]=192;

				/* increase input processing index */

				len++;

				/* done */

				break;
			}

			/* unescape character 219 */

			else if(in[len]==221)
			{
				/* write out character */

				dta->bfr[dta->length++]=219;

				/* increase input processing index */

				len++;

				/* done */
				break;
			}
		}

		/* copy current input byte to packet */

		dta->bfr[dta->length++]=in[len++];

		/* done */

		break;
	}

	/* finish */

	goto out;

	/* error processing: search for an end of frame marker in the
	   whole remaining input */

err:	while(len<ilen)if(in[len++]==192)
	{
		/* at end of frame reset the work area */

		dta->bfr[0]=0;
		dta->length=2;
		dta->state=NONE;

		/* finish */

		goto out;
	}

	/* set processing state to error state */

	dta->state=ERR;

out:	/* debug message */

	LEAVE("slip_to_pkg");

	/* return amount of input bytes consumed */

	return len;
}

/*
 * pkg_to_slip
 *
 * input:  pkg  - a pointer to the packet work area
 *	   in   - a pointer to the input bytes
 *	   ilen - the amount of input bytes available
 *
 * return: the amount of input bytes consumed
 *
 * This procedure converts packet format to a slip data stream.
 */

int pkg_to_slip(PKG *dta,WORD08 *in,int ilen)
{
	int len=0;	/* input processing index */


	/* debug message */

	ENTER("pkg_to_slip");

	/* if we are in error state continue with error processing */

	if(dta->state==ERR)goto err;

	/* act according to the amount of packet bytes processed */

	switch(dta->processed)
	{
	/* 0 bytes processed, get first header byte */

	case 0:	if(len<ilen)dta->bfr[0]=in[len++];
		else goto out;

	/* 1 byte processed, get second header byte */

	case 1:	if(len<ilen)dta->bfr[1]=in[len++];
		else goto out;

	/* 2 bytes processed, process header */

	case 2:	/* get packet type from packet header */

		dta->state=dta->bfr[0]>>4;

		/* get total length from packet header */

		dta->total=dta->bfr[0]&0xf;
		dta->total<<=8;
		dta->total|=dta->bfr[1];
		dta->total+=3;

		/* set up output base output length */

		dta->length=2;

		/* act according to packet type */

		switch(dta->state)
		{
		/* IPv4 */

		case IP:

		/* van jacobsen compressed tcp */

		case VJC_COMP:

		/* van jacobsen uncompressed tcp */

		case VJC_UNCOMP:

		/* compressed rtp */

		case RTP_COMP:

		/* uncompressed rtp */

		case RTP_UNCOMP:

			/* done */

			break;

		/* unknown packet type */

		default:/* truncate input length to remaining required packet
			   data */

			if(ilen>dta->total-dta->processed)
				ilen=dta->total-dta->processed;

			/* switch to error state */

			goto err;
		}

	/* 2 or more bytes processed */

	default:/* truncate input length to remaining required packet data */

		if(ilen>dta->total-dta->processed)
			ilen=dta->total-dta->processed;

		/* act according to packet type */

		switch(dta->state)
		{
			int l;	/* general purpose usage	*/
			int m;	/* counter and index		*/


		/* compressed rtp */

		case RTP_COMP:

		/* uncompressed rtp */

		case RTP_UNCOMP:

			/* if the cache is not yet full */

			if(dta->processed<102)
			{
				/* calculate maximum amount of input bytes
				   that can be consumed */

				if(ilen>102-dta->length)l=102-dta->processed;
				else l=ilen;

				/* copy input to cache */

				memcpy(dta->bfr+dta->processed,in,l);

				/* adjust amount of bytes processed value */

				dta->processed+=l;
			}

			/* as the cache is full don't consume input bytes */

			else l=0;

			/* return amount of newly cached data when either
			   the buffer is not sufficiently filled or the
			   end of the frame is not yet reached */

			if(dta->processed<102&&dta->total!=dta->processed)
				return l;

			/* unpack, handle errors */

			if(rtp_uncompress(dta,2))
			{
				/* restore amount of processed bytes */

				dta->processed-=l;

				/* switch to error state */

				goto err;
			}

			/* adjust amount of input bytes actually processed */

			len+=l;

			/* restore amount of bytes processed value */

			dta->processed-=l;

			/* get actual length of unpacked input */

			l=dta->length;

			/* restore output length */

			dta->length=2;

			/* process unpacked input */

			for(m=0;m<l;m++)switch(dta->hdr[m])
			{
			/* end of frame marker */

			case 192:
				/* escape character */

				dta->bfr[dta->length++]=219;
				dta->bfr[dta->length++]=220;

				/* done */

				break;

			/* escape character */

			case 219:
				/* escape character */

				dta->bfr[dta->length++]=219;
				dta->bfr[dta->length++]=221;

				/* done */

				break;

			/* all other data bytes */

			default:/* copy input byte to frame */

				dta->bfr[dta->length++]=dta->hdr[m];

				/* done */

				break;
			}

			/* done */

			break;
		}

		/* done */

		break;
	}

	/* process input */

	while(len<ilen)switch(in[len])
	{
	/* end of frame marker */

	case 192:
		/* escape character */

		dta->bfr[dta->length++]=219;
		dta->bfr[dta->length++]=220;

		/* increase input processing index */

		len++;

		/* done */

		break;

	/* escape character */

	case 219:
		/* escape character */

		dta->bfr[dta->length++]=219;
		dta->bfr[dta->length++]=221;

		/* increase input processing index */

		len++;

		/* done */

		break;

	/* all other data bytes */

	default:/* copy input byte to frame */

		dta->bfr[dta->length++]=in[len++];

		/* done */

		break;
	}

	/* if a complete packet is processed */

	if(dta->processed+len==dta->total)
	{
		/* convert packet tcp compression indicators to van
		   jacobsen compression indicators */

		switch(dta->state)
		{
		case VJC_COMP:
			dta->bfr[2]|=0x80;
			break;

		case VJC_UNCOMP:
			dta->bfr[2]|=0x70;
			break;
		}

		/* add end of frame marker and adjust output length */

		dta->bfr[dta->length--]=192;

		/* set up output data pointer */

		dta->data=dta->bfr+2;

		/* mark frame as complete */

		dta->state=STREAM;
	}

	/* finish */

	goto out;

	/* error processing: if the end of the packet is reached */

err:	if(dta->total-dta->processed==ilen)
	{
		/* at end of packet reset the work area */

		dta->state=NONE;
		dta->processed=0;

		/* debug message */

		LEAVE("pkg_to_slip");

		/* return amount of input bytes consumed */

		return ilen;
	}

	/* set processing state to error state */

	dta->state=ERR;

	/* adjust amount of packet bytes processed */

out:	dta->processed+=ilen;

	/* debug message */

	LEAVE("pkg_to_slip");

	/* return amount of input bytes consumed */

	return ilen;
}

/*
 * pkg_ppp_init
 *
 * input: dta - a pointer to the packet work area.
 *
 * This procedure resets a packet work area for use with ppp.
 */

void pkg_ppp_init(PKG *dta)
{
	/* debug message */

	ENTER("pkg_ppp_init");

	/* reset work area */

	dta->state=NONE;
	dta->length=1;
	dta->processed=0;
	dta->chksum=0xffff;
	dta->bfr[0]=0;

	/* debug message */

	LEAVE("pkg_ppp_init");
}

/*
 * ppp_to_pkg
 *
 * input:  pkg  - a pointer to the packet work area
 *	   in   - a pointer to the input bytes
 *	   ilen - the amount of input bytes available
 *
 * return: the amount of input bytes consumed
 *
 * This procedure converts a ppp data stream to packet format.
 */

int ppp_to_pkg(PKG *dta,WORD08 *in,int ilen)
{
	int len=0;	/* input processing index */


	/* debug message */

	ENTER("ppp_to_pkg");

	/* if we are in error state continue with error processing */

	if(dta->state==ERR)goto err;

	/* process input */

	while(len<ilen)switch(in[len])
	{
		int val;	/* general purpose usage */

	/* end of frame marker */

	case 0x7e:
		/* switch to error mode if not at least one data byte */

		if(dta->length<7)goto err;

		/* switch to error mode if checksum mismatch */

		if(dta->chksum!=0xf0b8)goto err;

		/* switch to error mode if ppp hdlc framing error */

		if(dta->bfr[1]!=0xff||dta->bfr[2]!=0x03)goto err;

		/* set up output data pointer and length */

		dta->data=dta->bfr+2;
		dta->length-=4;

		/* switch to error state if packet payload exceeds MAXBLOCK
		   bytes */

		if(dta->length>MAXBLOCK+2)goto err;

		/* if the ppp frame type consists of two bytes */

		if(!((val=dta->data[1])&1))
		{
			/* switch to error mode if not at least one data byte */

			if(dta->length<4)goto err;

			/* adjust packet length and output data pointer */

			dta->length--;
			dta->data++;

			/* get second byte of ppp frame type */

			val=(val<<8)|dta->data[1];
		}

		/* process ppp frame type and set packet type */

		switch(val)
		{
		/* IPv4 */

		case 0x0021:
			rtp_compress(dta);
			break;

		/* van jacobsen compressed tcp header */

		case 0x002d:
			dta->state=VJC_COMP;
			break;

		/* van jacobsen uncompressed tcp header */

		case 0x002f:
			dta->state=VJC_UNCOMP;
			break;

		/* switch to error mode if unknown frame type */

		default:goto err;
		}

		/* set up packet header */

		val=dta->length-3;
		dta->data[0]=(WORD08)((dta->state<<4)|(val>>8));
		dta->data[1]=(WORD08)(val);

		/* mark packet as complete */

		dta->state=STREAM;

		/* debug message */

		LEAVE("ppp_to_pkg");

		/* return amount of input bytes consumed */

		return len+1;

	/* escape character */

	case 0x7d:
		/* if we are not already in escape mode */

		if(!dta->bfr[0])
		{
			/* set up escape xor value */

			dta->bfr[0]=0x20;

			/* increase input processing index */

			len++;

			/* done */

			break;
		}

	/* all other data bytes */

	default:/* switch to error mode if maximum payload length is exceeded */

		if(dta->length==MAXBLOCK+7)goto err;

		/* copy (escaped) input byte to packet and adjust checksum */

		dta->chksum=(dta->chksum>>8)^crc16[(dta->chksum^
			(dta->bfr[dta->length++]=in[len++]^dta->bfr[0]))&0xff];

		/* reset escape value if necessary */

		if(dta->bfr[0])dta->bfr[0]=0;

		/* done */

		break;
	}

	/* finish */

	goto out;

	/* error processing: search for an end of frame marker in the
	   whole remaining input */

err:	while(len<ilen)if(in[len++]==0x7e)
	{
		/* at end of frame reset the work area */

		dta->bfr[0]=0;
		dta->length=1;
		dta->state=NONE;
		dta->chksum=0xffff;

		/* finish */

		goto out;
	}

	dta->state=ERR;

out:	/* debug message */

	LEAVE("ppp_to_pkg");

	/* return amount of input bytes consumed */

	return len;
}

/*
 * pkg_to_ppp
 *
 * input:  pkg  - a pointer to the packet work area
 *	   in   - a pointer to the input bytes
 *	   ilen - the amount of input bytes available
 *
 * return: the amount of input bytes consumed
 *
 * This procedure converts packet format to a ppp data stream.
 */

int pkg_to_ppp(PKG *dta,WORD08 *in,int ilen)
{
	int len=0;	/* input processing index */


	/* debug message */

	ENTER("pkg_to_ppp");

	/* if we are in error state continue with error processing */

	if(dta->state==ERR)goto err;

	/* act according to the amount of packet bytes processed */

	switch(dta->processed)
	{
		int val;	/* general purpose usage */

	/* 0 bytes processed, get first header byte */

	case 0:	if(len<ilen)dta->bfr[0]=in[len++];
		else goto out;

	/* 1 byte processed, get second header byte */

	case 1:	if(len<ilen)dta->bfr[1]=in[len++];
		else goto out;

	/* 2 bytes processed, process header */

	case 2:	/* get packet type from packet header */

		dta->state=dta->bfr[0]>>4;

		/* get total length from packet header */

		dta->total=dta->bfr[0]&0xf;
		dta->total<<=8;
		dta->total|=dta->bfr[1];
		dta->total+=3;

		/* set up output base output length */

		dta->length=6;

		/* add ppp hdlc frame intro to output */

		dta->bfr[2]=0xff;
		dta->bfr[3]=0x03;

		/* act according to packet type */

		switch(dta->state)
		{
		/* IPv4 */

		case IP:/* set up ppp frame type */

			val=0x0021;

			/* done */

			break;

		/* van jacobsen compressed tcp */

		case VJC_COMP:

			/* set up ppp frame type */

			val=0x002d;

			/* done */

			break;

		/* van jacobsen uncompressed tcp */

		case VJC_UNCOMP:

			/* set up ppp frame type */

			val=0x002f;

			/* done */

			break;

		/* compressed rtp */

		case RTP_COMP:

			/* set up ppp frame type */

			val=0x0021;

			/* done */

			break;

		/* uncompressed rtp */

		case RTP_UNCOMP:

			/* set up ppp frame type */

			val=0x0021;

			/* done */

			break;

		/* unknown packet type */

		default:/* truncate input length to remaining required packet
			    data */

			if(ilen>dta->total-dta->processed)
				ilen=dta->total-dta->processed;

			/* switch to error state */

			goto err;
		}

		/* add ppp frame type to ppp frame */

		dta->bfr[4]=(WORD08)(val>>8);
		dta->bfr[5]=(WORD08)(val);

		/* prepare ppp frame checksum */

		dta->chksum=0x3de3;
		dta->chksum=(dta->chksum>>8)^
				crc16[(dta->chksum^dta->bfr[4])&0xff];
		dta->chksum=(dta->chksum>>8)^
				crc16[(dta->chksum^dta->bfr[5])&0xff];

	/* 2 or more bytes processed */

	default:/* truncate input length to remaining required packet data */

		if(ilen>dta->total-dta->processed)
			ilen=dta->total-dta->processed;

		/* act according to packet type */

		switch(dta->state)
		{
			int l;	/* general purpose usage	*/
			int m;	/* counter and index		*/


		/* compressed rtp */

		case RTP_COMP:

		/* uncompressed rtp */

		case RTP_UNCOMP:

			/* if the cache is not yet full */

			if(dta->processed<102)
			{
				/* calculate maximum amount of input bytes
				   that can be consumed */

				if(ilen>102-dta->length)l=102-dta->processed;
				else l=ilen;

				/* copy input to cache */

				memcpy(dta->bfr+dta->processed+4+len,in+len,
					l-len);

				/* adjust amount of bytes processed value */

				dta->processed+=l;
			}

			/* as the cache is full don't consume input bytes */

			else l=0;

			/* return amount of newly cached data when either
			   the buffer is not sufficiently filled or the
			   end of the frame is not yet reached */

			if(dta->processed<102&&dta->total!=dta->processed)
				return l;

			/* unpack, handle errors */

			if(rtp_uncompress(dta,6))
			{
				/* restore amount of processed bytes */

				dta->processed-=l;

				/* switch to error state */

				goto err;
			}

			/* adjust amount of input bytes actually processed */

			len+=l;

			/* restore amount of bytes processed value */

			dta->processed-=l;

			/* get actual length of unpacked input */

			l=dta->length;

			/* restore output length */

			dta->length=6;

			/* process unpacked input */

			for(m=0;m<l;m++)
			{
				/* adjust checksum with current input byte */

				dta->chksum=(dta->chksum>>8)^
					crc16[(dta->chksum^dta->hdr[m])&0xff];


				/* act according to input byte */

				switch(dta->hdr[m])
				{
				/* escape character */

				case 0x7d:

				/* end of frame marker */

				case 0x7e:

					/* add escape character to output */

					dta->bfr[dta->length++]=0x7d;

					/* add escaped character to output */

					dta->bfr[dta->length++]=
						dta->hdr[m]^0x20;

					/* done */

					break;

				/* all other data bytes */

				default:/* copy input byte to frame */

					dta->bfr[dta->length++]=dta->hdr[m];

					/* done */

					break;
				}
			}

			/* done */

			break;
		}

		/* done */

		break;
	}

	/* process input */

	while(len<ilen)
	{
		/* adjust checksum with current input byte */

		dta->chksum=(dta->chksum>>8)^crc16[(dta->chksum^in[len])&0xff];

		/* act according to input byte */

		switch(in[len])
		{
		/* escape character */

		case 0x7d:

		/* end of frame marker */

		case 0x7e:

			/* add escape character to output frame */

			dta->bfr[dta->length++]=0x7d;

			/* add escaped input character to output frame */

			dta->bfr[dta->length++]=in[len++]^0x20;

			/* done */

			break;

		/* all other characters */

		default:/* copy input character to output frame */

			dta->bfr[dta->length++]=in[len++];

			/* done */

			break;
		}
	}

	/* if a complete packet is processed */

	if(dta->processed+len==dta->total)
	{
		/* finalize checksum */

		dta->chksum^=0xffff;

		/* add checksum low word to output frame */

		switch((WORD08)(dta->chksum))
		{
		/* frame end marker and escape character */

		case 0x7d:
		case 0x7e:
			/* copy escape character to output */

			dta->bfr[dta->length++]=0x7d;

			/* copy escaped value to output */

			dta->bfr[dta->length++]=(WORD08)(dta->chksum)^0x20;

			/* done */

			break;

		/* all other values */

		default:/* copy character to output */

			dta->bfr[dta->length++]=(WORD08)(dta->chksum);

			/* done */

			break;
		}

		/* add checksum high word to output frame */

		switch((WORD08)(dta->chksum>>8))
		{
		/* frame end marker and escape character */

		case 0x7d:
		case 0x7e:
			/* copy escape character to output */

			dta->bfr[dta->length++]=0x7d;

			/* copy escaped value to output */

			dta->bfr[dta->length++]=(WORD08)(dta->chksum>>8)^0x20;

			/* done */

			break;

		/* all other values */

		default:/* copy character to output */

			dta->bfr[dta->length++]=(WORD08)(dta->chksum>>8);

			/* done */

			break;
		}

		/* add end of frame marker and adjust output length */

		dta->bfr[dta->length--]=0x7e;

		/* set up output data pointer */

		dta->data=dta->bfr+2;

		/* mark frame as complete */

		dta->state=STREAM;
	}

	/* finish */

	goto out;

	/* error processing: if the end of the packet is reached */

err:	if(dta->total-dta->processed==ilen)
	{
		/* at end of packet reset the work area */

		dta->state=NONE;
		dta->processed=0;

		/* debug message */

		LEAVE("pkg_to_ppp");

		/* return amount of input bytes consumed */

		return ilen;
	}

	/* set processing state to error state */

	dta->state=ERR;

	/* adjust amount of packet bytes processed */

out:	dta->processed+=ilen;

	/* debug message */

	LEAVE("pkg_to_ppp");

	/* return amount of input bytes consumed */

	return ilen;
}
