/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, ast@domdv.de
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 * (c) 2001 Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* the big league: main                                                       */
/*============================================================================*/

#ifdef DEBUG
#define USAGE1 "       vpnd [-d number] [-p] [-l] [-r] [-n|-t] [-f configfile]\n\n"
#define USAGE2 "    -d print debug messages up to level <number>\n"
#else
#define USAGE1 "       vpnd [-p] [-l] [-r] [-n|-t] [-f configfile]\n\n"
#define USAGE2
#endif
#ifdef COMPRESS
#define COMPRESSINFO "\n    Compression is available.\n\n"
#else
#define COMPRESSINFO "\n    Compression is not available.\n\n"
#endif

#define USAGE  "usage: vpnd -h\n" \
	       "       vpnd -m [<master-key-file>]\n" \
	       "       vpnd -x <extended-master-key-directory>\n" \
	       "       vpnd -c <extended-master-key-file>\n" \
	       USAGE1 \
	       "    -h prints this info\n" \
	       "    -m creates master key file (default is /etc/vpnd.key),\n" \
	       "       note that the file must not yet exist\n" \
	       "    -x creates extended master key files named vpnd.lcl.key\n" \
	       "       and vpnd.rmt.key in the given directory,\n" \
	       "       note that the files must not yet exist\n" \
	       "    -c set/change/remove user password of extended master\n" \
	       "       key file, note that a newly created extended master\n" \
	       "       key file has no user password, to set a password\n" \
	       "       enter no password for the old password, to change\n" \
	       "       the password enter old and new password, to remove\n" \
	       "       the password enter no password for the new password\n" \
	       "       NOTE: no password verification is done, it is solely\n" \
	       "       your responsibility to enter and remember the\n" \
	       "       correct values, if you enter wrong values here\n" \
	       "       recovery is NOT possible, you must then create and\n" \
	       "       use a new set of extended master key files\n" \
	       USAGE2 \
	       "    -p request user password for extended master key file,\n" \
	       "       if standard input is a pipe the user password is\n" \
	       "       read from the pipe, otherwise the user is prompted\n" \
	       "       to enter the password\n" \
               "    -l allow dns lookups so host names can be used\n" \
               "    -r use realtime priority and lock memory (linux only)\n" \
	       "    -n do not become daemon\n" \
	       "    -t do modem init chat test\n" \
	       "    -f use configfile instead of /etc/vpnd.conf\n" \
	       COMPRESSINFO

/*
 * dummy
 *
 * This is just a dummy which is a replacement for real functions.
 */

void dummy(void)
{
}

/*
 * main
 *
 * input:  argc - the argument count
 *	   argv - argument vector pointer
 *
 * return: 1 in case of error, else
 *	   never returns
 *
 * This procedure is the main procedure of the
 * virtual provate network daemon (vpnd). It
 * takes care of command line processing,
 * configuration file parsing, daemonizing
 * as well as pid file creation and start 
 * of the main loop.
 */

int main(int argc,char *argv[])
{
	int fd;				/* pid file access */
	FILE *fp;			/* pid file access */
	WORD08 daemon;			/* detach flag     */
	WORD08 highprio;		/* high priority   */
	int i;				/* counter/index   */
	char *pwd;			/* user password   */
	struct stat stb;		/* stdin mode check*/
	char *pass;			/* user password   */
	struct rlimit lim;		/* 0 bytes core    */
#ifdef LINUX2
	struct sched_param sched;	/* for priority    */
#endif

	/* debug message */

	ENTER("main");

	/* prevent core dumps */

	lim.rlim_cur=lim.rlim_max=0;
	setrlimit(RLIMIT_CORE,&lim);

	/* preset global data */

	initvars(&globals);

	/* reset pid file handle(s) */

	fd=-1;
	fp=NULL;

	/* default is to detach */

	daemon=2;

	/* default is standard priority */

	highprio=0;

	/* set up pointer to user password buffer */

	pass=globals.u1.bfr;

	/* if new master key file is requested */

	if(argc==2||argc==3)if(!strcmp(argv[1],"-m"))
	{
		/* if pathname is given, use it instead of default */

		if(argc==3)globals.keyfile=argv[2];

		/* create basic mode master key file and exit */

		die(genmaster(&globals),0);
	}

	/* if new extended master key files are requested */

	if(argc==3)if(!strcmp(argv[1],"-x"))
	{
		/* make directory available to creation routine */

		globals.keyfile=argv[2];

		/* create extended mode master key files and exit */

		globals.extended=1;
		die(genmaster(&globals),0);
	}

	/* if extended master key file password change is requested */

	if(argc==3)if(!strcmp(argv[1],"-c"))
	{
		/* make pathname available to change routine */

		globals.keyfile=argv[2];

		/* change password and exit */

		globals.extended=1;
		die(chgpasswd(&globals),0);
	}

	/* process command line parameters */

	if(argc)while(--argc)
	{
		argv++;
		if(!strcmp(*argv,"-n"))daemon=0;
		else if(!strcmp(*argv,"-t"))verbose=1;
		else if(!strcmp(*argv,"-p"))globals.passwd=1;
		else if(!strcmp(*argv,"-l"))dns=1;
		else if(!strcmp(*argv,"-r"))highprio=1;
		else if(!strcmp(*argv,"-h"))goto usage;
		else if(!strcmp(*argv,"-f"))
		{
			if(!--argc)goto usage;
			globals.conffile=*(++argv);
		}
#ifdef DEBUG
		else if(!strcmp(*argv,"-d"))
		{
			if(!--argc)goto usage;
			debug=atoi(*(++argv));
		}
#endif
		else goto usage;
			
	}

	/* parse config file, terminate in case of error */

	if(!parse(&globals))goto err;

	/* in modem chat test mode open serial line,
	   call chatter, then close serial line, done */

	if(verbose)
	{
		/* assert that serial line mode is used */

		if(!globals.serialmode)
		{
			printf("-t only valid for modem init chat test\n");
			goto err;
		}

		/* set up signal handlers (default is ignore) */

		for(i=1;i<_NSIG;i++)setsig(i,SIG_IGN);
		setsig(SIGINT,sigdie);
		setsig(SIGTERM,sigdie);

		/* try to get serial line */

		if(!serialhandler(&globals))
		{
			printf("can't access %s\n",globals.serialdev);
			goto err;
		}

		/* print info */

		printf("starting modem init chat in test mode\n");

		/* do modem init chat in test mode */

		i=chathandler(3,&globals);

		/* release serial line */

		serialhandler(&globals);

		/* print chat test result */

		printf("modem init chat test returned %d\n",i);

		/* done */

		die(i,0);
	}

	/* now get and process user password if required */

	if(globals.passwd)
	{
		/* try to stat stdin, reset mode field if stat fails */

		if(fstat(0,&stb))stb.st_mode=0;

		/* in case stdin is a pipe */

		if(S_ISFIFO(stb.st_mode))
		{
			/* read password from stdin, reset password on error */

			if(!fgets(pass,128,stdin))*pass=0;

			/* remove trailing newline from password */

			if(*pass)if(pass[strlen(pass)-1]=='\n')
				pass[strlen(pass)-1]=0;

			/* set pointer for further password processing */

			pwd=pass;
		}

		/* in case stdin is not a pipe */

		else
		{
			/* request password from user */

			pwd=getpass("Enter password for master key: ");
		}

		/* reject empty password */

		if(!*pwd)
		{
			printf("Empty password, terminating...\n");
			goto err;
		}

		/* create key schedule from password */

		blowfish_key_schedule(pwd,strlen(pwd),globals.u3.sched);

		/* overwrite password memory */

		memset(pwd,0,strlen(pwd));
	}

	/* access pidfile, terminate in case of errors: on the one hand we
	   want to overwrite stale pid files, on the other hand we want
	   to prevent symlink attacks, so we first delete any old pid file
	   and then create the pid file */

	if(pidfile)
	{
		unlink(pidfile);
		if((fd=open(pidfile,O_CREAT|O_WRONLY|O_EXCL,0644))==-1)
		{
			printf("can't access '%s'\n",pidfile);
			pidfile=NULL;
			goto err;
		}
	}

	/* use stdio handle for pid file from now on */

	fp=fdopen(fd,"w");

	/* create control socket if requested, terminate in case of errors */

	if(control)if(controlhandler(0))
	{
		printf("can't create %s\n",control);
		if(fp)fclose(fp);
		goto err;
	}

	/* become daemon if not disabled */

	for(i=0;i<daemon;i++)switch(fork())
	{
		case 0:  break;
		case -1: printf("fork failed\n");
			 if(fp)fclose(fp);
			 goto err;
		default: if(fp)fclose(fp);
			 die(0,1);
	}

	/* write pid to pid file if file is open, then close file */

	if(fp)
	{
#ifdef SunOS
		fprintf(fp,"%lu",getpid());
#else
		fprintf(fp,"%u",getpid());
#endif
		fclose(fp);
	}

	/* detach from tty and then close possibly open file
	   handles except the control socket, if there is any
	   (wild guess but usually hits all files left open) */

	if(daemon)
	{
		i=1;
		ioctl(0,TIOCNOTTY,&i);
		for(i=0;i<256;i++)if(i!=ctrl)close(i);
		setsid();
	}

	/* open syslog */

	openlog("vpnd",daemon?LOG_PID:(LOG_PID|LOG_PERROR),facility);

	/* set up signal handlers (defualt is ignore) */

	for(i=1;i<_NSIG;i++)setsig(i,SIG_IGN);
	setsig(SIGINT,sigdie);
	setsig(SIGTERM,sigdie);

	/* funny but if you don't do this waitpid() fails... */

	blocksig(SIGCHLD);

	/* on linux try to lock into memory and increase priority
	   if requested, ignore errors */

#ifdef LINUX2
	if(highprio)
	{
		mlockall(MCL_CURRENT|MCL_FUTURE);
		memset(&sched,0,sizeof(sched));
		sched.sched_priority=1;
		sched_setscheduler(0,SCHED_RR,&sched);
	}
#endif

	/* do main job */

	sequencer(&globals);

	/* debug message */

	LEAVE("main");

	/* clean up and terminate */

	goto err;

	/* usage info */

usage:	printf("vpnd version %s\n%s",VERSION,USAGE);

	/* termination */

err:	die(1,0);

	/* not reached, just to shut up the compiler */

	return 1;
}
