/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, ast@domdv.de
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 * (c) 2001 Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* the verbose one: syslog messages                                           */
/*============================================================================*/

/*
 * logmsg
 *
 * input: type   - the message id
 *	  reason - the reason code for the message
 *	  anchor - pointer to vpnd global data
 *
 * This procedure writes a syslog message
 */

void logmsg(int type,int reason,VPN *anchor)
{
	struct in_addr addr;	/* used for ip to string conversion */


	/* debug message */

	ENTER("logmsg");

	/* print message according to message type given */

	switch(type)
	{
	case STARTUP:
#ifdef COMPRESS
		syslog(LOG_NOTICE,"version %s (compression enabled) starting.",VERSION);
#else
		syslog(LOG_NOTICE,"version %s (no compression) starting.",VERSION);
#endif
		break;
	case SHUTDOWN:
		syslog(LOG_NOTICE,"shutting down.");
		break;
	case SIGNAL:
		switch(reason)
		{
		case SIGINT:
			syslog(LOG_INFO,"Interrupt signal received.");
			break;
		case SIGTERM:
			syslog(LOG_INFO,"Termination signal received.");
			break;
		default:syslog(LOG_INFO,"signal %d received.",reason);
			break;
		}
		break;
	case SLIPFAIL:
		switch(reason)
		{
		case 1:	syslog(LOG_NOTICE,"%s link failed, not tty/pty " \
				"available, will retry in %d seconds.",
				anchor->iftype,anchor->retry);
			break;
		case 2:	syslog(LOG_NOTICE,"%s link failed, tty configuration " \
				"failed, will retry in %d seconds.",
				anchor->iftype,anchor->retry);
			break;
		case 3:	syslog(LOG_NOTICE,"%s link failed, saving or setting " \
				"the line discipline failed, will retry " \
				"in %d seconds.", anchor->iftype,anchor->retry);
			break;
		case 4:	syslog(LOG_NOTICE,"%s link failed, setting the " \
				"encapsulation or configuring the interface " \
				"failed, will retry in %d seconds.",
				anchor->iftype,anchor->retry);
			break;
		case 5:	syslog(LOG_NOTICE,"%s link failed, creating the " \
				"route to the peer failed, will retry " \
				"in %d seconds.",anchor->iftype,anchor->retry);
			break;
		case 6:	syslog(LOG_NOTICE,"%s link failed, creating the " \
				"priority route to the peer failed, " \
				"will retry in %d seconds.",
				anchor->iftype,anchor->retry);
			break;
		case 7:	syslog(LOG_NOTICE,"%s link failed, creation of " \
				"an additional route failed, " \
				"will retry in %d seconds.",
				anchor->iftype,anchor->retry);
			break;
		case 8:	syslog(LOG_NOTICE,"%s link failed, couldn't create " \
				"icmp socket, will retry in %d seconds.",
				anchor->iftype,anchor->retry);
			break;
		case 101:
			syslog(LOG_NOTICE,"%s link failed, exception " \
				"pending on interface, will retry in %d " \
				"seconds.",anchor->iftype,anchor->retry);
			break;
		case 102:
			syslog(LOG_NOTICE,"%s link failed, writing to " \
				"interface failed, will retry in %d seconds.",
				anchor->iftype,anchor->retry);
			break;
		case 103:
			syslog(LOG_NOTICE,"%s link failed, reading from " \
				"interface failed, will retry in %d seconds.",
				anchor->iftype,anchor->retry);
			break;
		default:syslog(LOG_NOTICE,"%s link failed, reason=%d, " \
				"will retry in %d seconds.",anchor->iftype,
				reason,anchor->retry);
			break;
		}
		break;
	case SLIPUP:
		syslog(LOG_INFO,"%s link established.",anchor->iftype);
		break;
	case SRVSFAIL:
		switch(reason)
		{
		case 1:	syslog(LOG_NOTICE,"server socket failed, no socket " \
				"available, will retry in %d seconds.",
				anchor->retry);
			break;
		case 2:	syslog(LOG_NOTICE,"server socket failed, bind " \
				"to requested address/port failed, " \
				"will retry in %d seconds.",anchor->retry);
			break;
		case 3:	syslog(LOG_NOTICE,"server socket failed, setting " \
				"a socket option failed, will retry in " \
				"%d seconds.",anchor->retry);
			break;
		case 4:	syslog(LOG_NOTICE,"server socket failed, listen " \
				"on the socket failed, will retry in " \
				"%d seconds.",anchor->retry);
			break;
		case 101:
			syslog(LOG_NOTICE,"server socket failed, exception " \
				"pending on socket, will retry in " \
				"%d seconds.",anchor->retry);
			break;
		default:syslog(LOG_NOTICE,"server socket failed, reason=%d, " \
				"will retry in %d seconds.",
				reason,anchor->retry);
				break;
		}
		break;
	case SRVSLSTN:
		syslog(LOG_INFO,"listening with address %s on port %d",
			anchor->server_ip,anchor->server_port);
		break;
	case PEERFAIL:
		switch(reason)
		{
		case 1:	syslog(LOG_INFO,"peer link failed, no socket " \
				"available, will retry in %d seconds.",
				anchor->retry);
			break;
		case 2:	syslog(LOG_INFO,"peer link failed, bind " \
				"to requested address/port failed, " \
				"will retry in %d seconds.",anchor->retry);
			break;
		case 3:	syslog(LOG_INFO,"peer link failed, setting " \
				"a socket option failed, will retry in " \
				"%d seconds.",anchor->retry);
			break;
		case 4:	syslog(LOG_INFO,"peer link failed, connecting " \
				"to the peer failed, will retry in " \
				"%d seconds.",anchor->retry);
			break;
		case 5:	syslog(LOG_INFO,"peer link failed, socks " \
				"protocol error, will retry in " \
				"%d seconds.",anchor->retry);
			break;
		case 6:	syslog(LOG_INFO,"peer link failed, connection " \
				"through socks server failed or was " \
				"rejected, will retry in %d seconds.",
				anchor->retry);
			break;
		case 101:
			syslog(LOG_INFO,"peer link failed, no icmp echo " \
				"replies received, will retry in %d seconds.",
				anchor->retry);
			break;
		case 102:
			syslog(LOG_INFO,"peer link failed, exception pending " \
				"on peer connection, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 103:
			syslog(LOG_INFO,"peer link failed, no response to " \
				"link test message, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 104:
			syslog(LOG_INFO,"peer link failed, failed to send " \
				"new key, will retry in %d seconds.",
				anchor->retry);
			break;
		case 105:
			syslog(LOG_INFO,"peer link failed, reading data " \
				"from peer failed, will retry in %d seconds.",
				anchor->retry);
			break;
		case 106:
			syslog(LOG_INFO,"peer link failed, sending data " \
				"to peer failed, will retry in %d seconds.",
				anchor->retry);
			break;
		case 111:
			syslog(LOG_INFO,"peer link failed, exception on " \
				"serial line, will retry in %d seconds.",
				anchor->retry);
			break;
		default:syslog(LOG_INFO,"peer link failed, reason=%d, " \
				"will retry in %d seconds.",reason,
				anchor->retry);
			break;
		}
		break;
	case CRYPTFAIL:
		switch(reason)
		{
		case 1:	syslog(LOG_NOTICE,"crypto init failed, send/receive " \
				"of new key or server iv failed, will retry " \
				"in %d seconds.",anchor->retry);
			break;
		case 2:	syslog(LOG_NOTICE,"crypto init failed, send/receive " \
				"of client iv failed, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 3:	syslog(LOG_NOTICE,"crypto init failed, master key " \
				"file read error, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 4:	syslog(LOG_NOTICE,"crypto init failed, master key " \
				"file version not supported, will retry " \
				"in %d seconds.",anchor->retry);
			break;
		case 5:	syslog(LOG_NOTICE,"crypto init failed, extended " \
				"information receive failed, will retry " \
				"in %d seconds.",anchor->retry);
			break;
		case 6:	syslog(LOG_NOTICE,"crypto init failed, extended " \
				"information send failed, will retry " \
				"in %d seconds.",anchor->retry);
			break;
		case 7:	syslog(LOG_NOTICE,"crypto init failed, key id " \
				"mismatch, will retry in %d seconds.",
				anchor->retry);
			break;
		case 8:	syslog(LOG_NOTICE,"crypto init failed, transcation " \
				"id mismatch, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 9:	syslog(LOG_NOTICE,"crypto init failed, sending " \
				"of remote iv to peer failed, will retry " \
				"in %d seconds.",anchor->retry);
			break;
		case 10:syslog(LOG_NOTICE,"crypto init failed, receive " \
				"of local iv from peer failed, will retry " \
				"in %d seconds.",anchor->retry);
			break;
		case 11:syslog(LOG_NOTICE,"crypto init failed, creating " \
				"new local iv failed, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 12:syslog(LOG_NOTICE,"crypto init failed, sending " \
				"new local iv failed, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 13:syslog(LOG_NOTICE,"crypto init failed, receiving " \
				"new remote iv failed, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 14:syslog(LOG_NOTICE,"crypto init failed, challenge " \
				"send error, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 15:syslog(LOG_NOTICE,"crypto init failed, challenge " \
				"receive error, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 16:syslog(LOG_NOTICE,"crypto init failed, challenge " \
				"comparison error, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 17:syslog(LOG_NOTICE,"crypto init failed, master " \
				"key file save error, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 18:syslog(LOG_NOTICE,"crypto init failed, peer " \
				"doesn't support hmac, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		default:syslog(LOG_NOTICE,"crypto init failed, reason=%d, " \
				"will retry in %d seconds.",reason,
				anchor->retry);
		}
		break;
	case CCONNECT:
		syslog(LOG_INFO,"connected to %s",anchor->server_ip);
		break;
	case DEVFAIL:
		syslog(LOG_INFO,"access to %s failed, " \
			"will retry in %d seconds.",
			anchor->serialdev,anchor->retry);
		break;
	case FORKFAIL:
		syslog(LOG_INFO,"fork failed, will retry in %d seconds.",
			anchor->retry);
		break;
	case INITCHAT:
		syslog(LOG_INFO,"serial line init chat started on %s",
			anchor->serialdev);
		break;
	case SELECTFAIL:
		syslog(LOG_NOTICE,"select failed, dying...");
		break;
	case CHATFAIL:
		switch(reason)
		{
		case 1:	syslog(LOG_INFO,"init chat failed, signal received, " \
				"will retry in %d seconds.",anchor->retry);
			break;
		case 2:	syslog(LOG_INFO,"init chat failed, changing serial " \
				"line mode to nonblocking failed, " \
				"will retry in %d seconds.",anchor->retry);
			break;
		case 3:	syslog(LOG_INFO,"init chat failed, can't access " \
				"init chat script, will retry in %d seconds.",
				anchor->retry);
			break;
		case 4:	syslog(LOG_INFO,"init chat failed, unknown command " \
				"in chat script, will retry in %d seconds.",
				anchor->retry);
			break;
		case 5:	syslog(LOG_INFO,"init chat failed, illegal command " \
				"format in chat script, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 6:	syslog(LOG_INFO,"init chat failed, illegal wait " \
				"time in chat script, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 7:	syslog(LOG_INFO,"init chat failed, illegal escape " \
				"sequence in chat script, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 8:	syslog(LOG_INFO,"init chat failed, can't get modem " \
				"control line status, will retry in %d " \
				"seconds.",anchor->retry);
			break;
		case 9:	syslog(LOG_INFO,"init chat failed, write to modem " \
				"failed, will retry in %d seconds.",
				anchor->retry);
			break;
		case 10:syslog(LOG_INFO,"init chat failed, read from modem " \
				"failed, will retry in %d seconds.",
				anchor->retry);
			break;
		case 11:syslog(LOG_INFO,"init chat failed, target 'fail' " \
				"encountered, will retry in %d seconds.",
				anchor->retry);
			break;
		case 12:syslog(LOG_INFO,"init chat failed, target not " \
				"found, will retry in %d seconds.",
				anchor->retry);
			break;
		case 256:	/* never used */
			syslog(LOG_INFO,"init chat failed, fork failure, " \
				"will retry in %d seconds.",anchor->retry);
			break;
		case 257:	/* never used */
			syslog(LOG_INFO,"init chat failed, child still " \
				"running, will retry in %d seconds.",
				anchor->retry);
			break;
		case 258:
			syslog(LOG_INFO,"init chat failed, system call " \
				"failed, will retry in %d seconds.",
				anchor->retry);
			break;
		case 259:
			syslog(LOG_INFO,"init chat failed, child terminated " \
				"abnormally, will retry in %d seconds.",
				anchor->retry);
			break;
		case 260:	/* internal error */
			syslog(LOG_INFO,"init chat failed, no action " \
				"specified, will retry in %d seconds.",
				anchor->retry);
			break;
		default:syslog(LOG_INFO,"init chat failed, reason %d, " \
				"will retry in %d seconds.",reason,
				anchor->retry);
			break;
		}
		break;
	case DCONNECT:
		syslog(LOG_INFO,"connection established on %s",
			anchor->serialdev);
		break;
	case SPEERFAIL:
		switch(reason)
		{
		case 101:
			syslog(LOG_INFO,"peer link failed, no icmp echo " \
				"replies received, closing connection.");
			break;
		case 102:
			syslog(LOG_INFO,"peer link failed, exception pending " \
				"on peer connection, closing connection.");
			break;
		case 103:
			syslog(LOG_INFO,"peer link failed, no response to " \
				"link test message, closing connection.");
			break;
		case 104:
			syslog(LOG_INFO,"peer link failed, failed to send " \
				"new key, closing connection.");
			break;
		case 105:
			syslog(LOG_INFO,"peer link failed, reading data " \
				"from peer failed, closing connection.");
			break;
		case 106:
			syslog(LOG_INFO,"peer link failed, sending data " \
				"to peer failed, closing connection.");
			break;
		default:syslog(LOG_INFO,"peer link failed, reason=%d, " \
				"closing connection.",reason);
			break;
		}
		break;
	case ACPTERR:
		syslog(LOG_INFO,"client connect failed, still waiting.");
		break;
	case ILLEGAL:
		addr.s_addr=reason;
		syslog(LOG_NOTICE,"illegal connect from %s, " \
			"closing connection.",inet_ntoa(addr));
		break;
	case PEEROPT:
		syslog(LOG_NOTICE,"peer socket option setup failed, " \
			"closing connection.");
		break;
	case PEERIOCTL:
		syslog(LOG_NOTICE,"peer socket ioctl failed, " \
			"closing connection.");
		break;
	case SCRYPT:
		switch(reason)
		{
		case 1:	syslog(LOG_NOTICE,"crypto init failed, send/receive " \
				"of new key or server iv failed, closing " \
				"connection.");
			break;
		case 2:	syslog(LOG_NOTICE,"crypto init failed, send/receive " \
				"of client iv failed, closing connection.");
			break;
		case 3:	syslog(LOG_NOTICE,"crypto init failed, master key " \
				"file read error, closing connection.");
			break;
		case 4:	syslog(LOG_NOTICE,"crypto init failed, master key " \
				"file version not supported, closing " \
				"connection.");
			break;
		case 5:	syslog(LOG_NOTICE,"crypto init failed, extended " \
				"information receive failed, closing " \
				"connection.");
			break;
		case 6:	syslog(LOG_NOTICE,"crypto init failed, extended " \
				"information send failed, closing connection.");
			break;
		case 7:	syslog(LOG_NOTICE,"crypto init failed, key id " \
				"mismatch, closing connection.");
			break;
		case 8:	syslog(LOG_NOTICE,"crypto init failed, transcation " \
				"id mismatch, closing connection.");
			break;
		case 9:	syslog(LOG_NOTICE,"crypto init failed, sending " \
				"of remote iv to peer failed, closing " \
				"connection.");
			break;
		case 10:syslog(LOG_NOTICE,"crypto init failed, receive " \
				"of local iv from peer failed, closing " \
				"connection.");
			break;
		case 11:syslog(LOG_NOTICE,"crypto init failed, creating " \
				"new local iv failed, closing connection.");
			break;
		case 12:syslog(LOG_NOTICE,"crypto init failed, sending " \
				"new local iv failed, closing connection.");
			break;
		case 13:syslog(LOG_NOTICE,"crypto init failed, receiving " \
				"new remote iv failed, closing connection.");
			break;
		case 14:syslog(LOG_NOTICE,"crypto init failed, challenge " \
				"send error, closing connection.");
			break;
		case 15:syslog(LOG_NOTICE,"crypto init failed, challenge " \
				"receive error, closing connection.");
			break;
		case 16:syslog(LOG_NOTICE,"crypto init failed, challenge " \
				"comparison error, closing connection.");
			break;
		case 17:syslog(LOG_NOTICE,"crypto init failed, master " \
				"key file save error, closing connection.");
			break;
		case 18:syslog(LOG_NOTICE,"crypto init failed, peer " \
				"doesn't support hmac, closing connection.");
			break;
		default:syslog(LOG_NOTICE,"crypto init failed, reason=%d, " \
				"closing connection.",reason);
		}
		break;
	case SCONNECT:
		addr.s_addr=reason;
		syslog(LOG_INFO,"connect from %s",inet_ntoa(addr));
		break;
	case KEYFAIL:
		syslog(LOG_NOTICE,"reading master key from file failed, " \
			"terminating...");
		break;
	case FALLBACK:
		syslog(LOG_NOTICE,"warning: transaction id mismatch, " \
			"using fallback data");
		break;
	case DCONTROL:
		syslog(LOG_INFO,"peer link closed by user, " \
			"earliest reconnect in %d seconds.",anchor->retry);
		break;
	case DSCONTROL:
		syslog(LOG_INFO,"peer link closed by user.");
		break;
	default:
		syslog(LOG_NOTICE,"unknown message with type %d and " \
			"reason %d",type,reason);
		break;
	}

	/* debug message */

	LEAVE("logmsg");
}
