/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, ast@domdv.de
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 * (c) 2001 Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* the big league: modem init chat                                            */
/*============================================================================*/

/* possible chat script commands , command id and amount of command 
   parameters required */

typedef struct
{
	char *cmd;
	WORD16 id;
	WORD16 parms;
} CMDLIST;

static CMDLIST cmdlist[]=
{
	{ "sleep",    0, 0 },
	{ "goto",     1, 1 },
	{ "dcdwait",  2, 2 },
	{ "dsrwait",  3, 2 },
	{ "ringwait", 4, 2 },
	{ "send",     5, 3 },
	{ "wait",     6, 3 },
	{ "expect",   7, 3 },
	{ NULL,       0, 0 }
};

/*
 * chathandler
 *
 * input:  action - action bit vector:
 *		    1 - start chat child process
 *		    2 - wait for child termination
 *		    4 - poll for child termination
 *
 * return: 0     - successful child start if action=1
 *	   0-255 - exit code of child process
 *	   256   - fork failed
 *	   257   - child still running
 *	   258   - system call error
 *	   259   - child didn't terminate normally
 *	   260   - no action specified
 *
 * This procedure runs the modem init chat which is always executed as a
 * child process.
 */

int chathandler(char action,VPN *anchor)
{
	int result;	/* child process exit code */


	/* start chat child process */

	if(action&1)
	{
		/* try to create child process */

		anchor->child=fork();

		/* if the child process creation failed */

		if(anchor->child==-1)
		{
			/* reset child process pid */

			anchor->child=0;

			/* print info if not in init chat test mode */

			if(!verbose)logmsg(FORKFAIL,0,anchor);

			/* signal fork error */

			return 256;
		}

		/* child executes chatter (never returns) */

		else if(!anchor->child)exit(chatter(anchor));

		/* parent prints info if not in init chat test mode */

		else if(!verbose)logmsg(INITCHAT,0,anchor);

		/* signal success, if in start only mode */

		if(action==1)return 0;
	}

	/* wait/poll for child termination */

	if(action&6)
	{
		/* wait for child to terminate, preprocess result */

		switch(waitpid(anchor->child,&result,(action&4)?WNOHANG:0))
		{
		/* child still running */

		case 0:	/* in poll mode signal child running */

			if(action&4)return 257;

		/* system call error */

		case -1:/* signal system call error */

			return 258;

		/* child termination */

		default:/* reset child process id */

			anchor->child=0;

			/* signal unspecified error if child did not
			   terminate regularly */

			if(!WIFEXITED(result))return 259;

			/* return exit code of child */

			else return WEXITSTATUS(result);
		}
	}

	/* signal no action taken */

	return 260;
}

/*
 * chatter
 *
 * input:  anchor - pointer to vpnd global data
 *
 * return: 0 in case of success, else <>0
 *
 * This procedure handles all serial line initalization chat.
 * It is run as a child process. If this process exits with 0
 * then initialization is assumed to be successful.
 */

int chatter(VPN *anchor)
{
	FILE *fp;		/* modem chat script file handle	*/
	int sline;		/* current chat script line number	*/
	char *item;		/* command as string			*/
	int delay;		/* numeric command wait time		*/
	int i;			/* general purpose usage		*/
	int j;			/* general purpose usage		*/
	char *ptr;		/* general purpose usage		*/
	int serial;		/* local handle to serial device	*/
	char *modemchat;	/* local pathname of init chat script	*/
	void (*msg)(char *,...);/* pointer either to printf or dummy	*/
	char *p[3];		/* pointers to command parameters	*/
	CMDLIST *cmd;		/* current command			*/
	char m;			/* unescape character buffer		*/
	char *bfr;		/* general purpose buffer		*/
	char *lbl;		/* general purpose buffer		*/
	char *val;		/* command parameter (unescaped)	*/


	/* debug message */

	ENTER("chatter");

	/* set up buffer pointers */

	bfr=anchor->u1.bfr;
	lbl=anchor->u2.bfr;
	val=anchor->u3.bfr;

	/* get local duplicate of handle to serial device */

	serial=dup(anchor->serial);

	/* get pointer to modem chat script pathnme */

	modemchat=anchor->modemchat;

	/* close all other open handles */

	closehandles();

	/* reset memory */

	clearmem();

	/* sigint and sigterm will terminate process immediately */

	setsig(SIGINT,SIG_DFL);
	setsig(SIGTERM,SIG_DFL);

	/* handle possible race condition */

	if(terminate)return 1;

	/* set up handler for alarm() timout */

	setsig(SIGALRM,chatalarm);

	/* select verbose or quiet mode */

	if(verbose)msg=(void (*)(char *,...))(printf);
	else msg=(void (*)(char *,...))(dummy);

	/* set serial line mode to nonblocking, handle errors */

#ifdef SunOS
	if(fcntl(serial,F_SETFL,O_NONBLOCK))
#else
	i=0;
	if(ioctl(serial,FIONBIO,&i))
#endif
	{
		msg("changing serial line to nonblocking failed\n");
		return 2;
	}

	/* done, if there's no chat script */

	if(!modemchat)RET("no chat script",0);

	/* reset timeout flag and line number */

	sline=timeout=0;

	/* try to open chat script file, handle errors */

	if((fp=fopen(modemchat,"r"))==NULL)
	{
		msg("accessing %s failed\n",modemchat);
		return 3;
	}

	/* for all lines of the chat script */

	while(fgets(bfr,sizeof(anchor->u1.bfr),fp))
	{
		/* increase line number */

		sline++;

		/* get command, skip empty lines */

		if((item=strtok(bfr," 	\n"))==NULL)continue;

		/* skip comment lines */

		else if(*item=='#')continue;

		/* if the current command is a jump target */

		else if(*item==':')
		{
			/* if looking for a jump target after
			   timeout and the target is found
			   reset timeout flag */

			if(timeout)if(!strcmp(lbl,item+1))timeout=0;

			/* print target if not in seek mode */

			if(!timeout)msg("%d: target %s\n",sline,item+1);

			/* continue with next line */

			continue;
		}

		/* don't process commands during label seek */

		else if(timeout)continue;

		/* search for command id */

		else for(cmd=cmdlist;cmd->cmd;cmd++)
			if(!strcmp(item,cmd->cmd))break;

		/* handle unknown commands */

		if(!cmd->cmd)
		{
			msg("unknown command in line %d of %s\n",sline,
				modemchat);
			return 4;
		}

		/* get all parameters, check, that all parameters are
		   given */

		for(delay=i=0;i<cmd->parms;i++)
			if((p[i]=strtok(NULL," 	\n"))==NULL)
		{
			msg("illegal command format in line %d of %s\n",
				sline,modemchat);
			return 5;
		}

		/* process current retrieved parameter */

		else switch(i)
		{
		/* convert wait time to numeric */

		case 1:	/* error if wait time < 1 */

			if((delay=atoi(*p))<=0)
			{
				msg("illegal wait time in line %d of %s\n",
					sline,modemchat);
				return 6;
			}

		/* get jump target */

		case 0:	/* save current target */

			strcpy(lbl,p[i]);

			/* done */

			break;

		/* get additional parameters for send/wait/expect cmd */

		case 2:	/* unescape value, handle errors */

			for(j=0,ptr=p[2];*ptr;ptr++)
			{
				/* in case of the escape character */

				if(*ptr=='\\')
				{
					/* process next character as
					   escape indentification,
					   handle errors */

					switch(*(++ptr))
					{
					case '\\':m='\\';
						  break;
					case 'b': m=' ';
						  break;
					case 't': m='\t';
						  break;
					case 'r': m='\r';
						  break;
					case 'n': m='\n';
						  break;
					default:  msg("illegal escape " \
							"sequence in " \
							"line %d of " \
							"%s\n",sline,modemchat);
						 return 7;
					}

					/* copy unescaped character */

					val[j++]=m;
				}

				/* just copy regular characters */

				else val[j++]=*ptr;
			}

			/* terminate unescaped value string */

			val[j]=0;

			/* done */

			break;
		}

		/* print current command in proper format */

		msg("%d: %s",sline,item);
		for(j=i=0;i<cmd->parms;i++)msg(" %s",p[i]);
		msg("\n");

		/* process current command with given timeout */

		alarm(delay);

		switch(cmd->id)
		{
		/* sleep */

		case 0:	/* sleep for a second */

			sleep(1);

			/* done */

			break;

		/* goto */

		case 1:	/* switch to label seek mode */

			timeout=1;

			/* done */

			break;

		/* dcdwait, dsrwait, ringwait */

		case 2:	if(!j)j=TIOCM_CAR;
		case 3:	if(!j)j=TIOCM_DSR;
		case 4:	if(!j)j=TIOCM_RNG;

			/* do, until line is set or a timeout occurs */

			while(!timeout)
			{
				/* get modem status, handle errors */

				if(ioctl(serial,TIOCMGET,&i)<0)
				{
					/* in test mode print message */

					msg("can't get modem control line " \
						"status\n");

					/* terminate */

					return 8;
				}

				/* exit loop if line is set */

				if(i&j)break;

				/* seep for 0.1 seconds */

				usleep(100000);
			}

			/* done */

			break;

		/* send */

		case 5: /* send unescaped string to modem,
			   handle errors */

			if(write(serial,val,strlen(val))!=strlen(val))
			{
				/* print message in test mode */

				msg("write failed\n");

				/* terminate if no timeout */

				if(!timeout)return 9;
			}

			/* done */

			break;

		/* wait/expect */

		case 6:
		case 7: /* do, until the given string is found
			   or a timeout occurs */

			while(1)
			{
				/* at maximum for the given buffer length - 1 */

				for(i=0,ptr=bfr;i<sizeof(anchor->u1.bfr)-1;)
				{
					/* read character from
					   modem, handle errors */

					if(read(serial,ptr,1)!=1)
					{
						/* in test mode print message */

						msg("read failed\n");

						/* terminate if no timeout */

						if(!timeout)return 10;
					}

					/* exit loop if timeout */

					if(timeout)break;

					/* skip NUL characters */

					if(!*ptr)continue;

					/* exit loop if newline read */

					if(*ptr++=='\n')break;

					/* increment read count */

					i++;
				}

				/* terminate line read */

				*ptr=0;

				/* exit line loop if timeout */

				if(timeout)break;

				/* print line in modem test mode */

				msg("read: %s",bfr);

				/* exit loop if given string found in line */

				if(strstr(bfr,val))break;

				/* exit loop in expect mode with error */

				if(cmd->id==7)
				{
					timeout=1;
					break;
				}
			}

			/* done */

			break;
		}

		/* reset alarm */

		alarm(0);

		/* in case of a timeout seek for jump label 
		   target from beginning of file, handle
		   special labels fail and done directly */

		if(timeout)
		{
			if(!strcmp(lbl,"fail"))return 11;
			if(!strcmp(lbl,"done"))return 0;
			fseek(fp,0,SEEK_SET);
			sline=0;
		}
	}

	/* close modem chat script */

	fclose(fp);

	/* handle label search errors */

	if(timeout)
	{
		msg("target %s not found\n",lbl);
		return 12;
	}

	/* debug message */

	LEAVE("chatter");

	/* signal success */

	return 0;
}
