/*
 * MD5 Production Implementation
 * This implementation: (c) 1999 Andreas Steinmetz
 * MD5 algorithm by Ron Rivest, see RFC1321
 * HMAC algorithm by Krawczyk, et. al., see RFC2104
 * Test cases for HMAC: see RFC2202
 *
 * License:
 * This code is under the GNU public license
 * for use with the virtual private network daemon (vpnd).
 * The copyright holder will however retain the copyright.
 * In addition to vpnd this code may be used
 * for all GPL/LGPL based open source software products.
 * For all other software products please contact ast@domdv.de
 * for a license agreement. There is no guarantee for the fitness
 * and usability of this code for any purpose. The author takes no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 *
 * This implementation is tuned for speed.
 *
 * Note that the implementation is processor independent.
 * It does not depend on any run time library and
 * should be 64 bit clean.
 *
 * Restriction:
 *
 * The maximum data length to be processed is 2^32-1 bytes but
 * this shouldn't be an issue for nearly all applications. If
 * it is an issue you will have to extend the variable 'total'
 * from 32 bits to 64 bits in the MD5 routines.
 *
 * Data sizes:
 *
 * data block for MD5 transformation	WORD08[64]
 * resulting MD5 hash			WORD08[16]
 *
 * WORD08 means an unsigned word of 8 bits length
 * WORD32 means an unsigned word of at least 32 bits length
 *
 * Mode of operation:
 *
 * MD5_FULL defined:
 *
 * 1. Call md5init
 * 2. For all data to be hashed call md5next
 * 3. To retrieve the hash call md5end
 *
 * MD5_FAST defined:
 *
 * Call md5 to get the hash for the specified data
 *
 * MD5_HMAC_FULL defined:
 *
 * 1. Call md5hmkey once to preprocess the selected key
 * 2. Call md5hminit
 * 3. For all data to be hashed call md5hmnext
 * 4. To retrieve the hash call md5hmend
 * 5. When the preprocessed key is no longer required
 *    reset the MD5HMDATA structure
 *
 * MD5_HMAC_FAST defined:
 *
 * 1. Call md5hmkey once to preprocess the selected key
 * 2. Call md5hmac to get the hash for the specified data
 * 3. When the preprocessed key is no longer required
 *    reset the MD5HMDATA structure
 */

/* sanity settings */

#if !defined(MD5_FULL) && !defined(MD5_FAST) && !defined(MD5_HMAC_FULL) && !defined(MD5_HMAC_FAST)
#define MD5_FULL
#define MD5_FAST
#define MD5_HMAC_FULL
#define MD5_HMAC_FAST
#define MD5_PARANOID
#endif

#if defined(MD5_HMAC_FULL) || defined(MD5_HMAC_FAST)
#ifndef MD5_HMAC
#define MD5_HMAC
#endif
#ifndef MD5_PARANOID
#define MD5_PARANOID
#endif
#endif

/* definitions */

#define MD5_SIZE 16
#define MD5(a) WORD08 a[MD5_SIZE]

/* required types */

typedef struct
{
	WORD32 md5[4];	/* current md5 result				*/
	WORD32 total;	/* total amount of bytes processed		*/
	union
	{
	  WORD32 l[16];	/* longword buffer for paranoia memory reset	*/
	  WORD08 b[64];	/* buffer for incomplete 64 byte message block	*/
	}bfr;
	WORD08 size;	/* size of data in buffer			*/
} MD5DATA;

#ifdef MD5_HMAC
typedef struct
{
	WORD32 imd5[4];	/* preprocessed HMAC key (inner)		*/
	WORD32 omd5[4];	/* preprocessed HMAC key (outer)		*/
} MD5HMDATA;
#endif

/* function prototypes */

#if defined(MD5_FULL) || defined(MD5_HMAC)
extern void CRYPTOCALL md5init(MD5DATA *ptr);
extern void CRYPTOCALL md5next(WORD08 *data,WORD32 length,MD5DATA *ptr);
extern void CRYPTOCALL md5end(WORD08 *result,MD5DATA *ptr);
#endif
#ifdef MD5_FAST
extern void CRYPTOCALL md5(WORD08 *data,WORD32 length,WORD08 *result);
#endif
#if defined(MD5_HMAC_FULL) || defined(MD5_HMAC_FAST)
extern void CRYPTOCALL md5hmkey(WORD08 *key,WORD32 keylength,MD5HMDATA *ptr);
#endif
#ifdef MD5_HMAC_FULL
extern void CRYPTOCALL md5hminit(MD5DATA *ptr,MD5HMDATA *key);
#define md5hmnext(a,b,c) md5next(a,b,c)
extern void CRYPTOCALL md5hmend(WORD08 *result,MD5DATA *ptr,MD5HMDATA *key);
#endif
#ifdef MD5_HMAC_FAST
extern void CRYPTOCALL md5hmac(WORD08 *data,WORD32 length,WORD08 *result,MD5HMDATA *key);
#endif

