/*
 * SHA1 Production Implementation
 * This implementation: (c) 1999 Andreas Steinmetz
 * SHA1 algorithm: see FIPS PUB 180-1
 * HMAC algorithm by Krawczyk, et. al., see RFC2104
 * Test cases for HMAC: see RFC2202
 *
 * License:
 * This code is under the GNU public license
 * for use with the virtual private network daemon (vpnd).
 * The copyright holder will however retain the copyright.
 * In addition to vpnd this code may be used
 * for all GPL/LGPL based open source software products.
 * For all other software products please contact ast@domdv.de
 * for a license agreement. There is no guarantee for the fitness
 * and usability of this code for any purpose. The author takes no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 *
 * This implementation is tuned for speed.
 *
 * Note that the implementation is processor independent.
 * It does not depend on any run time library and
 * should be 64 bit clean.
 *
 * Restriction:
 *
 * The maximum data length to be processed is 2^32-1 bytes but
 * this shouldn't be an issue for nearly all applications. If
 * it is an issue you will have to extend the variable 'total'
 * from 32 bits to 64 bits in the SHA1 routines.
 *
 * Data sizes:
 *
 * data block for SHA1 transformation	WORD08[64]
 * resulting SHA1 or HMAC hash		WORD08[20]
 *
 * WORD08 means an unsigned word of 8 bits length
 * WORD32 means an unsigned word of at least 32 bits length
 *
 * Mode of operation:
 *
 * SHA1_FULL defined:
 *
 * 1. Call sha1init
 * 2. For all data to be hashed call sha1next
 * 3. To retrieve the hash call sha1end
 *
 * SHA1_FAST defined:
 *
 * Call sha1 to get the hash for the specified data
 *
 * SHA1_HMAC_FULL defined:
 *
 * 1. Call sha1hmkey once to preprocess the selected key
 * 2. Call sha1hminit
 * 3. For all data to be hashed call sha1hmnext
 * 4. To retrieve the hash call sha1hmend
 * 5. When the preprocessed key is no longer required
 *    reset the SHA1HMDATA structure
 *
 * SHA1_HMAC_FAST defined:
 *
 * 1. Call sha1hmkey once to preprocess the selected key
 * 2. Call sha1hmac to get the hash for the specified data
 * 3. When the preprocessed key is no longer required
 *    reset the SHA1HMDATA structure
 */

#include "common.h"
#include "sha1.h"

/*
 * procedure sha1block
 *
 * input: block - pointer to the 64 byte message block
 *
 * inout: sha1 - pointer to the sha1 working buffer
 *
 * This procedure processes the 64 byte message block
 * and updates the current sha1 result.
 */

static void CRYPTOCALL sha1block(WORD32 *sha1,WORD08 *data)
{
	register WORD32 a;	/* work buffer	*/
	register WORD32 b;	/* work buffer	*/
	register WORD32 c;	/* work buffer	*/
	register WORD32 d;	/* work buffer	*/
	register WORD32 e;	/* work buffer	*/
	WORD32 w[16];		/* work buffer	*/


	/* get 64 byte message block into 32 bit array */

	a=data[0];
	a<<=8;
	a+=data[1];
	a<<=8;
	a+=data[2];
	a<<=8;
	w[0]=a+data[3];

	a=data[4];
	a<<=8;
	a+=data[5];
	a<<=8;
	a+=data[6];
	a<<=8;
	w[1]=a+data[7];

	a=data[8];
	a<<=8;
	a+=data[9];
	a<<=8;
	a+=data[10];
	a<<=8;
	w[2]=a+data[11];

	a=data[12];
	a<<=8;
	a+=data[13];
	a<<=8;
	a+=data[14];
	a<<=8;
	w[3]=a+data[15];

	a=data[16];
	a<<=8;
	a+=data[17];
	a<<=8;
	a+=data[18];
	a<<=8;
	w[4]=a+data[19];

	a=data[20];
	a<<=8;
	a+=data[21];
	a<<=8;
	a+=data[22];
	a<<=8;
	w[5]=a+data[23];

	a=data[24];
	a<<=8;
	a+=data[25];
	a<<=8;
	a+=data[26];
	a<<=8;
	w[6]=a+data[27];

	a=data[28];
	a<<=8;
	a+=data[29];
	a<<=8;
	a+=data[30];
	a<<=8;
	w[7]=a+data[31];

	a=data[32];
	a<<=8;
	a+=data[33];
	a<<=8;
	a+=data[34];
	a<<=8;
	w[8]=a+data[35];

	a=data[36];
	a<<=8;
	a+=data[37];
	a<<=8;
	a+=data[38];
	a<<=8;
	w[9]=a+data[39];

	a=data[40];
	a<<=8;
	a+=data[41];
	a<<=8;
	a+=data[42];
	a<<=8;
	w[10]=a+data[43];

	a=data[44];
	a<<=8;
	a+=data[45];
	a<<=8;
	a+=data[46];
	a<<=8;
	w[11]=a+data[47];

	a=data[48];
	a<<=8;
	a+=data[49];
	a<<=8;
	a+=data[50];
	a<<=8;
	w[12]=a+data[51];

	a=data[52];
	a<<=8;
	a+=data[53];
	a<<=8;
	a+=data[54];
	a<<=8;
	w[13]=a+data[55];

	a=data[56];
	a<<=8;
	a+=data[57];
	a<<=8;
	a+=data[58];
	a<<=8;
	w[14]=a+data[59];

	a=data[60];
	a<<=8;
	a+=data[61];
	a<<=8;
	a+=data[62];
	a<<=8;
	w[15]=a+data[63];

	/* copy old result 32 bit clean to work variables */

	a=sha1[0];
	b=sha1[1];
	c=sha1[2];
	d=sha1[3];
	e=sha1[4];

	/* process work buffer with optimized f(B,C,D)=(B&C)|((~B)&D) */

	e+=w[0];
	e+=0x5A827999;
	e+=(a<<5)|BITS32(a>>27);
	e+=((c^d)&b)^d;
	b=(b<<30)|BITS32(b>>2);

	d+=w[1];
	d+=0x5A827999;
	d+=(e<<5)|BITS32(e>>27);
	d+=((b^c)&a)^c;
	a=(a<<30)|BITS32(a>>2);

	c+=w[2];
	c+=0x5A827999;
	c+=(d<<5)|BITS32(d>>27);
	c+=((a^b)&e)^b;
	e=(e<<30)|BITS32(e>>2);

	b+=w[3];
	b+=0x5A827999;
	b+=(c<<5)|BITS32(c>>27);
	b+=((e^a)&d)^a;
	d=(d<<30)|BITS32(d>>2);

	a+=w[4];
	a+=0x5A827999;
	a+=(b<<5)|BITS32(b>>27);
	a+=((d^e)&c)^e;
	c=(c<<30)|BITS32(c>>2);

	e+=w[5];
	e+=0x5A827999;
	e+=(a<<5)|BITS32(a>>27);
	e+=((c^d)&b)^d;
	b=(b<<30)|BITS32(b>>2);

	d+=w[6];
	d+=0x5A827999;
	d+=(e<<5)|BITS32(e>>27);
	d+=((b^c)&a)^c;
	a=(a<<30)|BITS32(a>>2);

	c+=w[7];
	c+=0x5A827999;
	c+=(d<<5)|BITS32(d>>27);
	c+=((a^b)&e)^b;
	e=(e<<30)|BITS32(e>>2);

	b+=w[8];
	b+=0x5A827999;
	b+=(c<<5)|BITS32(c>>27);
	b+=((e^a)&d)^a;
	d=(d<<30)|BITS32(d>>2);

	a+=w[9];
	a+=0x5A827999;
	a+=(b<<5)|BITS32(b>>27);
	a+=((d^e)&c)^e;
	c=(c<<30)|BITS32(c>>2);

	e+=w[10];
	e+=0x5A827999;
	e+=(a<<5)|BITS32(a>>27);
	e+=((c^d)&b)^d;
	b=(b<<30)|BITS32(b>>2);

	d+=w[11];
	d+=0x5A827999;
	d+=(e<<5)|BITS32(e>>27);
	d+=((b^c)&a)^c;
	a=(a<<30)|BITS32(a>>2);

	c+=w[12];
	c+=0x5A827999;
	c+=(d<<5)|BITS32(d>>27);
	c+=((a^b)&e)^b;
	e=(e<<30)|BITS32(e>>2);

	b+=w[13];
	b+=0x5A827999;
	b+=(c<<5)|BITS32(c>>27);
	b+=((e^a)&d)^a;
	d=(d<<30)|BITS32(d>>2);

	a+=w[14];
	a+=0x5A827999;
	a+=(b<<5)|BITS32(b>>27);
	a+=((d^e)&c)^e;
	c=(c<<30)|BITS32(c>>2);

	e+=w[15];
	e+=0x5A827999;
	e+=(a<<5)|BITS32(a>>27);
	e+=((c^d)&b)^d;
	b=(b<<30)|BITS32(b>>2);

	w[0]^=w[13]^w[8]^w[2];
	w[0]=(w[0]<<1)|BITS32(w[0]>>31);
	d+=w[0];
	d+=0x5A827999;
	d+=(e<<5)|BITS32(e>>27);
	d+=((b^c)&a)^c;
	a=(a<<30)|BITS32(a>>2);

	w[1]^=w[14]^w[9]^w[3];
	w[1]=(w[1]<<1)|BITS32(w[1]>>31);
	c+=w[1];
	c+=0x5A827999;
	c+=(d<<5)|BITS32(d>>27);
	c+=((a^b)&e)^b;
	e=(e<<30)|BITS32(e>>2);

	w[2]^=w[15]^w[10]^w[4];
	w[2]=(w[2]<<1)|BITS32(w[2]>>31);
	b+=w[2];
	b+=0x5A827999;
	b+=(c<<5)|BITS32(c>>27);
	b+=((e^a)&d)^a;
	d=(d<<30)|BITS32(d>>2);

	w[3]^=w[0]^w[11]^w[5];
	w[3]=(w[3]<<1)|BITS32(w[3]>>31);
	a+=w[3];
	a+=0x5A827999;
	a+=(b<<5)|BITS32(b>>27);
	a+=((d^e)&c)^e;
	c=(c<<30)|BITS32(c>>2);

	/* process work buffer with f(B,C,D)=B^C^D */

	w[4]^=w[1]^w[12]^w[6];
	w[4]=(w[4]<<1)|BITS32(w[4]>>31);
	e+=w[4];
	e+=0x6ED9EBA1;
	e+=(a<<5)|BITS32(a>>27);
	e+=b^c^d;
	b=(b<<30)|BITS32(b>>2);

	w[5]^=w[2]^w[13]^w[7];
	w[5]=(w[5]<<1)|BITS32(w[5]>>31);
	d+=w[5];
	d+=0x6ED9EBA1;
	d+=(e<<5)|BITS32(e>>27);
	d+=a^b^c;
	a=(a<<30)|BITS32(a>>2);

	w[6]^=w[3]^w[14]^w[8];
	w[6]=(w[6]<<1)|BITS32(w[6]>>31);
	c+=w[6];
	c+=0x6ED9EBA1;
	c+=(d<<5)|BITS32(d>>27);
	c+=e^a^b;
	e=(e<<30)|BITS32(e>>2);

	w[7]^=w[4]^w[15]^w[9];
	w[7]=(w[7]<<1)|BITS32(w[7]>>31);
	b+=w[7];
	b+=0x6ED9EBA1;
	b+=(c<<5)|BITS32(c>>27);
	b+=d^e^a;
	d=(d<<30)|BITS32(d>>2);

	w[8]^=w[5]^w[0]^w[10];
	w[8]=(w[8]<<1)|BITS32(w[8]>>31);
	a+=w[8];
	a+=0x6ED9EBA1;
	a+=(b<<5)|BITS32(b>>27);
	a+=c^d^e;
	c=(c<<30)|BITS32(c>>2);

	w[9]^=w[6]^w[1]^w[11];
	w[9]=(w[9]<<1)|BITS32(w[9]>>31);
	e+=w[9];
	e+=0x6ED9EBA1;
	e+=(a<<5)|BITS32(a>>27);
	e+=b^c^d;
	b=(b<<30)|BITS32(b>>2);

	w[10]^=w[7]^w[2]^w[12];
	w[10]=(w[10]<<1)|BITS32(w[10]>>31);
	d+=w[10];
	d+=0x6ED9EBA1;
	d+=(e<<5)|BITS32(e>>27);
	d+=a^b^c;
	a=(a<<30)|BITS32(a>>2);

	w[11]^=w[8]^w[3]^w[13];
	w[11]=(w[11]<<1)|BITS32(w[11]>>31);
	c+=w[11];
	c+=0x6ED9EBA1;
	c+=(d<<5)|BITS32(d>>27);
	c+=e^a^b;
	e=(e<<30)|BITS32(e>>2);

	w[12]^=w[9]^w[4]^w[14];
	w[12]=(w[12]<<1)|BITS32(w[12]>>31);
	b+=w[12];
	b+=0x6ED9EBA1;
	b+=(c<<5)|BITS32(c>>27);
	b+=d^e^a;
	d=(d<<30)|BITS32(d>>2);

	w[13]^=w[10]^w[5]^w[15];
	w[13]=(w[13]<<1)|BITS32(w[13]>>31);
	a+=w[13];
	a+=0x6ED9EBA1;
	a+=(b<<5)|BITS32(b>>27);
	a+=c^d^e;
	c=(c<<30)|BITS32(c>>2);

	w[14]^=w[11]^w[6]^w[0];
	w[14]=(w[14]<<1)|BITS32(w[14]>>31);
	e+=w[14];
	e+=0x6ED9EBA1;
	e+=(a<<5)|BITS32(a>>27);
	e+=b^c^d;
	b=(b<<30)|BITS32(b>>2);

	w[15]^=w[12]^w[7]^w[1];
	w[15]=(w[15]<<1)|BITS32(w[15]>>31);
	d+=w[15];
	d+=0x6ED9EBA1;
	d+=(e<<5)|BITS32(e>>27);
	d+=a^b^c;
	a=(a<<30)|BITS32(a>>2);

	w[0]^=w[13]^w[8]^w[2];
	w[0]=(w[0]<<1)|BITS32(w[0]>>31);
	c+=w[0];
	c+=0x6ED9EBA1;
	c+=(d<<5)|BITS32(d>>27);
	c+=e^a^b;
	e=(e<<30)|BITS32(e>>2);

	w[1]^=w[14]^w[9]^w[3];
	w[1]=(w[1]<<1)|BITS32(w[1]>>31);
	b+=w[1];
	b+=0x6ED9EBA1;
	b+=(c<<5)|BITS32(c>>27);
	b+=d^e^a;
	d=(d<<30)|BITS32(d>>2);

	w[2]^=w[15]^w[10]^w[4];
	w[2]=(w[2]<<1)|BITS32(w[2]>>31);
	a+=w[2];
	a+=0x6ED9EBA1;
	a+=(b<<5)|BITS32(b>>27);
	a+=c^d^e;
	c=(c<<30)|BITS32(c>>2);

	w[3]^=w[0]^w[11]^w[5];
	w[3]=(w[3]<<1)|BITS32(w[3]>>31);
	e+=w[3];
	e+=0x6ED9EBA1;
	e+=(a<<5)|BITS32(a>>27);
	e+=b^c^d;
	b=(b<<30)|BITS32(b>>2);

	w[4]^=w[1]^w[12]^w[6];
	w[4]=(w[4]<<1)|BITS32(w[4]>>31);
	d+=w[4];
	d+=0x6ED9EBA1;
	d+=(e<<5)|BITS32(e>>27);
	d+=a^b^c;
	a=(a<<30)|BITS32(a>>2);

	w[5]^=w[2]^w[13]^w[7];
	w[5]=(w[5]<<1)|BITS32(w[5]>>31);
	c+=w[5];
	c+=0x6ED9EBA1;
	c+=(d<<5)|BITS32(d>>27);
	c+=e^a^b;
	e=(e<<30)|BITS32(e>>2);

	w[6]^=w[3]^w[14]^w[8];
	w[6]=(w[6]<<1)|BITS32(w[6]>>31);
	b+=w[6];
	b+=0x6ED9EBA1;
	b+=(c<<5)|BITS32(c>>27);
	b+=d^e^a;
	d=(d<<30)|BITS32(d>>2);

	w[7]^=w[4]^w[15]^w[9];
	w[7]=(w[7]<<1)|BITS32(w[7]>>31);
	a+=w[7];
	a+=0x6ED9EBA1;
	a+=(b<<5)|BITS32(b>>27);
	a+=c^d^e;
	c=(c<<30)|BITS32(c>>2);

	/* process work buffer with optimized f(B,C,D)=(B&C)|(B&D)^(C&D) */

	w[8]^=w[5]^w[0]^w[10];
	w[8]=(w[8]<<1)|BITS32(w[8]>>31);
	e+=w[8];
	e+=0x8F1BBCDC;
	e+=(a<<5)|BITS32(a>>27);
	e+=(b&c)|((b|c)&d);
	b=(b<<30)|BITS32(b>>2);

	w[9]^=w[6]^w[1]^w[11];
	w[9]=(w[9]<<1)|BITS32(w[9]>>31);
	d+=w[9];
	d+=0x8F1BBCDC;
	d+=(e<<5)|BITS32(e>>27);
	d+=(a&b)|((a|b)&c);
	a=(a<<30)|BITS32(a>>2);

	w[10]^=w[7]^w[2]^w[12];
	w[10]=(w[10]<<1)|BITS32(w[10]>>31);
	c+=w[10];
	c+=0x8F1BBCDC;
	c+=(d<<5)|BITS32(d>>27);
	c+=(e&a)|((e|a)&b);
	e=(e<<30)|BITS32(e>>2);

	w[11]^=w[8]^w[3]^w[13];
	w[11]=(w[11]<<1)|BITS32(w[11]>>31);
	b+=w[11];
	b+=0x8F1BBCDC;
	b+=(c<<5)|BITS32(c>>27);
	b+=(d&e)|((d|e)&a);
	d=(d<<30)|BITS32(d>>2);

	w[12]^=w[9]^w[4]^w[14];
	w[12]=(w[12]<<1)|BITS32(w[12]>>31);
	a+=w[12];
	a+=0x8F1BBCDC;
	a+=(b<<5)|BITS32(b>>27);
	a+=(c&d)|((c|d)&e);
	c=(c<<30)|BITS32(c>>2);

	w[13]^=w[10]^w[5]^w[15];
	w[13]=(w[13]<<1)|BITS32(w[13]>>31);
	e+=w[13];
	e+=0x8F1BBCDC;
	e+=(a<<5)|BITS32(a>>27);
	e+=(b&c)|((b|c)&d);
	b=(b<<30)|BITS32(b>>2);

	w[14]^=w[11]^w[6]^w[0];
	w[14]=(w[14]<<1)|BITS32(w[14]>>31);
	d+=w[14];
	d+=0x8F1BBCDC;
	d+=(e<<5)|BITS32(e>>27);
	d+=(a&b)|((a|b)&c);
	a=(a<<30)|BITS32(a>>2);

	w[15]^=w[12]^w[7]^w[1];
	w[15]=(w[15]<<1)|BITS32(w[15]>>31);
	c+=w[15];
	c+=0x8F1BBCDC;
	c+=(d<<5)|BITS32(d>>27);
	c+=(e&a)|((e|a)&b);
	e=(e<<30)|BITS32(e>>2);

	w[0]^=w[13]^w[8]^w[2];
	w[0]=(w[0]<<1)|BITS32(w[0]>>31);
	b+=w[0];
	b+=0x8F1BBCDC;
	b+=(c<<5)|BITS32(c>>27);
	b+=(d&e)|((d|e)&a);
	d=(d<<30)|BITS32(d>>2);

	w[1]^=w[14]^w[9]^w[3];
	w[1]=(w[1]<<1)|BITS32(w[1]>>31);
	a+=w[1];
	a+=0x8F1BBCDC;
	a+=(b<<5)|BITS32(b>>27);
	a+=(c&d)|((c|d)&e);
	c=(c<<30)|BITS32(c>>2);

	w[2]^=w[15]^w[10]^w[4];
	w[2]=(w[2]<<1)|BITS32(w[2]>>31);
	e+=w[2];
	e+=0x8F1BBCDC;
	e+=(a<<5)|BITS32(a>>27);
	e+=(b&c)|((b|c)&d);
	b=(b<<30)|BITS32(b>>2);

	w[3]^=w[0]^w[11]^w[5];
	w[3]=(w[3]<<1)|BITS32(w[3]>>31);
	d+=w[3];
	d+=0x8F1BBCDC;
	d+=(e<<5)|BITS32(e>>27);
	d+=(a&b)|((a|b)&c);
	a=(a<<30)|BITS32(a>>2);

	w[4]^=w[1]^w[12]^w[6];
	w[4]=(w[4]<<1)|BITS32(w[4]>>31);
	c+=w[4];
	c+=0x8F1BBCDC;
	c+=(d<<5)|BITS32(d>>27);
	c+=(e&a)|((e|a)&b);
	e=(e<<30)|BITS32(e>>2);

	w[5]^=w[2]^w[13]^w[7];
	w[5]=(w[5]<<1)|BITS32(w[5]>>31);
	b+=w[5];
	b+=0x8F1BBCDC;
	b+=(c<<5)|BITS32(c>>27);
	b+=(d&e)|((d|e)&a);
	d=(d<<30)|BITS32(d>>2);

	w[6]^=w[3]^w[14]^w[8];
	w[6]=(w[6]<<1)|BITS32(w[6]>>31);
	a+=w[6];
	a+=0x8F1BBCDC;
	a+=(b<<5)|BITS32(b>>27);
	a+=(c&d)|((c|d)&e);
	c=(c<<30)|BITS32(c>>2);

	w[7]^=w[4]^w[15]^w[9];
	w[7]=(w[7]<<1)|BITS32(w[7]>>31);
	e+=w[7];
	e+=0x8F1BBCDC;
	e+=(a<<5)|BITS32(a>>27);
	e+=(b&c)|((b|c)&d);
	b=(b<<30)|BITS32(b>>2);

	w[8]^=w[5]^w[0]^w[10];
	w[8]=(w[8]<<1)|BITS32(w[8]>>31);
	d+=w[8];
	d+=0x8F1BBCDC;
	d+=(e<<5)|BITS32(e>>27);
	d+=(a&b)|((a|b)&c);
	a=(a<<30)|BITS32(a>>2);

	w[9]^=w[6]^w[1]^w[11];
	w[9]=(w[9]<<1)|BITS32(w[9]>>31);
	c+=w[9];
	c+=0x8F1BBCDC;
	c+=(d<<5)|BITS32(d>>27);
	c+=(e&a)|((e|a)&b);
	e=(e<<30)|BITS32(e>>2);

	w[10]^=w[7]^w[2]^w[12];
	w[10]=(w[10]<<1)|BITS32(w[10]>>31);
	b+=w[10];
	b+=0x8F1BBCDC;
	b+=(c<<5)|BITS32(c>>27);
	b+=(d&e)|((d|e)&a);
	d=(d<<30)|BITS32(d>>2);

	w[11]^=w[8]^w[3]^w[13];
	w[11]=(w[11]<<1)|BITS32(w[11]>>31);
	a+=w[11];
	a+=0x8F1BBCDC;
	a+=(b<<5)|BITS32(b>>27);
	a+=(c&d)|((c|d)&e);
	c=(c<<30)|BITS32(c>>2);

	/* process work buffer with f(B,C,D)=B^C^D */

	w[12]^=w[9]^w[4]^w[14];
	w[12]=(w[12]<<1)|BITS32(w[12]>>31);
	e+=w[12];
	e+=0xCA62C1D6;
	e+=(a<<5)|BITS32(a>>27);
	e+=b^c^d;
	b=(b<<30)|BITS32(b>>2);

	w[13]^=w[10]^w[5]^w[15];
	w[13]=(w[13]<<1)|BITS32(w[13]>>31);
	d+=w[13];
	d+=0xCA62C1D6;
	d+=(e<<5)|BITS32(e>>27);
	d+=a^b^c;
	a=(a<<30)|BITS32(a>>2);

	w[14]^=w[11]^w[6]^w[0];
	w[14]=(w[14]<<1)|BITS32(w[14]>>31);
	c+=w[14];
	c+=0xCA62C1D6;
	c+=(d<<5)|BITS32(d>>27);
	c+=e^a^b;
	e=(e<<30)|BITS32(e>>2);

	w[15]^=w[12]^w[7]^w[1];
	w[15]=(w[15]<<1)|BITS32(w[15]>>31);
	b+=w[15];
	b+=0xCA62C1D6;
	b+=(c<<5)|BITS32(c>>27);
	b+=d^e^a;
	d=(d<<30)|BITS32(d>>2);

	w[0]^=w[13]^w[8]^w[2];
	w[0]=(w[0]<<1)|BITS32(w[0]>>31);
	a+=w[0];
	a+=0xCA62C1D6;
	a+=(b<<5)|BITS32(b>>27);
	a+=c^d^e;
	c=(c<<30)|BITS32(c>>2);

	w[1]^=w[14]^w[9]^w[3];
	w[1]=(w[1]<<1)|BITS32(w[1]>>31);
	e+=w[1];
	e+=0xCA62C1D6;
	e+=(a<<5)|BITS32(a>>27);
	e+=b^c^d;
	b=(b<<30)|BITS32(b>>2);

	w[2]^=w[15]^w[10]^w[4];
	w[2]=(w[2]<<1)|BITS32(w[2]>>31);
	d+=w[2];
	d+=0xCA62C1D6;
	d+=(e<<5)|BITS32(e>>27);
	d+=a^b^c;
	a=(a<<30)|BITS32(a>>2);

	w[3]^=w[0]^w[11]^w[5];
	w[3]=(w[3]<<1)|BITS32(w[3]>>31);
	c+=w[3];
	c+=0xCA62C1D6;
	c+=(d<<5)|BITS32(d>>27);
	c+=e^a^b;
	e=(e<<30)|BITS32(e>>2);

	w[4]^=w[1]^w[12]^w[6];
	w[4]=(w[4]<<1)|BITS32(w[4]>>31);
	b+=w[4];
	b+=0xCA62C1D6;
	b+=(c<<5)|BITS32(c>>27);
	b+=d^e^a;
	d=(d<<30)|BITS32(d>>2);

	w[5]^=w[2]^w[13]^w[7];
	w[5]=(w[5]<<1)|BITS32(w[5]>>31);
	a+=w[5];
	a+=0xCA62C1D6;
	a+=(b<<5)|BITS32(b>>27);
	a+=c^d^e;
	c=(c<<30)|BITS32(c>>2);

	w[6]^=w[3]^w[14]^w[8];
	w[6]=(w[6]<<1)|BITS32(w[6]>>31);
	e+=w[6];
	e+=0xCA62C1D6;
	e+=(a<<5)|BITS32(a>>27);
	e+=b^c^d;
	b=(b<<30)|BITS32(b>>2);

	w[7]^=w[4]^w[15]^w[9];
	w[7]=(w[7]<<1)|BITS32(w[7]>>31);
	d+=w[7];
	d+=0xCA62C1D6;
	d+=(e<<5)|BITS32(e>>27);
	d+=a^b^c;
	a=(a<<30)|BITS32(a>>2);

	w[8]^=w[5]^w[0]^w[10];
	w[8]=(w[8]<<1)|BITS32(w[8]>>31);
	c+=w[8];
	c+=0xCA62C1D6;
	c+=(d<<5)|BITS32(d>>27);
	c+=e^a^b;
	e=(e<<30)|BITS32(e>>2);

	w[9]^=w[6]^w[1]^w[11];
	w[9]=(w[9]<<1)|BITS32(w[9]>>31);
	b+=w[9];
	b+=0xCA62C1D6;
	b+=(c<<5)|BITS32(c>>27);
	b+=d^e^a;
	d=(d<<30)|BITS32(d>>2);

	w[10]^=w[7]^w[2]^w[12];
	w[10]=(w[10]<<1)|BITS32(w[10]>>31);
	a+=w[10];
	a+=0xCA62C1D6;
	a+=(b<<5)|BITS32(b>>27);
	a+=c^d^e;
	c=(c<<30)|BITS32(c>>2);

	w[11]^=w[8]^w[3]^w[13];
	w[11]=(w[11]<<1)|BITS32(w[11]>>31);
	e+=w[11];
	e+=0xCA62C1D6;
	e+=(a<<5)|BITS32(a>>27);
	e+=b^c^d;
	b=(b<<30)|BITS32(b>>2);

	w[12]^=w[9]^w[4]^w[14];
	w[12]=(w[12]<<1)|BITS32(w[12]>>31);
	d+=w[12];
	d+=0xCA62C1D6;
	d+=(e<<5)|BITS32(e>>27);
	d+=a^b^c;
	a=(a<<30)|BITS32(a>>2);

	w[13]^=w[10]^w[5]^w[15];
	w[13]=(w[13]<<1)|BITS32(w[13]>>31);
	c+=w[13];
	c+=0xCA62C1D6;
	c+=(d<<5)|BITS32(d>>27);
	c+=e^a^b;
	e=(e<<30)|BITS32(e>>2);

	w[14]^=w[11]^w[6]^w[0];
	w[14]=(w[14]<<1)|BITS32(w[14]>>31);
	b+=w[14];
	b+=0xCA62C1D6;
	b+=(c<<5)|BITS32(c>>27);
	b+=d^e^a;
	d=(d<<30)|BITS32(d>>2);

	w[15]^=w[12]^w[7]^w[1];
	w[15]=(w[15]<<1)|BITS32(w[15]>>31);
	a+=w[15];
	a+=0xCA62C1D6;
	a+=(b<<5)|BITS32(b>>27);
	a+=c^d^e;
	c=(c<<30)|BITS32(c>>2);

	/* create next hash result from previous result and work variables */

	sha1[0]+=a;
	sha1[1]+=b;
	sha1[2]+=c;
	sha1[3]+=d;
	sha1[4]+=e;
}

/*
 * procedure sha1init
 *
 * input: ptr - pointer to SHA1 internal data
 *
 * This procedure initializes the SHA1 internal data structure.
 */

#if defined(SHA1_FULL) || defined(SHA1_HMAC)

void CRYPTOCALL sha1init(register SHA1DATA *ptr)
{
	/* initialize all SHA1 internal data */

	ptr->total=ptr->size=0;
	ptr->sha1[0]=0x67452301;
	ptr->sha1[1]=0xEFCDAB89;
	ptr->sha1[2]=0x98BADCFE;
	ptr->sha1[3]=0x10325476;
	ptr->sha1[4]=0xC3D2E1F0;
}

#endif

/*
 * procedure sha1next
 *
 * input: data   - the data to be hashed
 *	  length - the length of the data to be hashed
 *	  ptr    - pointer to SHA1 internal data
 *
 * This procedure hashes the given block of data. Incomplete data blocks
 * are buffered in the SHA1 internal data structure.
 */

#if defined(SHA1_FULL) || defined(SHA1_HMAC)

void CRYPTOCALL sha1next(register WORD08 *data,register WORD32 length,
	register SHA1DATA *ptr)
{
	register WORD32 i;	/* buffer size/index	*/


	/* adjust total length */

	ptr->total+=length;

	/* while there is a remainder in the buffer and data are to be
	   processed and the buffer is notfull copy current byte to
	   buffer */

	for(i=ptr->size;(i&63)&&length;length--)ptr->bfr.b[i++]=*data++;

	/* if the buffer is full */

	if(i==64)
	{
		/* reset buffer size */

		i=0;

		/* hash buffer */

		sha1block(ptr->sha1,ptr->bfr.b);
	}

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)
		sha1block(ptr->sha1,data);

	/* copy remainder to buffer */

	for(;length;length--)ptr->bfr.b[i++]=*data++;

	/* save new buffer size/index */

	ptr->size=(WORD08)(i);
}

#endif

/*
 * procedure sha1end
 *
 * input:  ptr - pointer to SHA1 internal data
 *
 * output: result - the SHA1 hash
 *
 * This procedure finalizes and returns the SHA1 hash.
 */

#if defined(SHA1_FULL) || defined(SHA1_HMAC)

void CRYPTOCALL sha1end(register WORD08 *result,register SHA1DATA *ptr)
{
	register WORD32 i=ptr->size;		/* counter		*/


	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	ptr->bfr.b[i++]=0x80;
	if(i>56)
	{
		for(;i<64;i++)ptr->bfr.b[i]=0;
		i=0;
		sha1block(ptr->sha1,ptr->bfr.b);
	}
	for(;i<56;i++)ptr->bfr.b[i]=0;

	/* append bits 63 through 0 of total length in bits to buffer */

	ptr->bfr.b[56]=0;
	ptr->bfr.b[57]=0;
	ptr->bfr.b[58]=0;
	ptr->bfr.b[59]=(WORD08)(BITS32(ptr->total)>>29);
	ptr->bfr.b[60]=(WORD08)((ptr->total)>>21);
	ptr->bfr.b[61]=(WORD08)((ptr->total)>>13);
	ptr->bfr.b[62]=(WORD08)((ptr->total)>>5);
	ptr->bfr.b[63]=(WORD08)((ptr->total)<<3);

	/* hash buffer */

	sha1block(ptr->sha1,ptr->bfr.b);

	/* return final hash result to caller */

	result[ 0]=(WORD08)((ptr->sha1[0])>>24);
	result[ 1]=(WORD08)((ptr->sha1[0])>>16);
	result[ 2]=(WORD08)((ptr->sha1[0])>>8);
	result[ 3]=(WORD08) (ptr->sha1[0]);
	result[ 4]=(WORD08)((ptr->sha1[1])>>24);
	result[ 5]=(WORD08)((ptr->sha1[1])>>16);
	result[ 6]=(WORD08)((ptr->sha1[1])>>8);
	result[ 7]=(WORD08) (ptr->sha1[1]);
	result[ 8]=(WORD08)((ptr->sha1[2])>>24);
	result[ 9]=(WORD08)((ptr->sha1[2])>>16);
	result[10]=(WORD08)((ptr->sha1[2])>>8);
	result[11]=(WORD08) (ptr->sha1[2]);
	result[12]=(WORD08)((ptr->sha1[3])>>24);
	result[13]=(WORD08)((ptr->sha1[3])>>16);
	result[14]=(WORD08)((ptr->sha1[3])>>8);
	result[15]=(WORD08) (ptr->sha1[3]);
	result[16]=(WORD08)((ptr->sha1[4])>>24);
	result[17]=(WORD08)((ptr->sha1[4])>>16);
	result[18]=(WORD08)((ptr->sha1[4])>>8);
	result[19]=(WORD08) (ptr->sha1[4]);

	/* assert user data removal data from buffer */

#ifdef SHA1_PARANOID
	ptr->bfr.l[0]=0;
	ptr->bfr.l[1]=0;
	ptr->bfr.l[2]=0;
	ptr->bfr.l[3]=0;
	ptr->bfr.l[4]=0;
	ptr->bfr.l[5]=0;
	ptr->bfr.l[6]=0;
	ptr->bfr.l[7]=0;
	ptr->bfr.l[8]=0;
	ptr->bfr.l[9]=0;
	ptr->bfr.l[10]=0;
	ptr->bfr.l[11]=0;
	ptr->bfr.l[12]=0;
	ptr->bfr.l[13]=0;
	ptr->bfr.l[14]=0;
	ptr->bfr.l[15]=0;
#endif
}

#endif

/*
 * procedure sha1
 *
 * input:  data   - the data to be hashed
 *	   length - the length of the data to be hashed
 *
 * output: result - the SHA1 hash
 *
 * This procedure hashes the given data.
 */

#ifdef SHA1_FAST

void CRYPTOCALL sha1(register WORD08 *data,register WORD32 length,
	WORD08 *result)
{
	register WORD32 i;	/* counter and index		*/
	WORD32 total;		/* data length buffer		*/
	WORD32 sha1[5];		/* SHA1 current result buffer	*/
	union
	{
	  WORD32 l[16];		/* paranoia memory clear	*/
	  WORD08 b[64];		/* final block(s) buffer	*/
	}bfr;


	/* initialize SHA1 internal data */

	total=length;
	sha1[0]=0x67452301;
	sha1[1]=0xEFCDAB89;
	sha1[2]=0x98BADCFE;
	sha1[3]=0x10325476;
	sha1[4]=0xC3D2E1F0;

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)sha1block(sha1,data);

	/* copy remainder to buffer */

	for(i=0;length;length--)bfr.b[i++]=*data++;

	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	bfr.b[i++]=0x80;
	if(i>56)
	{
		for(;i<64;i++)bfr.b[i]=0;
		i=0;
		sha1block(sha1,bfr.b);
	}
	for(;i<56;i++)bfr.b[i]=0;

	/* append bits 63 through 0 of total length in bits to buffer */

	bfr.b[56]=0;
	bfr.b[57]=0;
	bfr.b[58]=0;
	bfr.b[59]=(WORD08)(BITS32(total)>>29);
	bfr.b[60]=(WORD08)(total>>21);
	bfr.b[61]=(WORD08)(total>>13);
	bfr.b[62]=(WORD08)(total>>5);
	bfr.b[63]=(WORD08)(total<<3);

	/* hash buffer */

	sha1block(sha1,bfr.b);

	/* return final hash result to caller */

	result[ 0]=(WORD08)((sha1[0])>>24);
	result[ 1]=(WORD08)((sha1[0])>>16);
	result[ 2]=(WORD08)((sha1[0])>>8);
	result[ 3]=(WORD08) (sha1[0]);
	result[ 4]=(WORD08)((sha1[1])>>24);
	result[ 5]=(WORD08)((sha1[1])>>16);
	result[ 6]=(WORD08)((sha1[1])>>8);
	result[ 7]=(WORD08) (sha1[1]);
	result[ 8]=(WORD08)((sha1[2])>>24);
	result[ 9]=(WORD08)((sha1[2])>>16);
	result[10]=(WORD08)((sha1[2])>>8);
	result[11]=(WORD08) (sha1[2]);
	result[12]=(WORD08)((sha1[3])>>24);
	result[13]=(WORD08)((sha1[3])>>16);
	result[14]=(WORD08)((sha1[3])>>8);
	result[15]=(WORD08) (sha1[3]);
	result[16]=(WORD08)((sha1[4])>>24);
	result[17]=(WORD08)((sha1[4])>>16);
	result[18]=(WORD08)((sha1[4])>>8);
	result[19]=(WORD08) (sha1[4]);

	/* assert user data removal data from buffer */

#ifdef SHA1_PARANOID
	bfr.l[0]=0;
	bfr.l[1]=0;
	bfr.l[2]=0;
	bfr.l[3]=0;
	bfr.l[4]=0;
	bfr.l[5]=0;
	bfr.l[6]=0;
	bfr.l[7]=0;
	bfr.l[8]=0;
	bfr.l[9]=0;
	bfr.l[10]=0;
	bfr.l[11]=0;
	bfr.l[12]=0;
	bfr.l[13]=0;
	bfr.l[14]=0;
	bfr.l[15]=0;
#endif

}

#endif

/*
 * procedure sha1hmkey
 *
 * input:  key    - the HMAC key
 *	   length - the length of the HMAC key
 *
 * output: ptr - the preprocessed HMAC key hashes
 *
 * This procedure processes a key for HMAC use.
 */

#ifdef SHA1_HMAC

void CRYPTOCALL sha1hmkey(WORD08 *key,WORD32 keylength,SHA1HMDATA *ptr)
{
	register WORD32 i;		/* counter and index	*/
	SHA1(hash);			/* hash of key		*/
	union
	{
		WORD08 pad[64];		/* padding buffer	*/
		SHA1DATA sha1data;	/* SHA1 internal data	*/
	}u;


	/* if the key is larger than 64 bytes hash it and use the
	   hash as the key */

	if(keylength>64)
	{
		sha1init(&u.sha1data);
		sha1next(key,keylength,&u.sha1data);
		sha1end(hash,&u.sha1data);
		key=hash;
		keylength=SHA1_SIZE;
		u.sha1data.sha1[0]=u.sha1data.sha1[1]=
			u.sha1data.sha1[2]=u.sha1data.sha1[3]=
			u.sha1data.sha1[4]=0;
	}

	/* copy initial values to HMAC internal data structure */

	ptr->isha1[0]=ptr->osha1[0]=0x67452301;
	ptr->isha1[1]=ptr->osha1[1]=0xEFCDAB89;
	ptr->isha1[2]=ptr->osha1[2]=0x98BADCFE;
	ptr->isha1[3]=ptr->osha1[3]=0x10325476;
	ptr->isha1[4]=ptr->osha1[4]=0xC3D2E1F0;

	/* copy key to padding buffer, fill up buffer
	   and pre-hash inner pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x36;
	for(;i<64;i++)u.pad[i]=0x36;
	sha1block(ptr->isha1,u.pad);

	/* copy key to padding buffer, fill up buffer
	   and pre-hash outer pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x5c;
	for(;i<64;i++)u.pad[i]=0x5c;
	sha1block(ptr->osha1,u.pad);

	/* clean up */

	for(i=0;i<SHA1_SIZE;i++)hash[i]=0;
	for(i=0;i<64;i++)u.pad[i]=0;
}

#endif

/*
 * procedure sha1hminit
 *
 * input:  key - the preprocessed HMAC key
 *
 * output: ptr - the SHA1 internal data
 *
 * This procedure initializes SHA1 data for HMAC use.
 */

#ifdef SHA1_HMAC_FULL

void CRYPTOCALL sha1hminit(SHA1DATA *ptr,SHA1HMDATA *key)
{
	ptr->total=64;
	ptr->size=0;
	ptr->sha1[0]=key->isha1[0];
	ptr->sha1[1]=key->isha1[1];
	ptr->sha1[2]=key->isha1[2];
	ptr->sha1[3]=key->isha1[3];
	ptr->sha1[4]=key->isha1[4];
}

#endif

/*
 * procedure sha1hmend
 *
 * input:  key - the preprocessed HMAC key
 *	   ptr - the SHA1 internal data
 *
 * output: result - the HMAC result (a SHA1 hash)
 *
 * This procedure finalizes and returns the HMAC.
 */

#ifdef SHA1_HMAC_FULL

void CRYPTOCALL sha1hmend(WORD08 *result,SHA1DATA *ptr,SHA1HMDATA *key)
{
	/* complete inner SHA1 */

	sha1end(result,ptr);

	/* initialize outer SHA1 */

	ptr->total=64;
	ptr->size=0;
	ptr->sha1[0]=key->osha1[0];
	ptr->sha1[1]=key->osha1[1];
	ptr->sha1[2]=key->osha1[2];
	ptr->sha1[3]=key->osha1[3];
	ptr->sha1[4]=key->osha1[4];

	sha1next(result,SHA1_SIZE,ptr);
	sha1end(result,ptr);
}

#endif

/*
 * procedure sha1hmac
 *
 * input:  data   - the data to be processed
 *	   length - the length of the data to be processed
 *	   key    - the preprocessed HMAC key
 *
 * output: result - the HMAC result (a SHA1 hash)
 *
 * This procedure creates and returns a HMAC.
 */

#ifdef SHA1_HMAC_FAST

void CRYPTOCALL sha1hmac(WORD08 *data,WORD32 length,WORD08 *result,
	SHA1HMDATA *key)
{
	SHA1DATA sha1data;	/* SHA1 internal data */


	/* initialize inner SHA1 */

	sha1data.total=64;
	sha1data.size=0;
	sha1data.sha1[0]=key->isha1[0];
	sha1data.sha1[1]=key->isha1[1];
	sha1data.sha1[2]=key->isha1[2];
	sha1data.sha1[3]=key->isha1[3];
	sha1data.sha1[4]=key->isha1[4];

	/* hash given data */

	sha1next(data,length,&sha1data);

	/* complete inner SHA1 */

	sha1end(result,&sha1data);

	/* initialize outer SHA1 */

	sha1data.total=64;
	sha1data.size=0;
	sha1data.sha1[0]=key->osha1[0];
	sha1data.sha1[1]=key->osha1[1];
	sha1data.sha1[2]=key->osha1[2];
	sha1data.sha1[3]=key->osha1[3];
	sha1data.sha1[4]=key->osha1[4];

	/* complete and return outer SHA1 */

	sha1next(result,SHA1_SIZE,&sha1data);
	sha1end(result,&sha1data);
}

#endif

