/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, ast@domdv.de
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 * (c) 2001 Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* slippery: slip interface creation and deletion                             */
/*============================================================================*/

/*
 * sliphandler
 *
 * input:  smode  - bitmap, see below
 *		    bit 0: 0=use pty, 1=use serial line
 *		    bit 1: 0=create interface, 1=delete interface
 *	   anchor - pointer to vpnd global data
 *
 * return: in interface creation mode
 *
 *	   0 - success
 *	   1 - no tty/pty available
 *	   2 - tty configuration failed
 *	   3 - saving/setting line discipline failed
 *	   4 - setting slip encapsulation/ifconfig failed
 *	   5 - creating route to partner failed
 *	   6 - creating priority route to peer failed
 *	   7 - creation of additional routes failed
 *	   8 - no icmp socket (with keepalive only)
 *
 *	   in interface deletion mode:
 *
 *	   0 - success
 *	   1 - tty/pty close error
 *	   2 - device config restore failed
 *	   3 - line discipline restore failed
 *	   4 - ifconfig down failed
 *	   5 - deleting route to partner failed
 *	   6 - deletion of priority route or additional routes failed
 *
 * This procedure creates or deletes the proxy slip interface.
 */

int sliphandler(int smode,VPN *anchor)
{
#ifdef SunOS
	return 3;
#else
	int i;			/* general purpose	*/
	int j;			/* general purpose	*/
	int k;			/* general purpose	*/
	char bfr[16];		/* string creation	*/
	int err;		/* return code		*/
	int quiet;		/* >0 for quiet errors	*/
	struct ifreq ifr;	/* used to set up if	*/
#if defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	struct ifaliasreq ifa;	/* used to set up if	*/
#endif
	struct termios t;	/* pty/tty settings	*/


	/* debug message */

	ENTER("sliphandler");

	/* reset return code */

	err=0;

	/* handle deletion requests in error handler tree */

	if(smode&2)goto delete;

	/* silence error handling */

	quiet=1;

	/* if a pseudo terminal has to be used */

	if(!(smode&1))
	{
		/* reset master to 'no device' */

		anchor->pty=-1;

		/* try to get pty/tty device pair (Unix98 style) */

#if defined(TIOCGPTN) && defined(TIOCSPTLCK)
		if((anchor->pty=open("/dev/ptmx",O_RDWR))>=0)
		{
			j=0;
			if(!ioctl(anchor->pty,TIOCGPTN,&i)&&
				!ioctl(anchor->pty,TIOCSPTLCK,&j))
			{
				j=1;
				sprintf(bfr,"/dev/pts/%d",i);
				if((anchor->tty=open(bfr,
					O_RDWR|O_NOCTTY|O_NDELAY))==-1)
				{
					close(anchor->pty);
					anchor->pty=-1;
				}
				else ioctl(anchor->pty,TIOCSPTLCK,&j);
			}
			else
			{
				close(anchor->pty);
				anchor->pty=-1;
			}
		}
#endif

		/* try to get pty/tty device pair (BSD style) */

		if(anchor->pty<0)
		    for(anchor->pty=-1,i='p';i<'t'&&anchor->pty<0;i++)
			for(j=0;j<16;j++)
		{
			sprintf(bfr,"/dev/pty%c%x",i,j);
			if((anchor->pty=open(bfr,O_RDWR))>=0)
			{
				bfr[5]='t';
				if((anchor->tty=open(bfr,
					O_RDWR|O_NOCTTY|O_NDELAY))>=0)
					break;
				close(anchor->pty);
				anchor->pty=-1;
			}
		}

		/* signal error in case of failure */

		if(anchor->pty<0)JUMP("open",err1);

		/* save tty device attributes, react to errors */

		if(tcgetattr(anchor->tty,&(anchor->original))<0)
			JUMP("tcgetattr",err2);

		/* set tty device attributes, react to errors */

		t.c_cflag=CS8|CREAD|HUPCL|CLOCAL;
		t.c_iflag=IGNBRK|IGNPAR;
		t.c_oflag=0;
		t.c_lflag=0;
		for(i=0;i<NCCS;i++)t.c_cc[i]=0;
		t.c_cc[VMIN]=1;
		t.c_cc[VTIME]=0;
		cfsetispeed(&t,B115200);
		cfsetospeed(&t,B115200);
		if(tcsetattr(anchor->tty,TCSAFLUSH,&t)<0)JUMP("tcsetattr",err2);
	}

	/* in serial line mode use serial line handle */

	else anchor->tty=anchor->serial;

	/* save old line discipline, react to errors */

	if(ioctl(anchor->tty,TIOCGETD,&(anchor->disc))<0)
		JUMP("ioctl(TIOCGETD)",err3);

	/* set line discipline to SLIP, react to errors */

#ifdef LINUX
	i=N_SLIP;
	if((anchor->proxy=ioctl(anchor->tty,TIOCSETD,&i))<0)
		JUMP("ioctl(TIOCSETD)",err3);
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	i=SLIPDISC;
	if(ioctl(anchor->tty,TIOCSETD,&i)<0)
		JUMP("ioctl(TIOCSETD)",err3);
	if(ioctl(anchor->tty,SLIOCGUNIT,&(anchor->proxy))<0)
		JUMP("ioctl(SLIOCGUNIT)",err3);
#endif

	/* set encapsulation, react to errors */

#ifdef LINUX
	i=anchor->cslip?SL_MODE_CSLIP:SL_MODE_SLIP;
	if(ioctl(anchor->tty,SIOCSIFENCAP,&i)<0)
		JUMP("ioctl(SIOCSIFENCAP)",err4);
#endif

	/* get control socket, react to errors */

	if((i=socket(AF_INET,SOCK_DGRAM,0))==-1)JUMP("socket",err4);

	/* reset error flag */

	j=0;

	/* create interface name */

	memset(&ifr,0,sizeof(ifr));
	sprintf(ifr.ifr_name,"sl%u",anchor->proxy);
#if defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	memset(&ifa,0,sizeof(ifa));
	sprintf(ifa.ifra_name,"sl%u",anchor->proxy);
#endif

	/* set up interface address, react to errors */

#ifdef LINUX
	((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->local_ip));
	if(ioctl(i,SIOCSIFADDR,&ifr)<0)j++;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_len=
		sizeof(struct sockaddr_in);
	((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->local_ip));
#endif

	/* set up interface peer address, react to errors */

#ifdef LINUX
	((struct sockaddr_in *)(&(ifr.ifr_dstaddr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_dstaddr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->remote_ip));
	if(!j)if(ioctl(i,SIOCSIFDSTADDR,&ifr)<0)j++;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_len=
		sizeof(struct sockaddr_in);
	((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->remote_ip));
#endif

	/* set up interface netmask, react to errors */

#ifdef LINUX
	((struct sockaddr_in *)(&(ifr.ifr_netmask)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_netmask)))->sin_addr.s_addr=
		htonl(0xffffffff);
	if(!j)if(ioctl(i,SIOCSIFNETMASK,&ifr)<0)j++;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	((struct sockaddr_in *)(&(ifa.ifra_mask)))->sin_len=
		sizeof(struct sockaddr_in);
	((struct sockaddr_in *)(&(ifa.ifra_mask)))->sin_addr.s_addr=
		htonl(0xffffffff);
	if(!j)if(ioctl(i,SIOCAIFADDR,&ifa)<0)j++;
#endif

	/* set up interface mtu, react to errors */

#ifndef OSOpenBSD
	ifr.ifr_mtu=anchor->mtu;
	if(!j)if(ioctl(i,SIOCSIFMTU,&ifr)<0)j++;
#endif

	/* activate interface, react to errors */

#ifdef LINUX
	ifr.ifr_flags=IFF_UP|IFF_RUNNING|IFF_POINTOPOINT|IFF_NOARP|
		IFF_MULTICAST;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	ifr.ifr_flags=IFF_UP|IFF_POINTOPOINT|IFF_NOARP|IFF_MULTICAST;
	if(anchor->cslip)ifr.ifr_flags|=SC_COMPRESS;
#endif
	if(!j)if(ioctl(i,SIOCSIFFLAGS,&ifr)<0)j++;

	/* close control socket */
 
	if(close(i))ERRNO("close");

	/* signal any errors encountered */

#ifdef LINUX
	if(j)JUMP("ioctl(SIOCSIFADDR,SIOCSIFDSTADDR,SIOCSIFNETMASK," \
		"SIOCSIFMTU,SIOCSIFFLAGS)",err4);
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	if(j)JUMP("ioctl(SIOCAIFADDR,SIOCSIFMTU,SIOCSIFFLAGS)",err4);
#endif

	/* set up route to partner, react to errors */

#ifdef LINUX
	if(!anchor->autoroute)
	{
		if(route(1,getaddr((unsigned char *)(anchor->remote_ip)),-1,0,
			(unsigned char *)(ifr.ifr_name)))
				GOTO("autoroute",err5);
	}
#endif

	/* set up priority route to peer, react to errors */

	if(anchor->peerroute)
		if(route(1,anchor->peerip,-1,
			anchor->peerroute>1?anchor->peergate:0,
			(unsigned char *)(anchor->peerdev)))
				GOTO("peerroute",err6);

	/* set up all additional routes */

	for(k=0;k<anchor->rmax;k++)
		if(route(1,anchor->rtab[k][0],anchor->rtab[k][1],
			anchor->rtab[k][2],NULL))
		GOTO("routex",err7);

	/* in case of keepalive packages when link idle open icmp socket */

	if(anchor->keepalive)if(icmphandler(anchor))GOTO("icmphandler",err8);

	/* mark slip link as established */

	anchor->slip=1;

	/* execute slip up process, if there is any */

	doexec(anchor->slipup,ifr.ifr_name);

	/* signal success */

	goto err0;

	/* done, if no slip link */

delete:	if(!anchor->slip)GOTO("slip link not up",err0);

	/* create interface name */

	memset(&ifr,0,sizeof(ifr));
	sprintf(ifr.ifr_name,"sl%u",anchor->proxy);
#if defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	memset(&ifa,0,sizeof(ifa));
	sprintf(ifa.ifra_name,"sl%u",anchor->proxy);
#endif

	/* execute slip down process, if there is any */

	doexec(anchor->slipdown,ifr.ifr_name);

	/* enable verbose mode */

	quiet=0;

	/* in case of keepalive packages when link idle close icmp socket */

	if(anchor->keepalive)icmphandler(anchor);

	/* error handling */

err8:	if(quiet)err++;
err7:	if(quiet)err++;

	/* delete additional routes */

	for(k=anchor->rmax-1;k>=0;k--)
		if(route(0,anchor->rtab[k][0],anchor->rtab[k][1],
			anchor->rtab[k][2],NULL))if(!quiet++)DEBUG1(3,"routex");

	/* delete priority route to peer */

	if(anchor->peerroute)
		if(route(0,anchor->peerip,-1,
			anchor->peerroute>1?anchor->peergate:0,
			(unsigned char *)(anchor->peerdev)))
				if(!quiet++)DEBUG1(3,"peerroute");

err6:	if(quiet)err++;

	/* delete route to partner, react to errors */

#ifdef LINUX
	if(!anchor->autoroute)
	{
		if(route(0,getaddr((unsigned char *)(anchor->remote_ip)),-1,0,
			(unsigned char *)(ifr.ifr_name)))
				if(!quiet++)DEBUG1(3,"autoroute");
	}

err5:	if(quiet)err++;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	if(quiet)err++;
#endif

	/* get control socket, react to errors */

	if((i=socket(AF_INET,SOCK_DGRAM,0))==-1)
	{
		if(!quiet++)ERRNO("socket");
	}

	/* if control socket could be created */

	else
	{
		/* deactivate interface, react to errors */

#if defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
		((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_len=
			sizeof(struct sockaddr_in);
		((struct sockaddr_in *)(&(ifa.ifra_addr)))
			->sin_family=AF_INET;
		((struct sockaddr_in *)(&(ifa.ifra_addr)))
			->sin_addr.s_addr=getaddr(anchor->local_ip);
		((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_len=
			sizeof(struct sockaddr_in);
		((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))
			->sin_family=AF_INET;
		((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))
			->sin_addr.s_addr=getaddr(anchor->remote_ip);
		((struct sockaddr_in *)(&(ifa.ifra_mask)))->sin_len=
			sizeof(struct sockaddr_in);
		((struct sockaddr_in *)(&(ifa.ifra_mask)))
			->sin_addr.s_addr=htonl(0xffffffff);
		if(ioctl(i,SIOCDIFADDR,&ifa)<0)
			ERRNO("ioctl(SIOCDIFADDR)");
#endif

		ifr.ifr_flags=0;
		if(ioctl(i,SIOCSIFFLAGS,&ifr)<0)if(!quiet++)
			ERRNO("ioctl(SIOCSIFFLAGS)");

		/* reset interface address, react to errors */

#ifdef LINUX
		((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_family=AF_INET;
		((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_addr.s_addr=0;
		if(ioctl(i,SIOCSIFADDR,&ifr)<0)if(!quiet++)
			ERRNO("ioctl(SIOCSIFADDR)");
#endif

		/* close control socket */
 
		if(close(i))if(!quiet++)ERRNO("close");
	}

err4:	if(quiet)err++;

	/* restore original line discipline, react to errors */

	if(ioctl(anchor->tty,TIOCSETD,&anchor->disc)<0)if(!quiet++)
		ERRNO("ioctl(TIOCSETD");

err3:	if(quiet)err++;

	/* restore old device characteristics, react to errors */

	if(!(smode&1))if(tcsetattr(anchor->tty,TCSAFLUSH,&(anchor->original))<0)
		if(!quiet++)ERRNO("tcsetattr");

err2:	if(quiet)err++;

	/* close tty and pty */

	if(!(smode&1))
	{
		if(close(anchor->tty))if(!quiet++)ERRNO("tty close");
		if(close(anchor->pty))if(!quiet++)ERRNO("pty close");
	}

err1:	if(quiet)err++;

	/*  mark slip link as down */

	anchor->slip=0;

	/* debug message */

err0:	LEAVE("sliphandler");

	/* signal success or error */

	return err;
#endif
}
