/*
 * virtual private network daemon (vpnd)
 *
 * (c) 2001 Andreas Steinmetz, ast@domdv.de
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */
 
#include "vpnd.h"
 
/*============================================================================*/
/* the manager: control socket utility                                        */
/*============================================================================*/

/*
 * expired
 *
 * input: unused - unused parameter
 *
 * This procedure is called by the alarm timer signal if there's a
 * transaction timeout on the control channel. This should never
 * happen except when a unix domain socket of some other process than
 * vpnd is given on the command line.
 */

void expired(int unused)
{
	fprintf(stderr,"Request timed out.\n");
	exit(1);
}

/*
 * usage
 *
 * This procedure prints a usage message and then terminates the process.
 */

void usage(void)
{
	fprintf(stderr,"Usage: vpnctl status|down|force <control-socket>\n");
	exit(1);
}

/*
 * main
 *
 * input:  argc - the argument count
 *         argv - argument vector pointer
 *
 * return: 0 in case of success, 1 in case of error
 *
 * This procedure is the main procedure of the
 * virtual private network daemon control utility
 * (vpnctl).
 */

int main(int argc,char *argv[])
{
	int s;			/* socket handle	*/
	struct sockaddr_un a;	/* socket address	*/
	int m;
	WORD08 b;
	struct stat stb;


	/* shut up compiler */

	m=0;

	/* check argument count */

	if(argc!=3)usage();

	/* check if valid command */

	if(!strcmp(argv[1],"status"))m=CTRL_STATUS;
	else if(!strcmp(argv[1],"down"))m=CTRL_DOWN;
	else if(!strcmp(argv[1],"force"))m=CTRL_FORCEDOWN;
	else usage();

	/* check socket existence and access (must not be used suid!!!) */

	if(strlen(argv[2])>=UNIX_PATH_MAX)usage();
	if(lstat(argv[2],&stb))usage();
	if(!S_ISSOCK(stb.st_mode))usage();
	if(access(argv[2],R_OK|W_OK))usage();

	/* set up command */

	b=(WORD08)(m);

	/* create socket, haqndle errors */

	if((s=socket(PF_UNIX,SOCK_STREAM,0))==-1)
	{
		perror("socket");
		exit(1);
	}

	/* create peer address */

	memset(&a,0,sizeof(a));
	a.sun_family=AF_UNIX;
	strcpy(a.sun_path,argv[2]);

	/* set transaction timeout handling */

	signal(SIGALRM,expired);
	alarm(5);

	/* connect to peer, handle errors */

	if(connect(s,(struct sockaddr *)(&a),sizeof(a)))
	{
		perror("connect");
		exit(1);
	}

	/* do transaction, handle errors */

	if(write(s,&b,1)!=1)
	{
		perror("write");
		exit(1);
	}

	if(read(s,&b,1)!=1)
	{
		perror("read");
		exit(1);
	}

	/* close control channel */

	close(s);

	/* disable timer */

	alarm(0);

	/* handle peer link status queries */

	if(m==CTRL_STATUS)switch(b)
	{
		/* peer link is up */

		case CTRL_ISUP:
			printf("Connected.\n");
			return 0;

		/* peer link is down */

		case CTRL_ISDOWN:
			printf("Disconnected.\n");
			return 0;

		/* unknown reply */

		default:fprintf(stderr,"Unkown reply received.\n");
			return 1;
	}

	/* handle status */

	switch(b)
	{
		/* successful request */

		case CTRL_OK:
			printf("ok.\n");
			return 0;

		/* failed request */

		case CTRL_FAIL:
			printf("failed.\n");
			return 1;

		/* unknown reply */

		default:fprintf(stderr,"Unkown reply received.\n");
			return 1;
	}
}
