/* BFD back-end for HP PA-RISC ELF files.
   Copyright (C) 1990, 91, 92, 93, 94, 95, 96, 97, 98, 99, 2000
   Free Software Foundation, Inc.

   Original code by
	Center for Software Science
	Department of Computer Science
	University of Utah
   Largely rewritten by Alan Modra <alan@linuxcare.com.au>

This file is part of BFD, the Binary File Descriptor library.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include "bfd.h"
#include "sysdep.h"
#include "libbfd.h"
#include "elf-bfd.h"
#include "elf/hppa.h"
#include "libhppa.h"
#include "elf32-hppa.h"
#define ARCH_SIZE		32
#include "elf-hppa.h"
#include "elf32-hppa.h"

/* In order to gain some understanding of code in this file without
   knowing all the intricate details of the linker, note the
   following:

   Functions named elf32_hppa_* are called by external routines, other
   functions are only called locally.  elf32_hppa_* functions appear
   in this file more or less in the order in which they are called
   from external routines.  eg. elf32_hppa_check_relocs is called
   early in the link process, elf32_hppa_finish_dynamic_sections is
   one of the last functions.  */

/* We use two hash tables to hold information for linking PA ELF objects.

   The first is the elf32_hppa_link_hash_table which is derived
   from the standard ELF linker hash table.  We use this as a place to
   attach other hash tables and static information.

   The second is the stub hash table which is derived from the
   base BFD hash table.  The stub hash table holds the information
   necessary to build the linker stubs during a link.

   There are a number of different stubs generated by the linker.

   Long branch stub:
   :		ldil LR'X,%r1
   :		be,n RR'X(%sr4,%r1)

   PIC long branch stub:
   :		b,l .+8,%r1
   :		addil L'X - ($PIC_pcrel$0 - 4),%r1
   :		be,n R'X - ($PIC_pcrel$0 - 8)(%sr4,%r1)

   Import stub to call shared library routine from normal object file
   (single sub-space version)
   :		addil L'lt_ptr+ltoff,%dp	; get procedure entry point
   :		ldw R'lt_ptr+ltoff(%r1),%r21
   :            bv %r0(%r21)
   :		ldw R'lt_ptr+ltoff+4(%r1),%r19	; get new dlt value.

   Import stub to call shared library routine from shared library
   (single sub-space version)
   :		addil L'ltoff,%r19		; get procedure entry point
   :		ldw R'ltoff(%r1),%r21
   :            bv %r0(%r21)
   :		ldw R'ltoff+4(%r1),%r19		; get new dlt value.

   Import stub to call shared library routine from normal object file
   (multiple sub-space support)
   :		addil L'lt_ptr+ltoff,%dp	; get procedure entry point
   :		ldw R'lt_ptr+ltoff(%r1),%r21
   :		ldw R'lt_ptr+ltoff+4(%r1),%r19	; get new dlt value.
   :		ldsid (%r21),%r1
   :		mtsp %r1,%sr0
   :		be 0(%sr0,%r21)			; branch to target
   :		stw %rp,-24(%sp)		; save rp

   Import stub to call shared library routine from shared library
   (multiple sub-space support)
   :		addil L'ltoff,%r19		; get procedure entry point
   :		ldw R'ltoff(%r1),%r21
   :		ldw R'ltoff+4(%r1),%r19		; get new dlt value.
   :		ldsid (%r21),%r1
   :		mtsp %r1,%sr0
   :		be 0(%sr0,%r21)			; branch to target
   :		stw %rp,-24(%sp)		; save rp

   Export stub to return from shared lib routine (multiple sub-space support)
   One of these is created for each exported procedure in a shared
   library (and stored in the shared lib).  Shared lib routines are
   called via the first instruction in the export stub so that we can
   do an inter-space return.  Not required for single sub-space.
   :		bl,n X,%rp			; trap the return
   :		nop
   :		ldw -24(%sp),%rp		; restore the original rp
   :		ldsid (%rp),%r1
   :		mtsp %r1,%sr0
   :		be,n 0(%sr0,%rp)		; inter-space return  */

#define PLT_ENTRY_SIZE 8
#define PLABEL_PLT_ENTRY_SIZE PLT_ENTRY_SIZE
#define GOT_ENTRY_SIZE 4
#define ELF_DYNAMIC_INTERPRETER "/lib/ld.so.1"

static const bfd_byte plt_stub[] =
{
  0x0e, 0x80, 0x10, 0x96,  /* 1: ldw	0(%r20),%r22		*/
  0xea, 0xc0, 0xc0, 0x00,  /*    bv	%r0(%r22)		*/
  0x0e, 0x88, 0x10, 0x95,  /*    ldw	4(%r20),%r21		*/
#define PLT_STUB_ENTRY (3*4)
  0xea, 0x9f, 0x1f, 0xdd,  /*    b,l	1b,%r20			*/
  0xd6, 0x80, 0x1c, 0x1e,  /*    depi	0,31,2,%r20		*/
  0x00, 0xc0, 0xff, 0xee,  /* 9: .word	fixup_func		*/
  0xde, 0xad, 0xbe, 0xef   /*    .word	fixup_ltp		*/
};

/* Section name for stubs is the associated section name plus this
   string.  */
#define STUB_SUFFIX ".stub"

/* Setting the following non-zero makes all long branch stubs
   generated during a shared link of the PIC variety.  This saves on
   relocs, but costs one extra instruction per stub.  */
#ifndef LONG_BRANCH_PIC_IN_SHLIB
#define LONG_BRANCH_PIC_IN_SHLIB 1
#endif

/* Set this non-zero to use import stubs instead of long branch stubs
   where a .plt entry exists for the symbol.  This is a fairly useless
   option as import stubs are bigger than PIC long branch stubs.  */
#ifndef LONG_BRANCH_VIA_PLT
#define LONG_BRANCH_VIA_PLT 0
#endif

/* We don't need to copy any PC- or GP-relative dynamic relocs into a
   shared object's dynamic section.  */
#ifndef RELATIVE_DYNAMIC_RELOCS
#define RELATIVE_DYNAMIC_RELOCS 0
#endif

enum elf32_hppa_stub_type {
  hppa_stub_long_branch,
  hppa_stub_long_branch_shared,
  hppa_stub_import,
  hppa_stub_import_shared,
  hppa_stub_export,
  hppa_stub_none
};

struct elf32_hppa_stub_hash_entry {

  /* Base hash table entry structure.  */
  struct bfd_hash_entry root;

  /* The stub section.  */
  asection *stub_sec;

#if ! LONG_BRANCH_PIC_IN_SHLIB
  /* It's associated reloc section.  */
  asection *reloc_sec;
#endif

  /* Offset within stub_sec of the beginning of this stub.  */
  bfd_vma stub_offset;

  /* Given the symbol's value and its section we can determine its final
     value when building the stubs (so the stub knows where to jump.  */
  bfd_vma target_value;
  asection *target_section;

  enum elf32_hppa_stub_type stub_type;

  /* The symbol table entry, if any, that this was derived from.  */
  struct elf32_hppa_link_hash_entry *h;

  /* Where this stub is being called from, or, in the case of combined
     stub sections, the first input section in the group.  */
  asection *id_sec;
};

struct elf32_hppa_link_hash_entry {

  struct elf_link_hash_entry elf;

  /* A pointer to the most recently used stub hash entry against this
     symbol.  */
  struct elf32_hppa_stub_hash_entry *stub_cache;

#if ! LONG_BRANCH_PIC_IN_SHLIB
  /* Used to track whether we have allocated space for a long branch
     stub relocation for this symbol in the given section.  */
  asection *stub_reloc_sec;
#endif

#if ! LONG_BRANCH_PIC_IN_SHLIB || RELATIVE_DYNAMIC_RELOCS
  /* Used to count relocations for delayed sizing of relocation
     sections.  */
  struct elf32_hppa_dyn_reloc_entry {

    /* Next relocation in the chain.  */
    struct elf32_hppa_dyn_reloc_entry *next;

    /* The section in dynobj.  */
    asection *section;

    /* Number of relocs copied in this section.  */
    bfd_size_type count;
  } *reloc_entries;
#endif

  /* Set during a static link if we detect a function is PIC.  */
  unsigned int pic_call:1;

  /* Set if this symbol is used by a plabel reloc.  */
  unsigned int plabel:1;

  /* Set if this symbol is an init or fini function and thus should
     use an absolute reloc.  */
  unsigned int plt_abs:1;
};

struct elf32_hppa_link_hash_table {

  /* The main hash table.  */
  struct elf_link_hash_table root;

  /* The stub hash table.  */
  struct bfd_hash_table stub_hash_table;

  /* Linker stub bfd.  */
  bfd *stub_bfd;

  /* Linker call-backs.  */
  asection * (*add_stub_section) PARAMS ((const char *, asection *));
  void (*layout_sections_again) PARAMS ((void));

  /* Array to keep track of which stub sections have been created, and
     information on stub grouping.  */
  struct map_stub {
    /* This is the section to which stubs in the group will be
       attached.  */
    asection *link_sec;
    /* The stub section.  */
    asection *stub_sec;
#if ! LONG_BRANCH_PIC_IN_SHLIB
    /* The stub section's reloc section.  */
    asection *reloc_sec;
#endif
  } *stub_group;

  /* Short-cuts to get to dynamic linker sections.  */
  asection *sgot;
  asection *srelgot;
  asection *splt;
  asection *srelplt;
  asection *sdynbss;
  asection *srelbss;

  /* Whether we support multiple sub-spaces for shared libs.  */
  unsigned int multi_subspace:1;

  /* Flags set when PCREL12F and PCREL17F branches detected.  Used to
     select suitable defaults for the stub group size.  */
  unsigned int has_12bit_branch:1;
  unsigned int has_17bit_branch:1;

  /* Set if we need a .plt stub to support lazy dynamic linking.  */
  unsigned int need_plt_stub:1;
};

/* Various hash macros and functions.  */
#define hppa_link_hash_table(p) \
  ((struct elf32_hppa_link_hash_table *) ((p)->hash))

#define hppa_stub_hash_lookup(table, string, create, copy) \
  ((struct elf32_hppa_stub_hash_entry *) \
   bfd_hash_lookup ((table), (string), (create), (copy)))

static struct bfd_hash_entry *stub_hash_newfunc
  PARAMS ((struct bfd_hash_entry *, struct bfd_hash_table *, const char *));

static struct bfd_hash_entry *hppa_link_hash_newfunc
  PARAMS ((struct bfd_hash_entry *, struct bfd_hash_table *, const char *));

static struct bfd_link_hash_table *elf32_hppa_link_hash_table_create
  PARAMS ((bfd *));

/* Stub handling functions.  */
static char *hppa_stub_name
  PARAMS ((const asection *, const asection *,
	   const struct elf32_hppa_link_hash_entry *,
	   const Elf_Internal_Rela *));

static struct elf32_hppa_stub_hash_entry *hppa_get_stub_entry
  PARAMS ((const asection *, const asection *,
	   struct elf32_hppa_link_hash_entry *,
	   const Elf_Internal_Rela *,
	   struct elf32_hppa_link_hash_table *));

static struct elf32_hppa_stub_hash_entry *hppa_add_stub
  PARAMS ((const char *, asection *, struct elf32_hppa_link_hash_table *));

static enum elf32_hppa_stub_type hppa_type_of_stub
  PARAMS ((asection *, const Elf_Internal_Rela *,
	   struct elf32_hppa_link_hash_entry *, bfd_vma));

static boolean hppa_build_one_stub
  PARAMS ((struct bfd_hash_entry *, PTR));

static boolean hppa_size_one_stub
  PARAMS ((struct bfd_hash_entry *, PTR));

/* BFD and elf backend functions.  */
static boolean elf32_hppa_object_p PARAMS ((bfd *));

static boolean elf32_hppa_add_symbol_hook
  PARAMS ((bfd *, struct bfd_link_info *, const Elf_Internal_Sym *,
	   const char **, flagword *, asection **, bfd_vma *));

static boolean elf32_hppa_create_dynamic_sections
  PARAMS ((bfd *, struct bfd_link_info *));

static boolean elf32_hppa_check_relocs
  PARAMS ((bfd *, struct bfd_link_info *,
	   asection *, const Elf_Internal_Rela *));

static asection *elf32_hppa_gc_mark_hook
  PARAMS ((bfd *, struct bfd_link_info *, Elf_Internal_Rela *,
	   struct elf_link_hash_entry *, Elf_Internal_Sym *));

static boolean elf32_hppa_gc_sweep_hook
  PARAMS ((bfd *, struct bfd_link_info *,
	   asection *, const Elf_Internal_Rela *));

static void elf32_hppa_hide_symbol
  PARAMS ((struct bfd_link_info *, struct elf_link_hash_entry *));

static boolean elf32_hppa_adjust_dynamic_symbol
  PARAMS ((struct bfd_link_info *, struct elf_link_hash_entry *));

static boolean hppa_handle_PIC_calls
  PARAMS ((struct elf_link_hash_entry *, PTR));

#if ((! LONG_BRANCH_PIC_IN_SHLIB && LONG_BRANCH_VIA_PLT) \
     || RELATIVE_DYNAMIC_RELOCS)
static boolean hppa_discard_copies
  PARAMS ((struct elf_link_hash_entry *, PTR));
#endif

static boolean clobber_millicode_symbols
  PARAMS ((struct elf_link_hash_entry *, struct bfd_link_info *));

static boolean elf32_hppa_size_dynamic_sections
  PARAMS ((bfd *, struct bfd_link_info *));

static bfd_reloc_status_type final_link_relocate
  PARAMS ((asection *, bfd_byte *, const Elf_Internal_Rela *,
	   bfd_vma, struct elf32_hppa_link_hash_table *, asection *,
	   struct elf32_hppa_link_hash_entry *));

static boolean elf32_hppa_relocate_section
  PARAMS ((bfd *, struct bfd_link_info *, bfd *, asection *,
	   bfd_byte *, Elf_Internal_Rela *, Elf_Internal_Sym *, asection **));

static boolean elf32_hppa_finish_dynamic_symbol
  PARAMS ((bfd *, struct bfd_link_info *,
	   struct elf_link_hash_entry *, Elf_Internal_Sym *));

static boolean elf32_hppa_finish_dynamic_sections
  PARAMS ((bfd *, struct bfd_link_info *));

static int elf32_hppa_elf_get_symbol_type
  PARAMS ((Elf_Internal_Sym *, int));

/* Assorted hash table functions.  */

/* Initialize an entry in the stub hash table.  */

static struct bfd_hash_entry *
stub_hash_newfunc (entry, table, string)
     struct bfd_hash_entry *entry;
     struct bfd_hash_table *table;
     const char *string;
{
  struct elf32_hppa_stub_hash_entry *ret;

  ret = (struct elf32_hppa_stub_hash_entry *) entry;

  /* Allocate the structure if it has not already been allocated by a
     subclass.  */
  if (ret == NULL)
    {
      ret = ((struct elf32_hppa_stub_hash_entry *)
	     bfd_hash_allocate (table,
				sizeof (struct elf32_hppa_stub_hash_entry)));
      if (ret == NULL)
	return NULL;
    }

  /* Call the allocation method of the superclass.  */
  ret = ((struct elf32_hppa_stub_hash_entry *)
	 bfd_hash_newfunc ((struct bfd_hash_entry *) ret, table, string));

  if (ret)
    {
      /* Initialize the local fields.  */
      ret->stub_sec = NULL;
#if ! LONG_BRANCH_PIC_IN_SHLIB
      ret->reloc_sec = NULL;
#endif
      ret->stub_offset = 0;
      ret->target_value = 0;
      ret->target_section = NULL;
      ret->stub_type = hppa_stub_long_branch;
      ret->h = NULL;
      ret->id_sec = NULL;
    }

  return (struct bfd_hash_entry *) ret;
}

/* Initialize an entry in the link hash table.  */

static struct bfd_hash_entry *
hppa_link_hash_newfunc (entry, table, string)
     struct bfd_hash_entry *entry;
     struct bfd_hash_table *table;
     const char *string;
{
  struct elf32_hppa_link_hash_entry *ret;

  ret = (struct elf32_hppa_link_hash_entry *) entry;

  /* Allocate the structure if it has not already been allocated by a
     subclass.  */
  if (ret == NULL)
    {
      ret = ((struct elf32_hppa_link_hash_entry *)
	     bfd_hash_allocate (table,
				sizeof (struct elf32_hppa_link_hash_entry)));
      if (ret == NULL)
	return NULL;
    }

  /* Call the allocation method of the superclass.  */
  ret = ((struct elf32_hppa_link_hash_entry *)
	 _bfd_elf_link_hash_newfunc ((struct bfd_hash_entry *) ret,
				     table, string));

  if (ret)
    {
      /* Initialize the local fields.  */
#if ! LONG_BRANCH_PIC_IN_SHLIB
      ret->stub_reloc_sec = NULL;
#endif
      ret->stub_cache = NULL;
#if ! LONG_BRANCH_PIC_IN_SHLIB || RELATIVE_DYNAMIC_RELOCS
      ret->reloc_entries = NULL;
#endif
      ret->pic_call = 0;
      ret->plabel = 0;
      ret->plt_abs = 0;
    }

  return (struct bfd_hash_entry *) ret;
}

/* Create the derived linker hash table.  The PA ELF port uses the derived
   hash table to keep information specific to the PA ELF linker (without
   using static variables).  */

static struct bfd_link_hash_table *
elf32_hppa_link_hash_table_create (abfd)
     bfd *abfd;
{
  struct elf32_hppa_link_hash_table *ret;

  ret = ((struct elf32_hppa_link_hash_table *) bfd_alloc (abfd, sizeof (*ret)));
  if (ret == NULL)
    return NULL;

  if (!_bfd_elf_link_hash_table_init (&ret->root, abfd, hppa_link_hash_newfunc))
    {
      bfd_release (abfd, ret);
      return NULL;
    }

  /* Init the stub hash table too.  */
  if (!bfd_hash_table_init (&ret->stub_hash_table, stub_hash_newfunc))
    return NULL;

  ret->stub_bfd = NULL;
  ret->add_stub_section = NULL;
  ret->layout_sections_again = NULL;
  ret->stub_group = NULL;
  ret->sgot = NULL;
  ret->srelgot = NULL;
  ret->splt = NULL;
  ret->srelplt = NULL;
  ret->sdynbss = NULL;
  ret->srelbss = NULL;
  ret->multi_subspace = 0;
  ret->has_12bit_branch = 0;
  ret->has_17bit_branch = 0;
  ret->need_plt_stub = 0;

  return &ret->root.root;
}

/* Build a name for an entry in the stub hash table.  */

static char *
hppa_stub_name (input_section, sym_sec, hash, rel)
     const asection *input_section;
     const asection *sym_sec;
     const struct elf32_hppa_link_hash_entry *hash;
     const Elf_Internal_Rela *rel;
{
  char *stub_name;
  size_t len;

  if (hash)
    {
      len = 8 + 1 + strlen (hash->elf.root.root.string) + 1 + 8 + 1;
      stub_name = bfd_malloc (len);
      if (stub_name != NULL)
	{
	  sprintf (stub_name, "%08x_%s+%x",
		   input_section->id & 0xffffffff,
		   hash->elf.root.root.string,
		   (int) rel->r_addend & 0xffffffff);
	}
    }
  else
    {
      len = 8 + 1 + 8 + 1 + 8 + 1 + 8 + 1;
      stub_name = bfd_malloc (len);
      if (stub_name != NULL)
	{
	  sprintf (stub_name, "%08x_%x:%x+%x",
		   input_section->id & 0xffffffff,
		   sym_sec->id & 0xffffffff,
		   (int) ELF32_R_SYM (rel->r_info) & 0xffffffff,
		   (int) rel->r_addend & 0xffffffff);
	}
    }
  return stub_name;
}

/* Look up an entry in the stub hash.  Stub entries are cached because
   creating the stub name takes a bit of time.  */

static struct elf32_hppa_stub_hash_entry *
hppa_get_stub_entry (input_section, sym_sec, hash, rel, hplink)
     const asection *input_section;
     const asection *sym_sec;
     struct elf32_hppa_link_hash_entry *hash;
     const Elf_Internal_Rela *rel;
     struct elf32_hppa_link_hash_table *hplink;
{
  struct elf32_hppa_stub_hash_entry *stub_entry;
  const asection *id_sec;

  /* If this input section is part of a group of sections sharing one
     stub section, then use the id of the first section in the group.
     Stub names need to include a section id, as there may well be
     more than one stub used to reach say, printf, and we need to
     distinguish between them.  */
  id_sec = hplink->stub_group[input_section->id].link_sec;

  if (hash != NULL && hash->stub_cache != NULL
      && hash->stub_cache->h == hash
      && hash->stub_cache->id_sec == id_sec)
    {
      stub_entry = hash->stub_cache;
    }
  else
    {
      char *stub_name;

      stub_name = hppa_stub_name (id_sec, sym_sec, hash, rel);
      if (stub_name == NULL)
	return NULL;

      stub_entry = hppa_stub_hash_lookup (&hplink->stub_hash_table,
					  stub_name, false, false);
      if (stub_entry == NULL)
	{
	  if (hash == NULL || hash->elf.root.type != bfd_link_hash_undefweak)
	    (*_bfd_error_handler) (_("%s(%s+0x%lx): cannot find stub entry %s"),
				   bfd_get_filename (input_section->owner),
				   input_section->name,
				   (long) rel->r_offset,
				   stub_name);
	}
      else
	{
	  if (hash != NULL)
	    hash->stub_cache = stub_entry;
	}

      free (stub_name);
    }

  return stub_entry;
}

/* Add a new stub entry to the stub hash.  Not all fields of the new
   stub entry are initialised.  */

static struct elf32_hppa_stub_hash_entry *
hppa_add_stub (stub_name, section, hplink)
     const char *stub_name;
     asection *section;
     struct elf32_hppa_link_hash_table *hplink;
{
  asection *link_sec;
  asection *stub_sec;
  struct elf32_hppa_stub_hash_entry *stub_entry;

  link_sec = hplink->stub_group[section->id].link_sec;
  stub_sec = hplink->stub_group[section->id].stub_sec;
  if (stub_sec == NULL)
    {
      stub_sec = hplink->stub_group[link_sec->id].stub_sec;
      if (stub_sec == NULL)
	{
	  size_t len;
	  char *s_name;

	  len = strlen (link_sec->name) + sizeof (STUB_SUFFIX);
	  s_name = bfd_alloc (hplink->stub_bfd, len);
	  if (s_name == NULL)
	    return NULL;

	  strcpy (s_name, link_sec->name);
	  strcpy (s_name + len - sizeof (STUB_SUFFIX), STUB_SUFFIX);
	  stub_sec = (*hplink->add_stub_section) (s_name, link_sec);
	  if (stub_sec == NULL)
	    return NULL;
	  hplink->stub_group[link_sec->id].stub_sec = stub_sec;
	}
      hplink->stub_group[section->id].stub_sec = stub_sec;
    }

  /* Enter this entry into the linker stub hash table.  */
  stub_entry = hppa_stub_hash_lookup (&hplink->stub_hash_table, stub_name,
				      true, false);
  if (stub_entry == NULL)
    {
      (*_bfd_error_handler) (_("%s: cannot create stub entry %s"),
			     bfd_get_filename (section->owner),
			     stub_name);
      return NULL;
    }

  stub_entry->stub_sec = stub_sec;
#if ! LONG_BRANCH_PIC_IN_SHLIB
  stub_entry->reloc_sec = hplink->stub_group[section->id].reloc_sec;
#endif
  stub_entry->stub_offset = 0;
  stub_entry->id_sec = link_sec;
  return stub_entry;
}

/* Determine the type of stub needed, if any, for a call.  */

static enum elf32_hppa_stub_type
hppa_type_of_stub (input_sec, rel, hash, destination)
     asection *input_sec;
     const Elf_Internal_Rela *rel;
     struct elf32_hppa_link_hash_entry *hash;
     bfd_vma destination;
{
  bfd_vma location;
  bfd_vma branch_offset;
  bfd_vma max_branch_offset;
  unsigned int r_type;

  if (hash != NULL
      && (((hash->elf.root.type == bfd_link_hash_defined
	    || hash->elf.root.type == bfd_link_hash_defweak)
	   && hash->elf.root.u.def.section->output_section == NULL)
	  || (hash->elf.root.type == bfd_link_hash_defweak
	      && hash->elf.dynindx != -1
	      && hash->elf.plt.offset != (bfd_vma) -1)
	  || hash->elf.root.type == bfd_link_hash_undefweak
	  || hash->elf.root.type == bfd_link_hash_undefined
	  || hash->pic_call))
    {
      /* If output_section is NULL, then it's a symbol defined in a
	 shared library.  We will need an import stub.  Decide between
	 hppa_stub_import and hppa_stub_import_shared later.  For
	 shared links we need stubs for undefined or weak syms too;
	 They will presumably be resolved by the dynamic linker.  */
      return hppa_stub_import;
    }

  /* Determine where the call point is.  */
  location = (input_sec->output_offset
	      + input_sec->output_section->vma
	      + rel->r_offset);

  branch_offset = destination - location - 8;
  r_type = ELF32_R_TYPE (rel->r_info);

  /* Determine if a long branch stub is needed.  parisc branch offsets
     are relative to the second instruction past the branch, ie. +8
     bytes on from the branch instruction location.  The offset is
     signed and counts in units of 4 bytes.  */
  if (r_type == (unsigned int) R_PARISC_PCREL17F)
    {
      max_branch_offset = (1 << (17-1)) << 2;
    }
  else if (r_type == (unsigned int) R_PARISC_PCREL12F)
    {
      max_branch_offset = (1 << (12-1)) << 2;
    }
  else /* R_PARISC_PCREL22F.  */
    {
      max_branch_offset = (1 << (22-1)) << 2;
    }

  if (branch_offset + max_branch_offset >= 2*max_branch_offset)
    {
#if LONG_BRANCH_VIA_PLT
      if (hash != NULL
	  && hash->elf.dynindx != -1
	  && hash->elf.plt.offset != (bfd_vma) -1)
	{
	  /* If we are doing a shared link and find we need a long
	     branch stub, then go via the .plt if possible.  */
	  return hppa_stub_import;
	}
      else
#endif
	return hppa_stub_long_branch;
    }
  return hppa_stub_none;
}

/* Build one linker stub as defined by the stub hash table entry GEN_ENTRY.
   IN_ARG contains the link info pointer.  */

#define LDIL_R1		0x20200000	/* ldil  LR'XXX,%r1		*/
#define BE_SR4_R1	0xe0202002	/* be,n  RR'XXX(%sr4,%r1)	*/

#define BL_R1		0xe8200000	/* b,l   .+8,%r1		*/
#define ADDIL_R1	0x28200000	/* addil L'XXX,%r1,%r1		*/
#define DEPI_R1		0xd4201c1e	/* depi  0,31,2,%r1		*/

#define ADDIL_DP	0x2b600000	/* addil L'XXX,%dp,%r1		*/
#define LDW_R1_R21	0x48350000	/* ldw   R'XXX(%sr0,%r1),%r21	*/
#define BV_R0_R21	0xeaa0c000	/* bv    %r0(%r21)		*/
#define LDW_R1_R19	0x48330000	/* ldw   R'XXX(%sr0,%r1),%r19	*/

#define ADDIL_R19	0x2a600000	/* addil L'XXX,%r19,%r1		*/
#define LDW_R1_DP	0x483b0000	/* ldw   R'XXX(%sr0,%r1),%dp	*/

#define LDSID_R21_R1	0x02a010a1	/* ldsid (%sr0,%r21),%r1	*/
#define MTSP_R1		0x00011820	/* mtsp  %r1,%sr0		*/
#define BE_SR0_R21	0xe2a00000	/* be    0(%sr0,%r21)		*/
#define STW_RP		0x6bc23fd1	/* stw   %rp,-24(%sr0,%sp)	*/

#define BL_RP		0xe8400002	/* b,l,n XXX,%rp		*/
#define NOP		0x08000240	/* nop				*/
#define LDW_RP		0x4bc23fd1	/* ldw   -24(%sr0,%sp),%rp	*/
#define LDSID_RP_R1	0x004010a1	/* ldsid (%sr0,%rp),%r1		*/
#define BE_SR0_RP	0xe0400002	/* be,n  0(%sr0,%rp)		*/

#ifndef R19_STUBS
#define R19_STUBS 1
#endif

#if R19_STUBS
#define LDW_R1_DLT	LDW_R1_R19
#else
#define LDW_R1_DLT	LDW_R1_DP
#endif

static boolean
hppa_build_one_stub (gen_entry, in_arg)
     struct bfd_hash_entry *gen_entry;
     PTR in_arg;
{
  struct elf32_hppa_stub_hash_entry *stub_entry;
  struct bfd_link_info *info;
  struct elf32_hppa_link_hash_table *hplink;
  asection *stub_sec;
  bfd *stub_bfd;
  bfd_byte *loc;
  bfd_vma sym_value;
  bfd_vma insn;
  int val;
  int size;

  /* Massage our args to the form they really have.  */
  stub_entry = (struct elf32_hppa_stub_hash_entry *) gen_entry;
  info = (struct bfd_link_info *) in_arg;

  hplink = hppa_link_hash_table (info);
  stub_sec = stub_entry->stub_sec;

  /* Make a note of the offset within the stubs for this entry.  */
  stub_entry->stub_offset = stub_sec->_raw_size;
  loc = stub_sec->contents + stub_entry->stub_offset;

  stub_bfd = stub_sec->owner;

  switch (stub_entry->stub_type)
    {
    case hppa_stub_long_branch:
      /* Create the long branch.  A long branch is formed with "ldil"
	 loading the upper bits of the target address into a register,
	 then branching with "be" which adds in the lower bits.
	 The "be" has its delay slot nullified.  */
      sym_value = (stub_entry->target_value
		   + stub_entry->target_section->output_offset
		   + stub_entry->target_section->output_section->vma);

      val = hppa_field_adjust (sym_value, (bfd_signed_vma) 0, e_lrsel);
      insn = hppa_rebuild_insn ((int) LDIL_R1, val, 21);
      bfd_put_32 (stub_bfd, insn, loc);

      val = hppa_field_adjust (sym_value, (bfd_signed_vma) 0, e_rrsel) >> 2;
      insn = hppa_rebuild_insn ((int) BE_SR4_R1, val, 17);
      bfd_put_32 (stub_bfd, insn, loc + 4);

#if ! LONG_BRANCH_PIC_IN_SHLIB
      if (info->shared)
	{
	  /* Output a dynamic relocation for this stub.  We only
	     output one PCREL21L reloc per stub, trusting that the
	     dynamic linker will also fix the implied PCREL17R for the
	     second instruction.  PCREL21L dynamic relocs had better
	     never be emitted for some other purpose...  */
	  asection *srel;
	  Elf_Internal_Rela outrel;

	  if (stub_entry->h == NULL)
	    {
	      (*_bfd_error_handler)
		(_("%s(%s+0x%lx): cannot relocate %s, recompile with -ffunction-sections"),
		 bfd_get_filename (stub_entry->target_section->owner),
		 stub_sec->name,
		 (long) stub_entry->stub_offset,
		 stub_entry->root.string);
	      bfd_set_error (bfd_error_bad_value);
	      return false;
	    }

	  srel = stub_entry->reloc_sec;
	  if (srel == NULL)
	    {
	      (*_bfd_error_handler)
		(_("Could not find relocation section for %s"),
		 stub_sec->name);
	      bfd_set_error (bfd_error_bad_value);
	      return false;
	    }

	  outrel.r_offset = (stub_entry->stub_offset
			     + stub_sec->output_offset
			     + stub_sec->output_section->vma);
	  outrel.r_info = ELF32_R_INFO (0, R_PARISC_PCREL21L);
	  outrel.r_addend = sym_value;
	  bfd_elf32_swap_reloca_out (stub_sec->output_section->owner,
				     &outrel,
				     ((Elf32_External_Rela *)
				      srel->contents + srel->reloc_count));
	  ++srel->reloc_count;
	}
#endif
      size = 8;
      break;

    case hppa_stub_long_branch_shared:
      /* Branches are relative.  This is where we are going to.  */
      sym_value = (stub_entry->target_value
		   + stub_entry->target_section->output_offset
		   + stub_entry->target_section->output_section->vma);

      /* And this is where we are coming from, more or less.  */
      sym_value -= (stub_entry->stub_offset
		    + stub_sec->output_offset
		    + stub_sec->output_section->vma);

      bfd_put_32 (stub_bfd, (bfd_vma) BL_R1, loc);
      val = hppa_field_adjust (sym_value, (bfd_signed_vma) -8, e_lrsel);
      insn = hppa_rebuild_insn ((int) ADDIL_R1, val, 21);
      bfd_put_32 (stub_bfd, insn, loc + 4);

      val = hppa_field_adjust (sym_value, (bfd_signed_vma) -8, e_rrsel) >> 2;
      insn = hppa_rebuild_insn ((int) BE_SR4_R1, val, 17);
      bfd_put_32 (stub_bfd, insn, loc + 8);
      size = 12;
      break;

    case hppa_stub_import:
    case hppa_stub_import_shared:
      sym_value = (stub_entry->h->elf.plt.offset
		   + hplink->splt->output_offset
		   + hplink->splt->output_section->vma
		   - elf_gp (hplink->splt->output_section->owner));

      insn = ADDIL_DP;
#if R19_STUBS
      if (stub_entry->stub_type == hppa_stub_import_shared)
	insn = ADDIL_R19;
#endif
      val = hppa_field_adjust (sym_value, (bfd_signed_vma) 0, e_lrsel),
      insn = hppa_rebuild_insn ((int) insn, val, 21);
      bfd_put_32 (stub_bfd, insn, loc);

      /* It is critical to use lrsel/rrsel here because we are using
	 two different offsets (+0 and +4) from sym_value.  If we use
	 lsel/rsel then with unfortunate sym_values we will round
	 sym_value+4 up to the next 2k block leading to a mis-match
	 between the lsel and rsel value.  */
      val = hppa_field_adjust (sym_value, (bfd_signed_vma) 0, e_rrsel);
      insn = hppa_rebuild_insn ((int) LDW_R1_R21, val, 14);
      bfd_put_32 (stub_bfd, insn, loc + 4);

      if (hplink->multi_subspace)
	{
	  val = hppa_field_adjust (sym_value, (bfd_signed_vma) 4, e_rrsel);
	  insn = hppa_rebuild_insn ((int) LDW_R1_DLT, val, 14);
	  bfd_put_32 (stub_bfd, insn, loc + 8);

	  bfd_put_32 (stub_bfd, (bfd_vma) LDSID_R21_R1, loc + 12);
	  bfd_put_32 (stub_bfd, (bfd_vma) MTSP_R1,      loc + 16);
	  bfd_put_32 (stub_bfd, (bfd_vma) BE_SR0_R21,   loc + 20);
	  bfd_put_32 (stub_bfd, (bfd_vma) STW_RP,       loc + 24);

	  size = 28;
	}
      else
	{
	  bfd_put_32 (stub_bfd, (bfd_vma) BV_R0_R21, loc + 8);
	  val = hppa_field_adjust (sym_value, (bfd_signed_vma) 4, e_rrsel);
	  insn = hppa_rebuild_insn ((int) LDW_R1_DLT, val, 14);
	  bfd_put_32 (stub_bfd, insn, loc + 12);

	  size = 16;
	}

      if (!info->shared
	  && stub_entry->h != NULL
	  && stub_entry->h->pic_call)
	{
	  /* Build the .plt entry needed to call a PIC function from
	     statically linked code.  We don't need any relocs.  */
	  bfd *dynobj;
	  struct elf32_hppa_link_hash_entry *eh;
	  bfd_vma value;

	  dynobj = hplink->root.dynobj;
	  eh = (struct elf32_hppa_link_hash_entry *) stub_entry->h;

	  BFD_ASSERT (eh->elf.root.type == bfd_link_hash_defined
		      || eh->elf.root.type == bfd_link_hash_defweak);

	  value = (eh->elf.root.u.def.value
		   + eh->elf.root.u.def.section->output_offset
		   + eh->elf.root.u.def.section->output_section->vma);

	  /* Fill in the entry in the procedure linkage table.

	     The format of a plt entry is
	     <funcaddr>
	     <__gp>.  */

	  bfd_put_32 (hplink->splt->owner, value,
		      hplink->splt->contents + eh->elf.plt.offset);
	  value = elf_gp (hplink->splt->output_section->owner);
	  bfd_put_32 (hplink->splt->owner, value,
		      hplink->splt->contents + eh->elf.plt.offset + 4);
	}
      break;

    case hppa_stub_export:
      /* Branches are relative.  This is where we are going to.  */
      sym_value = (stub_entry->target_value
		   + stub_entry->target_section->output_offset
		   + stub_entry->target_section->output_section->vma);

      /* And this is where we are coming from.  */
      sym_value -= (stub_entry->stub_offset
		    + stub_sec->output_offset
		    + stub_sec->output_section->vma);

      if (sym_value - 8 + 0x40000 >= 0x80000)
	{
	  (*_bfd_error_handler)
	    (_("%s(%s+0x%lx): cannot reach %s, recompile with -ffunction-sections"),
	     bfd_get_filename (stub_entry->target_section->owner),
	     stub_sec->name,
	     (long) stub_entry->stub_offset,
	     stub_entry->root.string);
	  bfd_set_error (bfd_error_bad_value);
	  return false;
	}

      val = hppa_field_adjust (sym_value, (bfd_signed_vma) -8, e_fsel) >> 2;
      insn = hppa_rebuild_insn ((int) BL_RP, val, 17);
      bfd_put_32 (stub_bfd, insn, loc);

      bfd_put_32 (stub_bfd, (bfd_vma) NOP,         loc + 4);
      bfd_put_32 (stub_bfd, (bfd_vma) LDW_RP,      loc + 8);
      bfd_put_32 (stub_bfd, (bfd_vma) LDSID_RP_R1, loc + 12);
      bfd_put_32 (stub_bfd, (bfd_vma) MTSP_R1,     loc + 16);
      bfd_put_32 (stub_bfd, (bfd_vma) BE_SR0_RP,   loc + 20);

      /* Point the function symbol at the stub.  */
      stub_entry->h->elf.root.u.def.section = stub_sec;
      stub_entry->h->elf.root.u.def.value = stub_sec->_raw_size;

      size = 24;
      break;

    default:
      BFD_FAIL ();
      return false;
    }

  stub_sec->_raw_size += size;
  return true;
}

#undef LDIL_R1
#undef BE_SR4_R1
#undef BL_R1
#undef ADDIL_R1
#undef DEPI_R1
#undef ADDIL_DP
#undef LDW_R1_R21
#undef LDW_R1_DLT
#undef LDW_R1_R19
#undef ADDIL_R19
#undef LDW_R1_DP
#undef LDSID_R21_R1
#undef MTSP_R1
#undef BE_SR0_R21
#undef STW_RP
#undef BV_R0_R21
#undef BL_RP
#undef NOP
#undef LDW_RP
#undef LDSID_RP_R1
#undef BE_SR0_RP

/* As above, but don't actually build the stub.  Just bump offset so
   we know stub section sizes.  */

static boolean
hppa_size_one_stub (gen_entry, in_arg)
     struct bfd_hash_entry *gen_entry;
     PTR in_arg;
{
  struct elf32_hppa_stub_hash_entry *stub_entry;
  struct elf32_hppa_link_hash_table *hplink;
  int size;

  /* Massage our args to the form they really have.  */
  stub_entry = (struct elf32_hppa_stub_hash_entry *) gen_entry;
  hplink = (struct elf32_hppa_link_hash_table *) in_arg;

  if (stub_entry->stub_type == hppa_stub_long_branch)
    {
#if ! LONG_BRANCH_PIC_IN_SHLIB
      if (stub_entry->reloc_sec != NULL)
	stub_entry->reloc_sec->_raw_size += sizeof (Elf32_External_Rela);
#endif
      size = 8;
    }
  else if (stub_entry->stub_type == hppa_stub_long_branch_shared)
    size = 12;
  else if (stub_entry->stub_type == hppa_stub_export)
    size = 24;
  else /* hppa_stub_import or hppa_stub_import_shared.  */
    {
      if (hplink->multi_subspace)
	size = 28;
      else
	size = 16;
    }

  stub_entry->stub_sec->_raw_size += size;
  return true;
}

/* Return nonzero if ABFD represents an HPPA ELF32 file.
   Additionally we set the default architecture and machine.  */

static boolean
elf32_hppa_object_p (abfd)
     bfd *abfd;
{
  unsigned int flags = elf_elfheader (abfd)->e_flags;

  switch (flags & (EF_PARISC_ARCH | EF_PARISC_WIDE))
    {
    case EFA_PARISC_1_0:
      return bfd_default_set_arch_mach (abfd, bfd_arch_hppa, 10);
    case EFA_PARISC_1_1:
      return bfd_default_set_arch_mach (abfd, bfd_arch_hppa, 11);
    case EFA_PARISC_2_0:
      return bfd_default_set_arch_mach (abfd, bfd_arch_hppa, 20);
    case EFA_PARISC_2_0 | EF_PARISC_WIDE:
      return bfd_default_set_arch_mach (abfd, bfd_arch_hppa, 25);
    }
  return true;
}

/* Undo the generic ELF code's subtraction of section->vma from the
   value of each external symbol.  */

static boolean
elf32_hppa_add_symbol_hook (abfd, info, sym, namep, flagsp, secp, valp)
     bfd *abfd ATTRIBUTE_UNUSED;
     struct bfd_link_info *info ATTRIBUTE_UNUSED;
     const Elf_Internal_Sym *sym ATTRIBUTE_UNUSED;
     const char **namep ATTRIBUTE_UNUSED;
     flagword *flagsp ATTRIBUTE_UNUSED;
     asection **secp;
     bfd_vma *valp;
{
  *valp += (*secp)->vma;
  return true;
}

/* Create the .plt and .got sections, and set up our hash table
   short-cuts to various dynamic sections.  */

static boolean
elf32_hppa_create_dynamic_sections (abfd, info)
     bfd *abfd;
     struct bfd_link_info *info;
{
  struct elf32_hppa_link_hash_table *hplink;

  /* Don't try to create the .plt and .got twice.  */
  hplink = hppa_link_hash_table (info);
  if (hplink->splt != NULL)
    return true;

  /* Call the generic code to do most of the work.  */
  if (! _bfd_elf_create_dynamic_sections (abfd, info))
    return false;

  hplink->splt = bfd_get_section_by_name (abfd, ".plt");
  hplink->srelplt = bfd_get_section_by_name (abfd, ".rela.plt");

  hplink->sgot = bfd_get_section_by_name (abfd, ".got");
  hplink->srelgot = bfd_make_section (abfd, ".rela.got");
  if (hplink->srelgot == NULL
      || ! bfd_set_section_flags (abfd, hplink->srelgot,
				  (SEC_ALLOC
				   | SEC_LOAD
				   | SEC_HAS_CONTENTS
				   | SEC_IN_MEMORY
				   | SEC_LINKER_CREATED
				   | SEC_READONLY))
      || ! bfd_set_section_alignment (abfd, hplink->srelgot, 2))
    return false;

  hplink->sdynbss = bfd_get_section_by_name (abfd, ".dynbss");
  hplink->srelbss = bfd_get_section_by_name (abfd, ".rela.bss");

  return true;
}

/* Look through the relocs for a section during the first phase, and
   allocate space in the global offset table or procedure linkage
   table.  At this point we haven't necessarily read all the input
   files.  */

static boolean
elf32_hppa_check_relocs (abfd, info, sec, relocs)
     bfd *abfd;
     struct bfd_link_info *info;
     asection *sec;
     const Elf_Internal_Rela *relocs;
{
  bfd *dynobj;
  Elf_Internal_Shdr *symtab_hdr;
  struct elf_link_hash_entry **sym_hashes;
  bfd_signed_vma *local_got_refcounts;
  const Elf_Internal_Rela *rel;
  const Elf_Internal_Rela *rel_end;
  struct elf32_hppa_link_hash_table *hplink;
  asection *sreloc;
  asection *stubreloc;

  if (info->relocateable)
    return true;

  hplink = hppa_link_hash_table (info);
  dynobj = hplink->root.dynobj;
  symtab_hdr = &elf_tdata (abfd)->symtab_hdr;
  sym_hashes = elf_sym_hashes (abfd);
  local_got_refcounts = elf_local_got_refcounts (abfd);
  sreloc = NULL;
  stubreloc = NULL;

  rel_end = relocs + sec->reloc_count;
  for (rel = relocs; rel < rel_end; rel++)
    {
      enum {
	NEED_GOT = 1,
	NEED_PLT = 2,
	NEED_DYNREL = 4,
#if LONG_BRANCH_PIC_IN_SHLIB
	NEED_STUBREL = 0,  /* We won't be needing them in this case.  */
#else
	NEED_STUBREL = 8,
#endif
	PLT_PLABEL = 16
      };

      unsigned int r_symndx, r_type;
      struct elf32_hppa_link_hash_entry *h;
      int need_entry;

      r_symndx = ELF32_R_SYM (rel->r_info);

      if (r_symndx < symtab_hdr->sh_info)
	h = NULL;
      else
	h = ((struct elf32_hppa_link_hash_entry *)
	     sym_hashes[r_symndx - symtab_hdr->sh_info]);

      r_type = ELF32_R_TYPE (rel->r_info);

      switch (r_type)
	{
	case R_PARISC_DLTIND14F:
	case R_PARISC_DLTIND14R:
	case R_PARISC_DLTIND21L:
	  /* This symbol requires a global offset table entry.  */
	  need_entry = NEED_GOT;

	  /* Mark this section as containing PIC code.  */
	  sec->flags |= SEC_HAS_GOT_REF;
	  break;

	case R_PARISC_PLABEL14R: /* "Official" procedure labels.  */
	case R_PARISC_PLABEL21L:
	case R_PARISC_PLABEL32:
	  /* If the addend is non-zero, we break badly.  */
	  BFD_ASSERT (rel->r_addend == 0);

	  /* If we are creating a shared library, then we need to
	     create a PLT entry for all PLABELs, because PLABELs with
	     local symbols may be passed via a pointer to another
	     object.  Additionally, output a dynamic relocation
	     pointing to the PLT entry.  */
	  need_entry = PLT_PLABEL | NEED_PLT | NEED_DYNREL;
	  break;

	case R_PARISC_PCREL12F:
	  hplink->has_12bit_branch = 1;
	  /* Fall thru.  */
	case R_PARISC_PCREL17C:
	case R_PARISC_PCREL17F:
	  hplink->has_17bit_branch = 1;
	  /* Fall thru.  */
	case R_PARISC_PCREL22F:
	  /* Function calls might need to go through the .plt, and
	     might require long branch stubs.  */
	  if (h == NULL)
	    {
	      /* We know local syms won't need a .plt entry, and if
		 they need a long branch stub we can't guarantee that
		 we can reach the stub.  So just flag an error later
		 if we're doing a shared link and find we need a long
		 branch stub.  */
	      continue;
	    }
	  else
	    {
	      /* Global symbols will need a .plt entry if they remain
		 global, and in most cases won't need a long branch
		 stub.  Unfortunately, we have to cater for the case
		 where a symbol is forced local by versioning, or due
		 to symbolic linking, and we lose the .plt entry.  */
	      need_entry = NEED_PLT | NEED_STUBREL;
	    }
	  break;

	case R_PARISC_SEGBASE: /* Used to set segment base.  */
	case R_PARISC_SEGREL32: /* Relative reloc.  */
	case R_PARISC_PCREL14F: /* PC relative load/store.  */
	case R_PARISC_PCREL14R:
	case R_PARISC_PCREL17R: /* External branches.  */
	case R_PARISC_PCREL21L: /* As above, and for load/store too.  */
	  /* We don't need to propagate the relocation if linking a
	     shared object since these are section relative.  */
	  continue;

	case R_PARISC_DPREL14F: /* Used for gp rel data load/store.  */
	case R_PARISC_DPREL14R:
	case R_PARISC_DPREL21L:
	  if (info->shared)
	    {
	      (*_bfd_error_handler)
		(_("%s: relocation %s can not be used when making a shared object; recompile with -fPIC"),
		 bfd_get_filename (abfd),
		 elf_hppa_howto_table[r_type].name);
	      bfd_set_error (bfd_error_bad_value);
	      return false;
	    }
	  /* Fall through.  */

	case R_PARISC_DIR17F: /* Used for external branches.  */
	case R_PARISC_DIR17R:
	case R_PARISC_DIR14F: /* Used for load/store from absolute locn.  */
	case R_PARISC_DIR14R:
	case R_PARISC_DIR21L: /* As above, and for ext branches too.  */
#if 1
	  /* Help debug shared library creation.  Any of the above
	     relocs can be used in shared libs, but they may cause
	     pages to become unshared.  */
	  if (info->shared)
	    {
	      (*_bfd_error_handler)
		(_("%s: relocation %s should not be used when making a shared object; recompile with -fPIC"),
		 bfd_get_filename (abfd),
		 elf_hppa_howto_table[r_type].name);
	    }
	  /* Fall through.  */
#endif

	case R_PARISC_DIR32: /* .word, PARISC.unwind relocs.  */
	  /* We may want to output a dynamic relocation later.  */
	  need_entry = NEED_DYNREL;
	  break;

	  /* This relocation describes the C++ object vtable hierarchy.
	     Reconstruct it for later use during GC.  */
	case R_PARISC_GNU_VTINHERIT:
	  if (!_bfd_elf32_gc_record_vtinherit (abfd, sec,
					       &h->elf, rel->r_offset))
	    return false;
	  continue;

	  /* This relocation describes which C++ vtable entries are actually
	     used.  Record for later use during GC.  */
	case R_PARISC_GNU_VTENTRY:
	  if (!_bfd_elf32_gc_record_vtentry (abfd, sec,
					     &h->elf, rel->r_addend))
	    return false;
	  continue;

	default:
	  continue;
	}

      /* Now carry out our orders.  */
      if (need_entry & NEED_GOT)
	{
	  /* Allocate space for a GOT entry, as well as a dynamic
	     relocation for this entry.  */
	  if (dynobj == NULL)
	    hplink->root.dynobj = dynobj = abfd;

	  if (hplink->sgot == NULL)
	    {
	      if (! elf32_hppa_create_dynamic_sections (dynobj, info))
		return false;
	    }

	  if (h != NULL)
	    {
	      if (h->elf.got.refcount == -1)
		{
		  h->elf.got.refcount = 1;

		  /* Make sure this symbol is output as a dynamic symbol.  */
		  if (h->elf.dynindx == -1)
		    {
		      if (! bfd_elf32_link_record_dynamic_symbol (info,
								  &h->elf))
			return false;
		    }

		  hplink->sgot->_raw_size += GOT_ENTRY_SIZE;
		  hplink->srelgot->_raw_size += sizeof (Elf32_External_Rela);
		}
	      else
		h->elf.got.refcount += 1;
	    }
	  else
	    {
	      /* This is a global offset table entry for a local symbol.  */
	      if (local_got_refcounts == NULL)
		{
		  size_t size;

		  /* Allocate space for local got offsets and local
		     plt offsets.  Done this way to save polluting
		     elf_obj_tdata with another target specific
		     pointer.  */
		  size = symtab_hdr->sh_info * 2 * sizeof (bfd_signed_vma);
		  local_got_refcounts = ((bfd_signed_vma *)
					 bfd_alloc (abfd, size));
		  if (local_got_refcounts == NULL)
		    return false;
		  elf_local_got_refcounts (abfd) = local_got_refcounts;
		  memset (local_got_refcounts, -1, size);
		}
	      if (local_got_refcounts[r_symndx] == -1)
		{
		  local_got_refcounts[r_symndx] = 1;

		  hplink->sgot->_raw_size += GOT_ENTRY_SIZE;
		  if (info->shared)
		    {
		      /* If we are generating a shared object, we need to
			 output a reloc so that the dynamic linker can
			 adjust this GOT entry (because the address
			 the shared library is loaded at is not fixed).  */
		      hplink->srelgot->_raw_size +=
			sizeof (Elf32_External_Rela);
		    }
		}
	      else
		local_got_refcounts[r_symndx] += 1;
	    }
	}

      if (need_entry & NEED_PLT)
	{
	  /* If we are creating a shared library, and this is a reloc
	     against a weak symbol or a global symbol in a dynamic
	     object, then we will be creating an import stub and a
	     .plt entry for the symbol.  Similarly, on a normal link
	     to symbols defined in a dynamic object we'll need the
	     import stub and a .plt entry.  We don't know yet whether
	     the symbol is defined or not, so make an entry anyway and
	     clean up later in adjust_dynamic_symbol.  */
	  if ((sec->flags & SEC_ALLOC) != 0)
	    {
	      if (h != NULL)
		{
		  if (h->elf.plt.refcount == -1)
		    {
		      h->elf.plt.refcount = 1;
		      h->elf.elf_link_hash_flags |= ELF_LINK_HASH_NEEDS_PLT;
		    }
		  else
		    h->elf.plt.refcount += 1;

		  /* If this .plt entry is for a plabel, mark it so
		     that adjust_dynamic_symbol will keep the entry
		     even if it appears to be local.  */
		  if (need_entry & PLT_PLABEL)
		    h->plabel = 1;
		}
	      else if (need_entry & PLT_PLABEL)
		{
		  int indx;

		  if (local_got_refcounts == NULL)
		    {
		      size_t size;

		      /* Allocate space for local got offsets and local
			 plt offsets.  */
		      size = symtab_hdr->sh_info * 2 * sizeof (bfd_signed_vma);
		      local_got_refcounts = ((bfd_signed_vma *)
					     bfd_alloc (abfd, size));
		      if (local_got_refcounts == NULL)
			return false;
		      elf_local_got_refcounts (abfd) = local_got_refcounts;
		      memset (local_got_refcounts, -1, size);
		    }
		  indx = r_symndx + symtab_hdr->sh_info;
		  if (local_got_refcounts[indx] == -1)
		    local_got_refcounts[indx] = 1;
		  else
		    local_got_refcounts[indx] += 1;
		}
	    }
	}

      if (need_entry & (NEED_DYNREL | NEED_STUBREL))
	{
	  /* Flag this symbol as having a non-got, non-plt reference
	     so that we generate copy relocs if it turns out to be
	     dynamic.  */
	  if (h != NULL)
	    h->elf.elf_link_hash_flags |= ELF_LINK_NON_GOT_REF;

	  /* If we are creating a shared library then we need to copy
	     the reloc into the shared library.  However, if we are
	     linking with -Bsymbolic, we need only copy absolute
	     relocs or relocs against symbols that are not defined in
	     an object we are including in the link.  PC- or DP- or
	     DLT-relative relocs against any local sym or global sym
	     with DEF_REGULAR set, can be discarded.  At this point we
	     have not seen all the input files, so it is possible that
	     DEF_REGULAR is not set now but will be set later (it is
	     never cleared).  We account for that possibility below by
	     storing information in the reloc_entries field of the
	     hash table entry.

	     A similar situation to the -Bsymbolic case occurs when
	     creating shared libraries and symbol visibility changes
	     render the symbol local.

	     As it turns out, all the relocs we will be creating here
	     are absolute, so we cannot remove them on -Bsymbolic
	     links or visibility changes anyway.  A STUB_REL reloc
	     is absolute too, as in that case it is the reloc in the
	     stub we will be creating, rather than copying the PCREL
	     reloc in the branch.  */
	  if ((sec->flags & SEC_ALLOC) != 0
	      && info->shared
#if RELATIVE_DYNAMIC_RELOCS
	      && (!info->symbolic
		  || is_absolute_reloc (r_type)
		  || (h != NULL
		      && ((h->elf.elf_link_hash_flags
			   & ELF_LINK_HASH_DEF_REGULAR) == 0)))
#endif
	      )
	    {
	      boolean doit;
	      asection *srel;

	      srel = sreloc;
	      if ((need_entry & NEED_STUBREL))
		srel = stubreloc;

	      /* Create a reloc section in dynobj and make room for
		 this reloc.  */
	      if (srel == NULL)
		{
		  char *name;

		  if (dynobj == NULL)
		    hplink->root.dynobj = dynobj = abfd;

		  name = bfd_elf_string_from_elf_section
		    (abfd,
		     elf_elfheader (abfd)->e_shstrndx,
		     elf_section_data (sec)->rel_hdr.sh_name);
		  if (name == NULL)
		    {
		      (*_bfd_error_handler)
			(_("Could not find relocation section for %s"),
			 sec->name);
		      bfd_set_error (bfd_error_bad_value);
		      return false;
		    }

		  if ((need_entry & NEED_STUBREL))
		    {
		      size_t len = strlen (name) + sizeof (STUB_SUFFIX);
		      char *newname = bfd_malloc (len);

		      if (newname == NULL)
			return false;
		      strcpy (newname, name);
		      strcpy (newname + len - sizeof (STUB_SUFFIX),
			      STUB_SUFFIX);
		      name = newname;
		    }

		  srel = bfd_get_section_by_name (dynobj, name);
		  if (srel == NULL)
		    {
		      flagword flags;

		      srel = bfd_make_section (dynobj, name);
		      flags = (SEC_HAS_CONTENTS | SEC_READONLY
			       | SEC_IN_MEMORY | SEC_LINKER_CREATED);
		      if ((sec->flags & SEC_ALLOC) != 0)
			flags |= SEC_ALLOC | SEC_LOAD;
		      if (srel == NULL
			  || !bfd_set_section_flags (dynobj, srel, flags)
			  || !bfd_set_section_alignment (dynobj, srel, 2))
			return false;
		    }
		  else if ((need_entry & NEED_STUBREL))
		    free (name);

		  if ((need_entry & NEED_STUBREL))
		    stubreloc = srel;
		  else
		    sreloc = srel;
		}

#if ! LONG_BRANCH_PIC_IN_SHLIB
	      /* If this is a function call, we only need one dynamic
		 reloc for the stub as all calls to a particular
		 function will go through the same stub.  Actually, a
		 long branch stub needs two relocations, but we count
		 on some intelligence on the part of the dynamic
		 linker.  */
	      if ((need_entry & NEED_STUBREL))
		{
		  doit = h->stub_reloc_sec != stubreloc;
		  h->stub_reloc_sec = stubreloc;
		}
	      else
#endif
		doit = 1;

	      if (doit)
		{
		  srel->_raw_size += sizeof (Elf32_External_Rela);

#if ! LONG_BRANCH_PIC_IN_SHLIB || RELATIVE_DYNAMIC_RELOCS
		  /* Keep track of relocations we have entered for
		     this global symbol, so that we can discard them
		     later if necessary.  */
		  if (h != NULL
		      && (0
#if RELATIVE_DYNAMIC_RELOCS
			  || ! is_absolute_reloc (rtype)
#endif
			  || (need_entry & NEED_STUBREL)))
		    {
		      struct elf32_hppa_dyn_reloc_entry *p;

		      for (p = h->reloc_entries; p != NULL; p = p->next)
			if (p->section == srel)
			  break;

		      if (p == NULL)
			{
			  p = ((struct elf32_hppa_dyn_reloc_entry *)
			       bfd_alloc (dynobj, sizeof *p));
			  if (p == NULL)
			    return false;
			  p->next = h->reloc_entries;
			  h->reloc_entries = p;
			  p->section = srel;
			  p->count = 0;
			}

		      /* NEED_STUBREL and NEED_DYNREL are never both
			 set.  Leave the count at zero for the
			 NEED_STUBREL case as we only ever have one
			 stub reloc per section per symbol, and this
			 simplifies code in hppa_discard_copies.  */
		      if (! (need_entry & NEED_STUBREL))
			++p->count;
		    }
#endif
		}
	    }
	}
    }

  return true;
}

/* Return the section that should be marked against garbage collection
   for a given relocation.  */

static asection *
elf32_hppa_gc_mark_hook (abfd, info, rel, h, sym)
     bfd *abfd;
     struct bfd_link_info *info ATTRIBUTE_UNUSED;
     Elf_Internal_Rela *rel;
     struct elf_link_hash_entry *h;
     Elf_Internal_Sym *sym;
{
  if (h != NULL)
    {
      switch ((unsigned int) ELF32_R_TYPE (rel->r_info))
	{
	case R_PARISC_GNU_VTINHERIT:
	case R_PARISC_GNU_VTENTRY:
	  break;

	default:
	  switch (h->root.type)
	    {
	    case bfd_link_hash_defined:
	    case bfd_link_hash_defweak:
	      return h->root.u.def.section;

	    case bfd_link_hash_common:
	      return h->root.u.c.p->section;

	    default:
	      break;
	    }
	}
    }
  else
    {
      if (!(elf_bad_symtab (abfd)
	    && ELF_ST_BIND (sym->st_info) != STB_LOCAL)
	  && ! ((sym->st_shndx <= 0 || sym->st_shndx >= SHN_LORESERVE)
		&& sym->st_shndx != SHN_COMMON))
	{
	  return bfd_section_from_elf_index (abfd, sym->st_shndx);
	}
    }

  return NULL;
}

/* Update the got and plt entry reference counts for the section being
   removed.  */

static boolean
elf32_hppa_gc_sweep_hook (abfd, info, sec, relocs)
     bfd *abfd;
     struct bfd_link_info *info ATTRIBUTE_UNUSED;
     asection *sec;
     const Elf_Internal_Rela *relocs;
{
  Elf_Internal_Shdr *symtab_hdr;
  struct elf_link_hash_entry **sym_hashes;
  bfd_signed_vma *local_got_refcounts;
  bfd_signed_vma *local_plt_refcounts;
  const Elf_Internal_Rela *rel, *relend;
  unsigned long r_symndx;
  struct elf_link_hash_entry *h;
  struct elf32_hppa_link_hash_table *hplink;
  bfd *dynobj;
  asection *sgot;
  asection *srelgot;

  symtab_hdr = &elf_tdata (abfd)->symtab_hdr;
  sym_hashes = elf_sym_hashes (abfd);
  local_got_refcounts = elf_local_got_refcounts (abfd);
  local_plt_refcounts = local_got_refcounts;
  if (local_plt_refcounts != NULL)
    local_plt_refcounts += symtab_hdr->sh_info;
  hplink = hppa_link_hash_table (info);
  dynobj = hplink->root.dynobj;
  if (dynobj == NULL)
    return true;

  sgot = hplink->sgot;
  srelgot = hplink->srelgot;

  relend = relocs + sec->reloc_count;
  for (rel = relocs; rel < relend; rel++)
    switch ((unsigned int) ELF32_R_TYPE (rel->r_info))
      {
      case R_PARISC_DLTIND14F:
      case R_PARISC_DLTIND14R:
      case R_PARISC_DLTIND21L:
	r_symndx = ELF32_R_SYM (rel->r_info);
	if (r_symndx >= symtab_hdr->sh_info)
	  {
	    h = sym_hashes[r_symndx - symtab_hdr->sh_info];
	    if (h->got.refcount > 0)
	      {
		h->got.refcount -= 1;
		if (h->got.refcount == 0)
		  {
		    sgot->_raw_size -= GOT_ENTRY_SIZE;
		    srelgot->_raw_size -= sizeof (Elf32_External_Rela);
		  }
	      }
	  }
	else if (local_got_refcounts != NULL)
	  {
	    if (local_got_refcounts[r_symndx] > 0)
	      {
		local_got_refcounts[r_symndx] -= 1;
		if (local_got_refcounts[r_symndx] == 0)
		  {
		    sgot->_raw_size -= GOT_ENTRY_SIZE;
		    if (info->shared)
		      srelgot->_raw_size -= sizeof (Elf32_External_Rela);
		  }
	      }
	  }
	break;

      case R_PARISC_PCREL12F:
      case R_PARISC_PCREL17C:
      case R_PARISC_PCREL17F:
      case R_PARISC_PCREL22F:
	r_symndx = ELF32_R_SYM (rel->r_info);
	if (r_symndx >= symtab_hdr->sh_info)
	  {
	    h = sym_hashes[r_symndx - symtab_hdr->sh_info];
	    if (h->plt.refcount > 0)
	      h->plt.refcount -= 1;
	  }
	break;

      case R_PARISC_PLABEL14R:
      case R_PARISC_PLABEL21L:
      case R_PARISC_PLABEL32:
	r_symndx = ELF32_R_SYM (rel->r_info);
	if (r_symndx >= symtab_hdr->sh_info)
	  {
	    h = sym_hashes[r_symndx - symtab_hdr->sh_info];
	    if (h->plt.refcount > 0)
	      h->plt.refcount -= 1;
	  }
	else if (local_plt_refcounts != NULL)
	  {
	    if (local_plt_refcounts[r_symndx] > 0)
	      local_plt_refcounts[r_symndx] -= 1;
	  }
	break;

      default:
	break;
      }

  return true;
}

/* Our own version of hide_symbol, so that we can keep plt entries for
   plabels.  */

static void
elf32_hppa_hide_symbol (info, h)
     struct bfd_link_info *info ATTRIBUTE_UNUSED;
     struct elf_link_hash_entry *h;
{
  h->dynindx = -1;
  if (! ((struct elf32_hppa_link_hash_entry *) h)->plabel)
    {
      h->elf_link_hash_flags &= ~ELF_LINK_HASH_NEEDS_PLT;
      h->plt.offset = (bfd_vma) -1;
    }
}

/* Adjust a symbol defined by a dynamic object and referenced by a
   regular object.  The current definition is in some section of the
   dynamic object, but we're not including those sections.  We have to
   change the definition to something the rest of the link can
   understand.  */

static boolean
elf32_hppa_adjust_dynamic_symbol (info, h)
     struct bfd_link_info *info;
     struct elf_link_hash_entry *h;
{
  bfd *dynobj;
  struct elf32_hppa_link_hash_table *hplink;
  asection *s;

  hplink = hppa_link_hash_table (info);
  dynobj = hplink->root.dynobj;

  /* If this is a function, put it in the procedure linkage table.  We
     will fill in the contents of the procedure linkage table later,
     when we know the address of the .got section.  */
  if (h->type == STT_FUNC
      || (h->elf_link_hash_flags & ELF_LINK_HASH_NEEDS_PLT) != 0)
    {
      if (h->plt.refcount <= 0
	  || ((h->elf_link_hash_flags & ELF_LINK_HASH_DEF_REGULAR) != 0
	      && h->root.type != bfd_link_hash_defweak
	      && ! ((struct elf32_hppa_link_hash_entry *) h)->plabel
	      && (!info->shared || info->symbolic)))
	{
	  /* The .plt entry is not needed when:
	     a) Garbage collection has removed all references to the
	     symbol, or
	     b) We know for certain the symbol is defined in this
	     object, and it's not a weak definition, nor is the symbol
	     used by a plabel relocation.  Either this object is the
	     application or we are doing a shared symbolic link.  */

	  /* As a special sop to the hppa ABI, we keep a .plt entry
	     for functions in sections containing PIC code.  */
	  if (!info->shared
	      && h->plt.refcount > 0
	      && (h->root.type == bfd_link_hash_defined
		  || h->root.type == bfd_link_hash_defweak)
	      && (h->root.u.def.section->flags & SEC_HAS_GOT_REF) != 0)
	    {
	      ((struct elf32_hppa_link_hash_entry *) h)->pic_call = 1;
	    }
	  else
	    {
	      h->plt.offset = (bfd_vma) -1;
	      h->elf_link_hash_flags &= ~ELF_LINK_HASH_NEEDS_PLT;
	      return true;
	    }
	}

      /* Make an entry in the .plt section.  */
      s = hplink->splt;
      h->plt.offset = s->_raw_size;
      if (PLABEL_PLT_ENTRY_SIZE != PLT_ENTRY_SIZE
	  && ((struct elf32_hppa_link_hash_entry *) h)->plabel
	  && (h->elf_link_hash_flags & ELF_LINK_FORCED_LOCAL) == 0)
	{
	  /* Add some extra space for the dynamic linker to use.  */
	  s->_raw_size += PLABEL_PLT_ENTRY_SIZE;
	}
      else
	s->_raw_size += PLT_ENTRY_SIZE;

      if (! ((struct elf32_hppa_link_hash_entry *) h)->pic_call)
	{
	  /* Make sure this symbol is output as a dynamic symbol.  */
	  if (h->dynindx == -1
	      && (h->elf_link_hash_flags & ELF_LINK_FORCED_LOCAL) == 0)
	    {
	      if (! bfd_elf32_link_record_dynamic_symbol (info, h))
		return false;
	    }

	  /* We also need to make an entry in the .rela.plt section.  */
	  s = hplink->srelplt;
	  s->_raw_size += sizeof (Elf32_External_Rela);

	  hplink->need_plt_stub = 1;
	}
      return true;
    }

  /* If this is a weak symbol, and there is a real definition, the
     processor independent code will have arranged for us to see the
     real definition first, and we can just use the same value.  */
  if (h->weakdef != NULL)
    {
      BFD_ASSERT (h->weakdef->root.type == bfd_link_hash_defined
		  || h->weakdef->root.type == bfd_link_hash_defweak);
      h->root.u.def.section = h->weakdef->root.u.def.section;
      h->root.u.def.value = h->weakdef->root.u.def.value;
      return true;
    }

  /* This is a reference to a symbol defined by a dynamic object which
     is not a function.  */

  /* If we are creating a shared library, we must presume that the
     only references to the symbol are via the global offset table.
     For such cases we need not do anything here; the relocations will
     be handled correctly by relocate_section.  */
  if (info->shared)
    return true;

  /* If there are no references to this symbol that do not use the
     GOT, we don't need to generate a copy reloc.  */
  if ((h->elf_link_hash_flags & ELF_LINK_NON_GOT_REF) == 0)
    return true;

  /* We must allocate the symbol in our .dynbss section, which will
     become part of the .bss section of the executable.  There will be
     an entry for this symbol in the .dynsym section.  The dynamic
     object will contain position independent code, so all references
     from the dynamic object to this symbol will go through the global
     offset table.  The dynamic linker will use the .dynsym entry to
     determine the address it must put in the global offset table, so
     both the dynamic object and the regular object will refer to the
     same memory location for the variable.  */

  s = hplink->sdynbss;

  /* We must generate a COPY reloc to tell the dynamic linker to
     copy the initial value out of the dynamic object and into the
     runtime process image.  We need to remember the offset into the
     .rela.bss section we are going to use.  */
  if ((h->root.u.def.section->flags & SEC_ALLOC) != 0)
    {
      asection *srel;

      srel = hplink->srelbss;
      srel->_raw_size += sizeof (Elf32_External_Rela);
      h->elf_link_hash_flags |= ELF_LINK_HASH_NEEDS_COPY;
    }

  {
    /* We need to figure out the alignment required for this symbol.  I
       have no idea how other ELF linkers handle this.  */
    unsigned int power_of_two;

    power_of_two = bfd_log2 (h->size);
    if (power_of_two > 3)
      power_of_two = 3;

    /* Apply the required alignment.  */
    s->_raw_size = BFD_ALIGN (s->_raw_size,
			      (bfd_size_type) (1 << power_of_two));
    if (power_of_two > bfd_get_section_alignment (dynobj, s))
      {
	if (! bfd_set_section_alignment (dynobj, s, power_of_two))
	  return false;
      }
  }
  /* Define the symbol as being at this point in the section.  */
  h->root.u.def.section = s;
  h->root.u.def.value = s->_raw_size;

  /* Increment the section size to make room for the symbol.  */
  s->_raw_size += h->size;

  return true;
}

/* Called via elf_link_hash_traverse to create .plt entries for an
   application that uses statically linked PIC functions.  Similar to
   the first part of elf32_hppa_adjust_dynamic_symbol.  */

static boolean
hppa_handle_PIC_calls (h, inf)
     struct elf_link_hash_entry *h;
     PTR inf;
{
  struct bfd_link_info *info;
  bfd *dynobj;
  struct elf32_hppa_link_hash_table *hplink;
  asection *s;

  if (! (h->plt.refcount > 0
	 && (h->root.type == bfd_link_hash_defined
	     || h->root.type == bfd_link_hash_defweak)
	 && (h->root.u.def.section->flags & SEC_HAS_GOT_REF) != 0))
    {
      h->plt.offset = (bfd_vma) -1;
      h->elf_link_hash_flags &= ~ELF_LINK_HASH_NEEDS_PLT;
      return true;
    }

  h->elf_link_hash_flags |= ELF_LINK_HASH_NEEDS_PLT;
  ((struct elf32_hppa_link_hash_entry *) h)->pic_call = 1;

  info = (struct bfd_link_info *) inf;
  hplink = hppa_link_hash_table (info);
  dynobj = hplink->root.dynobj;

  /* Make an entry in the .plt section.  */
  s = hplink->splt;
  h->plt.offset = s->_raw_size;
  s->_raw_size += PLT_ENTRY_SIZE;

  return true;
}

#if ((! LONG_BRANCH_PIC_IN_SHLIB && LONG_BRANCH_VIA_PLT) \
     || RELATIVE_DYNAMIC_RELOCS)
/* This function is called via elf_link_hash_traverse to discard space
   we allocated for relocs that it turned out we didn't need.  */

static boolean
hppa_discard_copies (h, inf)
     struct elf_link_hash_entry *h;
     PTR inf;
{
  struct elf32_hppa_dyn_reloc_entry *s;
  struct elf32_hppa_link_hash_entry *eh;
  struct bfd_link_info *info;

  eh = (struct elf32_hppa_link_hash_entry *) h;
  info = (struct bfd_link_info *) inf;

#if ! LONG_BRANCH_PIC_IN_SHLIB && LONG_BRANCH_VIA_PLT
  /* Handle the stub reloc case.  If we have a plt entry for the
     function, we won't be needing long branch stubs.  s->count will
     only be zero for stub relocs, which provides a handy way of
     flagging these relocs, and means we need do nothing special for
     the forced local and symbolic link case.  */
  if (eh->stub_reloc_sec != NULL
      && eh->elf.plt.offset != (bfd_vma) -1)
    {
      for (s = eh->reloc_entries; s != NULL; s = s->next)
	if (s->count == 0)
	  s->section->_raw_size -= sizeof (Elf32_External_Rela);
    }
#endif

#if RELATIVE_DYNAMIC_RELOCS
  /* If a symbol has been forced local or we have found a regular
     definition for the symbolic link case, then we won't be needing
     any relocs.  */
  if (eh->elf.dynindx == -1
      || ((eh->elf.elf_link_hash_flags & ELF_LINK_HASH_DEF_REGULAR) != 0
	  && !is_absolute_reloc (r_type)
	  && info->symbolic))
    {
      for (s = eh->reloc_entries; s != NULL; s = s->next)
	s->section->_raw_size -= s->count * sizeof (Elf32_External_Rela);
    }
#endif

  return true;
}
#endif

/* This function is called via elf_link_hash_traverse to force
   millicode symbols local so they do not end up as globals in the
   dynamic symbol table.  We ought to be able to do this in
   adjust_dynamic_symbol, but our adjust_dynamic_symbol is not called
   for all dynamic symbols.  Arguably, this is a bug in
   elf_adjust_dynamic_symbol.  */

static boolean
clobber_millicode_symbols (h, info)
     struct elf_link_hash_entry *h;
     struct bfd_link_info *info;
{
  /* Note!  We only want to remove these from the dynamic symbol
     table.  Therefore we do not set ELF_LINK_FORCED_LOCAL.  */
  if (h->type == STT_PARISC_MILLI)
    elf32_hppa_hide_symbol(info, h);
  return true;
}

/* Set the sizes of the dynamic sections.  */

static boolean
elf32_hppa_size_dynamic_sections (output_bfd, info)
     bfd *output_bfd;
     struct bfd_link_info *info;
{
  struct elf32_hppa_link_hash_table *hplink;
  bfd *dynobj;
  asection *s;
  boolean relocs;
  boolean reltext;

  hplink = hppa_link_hash_table (info);
  dynobj = hplink->root.dynobj;
  BFD_ASSERT (dynobj != NULL);

  if (hplink->root.dynamic_sections_created)
    {
      const char *funcname;
      bfd *i;

      /* Set the contents of the .interp section to the interpreter.  */
      if (! info->shared)
	{
	  s = bfd_get_section_by_name (dynobj, ".interp");
	  BFD_ASSERT (s != NULL);
	  s->_raw_size = sizeof ELF_DYNAMIC_INTERPRETER;
	  s->contents = (unsigned char *) ELF_DYNAMIC_INTERPRETER;
	}

      /* Force millicode symbols local.  */
      elf_link_hash_traverse (&hplink->root,
			      clobber_millicode_symbols,
			      info);

      /* DT_INIT and DT_FINI need a .plt entry.  Make sure they have
	 one.  */
      funcname = info->init_function;
      while (1)
	{
	  if (funcname != NULL)
	    {
	      struct elf_link_hash_entry *h;

	      h = elf_link_hash_lookup (&hplink->root,
					funcname,
					false, false, false);
	      if (h != NULL
		  && (h->elf_link_hash_flags & (ELF_LINK_HASH_REF_REGULAR
						| ELF_LINK_HASH_DEF_REGULAR)))
		{
		  if (h->plt.refcount <= 0)
		    {
		      h->elf_link_hash_flags |= ELF_LINK_HASH_NEEDS_PLT;

		      /* Make an entry in the .plt section.  We know
			 the function doesn't have a plabel by the
			 refcount.  */
		      s = hplink->splt;
		      h->plt.offset = s->_raw_size;
		      s->_raw_size += PLT_ENTRY_SIZE;

		      /* Make sure this symbol is output as a dynamic
			 symbol.  */
		      if (h->dynindx == -1)
			{
			  if (! bfd_elf32_link_record_dynamic_symbol (info, h))
			    return false;
			}

		      /* Make an entry for the reloc too.  */
		      s = hplink->srelplt;
		      s->_raw_size += sizeof (Elf32_External_Rela);
		    }

		  ((struct elf32_hppa_link_hash_entry *) h)->plt_abs = 1;
		}
	    }
	  if (funcname == info->fini_function)
	    break;
	  funcname = info->fini_function;
	}

      /* Set up .plt offsets for local plabels.  */
      for (i = info->input_bfds; i; i = i->link_next)
	{
	  bfd_signed_vma *local_plt;
	  bfd_signed_vma *end_local_plt;
	  bfd_size_type locsymcount;
	  Elf_Internal_Shdr *symtab_hdr;

	  local_plt = elf_local_got_refcounts (i);
	  if (!local_plt)
	    continue;

	  symtab_hdr = &elf_tdata (i)->symtab_hdr;
	  locsymcount = symtab_hdr->sh_info;
	  local_plt += locsymcount;
	  end_local_plt = local_plt + locsymcount;

	  for (; local_plt < end_local_plt; ++local_plt)
	    {
	      if (*local_plt > 0)
		{
		  s = hplink->splt;
		  *local_plt = s->_raw_size;
		  s->_raw_size += PLT_ENTRY_SIZE;
		  if (info->shared)
		    hplink->srelplt->_raw_size += sizeof (Elf32_External_Rela);
		}
	      else
		*local_plt = (bfd_vma) -1;
	    }
	}
    }
  else
    {
      /* Run through the function symbols, looking for any that are
	 PIC, and allocate space for the necessary .plt entries so
	 that %r19 will be set up.  */
      if (! info->shared)
	elf_link_hash_traverse (&hplink->root,
				hppa_handle_PIC_calls,
				info);

      /* We may have created entries in the .rela.got section.
	 However, if we are not creating the dynamic sections, we will
	 not actually use these entries.  Reset the size of .rela.got,
	 which will cause it to get stripped from the output file
	 below.  */
      hplink->srelgot->_raw_size = 0;
    }

#if ((! LONG_BRANCH_PIC_IN_SHLIB && LONG_BRANCH_VIA_PLT) \
     || RELATIVE_DYNAMIC_RELOCS)
  /* If this is a -Bsymbolic shared link, then we need to discard all
     relocs against symbols defined in a regular object.  We also need
     to lose relocs we've allocated for long branch stubs if we know
     we won't be generating a stub.  */
  if (info->shared)
    elf_link_hash_traverse (&hplink->root,
			    hppa_discard_copies,
			    info);
#endif

  /* The check_relocs and adjust_dynamic_symbol entry points have
     determined the sizes of the various dynamic sections.  Allocate
     memory for them.  */
  relocs = false;
  reltext = false;
  for (s = dynobj->sections; s != NULL; s = s->next)
    {
      const char *name;

      if ((s->flags & SEC_LINKER_CREATED) == 0)
	continue;

      /* It's OK to base decisions on the section name, because none
	 of the dynobj section names depend upon the input files.  */
      name = bfd_get_section_name (dynobj, s);

      if (strncmp (name, ".rela", 5) == 0)
	{
	  if (s->_raw_size != 0)
	    {
	      asection *target;
	      const char *outname;

	      /* Remember whether there are any reloc sections other
		 than .rela.plt.  */
	      if (strcmp (name+5, ".plt") != 0)
		relocs = true;

	      /* If this relocation section applies to a read only
		 section, then we probably need a DT_TEXTREL entry.  */
	      outname = bfd_get_section_name (output_bfd,
					      s->output_section);
	      target = bfd_get_section_by_name (output_bfd, outname + 5);
	      if (target != NULL
		  && (target->flags & SEC_READONLY) != 0
		  && (target->flags & SEC_ALLOC) != 0)
		reltext = true;

	      /* We use the reloc_count field as a counter if we need
		 to copy relocs into the output file.  */
	      s->reloc_count = 0;
	    }
	}
      else if (strcmp (name, ".plt") == 0)
	{
	  if (hplink->need_plt_stub)
	    {
	      /* Make space for the plt stub at the end of the .plt
		 section.  We want this stub right at the end, up
		 against the .got section.  */
	      int gotalign = bfd_section_alignment (dynobj, hplink->sgot);
	      int pltalign = bfd_section_alignment (dynobj, s);
	      bfd_size_type mask;

	      if (gotalign > pltalign)
		bfd_set_section_alignment (dynobj, s, gotalign);
	      mask = ((bfd_size_type) 1 << gotalign) - 1;
	      s->_raw_size = (s->_raw_size + sizeof (plt_stub) + mask) & ~mask;
	    }
	}
      else if (strcmp (name, ".got") == 0)
	;
      else
	{
	  /* It's not one of our sections, so don't allocate space.  */
	  continue;
	}

      if (s->_raw_size == 0)
	{
	  /* If we don't need this section, strip it from the
	     output file.  This is mostly to handle .rela.bss and
	     .rela.plt.  We must create both sections in
	     create_dynamic_sections, because they must be created
	     before the linker maps input sections to output
	     sections.  The linker does that before
	     adjust_dynamic_symbol is called, and it is that
	     function which decides whether anything needs to go
	     into these sections.  */
	  _bfd_strip_section_from_output (info, s);
	  continue;
	}

      /* Allocate memory for the section contents.  Zero it, because
	 we may not fill in all the reloc sections.  */
      s->contents = (bfd_byte *) bfd_zalloc (dynobj, s->_raw_size);
      if (s->contents == NULL && s->_raw_size != 0)
	return false;
    }

  if (hplink->root.dynamic_sections_created)
    {
      /* Like IA-64 and HPPA64, always create a DT_PLTGOT.  It
	 actually has nothing to do with the PLT, it is how we
	 communicate the LTP value of a load module to the dynamic
	 linker.  */
      if (! bfd_elf32_add_dynamic_entry (info, DT_PLTGOT, 0))
	return false;

      /* Add some entries to the .dynamic section.  We fill in the
	 values later, in elf32_hppa_finish_dynamic_sections, but we
	 must add the entries now so that we get the correct size for
	 the .dynamic section.  The DT_DEBUG entry is filled in by the
	 dynamic linker and used by the debugger.  */
      if (! info->shared)
	{
	  if (! bfd_elf32_add_dynamic_entry (info, DT_DEBUG, 0))
	    return false;
	}

      if (hplink->srelplt->_raw_size != 0)
	{
	  if (! bfd_elf32_add_dynamic_entry (info, DT_PLTRELSZ, 0)
	      || ! bfd_elf32_add_dynamic_entry (info, DT_PLTREL, DT_RELA)
	      || ! bfd_elf32_add_dynamic_entry (info, DT_JMPREL, 0))
	    return false;
	}

      if (relocs)
	{
	  if (! bfd_elf32_add_dynamic_entry (info, DT_RELA, 0)
	      || ! bfd_elf32_add_dynamic_entry (info, DT_RELASZ, 0)
	      || ! bfd_elf32_add_dynamic_entry (info, DT_RELAENT,
						sizeof (Elf32_External_Rela)))
	    return false;
	}

      if (reltext)
	{
	  if (! bfd_elf32_add_dynamic_entry (info, DT_TEXTREL, 0))
	    return false;
	  info->flags |= DF_TEXTREL;
	}
    }

  return true;
}

/* External entry points for sizing and building linker stubs.  */

/* Determine and set the size of the stub section for a final link.

   The basic idea here is to examine all the relocations looking for
   PC-relative calls to a target that is unreachable with a "bl"
   instruction.  */

boolean
elf32_hppa_size_stubs (output_bfd, stub_bfd, info, multi_subspace, group_size,
		       add_stub_section, layout_sections_again)
     bfd *output_bfd;
     bfd *stub_bfd;
     struct bfd_link_info *info;
     boolean multi_subspace;
     bfd_signed_vma group_size;
     asection * (*add_stub_section) PARAMS ((const char *, asection *));
     void (*layout_sections_again) PARAMS ((void));
{
  bfd *input_bfd;
  asection *section;
  asection **input_list, **list;
  Elf_Internal_Sym *local_syms, **all_local_syms;
  unsigned int bfd_indx, bfd_count;
  int top_id, top_index;
  struct elf32_hppa_link_hash_table *hplink;
  bfd_size_type stub_group_size;
  boolean stubs_always_before_branch;
  boolean stub_changed = 0;
  boolean ret = 0;

  hplink = hppa_link_hash_table (info);

  /* Stash our params away.  */
  hplink->stub_bfd = stub_bfd;
  hplink->multi_subspace = multi_subspace;
  hplink->add_stub_section = add_stub_section;
  hplink->layout_sections_again = layout_sections_again;
  stubs_always_before_branch = group_size < 0;
  if (group_size < 0)
    stub_group_size = -group_size;
  else
    stub_group_size = group_size;
  if (stub_group_size == 1)
    {
      /* Default values.  */
      stub_group_size = 8000000;
      if (hplink->has_17bit_branch || hplink->multi_subspace)
	stub_group_size = 250000;
      if (hplink->has_12bit_branch)
	stub_group_size = 7812;
    }

  /* Count the number of input BFDs and find the top input section id.  */
  for (input_bfd = info->input_bfds, bfd_count = 0, top_id = 0;
       input_bfd != NULL;
       input_bfd = input_bfd->link_next)
    {
      bfd_count += 1;
      for (section = input_bfd->sections;
	   section != NULL;
	   section = section->next)
	{
	  if (top_id < section->id)
	    top_id = section->id;
	}
    }

  hplink->stub_group
    = (struct map_stub *) bfd_zmalloc (sizeof (struct map_stub) * (top_id + 1));
  if (hplink->stub_group == NULL)
    return false;

  /* Make a list of input sections for each output section included in
     the link.

     We can't use output_bfd->section_count here to find the top output
     section index as some sections may have been removed, and
     _bfd_strip_section_from_output doesn't renumber the indices.  */
  for (section = output_bfd->sections, top_index = 0;
       section != NULL;
       section = section->next)
    {
      if (top_index < section->index)
	top_index = section->index;
    }

  input_list
    = (asection **) bfd_malloc (sizeof (asection *) * (top_index + 1));
  if (input_list == NULL)
    return false;

  /* For sections we aren't interested in, mark their entries with a
     value we can check later.  */
  list = input_list + top_index;
  do
    *list = bfd_abs_section_ptr;
  while (list-- != input_list);

  for (section = output_bfd->sections;
       section != NULL;
       section = section->next)
    {
      if ((section->flags & SEC_CODE) != 0)
	input_list[section->index] = NULL;
    }

  /* Now actually build the lists.  */
  for (input_bfd = info->input_bfds;
       input_bfd != NULL;
       input_bfd = input_bfd->link_next)
    {
      for (section = input_bfd->sections;
	   section != NULL;
	   section = section->next)
	{
	  if (section->output_section != NULL
	      && section->output_section->owner == output_bfd
	      && section->output_section->index <= top_index)
	    {
	      list = input_list + section->output_section->index;
	      if (*list != bfd_abs_section_ptr)
		{
		  /* Steal the link_sec pointer for our list.  */
#define PREV_SEC(sec) (hplink->stub_group[(sec)->id].link_sec)
		  /* This happens to make the list in reverse order,
		     which is what we want.  */
		  PREV_SEC (section) = *list;
		  *list = section;
		}
	    }
	}
    }

  /* See whether we can group stub sections together.  Grouping stub
     sections may result in fewer stubs.  More importantly, we need to
     put all .init* and .fini* stubs at the beginning of the .init or
     .fini output sections respectively, because glibc splits the
     _init and _fini functions into multiple parts.  Putting a stub in
     the middle of a function is not a good idea.  */
  list = input_list + top_index;
  do
    {
      asection *tail = *list;
      if (tail == bfd_abs_section_ptr)
	continue;
      while (tail != NULL)
	{
	  asection *curr;
	  asection *prev;
	  bfd_size_type total;

	  curr = tail;
	  if (tail->_cooked_size)
	    total = tail->_cooked_size;
	  else
	    total = tail->_raw_size;
	  while ((prev = PREV_SEC (curr)) != NULL
		 && ((total += curr->output_offset - prev->output_offset)
		     < stub_group_size))
	    curr = prev;

	  /* OK, the size from the start of CURR to the end is less
	     than 250000 bytes and thus can be handled by one stub
	     section.  (or the tail section is itself larger than
	     250000 bytes, in which case we may be toast.)
	     We should really be keeping track of the total size of
	     stubs added here, as stubs contribute to the final output
	     section size.  That's a little tricky, and this way will
	     only break if stubs added total more than 12144 bytes, or
	     1518 long branch stubs.  It seems unlikely for more than
	     1518 different functions to be called, especially from
	     code only 250000 bytes long.  */
	  do
	    {
	      prev = PREV_SEC (tail);
	      /* Set up this stub group.  */
	      hplink->stub_group[tail->id].link_sec = curr;
	    }
	  while (tail != curr && (tail = prev) != NULL);

	  /* But wait, there's more!  Input sections up to 250000
	     bytes before the stub section can be handled by it too.  */
	  if (!stubs_always_before_branch)
	    {
	      total = 0;
	      while (prev != NULL
		     && ((total += tail->output_offset - prev->output_offset)
			 < stub_group_size))
		{
		  tail = prev;
		  prev = PREV_SEC (tail);
		  hplink->stub_group[tail->id].link_sec = curr;
		}
	    }
	  tail = prev;
	}
    }
  while (list-- != input_list);
  free (input_list);
#undef PREV_SEC

  /* We want to read in symbol extension records only once.  To do this
     we need to read in the local symbols in parallel and save them for
     later use; so hold pointers to the local symbols in an array.  */
  all_local_syms
    = (Elf_Internal_Sym **) bfd_zmalloc (sizeof (Elf_Internal_Sym *)
					 * bfd_count);
  if (all_local_syms == NULL)
    return false;

  /* Walk over all the input BFDs, swapping in local symbols.
     If we are creating a shared library, create hash entries for the
     export stubs.  */
  for (input_bfd = info->input_bfds, bfd_indx = 0;
       input_bfd != NULL;
       input_bfd = input_bfd->link_next, bfd_indx++)
    {
      Elf_Internal_Shdr *symtab_hdr;
      Elf_Internal_Sym *isym;
      Elf32_External_Sym *ext_syms, *esym, *end_sy;

      /* We'll need the symbol table in a second.  */
      symtab_hdr = &elf_tdata (input_bfd)->symtab_hdr;
      if (symtab_hdr->sh_info == 0)
	continue;

      /* We need an array of the local symbols attached to the input bfd.
	 Unfortunately, we're going to have to read & swap them in.  */
      local_syms = (Elf_Internal_Sym *)
	bfd_malloc (symtab_hdr->sh_info * sizeof (Elf_Internal_Sym));
      if (local_syms == NULL)
	{
	  goto error_ret_free_local;
	}
      all_local_syms[bfd_indx] = local_syms;
      ext_syms = (Elf32_External_Sym *)
	bfd_malloc (symtab_hdr->sh_info * sizeof (Elf32_External_Sym));
      if (ext_syms == NULL)
	{
	  goto error_ret_free_local;
	}

      if (bfd_seek (input_bfd, symtab_hdr->sh_offset, SEEK_SET) != 0
	  || (bfd_read (ext_syms, 1,
			(symtab_hdr->sh_info * sizeof (Elf32_External_Sym)),
			input_bfd)
	      != (symtab_hdr->sh_info * sizeof (Elf32_External_Sym))))
	{
	  free (ext_syms);
	  goto error_ret_free_local;
	}

      /* Swap the local symbols in.  */
      isym = local_syms;
      esym = ext_syms;
      for (end_sy = esym + symtab_hdr->sh_info; esym < end_sy; esym++, isym++)
	bfd_elf32_swap_symbol_in (input_bfd, esym, isym);

      /* Now we can free the external symbols.  */
      free (ext_syms);

#if ! LONG_BRANCH_PIC_IN_SHLIB
      /* If this is a shared link, find all the stub reloc sections.  */
      if (info->shared)
	for (section = input_bfd->sections;
	     section != NULL;
	     section = section->next)
	  {
	    char *name;
	    asection *reloc_sec;

	    name = bfd_malloc (strlen (section->name)
			       + sizeof STUB_SUFFIX
			       + 5);
	    if (name == NULL)
	      return false;
	    sprintf (name, ".rela%s%s", section->name, STUB_SUFFIX);
	    reloc_sec = bfd_get_section_by_name (hplink->root.dynobj, name);
	    hplink->stub_group[section->id].reloc_sec = reloc_sec;
	    free (name);
	  }
#endif

      if (info->shared && hplink->multi_subspace)
	{
	  struct elf_link_hash_entry **sym_hashes;
	  struct elf_link_hash_entry **end_hashes;
	  unsigned int symcount;

	  symcount = (symtab_hdr->sh_size / sizeof (Elf32_External_Sym)
		      - symtab_hdr->sh_info);
	  sym_hashes = elf_sym_hashes (input_bfd);
	  end_hashes = sym_hashes + symcount;

	  /* Look through the global syms for functions;  We need to
	     build export stubs for all globally visible functions.  */
	  for (; sym_hashes < end_hashes; sym_hashes++)
	    {
	      struct elf32_hppa_link_hash_entry *hash;

	      hash = (struct elf32_hppa_link_hash_entry *) *sym_hashes;

	      while (hash->elf.root.type == bfd_link_hash_indirect
		     || hash->elf.root.type == bfd_link_hash_warning)
		hash = ((struct elf32_hppa_link_hash_entry *)
			hash->elf.root.u.i.link);

	      /* At this point in the link, undefined syms have been
		 resolved, so we need to check that the symbol was
		 defined in this BFD.  */
	      if ((hash->elf.root.type == bfd_link_hash_defined
		   || hash->elf.root.type == bfd_link_hash_defweak)
		  && hash->elf.type == STT_FUNC
		  && hash->elf.root.u.def.section->output_section != NULL
		  && (hash->elf.root.u.def.section->output_section->owner
		      == output_bfd)
		  && hash->elf.root.u.def.section->owner == input_bfd
		  && (hash->elf.elf_link_hash_flags & ELF_LINK_HASH_DEF_REGULAR)
		  && !(hash->elf.elf_link_hash_flags & ELF_LINK_FORCED_LOCAL)
		  && ELF_ST_VISIBILITY (hash->elf.other) == STV_DEFAULT)
		{
		  asection *sec;
		  const char *stub_name;
		  struct elf32_hppa_stub_hash_entry *stub_entry;

		  sec = hash->elf.root.u.def.section;
		  stub_name = hash->elf.root.root.string;
		  stub_entry = hppa_stub_hash_lookup (&hplink->stub_hash_table,
						      stub_name,
						      false, false);
		  if (stub_entry == NULL)
		    {
		      stub_entry = hppa_add_stub (stub_name, sec, hplink);
		      if (!stub_entry)
			goto error_ret_free_local;

		      stub_entry->target_value = hash->elf.root.u.def.value;
		      stub_entry->target_section = hash->elf.root.u.def.section;
		      stub_entry->stub_type = hppa_stub_export;
		      stub_entry->h = hash;
		      stub_changed = 1;
		    }
		  else
		    {
		      (*_bfd_error_handler) (_("%s: duplicate export stub %s"),
			     bfd_get_filename (input_bfd),
			     stub_name);
		    }
		}
	    }
	}
    }

  while (1)
    {
      asection *stub_sec;

      for (input_bfd = info->input_bfds, bfd_indx = 0;
	   input_bfd != NULL;
	   input_bfd = input_bfd->link_next, bfd_indx++)
	{
	  Elf_Internal_Shdr *symtab_hdr;

	  /* We'll need the symbol table in a second.  */
	  symtab_hdr = &elf_tdata (input_bfd)->symtab_hdr;
	  if (symtab_hdr->sh_info == 0)
	    continue;

	  local_syms = all_local_syms[bfd_indx];

	  /* Walk over each section attached to the input bfd.  */
	  for (section = input_bfd->sections;
	       section != NULL;
	       section = section->next)
	    {
	      Elf_Internal_Shdr *input_rel_hdr;
	      Elf32_External_Rela *external_relocs, *erelaend, *erela;
	      Elf_Internal_Rela *internal_relocs, *irelaend, *irela;

	      /* If there aren't any relocs, then there's nothing more
		 to do.  */
	      if ((section->flags & SEC_RELOC) == 0
		  || section->reloc_count == 0)
		continue;

	      /* If this section is a link-once section that will be
		 discarded, then don't create any stubs.  */
	      if (section->output_section == NULL
		  || section->output_section->owner != output_bfd)
		continue;

	      /* Allocate space for the external relocations.  */
	      external_relocs
		= ((Elf32_External_Rela *)
		   bfd_malloc (section->reloc_count
			       * sizeof (Elf32_External_Rela)));
	      if (external_relocs == NULL)
		{
		  goto error_ret_free_local;
		}

	      /* Likewise for the internal relocations.  */
	      internal_relocs = ((Elf_Internal_Rela *)
				 bfd_malloc (section->reloc_count
					     * sizeof (Elf_Internal_Rela)));
	      if (internal_relocs == NULL)
		{
		  free (external_relocs);
		  goto error_ret_free_local;
		}

	      /* Read in the external relocs.  */
	      input_rel_hdr = &elf_section_data (section)->rel_hdr;
	      if (bfd_seek (input_bfd, input_rel_hdr->sh_offset, SEEK_SET) != 0
		  || bfd_read (external_relocs, 1,
			       input_rel_hdr->sh_size,
			       input_bfd) != input_rel_hdr->sh_size)
		{
		  free (external_relocs);
		error_ret_free_internal:
		  free (internal_relocs);
		  goto error_ret_free_local;
		}

	      /* Swap in the relocs.  */
	      erela = external_relocs;
	      erelaend = erela + section->reloc_count;
	      irela = internal_relocs;
	      for (; erela < erelaend; erela++, irela++)
		bfd_elf32_swap_reloca_in (input_bfd, erela, irela);

	      /* We're done with the external relocs, free them.  */
	      free (external_relocs);

	      /* Now examine each relocation.  */
	      irela = internal_relocs;
	      irelaend = irela + section->reloc_count;
	      for (; irela < irelaend; irela++)
		{
		  unsigned int r_type, r_indx;
		  enum elf32_hppa_stub_type stub_type;
		  struct elf32_hppa_stub_hash_entry *stub_entry;
		  asection *sym_sec;
		  bfd_vma sym_value;
		  bfd_vma destination;
		  struct elf32_hppa_link_hash_entry *hash;
		  char *stub_name;
		  const asection *id_sec;

		  r_type = ELF32_R_TYPE (irela->r_info);
		  r_indx = ELF32_R_SYM (irela->r_info);

		  if (r_type >= (unsigned int) R_PARISC_UNIMPLEMENTED)
		    {
		      bfd_set_error (bfd_error_bad_value);
		      goto error_ret_free_internal;
		    }

		  /* Only look for stubs on call instructions.  */
		  if (r_type != (unsigned int) R_PARISC_PCREL12F
		      && r_type != (unsigned int) R_PARISC_PCREL17F
		      && r_type != (unsigned int) R_PARISC_PCREL22F)
		    continue;

		  /* Now determine the call target, its name, value,
		     section.  */
		  sym_sec = NULL;
		  sym_value = 0;
		  destination = 0;
		  hash = NULL;
		  if (r_indx < symtab_hdr->sh_info)
		    {
		      /* It's a local symbol.  */
		      Elf_Internal_Sym *sym;
		      Elf_Internal_Shdr *hdr;

		      sym = local_syms + r_indx;
		      hdr = elf_elfsections (input_bfd)[sym->st_shndx];
		      sym_sec = hdr->bfd_section;
		      if (ELF_ST_TYPE (sym->st_info) != STT_SECTION)
			sym_value = sym->st_value;
		      destination = (sym_value + irela->r_addend
				     + sym_sec->output_offset
				     + sym_sec->output_section->vma);
		    }
		  else
		    {
		      /* It's an external symbol.  */
		      int e_indx;

		      e_indx = r_indx - symtab_hdr->sh_info;
		      hash = ((struct elf32_hppa_link_hash_entry *)
			      elf_sym_hashes (input_bfd)[e_indx]);

		      while (hash->elf.root.type == bfd_link_hash_indirect
			     || hash->elf.root.type == bfd_link_hash_warning)
			hash = ((struct elf32_hppa_link_hash_entry *)
				hash->elf.root.u.i.link);

		      if (hash->elf.root.type == bfd_link_hash_defined
			  || hash->elf.root.type == bfd_link_hash_defweak)
			{
			  sym_sec = hash->elf.root.u.def.section;
			  sym_value = hash->elf.root.u.def.value;
			  if (sym_sec->output_section != NULL)
			    destination = (sym_value + irela->r_addend
					   + sym_sec->output_offset
					   + sym_sec->output_section->vma);
			}
		      else if (hash->elf.root.type == bfd_link_hash_undefweak)
			{
			  if (! info->shared)
			    continue;
			}
		      else if (hash->elf.root.type == bfd_link_hash_undefined)
			{
			  if (! (info->shared
				 && !info->no_undefined
				 && (ELF_ST_VISIBILITY (hash->elf.other)
				     == STV_DEFAULT)))
			    continue;
			}
		      else
			{
			  bfd_set_error (bfd_error_bad_value);
			  goto error_ret_free_internal;
			}
		    }

		  /* Determine what (if any) linker stub is needed.  */
		  stub_type = hppa_type_of_stub (section, irela, hash,
						 destination);
		  if (stub_type == hppa_stub_none)
		    continue;

		  /* Support for grouping stub sections.  */
		  id_sec = hplink->stub_group[section->id].link_sec;

		  /* Get the name of this stub.  */
		  stub_name = hppa_stub_name (id_sec, sym_sec, hash, irela);
		  if (!stub_name)
		    goto error_ret_free_internal;

		  stub_entry = hppa_stub_hash_lookup (&hplink->stub_hash_table,
						      stub_name,
						      false, false);
		  if (stub_entry != NULL)
		    {
		      /* The proper stub has already been created.  */
		      free (stub_name);
		      continue;
		    }

		  stub_entry = hppa_add_stub (stub_name, section, hplink);
		  if (stub_entry == NULL)
		    {
		      free (stub_name);
		      goto error_ret_free_local;
		    }

		  stub_entry->target_value = sym_value;
		  stub_entry->target_section = sym_sec;
		  stub_entry->stub_type = stub_type;
		  if (info->shared)
		    {
		      if (stub_type == hppa_stub_import)
			stub_entry->stub_type = hppa_stub_import_shared;
		      else if (stub_type == hppa_stub_long_branch
			       && (LONG_BRANCH_PIC_IN_SHLIB || hash == NULL))
			stub_entry->stub_type = hppa_stub_long_branch_shared;
		    }
		  stub_entry->h = hash;
		  stub_changed = 1;
		}

	      /* We're done with the internal relocs, free them.  */
	      free (internal_relocs);
	    }
	}

      if (!stub_changed)
	break;

      /* OK, we've added some stubs.  Find out the new size of the
	 stub sections.  */
      for (stub_sec = hplink->stub_bfd->sections;
	   stub_sec != NULL;
	   stub_sec = stub_sec->next)
	{
	  stub_sec->_raw_size = 0;
	  stub_sec->_cooked_size = 0;
	}
#if ! LONG_BRANCH_PIC_IN_SHLIB
      {
	int i;

	for (i = top_id; i >= 0; --i)
	  {
	    /* This will probably hit the same section many times..  */
	    stub_sec = hplink->stub_group[i].reloc_sec;
	    if (stub_sec != NULL)
	      {
		stub_sec->_raw_size = 0;
		stub_sec->_cooked_size = 0;
	      }
	  }
      }
#endif

      bfd_hash_traverse (&hplink->stub_hash_table,
			 hppa_size_one_stub,
			 hplink);

      /* Ask the linker to do its stuff.  */
      (*hplink->layout_sections_again) ();
      stub_changed = 0;
    }

  ret = 1;

 error_ret_free_local:
  while (bfd_count-- > 0)
    if (all_local_syms[bfd_count])
      free (all_local_syms[bfd_count]);
  free (all_local_syms);

  return ret;
}

/* For a final link, this function is called after we have sized the
   stubs to provide a value for __gp.  */

boolean
elf32_hppa_set_gp (abfd, info)
     bfd *abfd;
     struct bfd_link_info *info;
{
  struct elf32_hppa_link_hash_table *hplink;
  struct elf_link_hash_entry *h;
  asection *sec;
  bfd_vma gp_val;

  hplink = hppa_link_hash_table (info);
  h = elf_link_hash_lookup (&hplink->root, "$global$",
			    false, false, false);

  if (h != NULL && h->root.type == bfd_link_hash_defined)
    {
      gp_val = h->root.u.def.value;
      sec = h->root.u.def.section;
    }
  else
    {
      /* Choose to point our LTP at, in this order, one of .plt, .got,
	 or .data, if these sections exist.  In the case of choosing
	 .plt try to make the LTP ideal for addressing anywhere in the
	 .plt or .got with a 14 bit signed offset.  Typically, the end
	 of the .plt is the start of the .got, so choose .plt + 0x2000
	 if either the .plt or .got is larger than 0x2000.  If both
	 the .plt and .got are smaller than 0x2000, choose the end of
	 the .plt section.  */

      sec = hplink->splt;
      if (sec != NULL)
	{
	  gp_val = sec->_raw_size;
	  if (gp_val > 0x2000
	      || (hplink->sgot && hplink->sgot->_raw_size > 0x2000))
	    {
	      gp_val = 0x2000;
	    }
	}
      else
	{
	  gp_val = 0;
	  sec = hplink->sgot;
	  if (sec != NULL)
	    {
	      /* We know we don't have a .plt.  If .got is large,
		 offset our LTP.  */
	      if (sec->_raw_size > 0x2000)
		gp_val = 0x2000;
	    }
	  else
	    {
	      /* No .plt or .got.  Who cares what the LTP is?  */
	      sec = bfd_get_section_by_name (abfd, ".data");
	    }
	}
    }

  if (sec != NULL)
    gp_val += sec->output_section->vma + sec->output_offset;

  elf_gp (abfd) = gp_val;
  return true;
}

/* Build all the stubs associated with the current output file.  The
   stubs are kept in a hash table attached to the main linker hash
   table.  We also set up the .plt entries for statically linked PIC
   functions here.  This function is called via hppaelf_finish in the
   linker.  */

boolean
elf32_hppa_build_stubs (info)
     struct bfd_link_info *info;
{
  asection *stub_sec;
  struct bfd_hash_table *table;
  struct elf32_hppa_link_hash_table *hplink;

  hplink = hppa_link_hash_table (info);

  for (stub_sec = hplink->stub_bfd->sections;
       stub_sec != NULL;
       stub_sec = stub_sec->next)
    {
      size_t size;

      /* Allocate memory to hold the linker stubs.  */
      size = stub_sec->_raw_size;
      stub_sec->contents = (unsigned char *) bfd_zalloc (hplink->stub_bfd,
							 size);
      if (stub_sec->contents == NULL && size != 0)
	return false;
      stub_sec->_raw_size = 0;
    }

  /* Build the stubs as directed by the stub hash table.  */
  table = &hplink->stub_hash_table;
  bfd_hash_traverse (table, hppa_build_one_stub, info);

  return true;
}

/* Perform a relocation as part of a final link.  */

static bfd_reloc_status_type
final_link_relocate (input_section, contents, rel, value, hplink, sym_sec, h)
     asection *input_section;
     bfd_byte *contents;
     const Elf_Internal_Rela *rel;
     bfd_vma value;
     struct elf32_hppa_link_hash_table *hplink;
     asection *sym_sec;
     struct elf32_hppa_link_hash_entry *h;
{
  int insn;
  unsigned int r_type = ELF32_R_TYPE (rel->r_info);
  reloc_howto_type *howto = elf_hppa_howto_table + r_type;
  int r_format = howto->bitsize;
  enum hppa_reloc_field_selector_type_alt r_field;
  bfd *input_bfd = input_section->owner;
  bfd_vma offset = rel->r_offset;
  bfd_vma max_branch_offset = 0;
  bfd_byte *hit_data = contents + offset;
  bfd_signed_vma addend = rel->r_addend;
  bfd_vma location;
  struct elf32_hppa_stub_hash_entry *stub_entry = NULL;
  int val;

  if (r_type == R_PARISC_NONE)
    return bfd_reloc_ok;

  insn = bfd_get_32 (input_bfd, hit_data);

  /* Find out where we are and where we're going.  */
  location = (offset +
	      input_section->output_offset +
	      input_section->output_section->vma);

  switch (r_type)
    {
    case R_PARISC_PCREL12F:
    case R_PARISC_PCREL17F:
    case R_PARISC_PCREL22F:
      /* If this is a call to a function defined in another dynamic
	 library, or if it is a call to a PIC function in the same
	 object, or if this is a shared link and it is a call to a
	 weak symbol which may or may not be in the same object, then
	 find the import stub in the stub hash.  */
      if (sym_sec == NULL
	  || sym_sec->output_section == NULL
	  || (h != NULL &&
	      (h->pic_call
	       || (h->elf.root.type == bfd_link_hash_defweak
		   && h->elf.dynindx != -1
		   && h->elf.plt.offset != (bfd_vma) -1))))
	{
	  stub_entry = hppa_get_stub_entry (input_section, sym_sec,
					    h, rel, hplink);
	  if (stub_entry != NULL)
	    {
	      value = (stub_entry->stub_offset
		       + stub_entry->stub_sec->output_offset
		       + stub_entry->stub_sec->output_section->vma);
	      addend = 0;
	    }
	  else if (sym_sec == NULL && h != NULL
		   && h->elf.root.type == bfd_link_hash_undefweak)
	    {
	      /* It's OK if undefined weak.  Make undefined weak
		 branches go nowhere.  */
	      value = location;
	      addend = 0;
	    }
	  else
	    return bfd_reloc_notsupported;
	}
      /* Fall thru.  */

    case R_PARISC_PCREL21L:
    case R_PARISC_PCREL17C:
    case R_PARISC_PCREL17R:
    case R_PARISC_PCREL14R:
    case R_PARISC_PCREL14F:
      /* Make it a pc relative offset.  */
      value -= location;
      addend -= 8;
      break;

    case R_PARISC_DPREL21L:
    case R_PARISC_DPREL14R:
    case R_PARISC_DPREL14F:
    /* For all the DP relative relocations, we need to examine the symbol's
       section.  If it's a code section, then "data pointer relative" makes
       no sense.  In that case we don't adjust the "value", and for 21 bit
       addil instructions, we change the source addend register from %dp to
       %r0.  This situation commonly arises when a variable's "constness"
       is declared differently from the way the variable is defined.  For
       instance: "extern int foo" with foo defined as "const int foo".  */
      if (sym_sec == NULL)
	break;
      if ((sym_sec->flags & SEC_CODE) != 0)
	{
	  if ((insn & ((0x3f << 26) | (0x1f << 21)))
	      == (((int) OP_ADDIL << 26) | (27 << 21)))
	    {
	      insn &= ~ (0x1f << 21);
#if 1 /* debug them.  */
	      (*_bfd_error_handler)
		(_("%s(%s+0x%lx): fixing %s"),
		 bfd_get_filename (input_bfd),
		 input_section->name,
		 (long) rel->r_offset,
		 howto->name);
#endif
	    }
	  /* Now try to make things easy for the dynamic linker.  */

	  break;
	}
      /* Fall thru.  */

    case R_PARISC_DLTIND21L:
    case R_PARISC_DLTIND14R:
    case R_PARISC_DLTIND14F:
      value -= elf_gp (input_section->output_section->owner);
      break;

    default:
      break;
    }

  switch (r_type)
    {
    case R_PARISC_DIR32:
    case R_PARISC_DIR14F:
    case R_PARISC_DIR17F:
    case R_PARISC_PCREL17C:
    case R_PARISC_PCREL14F:
    case R_PARISC_DPREL14F:
    case R_PARISC_PLABEL32:
    case R_PARISC_DLTIND14F:
    case R_PARISC_SEGBASE:
    case R_PARISC_SEGREL32:
      r_field = e_fsel;
      break;

    case R_PARISC_DIR21L:
    case R_PARISC_PCREL21L:
    case R_PARISC_DPREL21L:
    case R_PARISC_PLABEL21L:
    case R_PARISC_DLTIND21L:
      r_field = e_lrsel;
      break;

    case R_PARISC_DIR17R:
    case R_PARISC_PCREL17R:
    case R_PARISC_DIR14R:
    case R_PARISC_PCREL14R:
    case R_PARISC_DPREL14R:
    case R_PARISC_PLABEL14R:
    case R_PARISC_DLTIND14R:
      r_field = e_rrsel;
      break;

    case R_PARISC_PCREL12F:
    case R_PARISC_PCREL17F:
    case R_PARISC_PCREL22F:
      r_field = e_fsel;

      if (r_type == (unsigned int) R_PARISC_PCREL17F)
	{
	  max_branch_offset = (1 << (17-1)) << 2;
	}
      else if (r_type == (unsigned int) R_PARISC_PCREL12F)
	{
	  max_branch_offset = (1 << (12-1)) << 2;
	}
      else
	{
	  max_branch_offset = (1 << (22-1)) << 2;
	}

      /* sym_sec is NULL on undefined weak syms or when shared on
	 undefined syms.  We've already checked for a stub for the
	 shared undefined case.  */
      if (sym_sec == NULL)
	break;

      /* If the branch is out of reach, then redirect the
	 call to the local stub for this function.  */
      if (value + addend + max_branch_offset >= 2*max_branch_offset)
	{
	  stub_entry = hppa_get_stub_entry (input_section, sym_sec,
					    h, rel, hplink);
	  if (stub_entry == NULL)
	    return bfd_reloc_notsupported;

	  /* Munge up the value and addend so that we call the stub
	     rather than the procedure directly.  */
	  value = (stub_entry->stub_offset
		   + stub_entry->stub_sec->output_offset
		   + stub_entry->stub_sec->output_section->vma
		   - location);
	  addend = -8;
	}
      break;

    /* Something we don't know how to handle.  */
    default:
      return bfd_reloc_notsupported;
    }

  /* Make sure we can reach the stub.  */
  if (max_branch_offset != 0
      && value + addend + max_branch_offset >= 2*max_branch_offset)
    {
      (*_bfd_error_handler)
	(_("%s(%s+0x%lx): cannot reach %s, recompile with -ffunction-sections"),
	 bfd_get_filename (input_bfd),
	 input_section->name,
	 (long) rel->r_offset,
	 stub_entry->root.string);
      return bfd_reloc_notsupported;
    }

  val = hppa_field_adjust (value, addend, r_field);

  switch (r_type)
    {
    case R_PARISC_PCREL12F:
    case R_PARISC_PCREL17C:
    case R_PARISC_PCREL17F:
    case R_PARISC_PCREL17R:
    case R_PARISC_PCREL22F:
    case R_PARISC_DIR17F:
    case R_PARISC_DIR17R:
      /* This is a branch.  Divide the offset by four.
	 Note that we need to decide whether it's a branch or
	 otherwise by inspecting the reloc.  Inspecting insn won't
	 work as insn might be from a .word directive.  */
      val >>= 2;
      break;

    default:
      break;
    }

  insn = hppa_rebuild_insn (insn, val, r_format);

  /* Update the instruction word.  */
  bfd_put_32 (input_bfd, (bfd_vma) insn, hit_data);
  return bfd_reloc_ok;
}

/* Relocate an HPPA ELF section.  */

static boolean
elf32_hppa_relocate_section (output_bfd, info, input_bfd, input_section,
			     contents, relocs, local_syms, local_sections)
     bfd *output_bfd;
     struct bfd_link_info *info;
     bfd *input_bfd;
     asection *input_section;
     bfd_byte *contents;
     Elf_Internal_Rela *relocs;
     Elf_Internal_Sym *local_syms;
     asection **local_sections;
{
  bfd *dynobj;
  bfd_vma *local_got_offsets;
  struct elf32_hppa_link_hash_table *hplink;
  Elf_Internal_Shdr *symtab_hdr;
  Elf_Internal_Rela *rel;
  Elf_Internal_Rela *relend;
  asection *sreloc;

  symtab_hdr = &elf_tdata (input_bfd)->symtab_hdr;

  hplink = hppa_link_hash_table (info);
  dynobj = hplink->root.dynobj;
  local_got_offsets = elf_local_got_offsets (input_bfd);
  sreloc = NULL;

  rel = relocs;
  relend = relocs + input_section->reloc_count;
  for (; rel < relend; rel++)
    {
      unsigned int r_type;
      reloc_howto_type *howto;
      unsigned int r_symndx;
      struct elf32_hppa_link_hash_entry *h;
      Elf_Internal_Sym *sym;
      asection *sym_sec;
      bfd_vma relocation;
      bfd_reloc_status_type r;
      const char *sym_name;
      boolean plabel;

      r_type = ELF32_R_TYPE (rel->r_info);
      if (r_type >= (unsigned int) R_PARISC_UNIMPLEMENTED)
	{
	  bfd_set_error (bfd_error_bad_value);
	  return false;
	}
      if (r_type == (unsigned int) R_PARISC_GNU_VTENTRY
	  || r_type == (unsigned int) R_PARISC_GNU_VTINHERIT)
	continue;

      r_symndx = ELF32_R_SYM (rel->r_info);

      if (info->relocateable)
	{
	  /* This is a relocateable link.  We don't have to change
	     anything, unless the reloc is against a section symbol,
	     in which case we have to adjust according to where the
	     section symbol winds up in the output section.  */
	  if (r_symndx < symtab_hdr->sh_info)
	    {
	      sym = local_syms + r_symndx;
	      if (ELF_ST_TYPE (sym->st_info) == STT_SECTION)
		{
		  sym_sec = local_sections[r_symndx];
		  rel->r_addend += sym_sec->output_offset;
		}
	    }
	  continue;
	}

      /* This is a final link.  */
      h = NULL;
      sym = NULL;
      sym_sec = NULL;
      if (r_symndx < symtab_hdr->sh_info)
	{
	  /* This is a local symbol, h defaults to NULL.  */
	  sym = local_syms + r_symndx;
	  sym_sec = local_sections[r_symndx];
	  relocation = ((ELF_ST_TYPE (sym->st_info) == STT_SECTION
			   ? 0 : sym->st_value)
			 + sym_sec->output_offset
			 + sym_sec->output_section->vma);
	}
      else
	{
	  int indx;

	  /* It's a global; Find its entry in the link hash.  */
	  indx = r_symndx - symtab_hdr->sh_info;
	  h = ((struct elf32_hppa_link_hash_entry *)
	       elf_sym_hashes (input_bfd)[indx]);
	  while (h->elf.root.type == bfd_link_hash_indirect
		 || h->elf.root.type == bfd_link_hash_warning)
	    h = (struct elf32_hppa_link_hash_entry *) h->elf.root.u.i.link;

	  relocation = 0;
	  if (h->elf.root.type == bfd_link_hash_defined
	      || h->elf.root.type == bfd_link_hash_defweak)
	    {
	      sym_sec = h->elf.root.u.def.section;
	      /* If sym_sec->output_section is NULL, then it's a
		 symbol defined in a shared library.  */
	      if (sym_sec->output_section != NULL)
		relocation = (h->elf.root.u.def.value
			      + sym_sec->output_offset
			      + sym_sec->output_section->vma);
	    }
	  else if (h->elf.root.type == bfd_link_hash_undefweak)
	    ;
	  else if (info->shared && !info->no_undefined
		   && ELF_ST_VISIBILITY (h->elf.other) == STV_DEFAULT)
	    {
	      if (info->symbolic)
		if (!((*info->callbacks->undefined_symbol)
		      (info, h->elf.root.root.string, input_bfd,
		       input_section, rel->r_offset, false)))
		  return false;
	    }
	  else
	    {
	      if (!((*info->callbacks->undefined_symbol)
		    (info, h->elf.root.root.string, input_bfd,
		     input_section, rel->r_offset, true)))
		return false;
	    }
	}

      /* Do any required modifications to the relocation value, and
	 determine what types of dynamic info we need to output, if
	 any.  */
      plabel = 0;
      switch (r_type)
	{
	case R_PARISC_DLTIND14F:
	case R_PARISC_DLTIND14R:
	case R_PARISC_DLTIND21L:
	  /* Relocation is to the entry for this symbol in the global
	     offset table.  */
	  if (h != NULL)
	    {
	      bfd_vma off;

	      off = h->elf.got.offset;
	      BFD_ASSERT (off != (bfd_vma) -1);

	      if (! hplink->root.dynamic_sections_created
		  || (info->shared
		      && (info->symbolic || h->elf.dynindx == -1)
		      && (h->elf.elf_link_hash_flags
			  & ELF_LINK_HASH_DEF_REGULAR) != 0))
		{
		  /* This is actually a static link, or it is a
		     -Bsymbolic link and the symbol is defined
		     locally, or the symbol was forced to be local
		     because of a version file.  We must initialize
		     this entry in the global offset table.  Since the
		     offset must always be a multiple of 4, we use the
		     least significant bit to record whether we have
		     initialized it already.

		     When doing a dynamic link, we create a .rela.got
		     relocation entry to initialize the value.  This
		     is done in the finish_dynamic_symbol routine.  */
		  if ((off & 1) != 0)
		    off &= ~1;
		  else
		    {
		      bfd_put_32 (output_bfd, relocation,
				  hplink->sgot->contents + off);
		      h->elf.got.offset |= 1;
		    }
		}

	      relocation = off;
	    }
	  else
	    {
	      /* Local symbol case.  */
	      bfd_vma off;

	      BFD_ASSERT (local_got_offsets != NULL
			  && local_got_offsets[r_symndx] != (bfd_vma) -1);

	      off = local_got_offsets[r_symndx];

	      /* The offset must always be a multiple of 4.  We use
		 the least significant bit to record whether we have
		 already generated the necessary reloc.  */
	      if ((off & 1) != 0)
		off &= ~1;
	      else
		{
		  bfd_put_32 (output_bfd, relocation,
			      hplink->sgot->contents + off);

		  if (info->shared)
		    {
		      /* Output a dynamic *ABS* relocation for this
			 GOT entry.  In this case it is relative to
			 the base of the object because the symbol
			 index is zero.  */
		      Elf_Internal_Rela outrel;
		      asection *srelgot = hplink->srelgot;

		      outrel.r_offset = (off
					 + hplink->sgot->output_offset
					 + hplink->sgot->output_section->vma);
		      outrel.r_info = ELF32_R_INFO (0, R_PARISC_DIR32);
		      outrel.r_addend = relocation;
		      bfd_elf32_swap_reloca_out (output_bfd, &outrel,
						 ((Elf32_External_Rela *)
						  srelgot->contents
						  + srelgot->reloc_count));
		      ++srelgot->reloc_count;
		    }

		  local_got_offsets[r_symndx] |= 1;
		}

	      relocation = off;
	    }

	  /* Add the base of the GOT to the relocation value.  */
	  relocation += (hplink->sgot->output_offset
			 + hplink->sgot->output_section->vma);
	  break;

	case R_PARISC_PLABEL14R:
	case R_PARISC_PLABEL21L:
	case R_PARISC_PLABEL32:
	  if (hplink->root.dynamic_sections_created)
	    {
	      bfd_vma off;

	      /* If we have a global symbol with a PLT slot, then
		 redirect this relocation to it.  */
	      if (h != NULL)
		{
		  off = h->elf.plt.offset;
		}
	      else
		{
		  int indx;

		  indx = r_symndx + symtab_hdr->sh_info;
		  off = local_got_offsets[indx];

		  /* As for the local .got entry case, we use the last
		     bit to record whether we've already initialised
		     this local .plt entry.  */
		  if ((off & 1) != 0)
		    off &= ~1;
		  else
		    {
		      bfd_put_32 (output_bfd,
				  relocation,
				  hplink->splt->contents + off);
		      bfd_put_32 (output_bfd,
				  elf_gp (hplink->splt->output_section->owner),
				  hplink->splt->contents + off + 4);

		      if (info->shared)
			{
			  /* Output a dynamic IPLT relocation for this
			     PLT entry.  */
			  Elf_Internal_Rela outrel;
			  asection *srelplt = hplink->srelplt;

			  outrel.r_offset = (off
					     + hplink->splt->output_offset
					     + hplink->splt->output_section->vma);
			  outrel.r_info = ELF32_R_INFO (0, R_PARISC_IPLT);
			  outrel.r_addend = relocation;
			  bfd_elf32_swap_reloca_out (output_bfd, &outrel,
						     ((Elf32_External_Rela *)
						      srelplt->contents
						      + srelplt->reloc_count));
			  ++srelplt->reloc_count;
			}

		      local_got_offsets[indx] |= 1;
		    }
		}

	      BFD_ASSERT (off < (bfd_vma) -2);

	      /* PLABELs contain function pointers.  Relocation is to
		 the entry for the function in the .plt.  The magic +2
		 offset signals to $$dyncall that the function pointer
		 is in the .plt and thus has a gp pointer too.
		 Exception:  Undefined PLABELs should have a value of
		 zero.  */
	      if (h == NULL
		  || (h->elf.root.type != bfd_link_hash_undefweak
		      && h->elf.root.type != bfd_link_hash_undefined))
		{
		  relocation = (off
				+ hplink->splt->output_offset
				+ hplink->splt->output_section->vma
				+ 2);
		}
	      plabel = 1;
	    }
	  /* Fall through and possibly emit a dynamic relocation.  */

	case R_PARISC_DIR17F:
	case R_PARISC_DIR17R:
	case R_PARISC_DIR14F:
	case R_PARISC_DIR14R:
	case R_PARISC_DIR21L:
	case R_PARISC_DPREL14F:
	case R_PARISC_DPREL14R:
	case R_PARISC_DPREL21L:
	case R_PARISC_DIR32:
	  /* The reloc types handled here and this conditional
	     expression must match the code in check_relocs and
	     hppa_discard_copies.  ie. We need exactly the same
	     condition as in check_relocs, with some extra conditions
	     (dynindx test in this case) to cater for relocs removed
	     by hppa_discard_copies.  */
	  if ((input_section->flags & SEC_ALLOC) != 0
	      && info->shared
#if RELATIVE_DYNAMIC_RELOCS
	      && (is_absolute_reloc (r_type)
		  || ((!info->symbolic
		       || (h != NULL
			   && ((h->elf.elf_link_hash_flags
				& ELF_LINK_HASH_DEF_REGULAR) == 0
			       || h->elf.root.type == bfd_link_hash_defweak)))
		      && (h == NULL || h->elf.dynindx != -1)))
#endif
	      )
	    {
	      Elf_Internal_Rela outrel;
	      boolean skip;

	      /* When generating a shared object, these relocations
		 are copied into the output file to be resolved at run
		 time.  */

	      if (sreloc == NULL)
		{
		  const char *name;

		  name = (bfd_elf_string_from_elf_section
			  (input_bfd,
			   elf_elfheader (input_bfd)->e_shstrndx,
			   elf_section_data (input_section)->rel_hdr.sh_name));
		  if (name == NULL)
		    return false;
		  sreloc = bfd_get_section_by_name (dynobj, name);
		  BFD_ASSERT (sreloc != NULL);
		}

	      outrel.r_offset = rel->r_offset;
	      outrel.r_addend = rel->r_addend;
	      skip = false;
	      if (elf_section_data (input_section)->stab_info != NULL)
		{
		  bfd_vma off;

		  off = (_bfd_stab_section_offset
			 (output_bfd, &hplink->root.stab_info,
			  input_section,
			  &elf_section_data (input_section)->stab_info,
			  rel->r_offset));
		  if (off == (bfd_vma) -1)
		    skip = true;
		  outrel.r_offset = off;
		}

	      outrel.r_offset += (input_section->output_offset
				  + input_section->output_section->vma);

	      if (skip)
		{
		  memset (&outrel, 0, sizeof (outrel));
		}
	      else if (h != NULL
		       && h->elf.dynindx != -1
		       && (plabel
			   || !info->symbolic
			   || (h->elf.elf_link_hash_flags
			       & ELF_LINK_HASH_DEF_REGULAR) == 0))
		{
		  outrel.r_info = ELF32_R_INFO (h->elf.dynindx, r_type);
		}
	      else /* It's a local symbol, or one marked to become local.  */
		{
		  int indx = 0;

		  /* Add the absolute offset of the symbol.  */
		  outrel.r_addend += relocation;

		  /* Global plabels need to be processed by the
		     dynamic linker so that functions have at most one
		     fptr.  For this reason, we need to differentiate
		     between global and local plabels, which we do by
		     providing the function symbol for a global plabel
		     reloc, and no symbol for local plabels.  */
		  if (! plabel
		      && sym_sec != NULL
		      && sym_sec->output_section != NULL
		      && ! bfd_is_abs_section (sym_sec))
		    {
		      indx = elf_section_data (sym_sec->output_section)->dynindx;
		      /* We are turning this relocation into one
			 against a section symbol, so subtract out the
			 output section's address but not the offset
			 of the input section in the output section.  */
		      outrel.r_addend -= sym_sec->output_section->vma;
		    }

		  outrel.r_info = ELF32_R_INFO (indx, r_type);
		}

	      bfd_elf32_swap_reloca_out (output_bfd, &outrel,
					 ((Elf32_External_Rela *)
					  sreloc->contents
					  + sreloc->reloc_count));
	      ++sreloc->reloc_count;
	    }
	  break;

	default:
	  break;
	}

      r = final_link_relocate (input_section, contents, rel, relocation,
			       hplink, sym_sec, h);

      if (r == bfd_reloc_ok)
	continue;

      if (h != NULL)
	sym_name = h->elf.root.root.string;
      else
	{
	  sym_name = bfd_elf_string_from_elf_section (input_bfd,
						      symtab_hdr->sh_link,
						      sym->st_name);
	  if (sym_name == NULL)
	    return false;
	  if (*sym_name == '\0')
	    sym_name = bfd_section_name (input_bfd, sym_sec);
	}

      howto = elf_hppa_howto_table + r_type;

      if (r == bfd_reloc_undefined || r == bfd_reloc_notsupported)
	{
	  (*_bfd_error_handler)
	    (_("%s(%s+0x%lx): cannot handle %s for %s"),
	     bfd_get_filename (input_bfd),
	     input_section->name,
	     (long) rel->r_offset,
	     howto->name,
	     sym_name);
	}
      else
	{
	  if (!((*info->callbacks->reloc_overflow)
		(info, sym_name, howto->name, (bfd_vma) 0,
		 input_bfd, input_section, rel->r_offset)))
	    return false;
	}
    }

  return true;
}

/* Finish up dynamic symbol handling.  We set the contents of various
   dynamic sections here.  */

static boolean
elf32_hppa_finish_dynamic_symbol (output_bfd, info, h, sym)
     bfd *output_bfd;
     struct bfd_link_info *info;
     struct elf_link_hash_entry *h;
     Elf_Internal_Sym *sym;
{
  struct elf32_hppa_link_hash_table *hplink;
  bfd *dynobj;

  hplink = hppa_link_hash_table (info);
  dynobj = hplink->root.dynobj;

  if (h->plt.offset != (bfd_vma) -1)
    {
      bfd_vma value;

      /* This symbol has an entry in the procedure linkage table.  Set
	 it up.

	 The format of a plt entry is
	 <funcaddr>
	 <__gp>
      */
      value = 0;
      if (h->root.type == bfd_link_hash_defined
	  || h->root.type == bfd_link_hash_defweak)
	{
	  value = h->root.u.def.value;
	  if (h->root.u.def.section->output_section != NULL)
	    value += (h->root.u.def.section->output_offset
		      + h->root.u.def.section->output_section->vma);
	}

      if (! ((struct elf32_hppa_link_hash_entry *) h)->pic_call)
	{
	  Elf_Internal_Rela rel;

	  /* Create a dynamic IPLT relocation for this entry.  */
	  rel.r_offset = (h->plt.offset
			  + hplink->splt->output_offset
			  + hplink->splt->output_section->vma);
	  if (! ((struct elf32_hppa_link_hash_entry *) h)->plt_abs
	      && h->dynindx != -1)
	    {
	      /* To support lazy linking, the function pointer is
		 initialised to point to a special stub stored at the
		 end of the .plt.  This is only done for plt entries
		 with a non-*ABS* dynamic relocation.  */
	      value = (hplink->splt->output_offset
		       + hplink->splt->output_section->vma
		       + hplink->splt->_raw_size
		       - sizeof (plt_stub)
		       + PLT_STUB_ENTRY);
	      rel.r_info = ELF32_R_INFO (h->dynindx, R_PARISC_IPLT);
	      rel.r_addend = 0;
	    }
	  else
	    {
	      /* This symbol has been marked to become local, and is
		 used by a plabel so must be kept in the .plt.  */
	      rel.r_info = ELF32_R_INFO (0, R_PARISC_IPLT);
	      rel.r_addend = value;
	    }

	  bfd_elf32_swap_reloca_out (hplink->splt->output_section->owner,
				     &rel,
				     ((Elf32_External_Rela *)
				      hplink->srelplt->contents
				      + hplink->srelplt->reloc_count));
	  hplink->srelplt->reloc_count++;
	}

      bfd_put_32 (hplink->splt->owner,
		  value,
		  hplink->splt->contents + h->plt.offset);
      bfd_put_32 (hplink->splt->owner,
		  elf_gp (hplink->splt->output_section->owner),
		  hplink->splt->contents + h->plt.offset + 4);
      if (PLABEL_PLT_ENTRY_SIZE != PLT_ENTRY_SIZE
	  && ((struct elf32_hppa_link_hash_entry *) h)->plabel
	  && h->dynindx != -1)
	{
	  memset (hplink->splt->contents + h->plt.offset + 8,
		  0, PLABEL_PLT_ENTRY_SIZE - PLT_ENTRY_SIZE);
	}

      if ((h->elf_link_hash_flags & ELF_LINK_HASH_DEF_REGULAR) == 0)
	{
	  /* Mark the symbol as undefined, rather than as defined in
	     the .plt section.  Leave the value alone.  */
	  sym->st_shndx = SHN_UNDEF;
	}
    }

  if (h->got.offset != (bfd_vma) -1)
    {
      Elf_Internal_Rela rel;

      /* This symbol has an entry in the global offset table.  Set it
	 up.  */

      rel.r_offset = ((h->got.offset &~ (bfd_vma) 1)
		      + hplink->sgot->output_offset
		      + hplink->sgot->output_section->vma);

      /* If this is a static link, or it is a -Bsymbolic link and the
	 symbol is defined locally or was forced to be local because
	 of a version file, we just want to emit a RELATIVE reloc.
	 The entry in the global offset table will already have been
	 initialized in the relocate_section function.  */
      if (! hplink->root.dynamic_sections_created
	  || (info->shared
	      && (info->symbolic || h->dynindx == -1)
	      && (h->elf_link_hash_flags & ELF_LINK_HASH_DEF_REGULAR)))
	{
	  rel.r_info = ELF32_R_INFO (0, R_PARISC_DIR32);
	  rel.r_addend = (h->root.u.def.value
			  + h->root.u.def.section->output_offset
			  + h->root.u.def.section->output_section->vma);
	}
      else
	{
	  BFD_ASSERT((h->got.offset & 1) == 0);
	  bfd_put_32 (output_bfd, (bfd_vma) 0,
		      hplink->sgot->contents + h->got.offset);
	  rel.r_info = ELF32_R_INFO (h->dynindx, R_PARISC_DIR32);
	  rel.r_addend = 0;
	}

      bfd_elf32_swap_reloca_out (output_bfd, &rel,
				 ((Elf32_External_Rela *)
				  hplink->srelgot->contents
				  + hplink->srelgot->reloc_count));
      ++hplink->srelgot->reloc_count;
    }

  if ((h->elf_link_hash_flags & ELF_LINK_HASH_NEEDS_COPY) != 0)
    {
      asection *s;
      Elf_Internal_Rela rel;

      /* This symbol needs a copy reloc.  Set it up.  */

      BFD_ASSERT (h->dynindx != -1
		  && (h->root.type == bfd_link_hash_defined
		      || h->root.type == bfd_link_hash_defweak));

      s = hplink->srelbss;

      rel.r_offset = (h->root.u.def.value
		      + h->root.u.def.section->output_offset
		      + h->root.u.def.section->output_section->vma);
      rel.r_addend = 0;
      rel.r_info = ELF32_R_INFO (h->dynindx, R_PARISC_COPY);
      bfd_elf32_swap_reloca_out (output_bfd, &rel,
				 ((Elf32_External_Rela *) s->contents
				  + s->reloc_count));
      ++s->reloc_count;
    }

  /* Mark _DYNAMIC and _GLOBAL_OFFSET_TABLE_ as absolute.  */
  if (h->root.root.string[0] == '_'
      && (strcmp (h->root.root.string, "_DYNAMIC") == 0
	  || strcmp (h->root.root.string, "_GLOBAL_OFFSET_TABLE_") == 0))
    {
      sym->st_shndx = SHN_ABS;
    }

  return true;
}

/* Finish up the dynamic sections.  */

static boolean
elf32_hppa_finish_dynamic_sections (output_bfd, info)
     bfd *output_bfd;
     struct bfd_link_info *info;
{
  bfd *dynobj;
  struct elf32_hppa_link_hash_table *hplink;
  asection *sdyn;

  hplink = hppa_link_hash_table (info);
  dynobj = hplink->root.dynobj;

  sdyn = bfd_get_section_by_name (dynobj, ".dynamic");

  if (hplink->root.dynamic_sections_created)
    {
      Elf32_External_Dyn *dyncon, *dynconend;

      BFD_ASSERT (sdyn != NULL);

      dyncon = (Elf32_External_Dyn *) sdyn->contents;
      dynconend = (Elf32_External_Dyn *) (sdyn->contents + sdyn->_raw_size);
      for (; dyncon < dynconend; dyncon++)
	{
	  Elf_Internal_Dyn dyn;
	  asection *s;

	  bfd_elf32_swap_dyn_in (dynobj, dyncon, &dyn);

	  switch (dyn.d_tag)
	    {
	    default:
	      break;

	    case DT_PLTGOT:
	      /* Use PLTGOT to set the GOT register.  */
	      dyn.d_un.d_ptr = elf_gp (output_bfd);
	      bfd_elf32_swap_dyn_out (output_bfd, &dyn, dyncon);
	      break;

	    case DT_JMPREL:
	      s = hplink->srelplt;
	      dyn.d_un.d_ptr = s->output_section->vma + s->output_offset;
	      bfd_elf32_swap_dyn_out (output_bfd, &dyn, dyncon);
	      break;

	    case DT_PLTRELSZ:
	      s = hplink->srelplt;
	      if (s->_cooked_size != 0)
		dyn.d_un.d_val = s->_cooked_size;
	      else
		dyn.d_un.d_val = s->_raw_size;
	      bfd_elf32_swap_dyn_out (output_bfd, &dyn, dyncon);
	      break;

	    case DT_INIT:
	    case DT_FINI:
	      {
		struct elf_link_hash_entry *h;
		const char *funcname;

		if (dyn.d_tag == DT_INIT)
		  funcname = info->init_function;
		else
		  funcname = info->fini_function;

		h = elf_link_hash_lookup (&hplink->root, funcname,
					  false, false, false);

		/* This is a function pointer.  The magic +2 offset
		   signals to $$dyncall that the function pointer
		   is in the .plt and thus has a gp pointer too.  */
		dyn.d_un.d_ptr = (h->plt.offset
				  + hplink->splt->output_offset
				  + hplink->splt->output_section->vma
				  + 2);
		bfd_elf32_swap_dyn_out (output_bfd, &dyn, dyncon);
		break;
	      }
	    }
	}
    }

  if (hplink->sgot->_raw_size != 0)
    {
      /* Fill in the first entry in the global offset table.
	 We use it to point to our dynamic section, if we have one.  */
      bfd_put_32 (output_bfd,
		  (sdyn != NULL
		   ? sdyn->output_section->vma + sdyn->output_offset
		   : (bfd_vma) 0),
		  hplink->sgot->contents);

      /* The second entry is reserved for use by the dynamic linker.  */
      memset (hplink->sgot->contents + GOT_ENTRY_SIZE, 0, GOT_ENTRY_SIZE);

      /* Set .got entry size.  */
      elf_section_data (hplink->sgot->output_section)
	->this_hdr.sh_entsize = GOT_ENTRY_SIZE;
    }

  if (hplink->splt->_raw_size != 0)
    {
      /* Set plt entry size.  */
      elf_section_data (hplink->splt->output_section)
	->this_hdr.sh_entsize = PLT_ENTRY_SIZE;

      if (hplink->need_plt_stub)
	{
	  /* Set up the .plt stub.  */
	  memcpy (hplink->splt->contents
		  + hplink->splt->_raw_size - sizeof (plt_stub),
		  plt_stub, sizeof (plt_stub));

	  if ((hplink->splt->output_offset
	       + hplink->splt->output_section->vma
	       + hplink->splt->_raw_size)
	      != (hplink->sgot->output_offset
		  + hplink->sgot->output_section->vma))
	    {
	      (*_bfd_error_handler)
		(_(".got section not immediately after .plt section"));
	      return false;
	    }
	}
    }

  return true;
}

/* Called when writing out an object file to decide the type of a
   symbol.  */
static int
elf32_hppa_elf_get_symbol_type (elf_sym, type)
     Elf_Internal_Sym *elf_sym;
     int type;
{
  if (ELF_ST_TYPE (elf_sym->st_info) == STT_PARISC_MILLI)
    return STT_PARISC_MILLI;
  else
    return type;
}

/* Misc BFD support code.  */
#define bfd_elf32_bfd_is_local_label_name    elf_hppa_is_local_label_name
#define bfd_elf32_bfd_reloc_type_lookup	     elf_hppa_reloc_type_lookup
#define elf_info_to_howto		     elf_hppa_info_to_howto
#define elf_info_to_howto_rel		     elf_hppa_info_to_howto_rel

/* Stuff for the BFD linker.  */
#define bfd_elf32_bfd_final_link	     _bfd_elf32_gc_common_final_link
#define bfd_elf32_bfd_link_hash_table_create elf32_hppa_link_hash_table_create
#define elf_backend_add_symbol_hook	     elf32_hppa_add_symbol_hook
#define elf_backend_adjust_dynamic_symbol    elf32_hppa_adjust_dynamic_symbol
#define elf_backend_check_relocs	     elf32_hppa_check_relocs
#define elf_backend_create_dynamic_sections  elf32_hppa_create_dynamic_sections
#define elf_backend_fake_sections	     elf_hppa_fake_sections
#define elf_backend_relocate_section	     elf32_hppa_relocate_section
#define elf_backend_hide_symbol		     elf32_hppa_hide_symbol
#define elf_backend_finish_dynamic_symbol    elf32_hppa_finish_dynamic_symbol
#define elf_backend_finish_dynamic_sections  elf32_hppa_finish_dynamic_sections
#define elf_backend_size_dynamic_sections    elf32_hppa_size_dynamic_sections
#define elf_backend_gc_mark_hook	     elf32_hppa_gc_mark_hook
#define elf_backend_gc_sweep_hook	     elf32_hppa_gc_sweep_hook
#define elf_backend_object_p		     elf32_hppa_object_p
#define elf_backend_final_write_processing   elf_hppa_final_write_processing
#define elf_backend_get_symbol_type	     elf32_hppa_elf_get_symbol_type

#define elf_backend_can_gc_sections	     1
#define elf_backend_plt_alignment	     2
#define elf_backend_want_got_plt	     0
#define elf_backend_plt_readonly	     0
#define elf_backend_want_plt_sym	     0
#define elf_backend_got_header_size	     8

#define TARGET_BIG_SYM		bfd_elf32_hppa_vec
#define TARGET_BIG_NAME		"elf32-hppa"
#define ELF_ARCH		bfd_arch_hppa
#define ELF_MACHINE_CODE	EM_PARISC
#define ELF_MAXPAGESIZE		0x1000

#include "elf32-target.h"
