#include <lct/local.h>

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/stat.h>

#include <lct/generic.h>
#include <lct/font.h>

/* defined later */
static void position_codepage(int requested_height, FILE *fpi);

/*@null@*/
simple_font* read_simple_font (FILE* fontfile, int font_format)
{
  struct stat stbuf;
  off_t filesize;
  int has_internal_sfm = 0;
  simple_font* the_font;

  assert (INITIAL_CHARSLOTS >= 256);
  
  /*
   * check arguments
   */

  /* fontfile is required */
  if (fontfile == NULL)
    {
      errno = EINVAL;
      return NULL;
    }

  /*
   * init
   */
  
  the_font = (simple_font*)calloc (1, sizeof(simple_font));
  if (!the_font) return NULL;

  the_font->font.chardata = (char*)malloc (FONTDATA_CHARPADHEIGHT * INITIAL_CHARSLOTS);
  if (!the_font->font.chardata) return NULL;

  /*
   * get filesize
   */
  
  /* FIXME: should not use _fileno ! */
  if (fstat(fontfile->_fileno, &stbuf) == -1)
    goto rsf_return_error;
  
  if (S_ISREG(stbuf.st_mode))
    filesize = stbuf.st_size;
  else
    {
      /* reading from pipe */
      errno = 0;
      filesize = -1;		       /* cannot know, so say we don't know */
    }

  fprintf (stderr, "Format: %u\n", font_format);
  
  /*
   * Position `fontfile' on fontdata for loading;
   * set relevant parameters accoring to font_format
   */
  switch (font_format)
    {
    case FFF_PSF:
      {
	struct psf_header psfhdr;

	if (psf_header_read (fontfile, &psfhdr) != 0)
	  goto rsf_return_error;

	if (PSF_MODE_HAS512(psfhdr.mode))
	  {
#if (INITIAL_CHARSLOTS < 512)
	    if (NULL == (the_font->font.chardata =
			 realloc (the_font->font.chardata, FONTDATA_CHARPADHEIGHT * 512)))
	      return NULL; /* FIXME: memory leak if realloc failed */
#endif
	    the_font->font.charcount = 512;
	  }
	else
	  the_font->font.charcount = 256;
	
	has_internal_sfm = (PSF_MODE_HASSFM(psfhdr.mode));
	the_font->font.charheight = psfhdr.charheight;
	the_font->font.charwidth = 8;

	fontdata_read_binary (fontfile, &(the_font->font) );

	break;
      }


    case FFF_BDF:
      if (bdf_read (fontfile, the_font) != 0)
	goto rsf_return_error;
      
      break;
      

    case FFF_RAW:
#if 1	/* RAW filesize check */
      /* raw font */
      if ((filesize > 0) && (filesize & 0377)) 
	{
	  errno = EBFONT;
	  goto rsf_return_error;
	}
#endif
      if (filesize > 0)
	{
	  the_font->font.charheight = filesize/256;
	  the_font->font.charwidth = 8;
	  the_font->font.charcount = 256;
	}
      else
	{
	  fprintf (stderr, _("Cannot (yet) load a non-seekable RAW file\n"));
	  errno = EINVAL;
	  goto rsf_return_error;
	}


      fontdata_read_binary (fontfile, &(the_font->font) );
      
      /*
       * sanity check for file-size on old font-formats
       */
      if (font_format == FFF_RAW)
	{
	  long old_pos = ftell (fontfile);
      
	  xfseek (fontfile, 0, SEEK_END);
      
	  if (old_pos != ftell (fontfile))
	    {
	      fprintf (stderr, _("Bad font input font-file length. Aborting.\n"));
	      exit (1);
	    }
	}

      break;
	

    default:
      fprintf (stderr, _("I can't handle this font-file format yet.\n"));
      errno = EINVAL;
      goto rsf_return_error;
    }


  /* read provided SFM if there is one */
  if ((font_format == FFF_PSF) && has_internal_sfm)
    {
      if (-1 == sfm_read_binary (fontfile, &(the_font->sfm), the_font->font.charcount))
	goto rsf_return_error;

#ifndef NDEBUG
      /* if (verbose)*/
      fprintf(stderr, _("Reading SFM from font file.\n"));
#endif
    }

  
  return the_font; /* success */

rsf_return_error:
  {
    int saved_errno = errno;

    free_simple_font (the_font);

    errno = saved_errno;
  }
  return NULL;
}


void free_simple_font (simple_font* font)
{
  if (font->font.chardata)
    free (font->font.chardata);
  free (font);
}

font_group* read_font_group (FILE* fontfile, int font_format)
{
  struct stat stbuf;
  off_t filesize;
  font_group* the_fonts;

  /*
   * check arguments
   */
  
  /* fontfile is required */
  if (fontfile == NULL)
    {
      errno = EINVAL;
      return NULL;
    }

  /*
   * init
   */
  
  the_fonts = (font_group*)calloc (1, sizeof(font_group));
  if (!the_fonts) return NULL;
  the_fonts->fonts = (cfontdesc*)malloc (INITIAL_FONTSLOTS * sizeof (cfontdesc));
  the_fonts->allocated_fontslots = INITIAL_FONTSLOTS;

  /*
   * get filesize
   */
  
  /* FIXME: should not use _fileno ! */
  if (fstat(fontfile->_fileno, &stbuf) == -1)
    goto rfg_return_error;
  
  if (S_ISREG(stbuf.st_mode))
    filesize = stbuf.st_size;
  else
    {
      /* reading from pipe */
      errno = 0;
      filesize = -1;		       /* cannot know, so say we don't know */
    }

  switch (font_format)
    {
    case FFF_CP:
      /* CP file with 3 256-chars code pages (8, 14, 16 scanlines) ? */
#if 1
      /* file has no magic */
      if (filesize == -1)
	{
	  fprintf(stderr,
		  _("setfont: read from pipe - "
		    "cannot test for CP or RAW format - aborting.\n"));
	  exit(1);
	}
      
      /* filesize != -1 : we know we can rewind */
      rewind(fontfile);
#endif

      {
	int nb_cp_heights = 3;
	int cp_heights[3] = { 16, 14, 8 };
	int i;

	for (i=0; i<nb_cp_heights; i++)
	  {
	    position_codepage(cp_heights[i], fontfile);
	    the_fonts->fonts[i].charheight = cp_heights[i];
	    the_fonts->fonts[i].charwidth = 8;
	    the_fonts->fonts[i].charcount = 256;

	      /* allocate ( unit_max = 32 ) * (fontsize_max = 512)
	       * FIXME: could be scaled down to (32 * fontsize) ?
	       * could that break PIO_FONT 
	       * maybe 32 * min (fontsize, 256) ? */
	    the_fonts->fonts[i].chardata = (char*)malloc (16384);
	    
	    if (fontdata_read_binary (fontfile, &(the_fonts->fonts[i])) != 0)
	      goto rfg_return_error;
	    
	    the_fonts->fontcount = i+1;
	  }
      }
      
      break;
      
    default:
      fprintf (stderr, _("I can't handle this font-file format yet.\n"));
      errno = EINVAL;
      goto rfg_return_error;
    }

  return the_fonts; /* success */

rfg_return_error:
  {
    int saved_errno = errno;

    free_font_group (the_fonts);
    
    errno = saved_errno;
  }
  return NULL;
}


void free_font_group (font_group* group)
{
  int i;

  if (group->fonts)
    {
      for (i=0; i<group->allocated_fontslots; i++)
	if (group->fonts[i].chardata)
	  free (group->fonts[i].chardata);
      free (group->fonts);
    }
  free (group);
}


static void position_codepage(int requested_height, FILE *fpi) 
{
  int offset;

  /* code page: first 40 bytes, then 8x16 font,
   * then 6 bytes, then 8x14 font,
   * then 6 bytes, then 8x8 font
   */

  /* if we're there we already read sizeof(psf_header) bytes. */
  
  if (!requested_height) 
    {
      fprintf(stderr,
	      _("This file contains 3 fonts: 8x8, 8x14 and 8x16. Please indicate\n"
		"using a --char-height option of 8, 14, or 16 which one you want to load.\n"));
      exit(1);
    }
  switch (requested_height) 
    {
    case 8:
      offset = 7732; break;
    case 14:
      offset = 4142; break;
    case 16:
      offset = 40; break;
    default:
      fprintf(stderr, 
	      _("You asked for font height %d, but only 8, 14, 16 are possible here.\n"),
	      requested_height);
      exit(1);
    }
  
  /* use SEEK_CUR to allow for pipes */
  if (xfseek(fpi, offset, SEEK_SET))
    {
      perror(_("seek error on input file"));
      exit(1);
    }
}
