/*
 * (c) Copyright 1992 by Panagiotis Tsirigotis
 * (c) Sections Copyright 1998-2001 by Rob Braun
 * All rights reserved.  The file named COPYRIGHT specifies the terms 
 * and conditions for redistribution.
 */

static char RCSid[] = "$Id: access.c,v 1.3 1999/10/13 02:51:03 bbraun Exp $" ;

#include <syslog.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include "config.h"

#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif

#ifdef LIBWRAP
#include <tcpd.h>
int deny_severity = LOG_INFO;
int allow_severity = LOG_INFO;
#endif

#include "connection.h"
#include "service.h"
#include "server.h"
#include "state.h"
#include "addr.h"
#include "access.h"
#include "sconst.h"
#include "str.h"
#include "servicedefs.h"

#if !defined(NAME_MAX)
      #ifdef FILENAME_MAX
      #define NAME_MAX FILENAME_MAX
      #else
      #define NAME_MAX 256
      #endif
#endif

/*
 * This is the globals for the Sensor. The Sensor will add the incoming IP
 * address to the global_no_access table for whatever the configured time is.
 */
#define MAX_GLOBAL_NO_ACCESS      10240
pset_h global_no_access = NULL;      /* global no_access listn   */
pset_h global_no_access_time = NULL;   /* time of the infraction   */


bool_int ti_current_time_check( pset_h );

#ifdef HAVE_LOADAVG
extern double xgetloadavg() ;
#endif

struct name_value access_code_names[] =
   {
      { "address",                  (int) AC_ADDRESS         },
      { "time",                     (int) AC_TIME            },
      { "fork",                     (int) AC_FORK            },
      { "service_limit",            (int) AC_SERVICE_LIMIT   },
      { "per_source_limit",         (int) AC_PER_SOURCE_LIMIT},
      { "process_limit",            (int) AC_PROCESS_LIMIT   },
      { "libwrap",                  (int) AC_LIBWRAP         },
      { "load",                     (int) AC_LOAD            },
      { "connections per second",   (int) AC_CPS             },
      { CHAR_NULL,                  1                        },
      { "UNKNOWN",                  0                        }
   } ;

/* alarm handler for re-enabling disabled services (for cps) */
PRIVATE void alrm_ena(int signo)
{
   int i;
   time_t nowtime;

   nowtime = time(NULL);
   for( i=0; i < pset_count( SERVICES(ps) ); i++ ) {
      struct service *sp;
      struct service_config *scp;

      sp = pset_pointer( SERVICES(ps), i);
      scp = SVC_CONF( sp );

      if( sp->svc_state == SVC_DISABLED ) {
         if ( scp->sc_time_reenable <= nowtime ) {
            /* re-enable the service */
            if( svc_activate(sp) == OK ) {
               syslog(LOG_ERR, 
               "Activating service %s", scp->sc_name);
            } else {
               syslog(LOG_ERR, 
               "Error activating service %s", 
               scp->sc_name) ;
            } /* else */
         }
      }
   } /* for */
}

/*
 * Returns OK if the IP address in sinp is acceptable to the access control
 * lists of the specified service.
 */
#ifdef INET6
PRIVATE status_e remote_address_check(struct service *sp, struct sockaddr_in6 *sinp)
#else
PRIVATE status_e remote_address_check(struct service *sp, struct sockaddr_in *sinp)
#endif
{
   /*
    * of means only_from, na means no_access
    */
   char *func = "remote_address_check";
   bool_int   of_matched  = FALSE;
   bool_int   na_matched  = FALSE;
#ifdef INET6
   struct in6_addr      of_match ;
   struct in6_addr      na_match ;
   struct in6_addr      *addr ;

   bzero(&of_match, sizeof(of_match));
   bzero(&na_match, sizeof(na_match));
   addr = &sinp->sin6_addr ;
#else
   uint32_t           of_match ;
   uint32_t           na_match ;
   struct in_addr      *addr ;
   addr = &sinp->sin_addr ;
#endif
   if ( SC_SENSOR(sp->svc_conf) )
   {   /* They hit a sensor...   */
      if (sp->svc_conf->sc_deny_time != 0)   /* 0 simply logs it   */
      {
         if ( pset_count( global_no_access ) < MAX_GLOBAL_NO_ACCESS)
         {
            int item_matched = addrlist_match( global_no_access, addr, &na_match );
            if ( item_matched == 0)
            {   /* no match...adding to the list   */
               char *dup_addr = new_string(xntoa(SA( sinp )) );
               if (addrlist_add(global_no_access, dup_addr) == FAILED)
                  msg(LOG_ERR, func,
                  "Failed adding %s to the global_no_access list", dup_addr);
               else
               {
                  time_t nowtime;
                  char time_buf[40];

                  msg(LOG_CRIT, func,
                     "Adding %s to the global_no_access list for %d minutes",
                  dup_addr, sp->svc_conf->sc_deny_time);
                  nowtime = time(NULL);

                  if (sp->svc_conf->sc_deny_time == -1)
                     strcpy(time_buf, "-1");
                  else
                     strx_nprint(time_buf, 38, "%ld", 
                        (time_t)nowtime+(60*sp->svc_conf->sc_deny_time));
                  if (pset_add(global_no_access_time, new_string(time_buf)) == FAILED)
                  {
                     msg(LOG_ERR, func,
                        "Failed adding %s to the global_no_access_time list. global_no_access list is broken, xinetd needs restarting.", dup_addr);
                     /* ideally, we should rollback the previous addr addition.   */
                  }
               }
               if (dup_addr)
                  free(dup_addr);
            }
            else
            {   
               /* Here again, eh?...update time stamp. Note: We are in 
                * parent's context.   
                */
               char *exp_time;
               time_t stored_time;

               exp_time = pset_pointer( global_no_access_time, item_matched ) ;
               stored_time = atol(exp_time);
               if (stored_time != -1)   /* if never let them off, bypass */
               {
                  time_t nowtime, new_time;

                  nowtime = time(NULL);
                  new_time = (time_t)nowtime+(60*sp->svc_conf->sc_deny_time);
                  if (difftime(new_time, (time_t)stored_time) > 0.0)
                  {   /* new_time is longer save it   */
                     char time_buf[40];

                     strx_nprint(time_buf, 38, "%ld", new_time);
                     free(exp_time);
                     exp_time = new_string(time_buf);
                  }
               }
            }
         }
         else
         {
            msg(LOG_ERR, func, "Maximum global_no_access count reached.");
         }
      }
      return ( FAILED );
   }
   else if ( (global_no_access) && pset_count( global_no_access ) )
   {   
      /* They hit a real server...note, this is likely to be a child process. */
      if (addrlist_match( global_no_access, addr, &na_match ))
         return ( FAILED );
   }

   if ( sp->svc_no_access != NULL )
      na_matched = addrlist_match( sp->svc_no_access, addr, &na_match ) ;

   if ( sp->svc_only_from != NULL )
      of_matched = addrlist_match( sp->svc_only_from, addr, &of_match ) ;

   /*
    * Check if the specified address is in both lists
    */
   if ( na_matched && of_matched )
   {
      /*
       * The greater match wins.
       * If the matches are equal, this is an error in the service entry
       * and we cannot allow a server to start.
       * We do not disable the service entry (not our job).
       */
#ifdef INET6
      int ret = memcmp(&na_match, &of_match, sizeof(na_match));

      if( ret == 0 )
         msg( LOG_ERR, func,
         "Service=%s: only_from list and no_access list match equally the address %s",
            SVC_ID( sp ), xntoa( SA(sinp) ) ) ;
      else if ( ret < 0 )
         return OK;
      else
         return FAILED;
#else
      if ( na_match == of_match )
         msg( LOG_ERR, func,
"Service=%s: only_from list and no_access list match equally the address %s",
            SVC_ID( sp ), 
            xntoa( SA(sinp) ) ) ;
      return( ( of_match > na_match ) ? OK : FAILED ) ;
#endif
   }

   if ( sp->svc_no_access != NULL && na_matched )
      return( FAILED ) ;
   if ( sp->svc_only_from != NULL && ! of_matched )
      return( FAILED ) ;

   /*
    * If no lists were specified, the default is to allow starting a server
    */
   return( OK ) ;
}

void scrub_global_access_lists( connection_s *cp )
{
   int count;

   if( global_no_access == NULL )
      count = 0;
   else
      count = pset_count( global_no_access );
   if (count )
   {   
      int found_one = 0;
      unsigned u;
      time_t nowtime = time(NULL);

      for (u=0; u < count; u++)
      {
         char *exp_time;
         time_t stored_time;
         exp_time = pset_pointer( global_no_access_time, u ) ;
         stored_time = atol(exp_time);
         if (stored_time == -1)   /* never let them off   */
            continue;
         if (difftime(nowtime, (time_t)stored_time) > 0.0)
         {
            __pset_pointer ptr;

            pset_pointer(global_no_access, u) = NULL;

            ptr = global_no_access_time->ptrs[ u ];
            free(ptr);
            pset_pointer(global_no_access_time, u ) = NULL;
            found_one = 1;
         }
      }
      if (found_one)
      {
         pset_compact( global_no_access );
         pset_compact( global_no_access_time );
      }
   }
}

/*
 * mp is the mask pointer, t is the check type
 */
#define CHECK( mp, t )      ( ( (mp) == NULL ) || M_IS_SET( *(mp), t ) )

/*
 * Perform the access controls specified by check_mask.
 * If check_mask is NULL, perform all access controls
 */
access_e access_control( struct service *sp, 
                         connection_s *cp, 
                         mask_t *check_mask )
{
   struct service_config   *scp = SVC_CONF( sp ) ;
#ifdef INET6
   struct sockaddr_in6     *sinp = SOCKADDRIN_NULL;
#else
   struct sockaddr_in      *sinp = SOCKADDRIN_NULL;
#endif

   /* make sure it's not one of the special pseudo services */
   if( (strncmp(SC_NAME( scp ), INTERCEPT_SERVICE_NAME, sizeof(INTERCEPT_SERVICE_NAME)) == 0) || (strncmp(SC_NAME( scp ), LOG_SERVICE_NAME, sizeof(LOG_SERVICE_NAME)) == 0) || (strncmp(SC_NAME( scp ), SHUTDOWN_SERVICE_NAME, sizeof(SHUTDOWN_SERVICE_NAME)) == 0) ) {
      return (AC_OK);
   }

   /* This has to be before the TCP_WRAPPERS stuff to make sure that
      the sensor gets a chance to see the address */
   if ( CHECK( check_mask, CF_ADDRESS ) &&
         ( sinp = conn_address( cp ) ) != SOCKADDRIN_NULL &&
         remote_address_check( sp, sinp ) == FAILED )
      return( AC_ADDRESS ) ;

   if( ! SC_NOLIBWRAP( scp ) ) 
   { /* LIBWRAP code block */
#ifdef LIBWRAP
   struct request_info req;
   char *server = NULL;

   /* get the server name to pass to libwrap */
   if( SC_NAMEINARGS( scp ) )
      server = strrchr( scp->sc_server_argv[0], '/' );
   else {
      if( scp->sc_server == NULL ) {
         /* probably an internal server, use the service id instead */
         server = scp->sc_id;
         server--;  /* nasty.  we increment it later... */
      } else {
         server = strrchr( scp->sc_server, '/' );
      }
   }

   /* If this is a redirection, go by the service name,
    * since the server name will be bogus.
    */
   if( scp->sc_redir_addr != NULL ) {
      server = scp->sc_name;
      server--; /* nasty but ok. */
   }

   if( server == NULL )
      server = scp->sc_server_argv[0];
   else
      server++;

   request_init(&req, RQ_DAEMON, server, RQ_FILE, cp->co_descriptor, 0);
   fromhost(&req);
   if (!hosts_access(&req)) {
      syslog(deny_severity, "libwrap refused connection from %s", conn_addrstr(cp));
      return(AC_LIBWRAP);
   }
#endif
   } /* LIBWRAP code block */

   return( AC_OK ) ;
}


/* Do the "light weight" access control here */
access_e parent_access_control( struct service *sp, connection_s *cp )
{
   struct service_config *scp = SVC_CONF( sp ) ;
   int u, n;
   time_t nowtime;
#ifdef INET6
   struct sockaddr_in6   *sinp = SOCKADDRIN_NULL;
#else
   struct sockaddr_in    *sinp = SOCKADDRIN_NULL;
#endif
   /* make sure it's not one of the special pseudo services */
   if( (strncmp(SC_NAME( scp ), INTERCEPT_SERVICE_NAME, sizeof(INTERCEPT_SERVICE_NAME)) == 0) || (strncmp(SC_NAME( scp ), LOG_SERVICE_NAME, sizeof(LOG_SERVICE_NAME)) == 0) || (strncmp(SC_NAME( scp ), SHUTDOWN_SERVICE_NAME, sizeof(SHUTDOWN_SERVICE_NAME)) == 0) ) 
      return (AC_OK);

   /* CPS handler */
   if( scp->sc_time_conn_max != 0 ) {
      nowtime = time(NULL);
      if( ((scp->sc_time_limit - nowtime) > 2) || 
         ( (scp->sc_time_limit - nowtime) < 0)) {

         scp->sc_time_limit = nowtime;
         scp->sc_time_conn = 1;
      } else {
         scp->sc_time_conn ++;
         if( scp->sc_time_conn > scp->sc_time_conn_max ) {
            svc_deactivate(sp);
            /* Need to implement a restart timer */
            syslog(LOG_ERR, "Deactivating service %s due to excessive incoming connections.  Restarting in %d seconds.", scp->sc_name, (int)scp->sc_time_wait);
            if( signal( SIGALRM, alrm_ena) == SIG_ERR ) {
               syslog(LOG_ERR, "Error setting up timer for reactivating service %s", scp->sc_name);
               return(AC_CPS);
            }
            nowtime = time(NULL);
            scp->sc_time_reenable = nowtime + scp->sc_time_wait;
            alarm(scp->sc_time_wait);
            return(AC_CPS);
         }
      }
   }

#ifdef HAVE_LOADAVG
   if ( scp->sc_max_load != 0 ) {
      if ( xgetloadavg() >= scp->sc_max_load ) {
         syslog(LOG_ERR, 
            "refused connect from %s due to excessive load", 
            conn_addrstr(cp));
         return( AC_LOAD );
      }
   }
#endif

   if ( SC_ACCESS_TIMES( scp ) != NULL && 
         ! ti_current_time_check( SC_ACCESS_TIMES( scp ) ) )
      return( AC_TIME ) ;

   if ( SVC_RUNNING_SERVERS( sp ) >= SC_INSTANCES( scp ) )
      return( AC_SERVICE_LIMIT ) ;

   if( scp->sc_per_source != UNLIMITED ) {
      if ( sinp == SOCKADDRIN_NULL ) sinp = conn_address ( cp ) ;
      if ( sinp != SOCKADDRIN_NULL ) {
         n = 0 ;
         for ( u = 0 ; u < pset_count( SERVERS( ps ) ) ; u++ ) {
            struct server *serp ;
            connection_s *cop ;
            serp = SERP( pset_pointer( SERVERS( ps ), u ) ) ;
            if ( (SERVER_SERVICE( serp ) == sp) &&
               ( cop = SERVER_CONNECTION( serp ) ) ) {

#ifdef INET6
               if ( IN6_ARE_ADDR_EQUAL( &(cop->co_remote_address.sin6_addr) , &(sinp->sin6_addr) ) )
#else
               if ( cop->co_remote_address.sin_addr.s_addr == sinp->sin_addr.s_addr)
#endif
                  n++;
            }
         }

         if ( n >= scp->sc_per_source )
            return( AC_PER_SOURCE_LIMIT ) ;
      }
   }

   if ( ps.ros.process_limit ) {
      unsigned processes_to_create = SC_IS_INTERCEPTED( scp ) ? 2 : 1 ;

      if ( pset_count( SERVERS( ps ) ) + processes_to_create > 
         ps.ros.process_limit ) {
         return( AC_PROCESS_LIMIT ) ;
      }
   }

   return (AC_OK);
}
