/* get_user_sid.c - contains implementation of functions to get a SID for
 *                  a user.  Used by login program.
 */

#include "get_user_sid.h"

/*
 * get_default_user_sid - given username - the login id of the user, sets user_sid
 *                to the SID of the user
 *                Looks up the default security context.
 *                Returns 1 if a SID was found, 0 otherwise
 */
int get_default_user_sid (const char* username, const int len, 
                  security_context_t* user_context, security_id_t* user_sid)
{
  FILE* fseccon;                 /* default security context file */

  *user_context=NULL;

  /* Open file */
  if (!(fseccon = fopen (_SECCONTEXT_PATH, "r"))) {
      fprintf (stderr, "Couldn't open default security context file %s\n", 
	       _SECCONTEXT_PATH);
      return 0;
    }

  if (!find_sec_context (fseccon, username, len, user_context)) {
      /* Obtain default security context - set to empty string if not found */
      fprintf (stderr,
	       "Unable to find default security context for user %s\n", 
	       username);
      fclose (fseccon);

      return 0;
    }
  fclose (fseccon);

  if (*user_context==NULL || security_context_to_sid(*user_context,strlen(*user_context)+1,user_sid) ) {
    fprintf (stderr, "Unable to obtain sid for %s\n", *user_context!=NULL?*user_context:" a null context" );
    return 0;
  }

 return 1;
}


/*
 * get_user_sid - given username - the login id of the user, sets user_sid
 *                to the SID of the user
 *                Looks up the default security context and allows user to
 *                enter a different security context.  Then queries the 
 *                security server to obtain the sid matching the requested
 *                security context.
 *                Returns 1 if a SID was found, 0 otherwise
 */
int get_user_sid (const char* username, const int len, 
                  security_context_t* user_context, security_id_t* user_sid)
{
    FILE* fseccon;                 /* default security context file */
    security_context_t defsec_context = 0;  /* default security context */
 
    
    /* Open file */
    if (!(fseccon = fopen (_SECCONTEXT_PATH, "r")))
    {
        fprintf (stderr, "Couldn't open default security context file %s\n", 
                            _SECCONTEXT_PATH);
        defsec_context = (security_context_t) malloc (sizeof(char));
        if (defsec_context == NULL)
        {
            fprintf (stderr, "Unable to malloc\n");
            return 0;
        }
        defsec_context[0] = 0;
    }
    else if (!find_sec_context (fseccon, username, len, &defsec_context))
    {
        /* Obtain default security context - set to empty string if not found */
        fprintf (stderr,
                "Unable to find default security context for user %s\n", 
                username);
        defsec_context = (security_context_t) malloc (sizeof(char));
        if (defsec_context == NULL)
        {
            fprintf (stderr, "Unable to malloc\n");
            return 0;
        }
        defsec_context[0] = 0;
    }

    /* Get a SID for the user - may be based on default security context
       or the user may request a new security context                    */
    if (!obtain_user_sid (username, defsec_context, user_context, user_sid))
    {
        if (defsec_context != NULL)
            free (defsec_context);
        if (fseccon != NULL)
            fclose (fseccon);
        return 0;
    }

    if (defsec_context != NULL)
        free (defsec_context);
    if (fseccon != NULL)
        fclose (fseccon);
    return 1;
}


/* find_sec_context - given a pointer to the default security context file
 *                    and a username, finds the line of the file that 
 *                    corresponds to the user.  That line without any 
 *                    leading blanks is set to be the default security
 *                    context.
 *                    Returns 1 if a line matching the user is found, 0
 *                    otherwise
 */
int find_sec_context (FILE *fseccon, const char* username, const int len, 
                      security_context_t* def)
{
    char buf[250];    /* contains the line read from the file       */
    int i;            /* index into buf                             */
    int found = 0;    /* true if the line corresponding to the user 
                         has been found                             */

    /* Make sure to start at the beginning of the file */
    rewind (fseccon);

    /* Search line by line until the username is found or eof is reached */
    while ((!found) && (!feof (fseccon)))
    {
        fgets (buf, 250, fseccon);
        i = 0;

        /* Make sure there are no leading spaces in the buffer */
        while (buf[i] == ' ')
            i++;
    
        if ((strncmp (username, &buf[i], len) == 0) && (buf[i+len] == ':'))
            found = 1;  /* This line goes with username */
    }

    if (found)  /* A line matching username has been found */
    {
        /* malloc space for the security context */
        (*def) = (security_context_t) malloc (sizeof(char) * (1+strlen(buf)-i));
        if ((*def) == NULL)
        {
            fprintf (stderr, "Unable to malloc\n");
            return 0;
        }

        /* copy the line into the space pointed to by default */
        strcpy ((*def), &buf[i]);
        if ((*def)[strlen(*def)-1] == '\n')
            (*def)[strlen(*def)-1] = '\0';
    }

    return found;
}


/* 
 * obtain_user_sid - given a default security context, query the user to
 *                   see if they want to enter a different security context
 *                   and set user_sid to the SID corresponding to the 
 *                   security context selected by the user
 *                   Returns 1 if a SID is found, 0 otherwise
 */
int obtain_user_sid (const char* username, 
                     const security_context_t defsec_context, 
                     security_context_t* user_context,
                     security_id_t* user_sid)
{
    int count = 0;        /* number of times a user has entered a security 
                             context */
    char response[100];   /* used to read the user's response               */
    int trydef = 0;       /* 1 if default security context has been tried   */
    security_context_t trysec_context = 0;  /* the security context entered
                                           by the user                      */
    int done = 0;         /* 1 if a valid SID has been found                */
    int retval;           /* the value returned for security_context_to_sid */


    /* Make sure username is non-null */
    if (!username)
    {
        /* obtain_user_sid called with no user name */
        return 0;
    }

    /* Make sure user_context and user_sid are not NULL pointers */
    if ((!user_context) || (!user_sid))
    {
        /* output parameter is 0 pointer */
        return 0;
    }

    /* Check to see if there is a default security context and make sure
       the user part of it matches username                               */
    if ((!defsec_context) || (strlen(defsec_context) <=0 ))
    {
        printf ("\nYou do not have a default security context\n");
        trydef = 1;
    }
    else if ((strncmp (username, defsec_context, strlen(username)) != 0)
           || (defsec_context[strlen(username)] != ':'))
    {
        printf ("\nusername does not match name in security context\n");
        trydef = 1;
    }
    else
        printf ("\nYour default security context is %s\n", defsec_context);

    /* Allow user to enter a new security context */
    do
    {
        count ++;  /* Increment number of tries */

        printf ("Do you want to enter a new security context?[n] ");
        fflush (stdin);
        fgets (response, 100, stdin);
        fflush (stdin);

        if ((response[0] == 'y') || (response[0] == 'Y'))
        {
            /* Get security context from user */
            query_user_sec_context (username, defsec_context, &trysec_context);

            /* Obtain sid for security context */
            retval = security_context_to_sid(trysec_context, 
                                       strlen(trysec_context)+1, user_sid);
        
            /* See if a valid SID was found */
            if (retval != -1)
            {
                (*user_context)=(security_context_t)malloc
                                                    (strlen(trysec_context) +1);
                if ((*user_context) == NULL)
                {
                    fprintf (stderr, "Unable to malloc\n");
                    return 0;
                }
                strcpy ((*user_context), trysec_context);
                done = 1;
            }
            else
            {
                fprintf (stderr, "Unable to obtain SID for %s\n",
                        trysec_context);
            }

            if (trysec_context != NULL)
                free (trysec_context);
        }
        else if (!trydef && (strlen (defsec_context) >0))
        {
            /* Obtain sid for default security context */
            trydef = 1;
            retval = security_context_to_sid(defsec_context, 
                                       strlen(defsec_context)+1, user_sid);
            if (retval != -1)
            {
                (*user_context)=(security_context_t)malloc
                                                    (strlen(defsec_context)+1);
                if ((*user_context) == NULL)
                {
                    fprintf (stderr, "Unable to malloc\n");
                    return 0;
                }
                strcpy ((*user_context), defsec_context);
                done = 1;
            }
            else
            {
                fprintf(stderr,"Unable to obtain SID for default context %s\n",
                        defsec_context);
            }
        }
        else
        {
            /* Unable to obtain SID */
            return 0;
        }
       
        if (!done)
            printf ("That security context is invalid.\n");
    }
    while ((!done) && (count < MAX_SECURITY_CONTEXT_TRIES));

    return done;
}


/* query_user_sec_context - given a default security context, allow
 *                          the user to enter a new security context.
 *                          Store the user's choice (even if it's the
 *                          default) in trysec_context
 */
void query_user_sec_context (const char* username, 
                             const security_context_t defsec_context, 
                             security_context_t* trysec_context)
{
    char* defuser = 0;         /* the default user name - should match username */
    char* defrole = 0;         /* the default role                         */
    char* deftype = 0;         /* the default type                         */
    char* deflevel = 0;        /* the default mls level                    */
    char newrole[100];     /* role selected by user                    */
    char newtype[100];     /* type selected by user                    */
    char newlevel[100];    /* level  selected by user                  */


    /* Make sure we have a valid username */
    if (username)
    {
        /* Try to separate the security context into its individual fields */
        if (!security_context_to_parts (defsec_context,&defuser,&defrole,
                                        &deftype,&deflevel))
        {
            /* Bad default security context */
            if (defuser != NULL)
            {
                free (defuser);
                defuser = 0;
            }
            if (defrole != NULL)
            {
                free (defrole);
                defrole = 0;
            }
            if (deftype != NULL)
            {
                free (deftype);
                deftype = 0;
            }
            if (deflevel != NULL)
            {
                free (deflevel);
                deflevel = 0;
            }
        }
        else if (strcmp (username, defuser))
        {
            /* Security context not for this user, but we will allow user
               to enter new fields */
            if (defuser != NULL)
            {
                free (defuser);
                defuser = 0;
            }
            if (defrole != NULL)
            {
                free (defrole);
                defrole = 0;
            }
            if (deftype != NULL)
            {
                free (deftype);
                deftype = 0;
            }
            if (deflevel != NULL)
            {
                free (deflevel);
                deflevel = 0;
            }
        }

        /* Allow the user a chance to enter each field */
        printf ("\n");
        get_field ("role", defrole, newrole);

	if( strcmp(defrole,newrole) ) {
	  char *newdef;

	  if( !get_default_type(newrole,strlen(newrole),&newdef) ) {
	    free(deftype);
	    deftype=newdef;
	  }
	}

        get_field ("type", deftype, newtype);

#ifdef CONFIG_FLASK_MLS
        get_field ("level", deflevel, newlevel);
#endif
 
        /* Combine the the new fields into a security context */
        if (!parts_to_security_context (username, newrole, newtype, newlevel,
                                        trysec_context))
        {
            /* Unable to create new security context */
            if ((*trysec_context) != NULL)
            {
                free (trysec_context);
                *trysec_context = 0;
            }
        }
    }
    else  /* Make sure no one thinks we created a valid security context */
    {
        /* Must have a valid username */
        if ((*trysec_context) != NULL)
        {
            free (trysec_context);
            *trysec_context = 0;
        }
    }

    if (defuser != NULL)
    {
        free (defuser);
        defuser = 0;
    }
    if (defrole != NULL)
    {
        free (defrole);
        defrole = 0;
    }
    if (deftype != NULL)
    {
        free (deftype);
        deftype = 0;
    }
    if (deflevel != NULL)
    {
        free (deflevel);
        deflevel = 0;
    }
}


/* get_field - given fieldstr - the "name" of a field, deffield - the 
 *             default value of the field, query the user and set the
 *             new value of the field
 */
void get_field (const char* fieldstr, const char* deffield, char* newfield)
{
    int done = 0;  /* true if a non-empty field has been obtain */

    while (!done)  /* Keep going until we get a value for the field */
    {
        printf ("\tEnter %s ", fieldstr);
        if (deffield)
            if (strlen(deffield) > 0)
                printf (" (return for default) [%s] ", deffield);
        fflush (stdin);
        fgets (newfield, 100, stdin);
        fflush (stdin);
        newfield[strlen(newfield)-1] = '\0';

        if (strlen(newfield) == 0)  /* user selected default */
        {
            if (!deffield) /* Don't have a default */
            {
                printf ("Sorry default unavailable.  You must enter a %s\n",
                         fieldstr);
            }
            else
            {
                /* The user wants the default so copy that into newuser */
                strcpy (newfield, deffield);
            }
        }

        if (strlen(newfield) > 0)
            done = 1;
    }
}


/* security_context_to_parts - given a security context, parse it into
 *                             its individual fields - user, role, type
 *                             level
 *                             Returns 1 if able to successfully parse
 *                             the security context, 0 otherwise
 */
int security_context_to_parts (const security_context_t buf, char** user, 
                               char** role, char** type, char** level)
{
    int i, j;          /* Used as indices into buf - i points to colon
                          before current field, j points to colon after
                          current field                                 */
    int retval = 1;    /* The value to be returned from this function   */
    char* colon;       /* pointer to the "next" colon in buf            */

    /* Make sure buf points to something */
    if (!buf)
    {
        /* Invalid security context */
        retval = 0;
    }

    /* Make sure user, role, type, and level point to something */
    if (!user || !role || !type)
    {
        /* Bad output parameters */
        retval = 0;
    }

#ifdef CONFIG_FLASK_MLS
    if (!level)
    {
        /* Bad output parameter */
        retval = 0;
    }
#endif

    if (retval != 0)
    {
        /* Find the position of the first colon in buf */
        /* Set j so that buf[j] == ':'                 */
        i = 0;
        colon = index (buf, ':');
        j = colon - buf;
        if (j <= -1)
        {
            /* Invalid security context */
            retval = 0;
        }
    }

    if (retval != 0)
    {
        /* malloc space for the user field.  This will be the size of j+1 */
        (*user) = (char *) malloc (sizeof(char) * (j+1));
        if ((*user) == NULL)
        {
            fprintf (stderr, "Unable to malloc\n");
            retval = 0;
        }
    }

    if (retval != 0)
    {
        /* copy the appropriate part of buf into user and set the 
           terminating null */
        strncpy ((*user), buf, j);
        (*user)[j] = '\0';
  
        /* change i to index the colon at the end of user and set j to
           index the next colon in buf */
        i = j;
        colon = index (&buf[i+1], ':');
        j = colon - buf;
        if (j <= -1)
        {
            /* Invalid security context */
            retval = 0;
        }
    }

    if (retval != 0)
    {
        /* malloc space for the role field - size j-i (the space between
           the colons */
        (*role) = (char*) malloc (sizeof(char) * (j-i));
        if ((*role) == NULL)
        {
            fprintf (stderr, "Unable to malloc\n");
            retval = 0;
        }
    }
    
    if (retval != 0)
    {
        /* copy the appropriate part of buf into role and set the 
           terminating null */
        strncpy ((*role), &buf[i+1], j-i-1);
        (*role)[j-i-1] = '\0';
  
        /* change i to index the colon at the end of role and set j to
           index the next colon in buf */
        i = j;
#ifdef CONFIG_FLASK_MLS
        colon = index (&buf[i+1], ':');
        j = colon - buf;
#else
        j = strlen (buf);
#endif
        if (j <= -1)
        {
            /* Invalid security context */
            retval = 0;
        }
    }

    if (retval != 0)
    {
        /* malloc space for the type field - size j-i (the space between
           the colons */
        (*type) = (char*) malloc (sizeof(char) * (j-i));
        if ((*type) == NULL)
        {
            fprintf (stderr, "Unable to malloc\n");
            retval = 0;
        }
    }
    
    if (retval != 0)
    {
        /* copy the appropriate part of buf into type and set the 
           terminating null */
        strncpy ((*type), &buf[i+1], j-i-1);
        (*type)[j-i-1] = '\0';
  
#ifdef CONFIG_FLASK_MLS

        /* change i to index the colon at the end of role and set j to
           the length of buf (it will index the space after the end of
           buf */
        i = j;
        j = strlen (buf);
        if (j <= -1)
        {
            /* Invalid security context */
            retval = 0;
        }
#endif
    }

#ifdef CONFIG_FLASK_MLS
    if (retval != 0)
    {
        /* malloc space for the level field - size j-i (the space between
           the final colon and the end of buf */
        (*level) = (char*) malloc (sizeof(char) * (j-i));
        if ((*level) == NULL)
        {
            fprintf (stderr, "Unable to malloc\n");
            retval = 0;
        }
    }

    if (retval != 0)
    {
        /* copy the appropriate part of buf into level and set the 
           terminating null */
        strncpy ((*level), &buf[i+1], j-i-1);
        (*level)[j-i-1] = '\0';
    }
#endif

    return retval;
}


/* parts_to_security_context - given char strings for user, role, type,
 *                             and level, combine them to form a security
 *                             context of the form user:role:type:level
 *                             Store it in new
 *                             Return 1 if a new security context was
 *                             successfully formed, 0 otherwise
 */
int parts_to_security_context (const char* user, const char* role, 
                               const char* type,
                               const char* level, security_context_t* new)
{
    int userlen,   /* the length of the user field           */
        rolelen,   /* the length of the role field           */
        typelen,   /* the length of the type field           */ 
        levellen,  /* the length of the level field          */
        newlen;    /* the length of the new security context */

    /* Make sure parts actually point to something */
    if (!user || !role || !type)
    {
        /* Bad input */
        return 0;
    }

    /* Get the lengths of the fields */
    userlen = strlen (user);
    rolelen = strlen (role);
    typelen = strlen (type);

#ifdef CONFIG_FLASK_MLS
    if (!level)
    {
        /* Bad input */
        return 0;
    }
    levellen = strlen (level);
    newlen = userlen + rolelen + typelen + levellen + 4;
#else
    newlen = userlen + rolelen + typelen + 3;
#endif

    /* Malloc space for the new security context */
    (*new) = (security_context_t)malloc (sizeof(char) * newlen);
    if ((*new) == NULL)
    {
        fprintf (stderr, "Unable to malloc\n");
        return 0;
    }

    /* copy the parts into the new security context with :'s separating them */
    strncpy ((*new), user, userlen);
    (*new)[userlen] = ':';
    strncpy (&(*new)[userlen+1], role, rolelen);
    (*new)[userlen+rolelen+1] = ':';
    strncpy (&(*new)[userlen+rolelen+2], type, typelen);

#ifdef CONFIG_FLASK_MLS
    (*new)[userlen+rolelen+typelen+2] = ':';
    strncpy (&(*new)[userlen+rolelen+typelen+3], level, levellen);
    (*new)[userlen+rolelen+typelen+levellen+3] = '\0';
#else
    (*new)[userlen+rolelen+typelen+2] = '\0';
#endif

    /* If we've made it this far, we're successful */
    return 1;
}

