/* chsidfs.c
 *
 * used to change the SID of a file system using the fs_sid parameter
 * and/or the SID to use for unlabeled files using the f_sid parameter
 * with -- sid option, expects fs_sid and f_sid to be SIDs
 * with -- context option, fs_sid and f_sid should be security contexts
 */

#include <errno.h>
#include <fs_secure.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <sys/types.h>
 
void usage(char* program_name)
{
  printf("usage:  %s --sid FS_SID F_SID PATHNAME(s) \n", program_name);
  printf("or:     %s --context FS_CONTEXT F_CONTEXT PATHNAME(s)\n", program_name);
}
 
int main(int argc, char **argv) 
{
 
  security_id_t fs_sid = -1;   /* SID of file system */
  security_id_t f_sid = -1;    /* SID to use for unlabeled files */
  char *scontext = NULL;       /* security context */
  int ctxtlen = 256;           /* max allowed length for scontext */
  int rv;
  int optc;
  char *ep;
  int sidflag = -1;  /* 1 if --sid option, 0 if --context option */
 
  static struct option const longopts[] =
  {
    {"sid",     required_argument, NULL, 's'},
    {"context", required_argument, NULL, 'c'},
    {NULL, 0, NULL, 0}
  };
 
 
 
  while ((optc = getopt_long (argc, argv, "pm:s:c:", longopts, NULL)) != -1)
    {
      switch (optc)
        {
        case 's':
          sidflag = 1;
          if ( ((int) fs_sid > 0) || (scontext != NULL) ) {
            (void) fprintf(stderr, "%s: --sid (-s) and --context (-c) are ", argv[0]);
            (void) fprintf(stderr, "mutually exclusive\n");
            exit( 1 );
          }
            /* check for typos */
            fs_sid = (security_id_t) strtol(optarg, &ep, 10);
            if ( *ep ) {
              (void) fprintf(stderr, "%s: non-numeric SID '%s'\n", argv[0], optarg);
              exit( 1 );
            }
          /* do sanity check, save result on success */
          scontext = calloc(1, ctxtlen+1);
          if ( scontext != NULL ) {
            if ( security_sid_to_context(fs_sid, scontext, &ctxtlen) ) {
              if ( errno != ENOSPC ) {
                (void) fprintf(stderr, "%s: security_sid_to_context(%d): '%s'\n", argv[0],
                               (int) fs_sid, strerror(errno));
                exit( 1 );
              }
              free(scontext);
              scontext = calloc(1, ctxtlen+1);
              /* nonfatal, so if there's an error we punt */
              if ( scontext != NULL )
                if ( security_sid_to_context(fs_sid, scontext, &ctxtlen) ) {
                  (void) fprintf(stderr, "%s: security_sid_to_context(%d): %s\n", argv[0],
                                 (int) fs_sid, strerror(errno));
                  exit( 1 );
                }
            }
          }
          break;
        case 'c':
          sidflag = 0;
          if ( ((int) fs_sid >= 0) || (scontext != NULL) ) {
            (void) fprintf(stderr, "%s: --context (-c) and --sid (-s) are mutually", argv[0]);
            (void) fprintf(stderr, " exclusive\n");
            exit( 1 );
          }
          scontext = optarg;
          /* sanity check */
          rv = security_context_to_sid(scontext, strlen(scontext)+1, &fs_sid);
          if ( rv ) {
            (void) fprintf(stderr, "%s: security_context_to_sid(%s): %s\n", argv[0],
                           scontext, strerror(errno));
            exit( 1 );
          }
          break;
        default:
          usage(argv[0]);
          exit(1);
    }
  }
 
  if ((optind == argc) || ((int)fs_sid < 0))
    {
      usage(argv[0]);
      exit(1);
    }
 
  /* Need to get the second sid or context */
  if (sidflag == 0)
  {
    rv = security_context_to_sid (argv[optind], strlen(argv[optind])+1, &f_sid);
    if (rv) {
      (void) fprintf(stderr, "%s: security_context_t_sid(%s): %s\n", argv[0],
                     argv[optind], strerror(errno));
      exit(1);
    }
  }
  else
  {
    f_sid = (security_id_t) strtol (argv[optind], &ep, 10);
    if ( *ep ) {
      (void) fprintf(stderr, "%s: non-numeric SID '%s'\n", argv[0], optarg);
      exit( 1 );
    }
  }
  
  optind++;

  for (; optind < argc; ++optind)
    {
      if ( (int) fs_sid > 0 ) {
        if (chsidfs(argv[optind], fs_sid, f_sid) < 0) {
          perror("chsidfs");
          exit(2);
        }
      }
      else
        {
          usage(argv[0]);
          exit(1);
        }
    }
 
  exit(0);
}

