/*
 * semget.c
 *
 * Author: Howard Holm (NSA), <hdholm@epoch.ncsc.mil>
 * 
 * Create a semaphore set (possibly at a different context.)  If a set already
 * exists with the (optional) key then return it's identifier.
 *
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <linux/flask/flask.h>
#include <sys/sem.h>
#include <ipc_secure.h>

/* print the usage information for this program */
void usage(char *progname) 
{
  fprintf(stderr, "usage: %s [-k key] [-c context | -s sid] [-n num_semaphores]\n", progname);
  exit(1);
}


int main(int argc, char **argv)
{
  int have_sid = 0;
  int nsems = 1;
  int valid_ipc_key = 0;
  key_t ipc_key;
  int ret;
  security_id_t sid;
  char c;
  int id;

  while ( (c = getopt(argc, argv, "k:c:s:n:")) != EOF) {
    switch (c) {
    case 'k':
      ipc_key = atoi(optarg);
      valid_ipc_key = 1;
      break;
    case 'c':
      if (have_sid) {
	fprintf (stderr, "May specify only a sid (-s) or context (-c) not both\n");
	usage (argv[0]);
      } else {
	have_sid = 1;
	ret = security_context_to_sid(optarg,strlen(optarg)+1,&sid);
	if (ret) {
	  perror("Invalid context (-c) given");
	  exit(ret);
	}
      }
      break;
    case 's':
      if (have_sid) {
	fprintf (stderr, "May specify only a sid (-s) or context (-c) not both\n");
	usage (argv[0]);
      } else {
	have_sid = 1;
	sid = atoi(optarg);
      }
      break;
    case 'n':
      nsems = atoi(optarg);
      break;
    default:
      usage(argv[0]);
    }
  }

  if (have_sid) {
    if (!valid_ipc_key)
      ipc_key = IPC_PRIVATE;
    if ((id = semget_secure (ipc_key, nsems, IPC_CREAT|0777, sid)) == -1){
      perror ("semget_secure failed");
      exit(1);
    } else
      fprintf (stderr, "%s:semget_secure succeeded key = %d, id = %d\n", argv[0], ipc_key, id);
  } else {
    if (!valid_ipc_key)
      ipc_key = IPC_PRIVATE;
    if ((id = semget (ipc_key, nsems, IPC_CREAT|0777)) == -1){
      perror ("semget failed");
      exit (1);
    } else
      fprintf (stderr, "%s:semget succeeded key = %d, id = %d\n", argv[0], ipc_key, id);
  }
}
