/*
 * semread.c
 *
 * Author: Howard Holm (NSA), <hdholm@epoch.ncsc.mil>
 * 
 * Read a semaphore set with the given id
 *
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/sem.h>
#include <ipc_secure.h>

#if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
  /* union semun is defined by including <sys/sem.h> */
#else
  /* according to X/OPEN we have to define it ourselves */
  union semun {
    int val;                    /* value for SETVAL */
    struct semid_ds *buf;       /* buffer for IPC_STAT, IPC_SET */
    unsigned short int *array;  /* array for GETALL, SETALL */
    struct seminfo *__buf;      /* buffer for IPC_INFO */
  };
#endif

#define MAXSEMSTESTED 20

/* print the usage information for this program */
void usage(char *progname) 
{
  fprintf(stderr, "usage:  %s id semaphore_numbers\n", progname);
  exit(1);
}

int main(int argc, char **argv)
{
  int id, semid;
  int i, rtn;
  union semun result;
  int nsems = 0;
  
  if (argc < 3) {
    fprintf (stderr, "Semaphore set id and semaphore numbers are requried.\n");
    usage(argv[0]);
  }
  id = atoi (argv[1]);
  if (argc > MAXSEMSTESTED + 2) {
    fprintf (stderr, "Only %d semaphores may be used.\n", MAXSEMSTESTED);
    usage(argv[0]);
    exit(1);
  }

  /* Read attributes for a specific semaphore within a set */
  for ( i=2; i < argc; i++) {
    semid = atoi (argv[i]);

    if ((rtn = semctl (id, semid, GETPID, &result)) < 0)
      perror ("semctl(GETPID) failed");
    else
      fprintf (stderr, "PID of last process to alter sem %d: %d\n", semid, rtn);
  
    if ((rtn = semctl (id, semid, GETNCNT, &result)) < 0)
      perror ("semctl(GETNCNT) failed");
    else
      fprintf (stderr, "semctl(GETNCNT) number of processes waiting on sem %d: %d\n", semid, rtn);
  
    if ((rtn = semctl (id, semid, GETZCNT, &result)) < 0)
      perror ("semctl(GETZCNT) failed");
    else
      fprintf (stderr, "semctl(GETZCNT) number of processes waiting for zero on sem %d: %d\n",
	       semid, rtn);
  }

  if ((result.array = (unsigned short int *) malloc (sizeof (int) * MAXSEMSTESTED)) == NULL) {
    perror ("malloc failed");
    exit (1);
  }
  if (semctl (id, 0, GETALL, result) < 0)
    perror ("semctl(GETALL) failed");
  else 
    for (i = 0, fprintf (stderr, "Semaphores now:"); i < MAXSEMSTESTED; i++) {
      fprintf (stderr, " %2d", result.array[i]);
    }
  fprintf (stderr, "\n");
  free (result.array);

}


