/*
 * unix_stream_server
 * 
 * Create a listening socket on a specified port, 
 * accept connections and echo a message.
 *
 * Permit testing of socket_secure, listen_secure, 
 * accept_secure, getsockname_secure and getpeername_secure.
 * 
 * Calls socket_secure if '-s socket_context' is specified.
 * Calls listen_secure if '-u' or '-n newconn_context' is specified.
 * Uses the abstract name space if '-a' is specified.
 * Receives a descriptor if '-f' is specified.
 *
 * Usage:  unix_stream_server [-f] [-a] [-s socket_context] [-u] [-n newconn_context] pathname
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/un.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>
#include <stdio.h>

#include <socket_secure.h>
#include <ss.h>

int abstract = 0;
char my_path[1024];
#define CLEANUP_AND_EXIT do { if (!abstract) unlink(my_path); exit(1); } while (0)

void handler(int sig)
{
	printf("Received signal %d, exiting...\n", sig);
	CLEANUP_AND_EXIT;
}

void usage(char *progname) 
{
	printf("usage:  %s [-f] [-a] [-s socket_context] [-u] [-n newconn_context] pathname\n", progname);
	exit(1);
}

int main(int argc, char **argv)
{
	struct sockaddr_un sun;
	char buf[1024], context[255];
	int i, s, sunlen, ret, buflen, ctrl, c, useclient = 0, recvfd = 0, contextlen;
	security_id_t so_sid = 0, newconn_sid = 0, peer_sid = 0, tmp_sid = 0; 
	struct msghdr msg = { 0 };
	struct iovec iov;
	struct cmsghdr *cmsg;
	char cmsgbuf[CMSG_SPACE(sizeof(int))];

	while ( (c = getopt(argc, argv, "s:n:uaf")) != EOF) {
		switch (c) {
		case 's':
		case 'n':
			ret = security_context_to_sid(optarg,strlen(optarg)+1,&tmp_sid);
			if (ret) {
				perror(optarg);
				exit(ret);
			}
			if (c == 's')
				so_sid = tmp_sid;
			else if (c == 'n')
				newconn_sid = tmp_sid;
			break;
		case 'u':
			useclient = 1;
			break;
		case 'a':
			abstract = 1;
			break;
		case 'f':
			recvfd = 1;
			break;
		default:
			usage(argv[0]);
		}
	}

	if (optind != (argc - 1)) {
		usage(argv[0]);
	}

	for (i = 0; i < 32; i++) {
		signal(i, handler);
	}

	/* Create the listening socket */
	if (so_sid) 
		s = socket_secure(AF_UNIX, SOCK_STREAM, 0, so_sid);
	else 
		s = socket(AF_UNIX, SOCK_STREAM, 0);
	if (s == -1) {
		perror("socket");
		exit(1);
	}

	/* Obtain the security context of the listening socket */
	sun.sun_family = AF_UNIX;
	sunlen = sizeof(struct sockaddr_un);
	ret = getsockname_secure(s, &sun, &sunlen, &tmp_sid);
	if (ret) {
		perror("getsockname_secure");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(tmp_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}
	
	printf("Created socket with context %s\n", context);

	/* Bind to the specified name */
	sun.sun_family = AF_UNIX;
	if (abstract) {
		sun.sun_path[0] = 0;
		strcpy(sun.sun_path+1, argv[optind++]);
		sunlen = strlen(sun.sun_path+1) + 1 + sizeof(short);
	} else {
		strcpy(sun.sun_path, argv[optind++]);
		sunlen = strlen(sun.sun_path) + 1 + sizeof(short);
		strcpy(my_path, sun.sun_path);
	}
	ret = bind(s, &sun, sunlen);
	if (ret == -1) {
		perror("bind");
		exit(1);
	}

	/* Set state of socket to accept connections */
	if (newconn_sid || useclient) 
		ret = listen_secure(s, 5, newconn_sid, useclient);
	else
		ret = listen(s, 5);
	if (ret == -1) {
		perror("listen");
		CLEANUP_AND_EXIT;
	}
	
	while (1) {
		/* Accept a new connection and obtain the peer socket context */
		sunlen = sizeof(struct sockaddr_un);
		ctrl = accept_secure(s, (struct sockaddr*)&sun, &sunlen, &peer_sid);
		if (ctrl == -1) {
			perror("accept_secure");
			CLEANUP_AND_EXIT;
		}

		contextlen = sizeof(context);
		ret = security_sid_to_context(peer_sid, context, &contextlen);
		if (ret) {
			perror("security_sid_to_context");
			CLEANUP_AND_EXIT;
		}

		if (sun.sun_path[0] == 0) {
			sun.sun_path[0] = '@';
			sun.sun_path[sunlen - sizeof(short)] = 0;
		}
	
		printf("Accepted connection from peer (%s) with context %s\n", 
		       sun.sun_path, context);

		/* Obtain the new connection socket security context */
		sunlen = sizeof(struct sockaddr_un);
		ret = getsockname_secure(ctrl, &sun, &sunlen, &tmp_sid);
		if (ret) {
			perror("getsockname_secure");
			CLEANUP_AND_EXIT;
		}

		contextlen = sizeof(context);
		ret = security_sid_to_context(tmp_sid, context, &contextlen);
		if (ret) {
			perror("security_sid_to_context");
			CLEANUP_AND_EXIT;
		}

		printf("New socket created by connection has context %s\n", 
		       context);

		/* Verify that getpeername_secure is consistent with accept_secure */
		sunlen = sizeof(struct sockaddr_un);
		ret = getpeername_secure(ctrl, (struct sockaddr*)&sun, &sunlen, &tmp_sid);
		if (ret) {
			perror("getpeername_secure");
			CLEANUP_AND_EXIT;
		}

		if (tmp_sid != peer_sid) {
			printf("Getpeername_secure returned a different SID than accept_secure!\n");
			CLEANUP_AND_EXIT;
		}

		/* Receive and echo a message */
		buf[0] = 0;
		msg.msg_controllen = 0;
		if (recvfd) {
			iov.iov_base = buf;
			iov.iov_len = sizeof(buf);
			msg.msg_name = 0;
			msg.msg_namelen = 0;
			msg.msg_iov = &iov;
			msg.msg_iovlen = 1;
			msg.msg_control = cmsgbuf;
			msg.msg_controllen = sizeof cmsgbuf;
			ret = recvmsg(ctrl, &msg, 0);
		} else {
			ret = recv(ctrl, buf, sizeof(buf), 0);
		}
		if (ret == -1) {
			perror("recv");
			CLEANUP_AND_EXIT;
		}		    

		printf("Received message ##%s##", buf);
		if (msg.msg_controllen)
			printf(" and a descriptor");
		printf("\n");

		ret = send(ctrl, buf, sizeof(buf), 0);
		if (ret == -1) {
			perror("send");
			CLEANUP_AND_EXIT;
		}

		/* Close the connection */
		close(ctrl);
	}
}
